﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../eth_Gmii.h"
#include "../eth_Packet.h"
#include "../eth_Worker.h"
#include "../eth_SocketIf.h"
#include "../eth_Allocator.h"

#include "eth_UsbDevice.h"

namespace nn     {
namespace eth    {
namespace device {
namespace usb    {

enum
{
    ASIX_88772_VID = 0x0b95,
    ASIX_88772_PID = 0x7720,
};

class UsbDevice;

class Asix88772 :
    public  Allocator,
    private Gmii,
    private SocketIf
{

private:

    enum
    {
        BulkOutEp     = 0,
        BulkInEp      = 1,
        InterruptInEp = 2,
        EndpointCount
    };

    static const size_t PacketSizeIn         = (16 * 1024);
    static const size_t PacketSizeOut        = (16 * 1024);
    static const size_t PacketSizeInterrupt  = (4 * 1024);
    static const int    PollRateMs           = 128;
    static const uint32_t AsyncPacketCount   = 4;

    int                 m_PhyAddress;
    uint8_t             m_LinkState;
    bool                m_Running;
    nn::os::SystemEvent m_DataEvent;
    Worker              m_BulkInThread;
    Worker              m_BulkOutThread;
    Worker              m_InterruptThread;
    UsbDevice*          m_pUsbDevice;


    static void DriverSleepMs(uint32_t timeoutMilliSeconds)
    NN_NOEXCEPT
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(timeoutMilliSeconds));
    }

    //
    // Register/memory read/writes
    //
    nn::Result AxeCommand(int command, int index, int value, void* pData) NN_NOEXCEPT;
    nn::Result PhyReadReg(uint16_t miiRegister, uint16_t* miiValue) NN_NOEXCEPT;
    nn::Result PhyWriteReg(uint16_t miiRegister, uint16_t miiValue) NN_NOEXCEPT;

    //
    // Phy access functions (overrides for Gmii)
    //
    nn::Result PhyReset() NN_NOEXCEPT;
    nn::Result PhyResetLink() NN_NOEXCEPT;
    nn::Result PhyUpdateMtu(int mtu) NN_NOEXCEPT;
    nn::Result PhyGetMedia(MediaType* pMediaType) NN_NOEXCEPT;
    nn::Result PhySetMedia(MediaType type) NN_NOEXCEPT;

    //
    // Usb data transfers
    //
    void WriteThread() NN_NOEXCEPT;
    void ReadThread() NN_NOEXCEPT;
    void InterruptThread() NN_NOEXCEPT;
    void GetFramesFromStack(Packet* pPacket) NN_NOEXCEPT;
    void SendFramesToStack(Packet* pPacket) NN_NOEXCEPT;
    void LinkInterrupt(Packet* pPacket) NN_NOEXCEPT;

    //
    // Socket related (overrides for SocketIf)
    //
    void Start() NN_NOEXCEPT;
    nn::Result UpdateMultiFilter(struct ifnet *pIfnet) NN_NOEXCEPT;

public:
    // Usb endpoint description
    static const EndPointDescriptor EndPoints[EndpointCount];
    static const int NumberOfEndPoints;

    explicit Asix88772(UsbDevice* usb) NN_NOEXCEPT;
    ~Asix88772() NN_NOEXCEPT;
    nn::Result Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

};

}}}}
