﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/eth/eth_EthTypes.h>
#include <net/if_dl.h>

#include "../mii_Registers.h"
#include "eth_Asix88772.reg.h"
#include "eth_Asix88772.h"

#include <siglo/device.h>

namespace nn     {
namespace eth    {
namespace device {
namespace usb    {

#define RETURN_IF_FAILURE(f)    \
do                              \
{                               \
    nn::Result result = (f);    \
    if (result.IsFailure())     \
    {                           \
        return result;          \
    }                           \
}                               \
while(NN_STATIC_CONDITION(0))

const int Asix88772::NumberOfEndPoints = EndpointCount;

const EndPointDescriptor Asix88772::EndPoints[EndpointCount]
{
    [BulkOutEp] =
    {
        .type      = nn::usb::UsbEndpointType_Bulk,
        .direction = nn::usb::UsbEndpointDirection_ToDevice,
    },
    [BulkInEp] =
    {
        .type      = nn::usb::UsbEndpointType_Bulk,
        .direction = nn::usb::UsbEndpointDirection_ToHost,
    },
    [InterruptInEp] =
    {
        .type      = nn::usb::UsbEndpointType_Int,
        .direction = nn::usb::UsbEndpointDirection_ToHost,
    },
};

nn::Result Asix88772::AxeCommand(int command, int index, int value, void* pData)
NN_NOEXCEPT
{
    size_t lengthOut;

    return AXE_CMD_IS_WRITE(command) ?
           m_pUsbDevice->SubmitControlOutRequest(
                            &lengthOut, pData, AXE_CMD_CMD(command),
                            UT_WRITE_VENDOR_DEVICE, value, index, AXE_CMD_LEN(command)
           ):
           m_pUsbDevice->SubmitControlInRequest(
                            &lengthOut, pData, AXE_CMD_CMD(command),
                            UT_READ_VENDOR_DEVICE,  value, index, AXE_CMD_LEN(command)
           );
}

nn::Result Asix88772::PhyReadReg(uint16_t miiRegister, uint16_t* pMiiValue)
NN_NOEXCEPT
{
    uint16_t value;

    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_MII_OPMODE_SW, 0, 0, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_MII_READ_REG, miiRegister, m_PhyAddress, &value));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_MII_OPMODE_HW, 0, 0, nullptr));

    /*
     * BMSR of AX88772 indicates that it supports extended
     * capability but the extended status register is
     * reserved for embedded ethernet PHY. So clear the
     * extended capability bit of BMSR.
     */
    if (miiRegister == MII_BMSR)
    {
        value &= ~BMSR_EXTCAP;
    }

    *pMiiValue = value;

    return ResultSuccess();
}

nn::Result Asix88772::PhyWriteReg(uint16_t miiRegister, uint16_t miiValue)
NN_NOEXCEPT
{
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_MII_OPMODE_SW, 0, 0, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_MII_WRITE_REG, miiRegister, m_PhyAddress, &miiValue));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_MII_OPMODE_HW, 0, 0, nullptr));
    return ResultSuccess();
}

nn::Result Asix88772::PhyReset()
NN_NOEXCEPT
{
    uint16_t temp;

    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_READ_PHYID, 0, 0, &temp));
    m_PhyAddress = (temp>>8);

    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_SW_PHY_SELECT, 0, AXE_PHY_SEL_PRI, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_SW_RESET_REG,  0, AXE_SW_RESET_IPPD, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_SW_RESET_REG,  0, 0, nullptr));
    DriverSleepMs(100);

    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_SW_RESET_REG, 0, AXE_SW_RESET_IPRL | AXE_SW_RESET_PRL, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_SW_RESET_REG, 0, AXE_SW_RESET_PRL, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_SW_RESET_REG, 0, AXE_SW_RESET_IPRL | AXE_SW_RESET_PRL, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_RXCTL_WRITE,  0, AXE_RXCMD_ENABLE, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_READ_NODEID,  0, 0, Gmii::m_PhyMacAddress));

    return ResultSuccess();
}

nn::Result Asix88772::PhySetMedia(MediaType type)
NN_NOEXCEPT
{
    uint16_t mediaValue = AXE_MEDIA_MAGIC | AXE_MEDIA_ENCK | AXE_MEDIA_SUPERMAC;
    uint16_t rxctlValue = AXE_RXCMD_MULTICAST | AXE_RXCMD_BROADCAST | AXE_RXCMD_ENABLE | AXE_RXCMD_MFB_16384;

    if (EthLinkUp(type))
    {
        mediaValue |= AXE_MEDIA_RX_EN;
    }

    if (type == MediaType_AUTO)
    {
        mediaValue |= AXE_MEDIA_100TX | AXE_MEDIA_FULL_DUPLEX | AXE_MEDIA_TXFLOW_CONTROL_EN | AXE_MEDIA_RXFLOW_CONTROL_EN;
    }
    else
    {
        if (EthFullDuplex(type))
        {
            mediaValue |= AXE_MEDIA_FULL_DUPLEX;
        }

        if (EthFlow(type))
        {
            mediaValue |= AXE_MEDIA_TXFLOW_CONTROL_EN | AXE_MEDIA_RXFLOW_CONTROL_EN;
        }

        switch (EthSpeed(type))
        {
        case 100:
            mediaValue |= AXE_MEDIA_100TX;
            break;
        default:
            break;
        }
    }

    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_WRITE_MEDIA, 0, mediaValue, nullptr));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_RXCTL_WRITE, 0, rxctlValue, nullptr));

    return ResultSuccess();
}

nn::Result Asix88772::PhyGetMedia(MediaType* pType)
NN_NOEXCEPT
{
    uint16_t bmcr;
    uint16_t anlpar;
    uint16_t bmsr;

    RETURN_IF_FAILURE(PhyReadReg(MII_BMCR, &bmcr));
    RETURN_IF_FAILURE(PhyReadReg(MII_ANLPAR, &anlpar));
    RETURN_IF_FAILURE(PhyReadReg(MII_BMSR, &bmsr));

    *pType = (MediaType)EthMakeMediaType(
                (bmcr & BMCR_SPEED0         ? Type_100_TX  : Type_10_T),
                (bmcr & BMCR_AUTOEN         ? SubType_AUTO : SubType_MANUAL),
                (bmcr & BMCR_FDX            ? Option_FDX   : Option_HDX) |
                (anlpar & ANLPAR_PAUSE_MASK ? Option_FLOW  : 0),
                (bmsr & BMSR_LINK           ? LinkState_Up : LinkState_Down));

    return ResultSuccess();
}


void Asix88772::LinkInterrupt(Packet* packet)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(packet->m_ValidLength == sizeof(LinkState));

    LinkState* state = reinterpret_cast<LinkState*>(packet->m_pData);

    if ((m_LinkState ^ state->link) & AX_INT_PPLS_LINK)
    {
        m_LinkState = state->link;

        if (m_LinkState & AX_INT_PPLS_LINK)
        {
            SocketIf::SetLinkState(true);
            Gmii::UpdateLinkStatus();
        }
        else
        {
            Gmii::UpdateLinkStatus();
            SocketIf::SetLinkState(false);
        }
    }

    Tick();
}

void Asix88772::GetFramesFromStack(Packet* pPacket)
NN_NOEXCEPT
{
    PacketHeader* pHeader;
    uint8_t* buffer    = pPacket->m_pData;
    int      remain    = static_cast<int>(pPacket->m_TotalLength);
    int      frameSize = SocketIf::GetMtu() + sizeof(PacketHeader);
    size_t   length;

    while (remain >= frameSize)
    {
        pHeader = reinterpret_cast<PacketHeader*>(buffer);

        if (SocketIf::GetFrameFromStack(
                &length,
                buffer + sizeof(PacketHeader),
                remain - sizeof(PacketHeader)
                ).IsFailure() || length == 0)
        {
            break;
        }

        pHeader->packetLength  =  length;
        pHeader->ipacketLength = ~length;

        length  = (length + 1) & ~0x1;
        buffer += length + sizeof(PacketHeader);
        remain -= length + sizeof(PacketHeader);

        if (remain < static_cast<int>(sizeof(PacketHeader)))
        {
            remain = 0;
            break;
        }

        if (!((pPacket->m_TotalLength - remain) % 512))
        {
            pHeader = reinterpret_cast<PacketHeader*>(buffer);
            pHeader->packetLength  = 0;
            pHeader->ipacketLength = 0xffff;
            buffer += sizeof(PacketHeader);
            remain -= sizeof(PacketHeader);
        }
    }

    if (remain < 0)
    {
        remain = 0;
    }

    pPacket->m_ValidLength = pPacket->m_TotalLength - remain;
}

void Asix88772::SendFramesToStack(Packet* pPacket)
NN_NOEXCEPT
{
    uint8_t*      buffer = pPacket->m_pData;
    int           remain = static_cast<int>(pPacket->m_ValidLength);
    PacketHeader* pHeader;

    while (remain > static_cast<int>(sizeof(PacketHeader)))
    {
        pHeader = reinterpret_cast<PacketHeader*>(buffer);
        buffer += sizeof(PacketHeader);
        remain -= sizeof(PacketHeader);

        if ((pHeader->packetLength ^ pHeader->ipacketLength) != 0xffff)
        {
            break;
        }

        if (remain < pHeader->packetLength)
        {
            break;
        }

        if (SocketIf::SendFrameToStack(buffer, pHeader->packetLength).IsFailure())
        {
            break;
        }

        buffer += pHeader->packetLength;
        remain -= pHeader->packetLength;
    }
}

void Asix88772::ReadThread()
NN_NOEXCEPT
{
    Packet     packets[AsyncPacketCount];
    nn::Result result;

    for (uint32_t i = 0; i < sizeof(packets) / sizeof(packets[0]); i++)
    {
        result = packets[i].Initialize(PacketSizeIn);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    m_pUsbDevice->RunAsyncInOperation(
                    packets,
                    sizeof(packets) / sizeof(packets[0]),
                    std::bind(&Asix88772::SendFramesToStack, this, std::placeholders::_1),
                    BulkInEp);

    for (uint32_t i = 0; i < sizeof(packets) / sizeof(packets[0]); i++)
    {
        packets[i].Finalize();
    }
}

void Asix88772::WriteThread()
NN_NOEXCEPT
{
    nn::Result result;
    Packet     packet;

    result = packet.Initialize(PacketSizeOut);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    while (m_Running)
    {
        GetFramesFromStack(&packet);

        if (packet.m_ValidLength == 0)
        {
            m_DataEvent.Wait();
            continue;
        }

        if (m_pUsbDevice->SubmitOutRequest(
                &packet.m_ValidLength,
                BulkOutEp,
                packet.m_pData,
                packet.m_ValidLength).IsFailure())
        {
            break;
        }
    }

    packet.Finalize();
}

void Asix88772::InterruptThread()
NN_NOEXCEPT
{
    nn::Result result;
    Packet     packet;
    nn::os::SystemEventType* event;

    result = packet.Initialize(PacketSizeInterrupt);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    event = m_pUsbDevice->GetTerminateEventPointer();

    while (m_Running)
    {
        if (nn::os::TimedWaitSystemEvent(event, nn::TimeSpan::FromMilliSeconds(PollRateMs)))
        {
            break;
        }

        if ((result = m_pUsbDevice->SubmitInRequest(
                &packet.m_ValidLength,
                InterruptInEp,
                packet.m_pData,
                sizeof(LinkState))).IsFailure())
        {
            NN_SDK_LOG("[usb eth]: terminating (%d:%d)\n", result.GetModule(), result.GetDescription());
            break;
        }

        if (packet.m_ValidLength > 0)
        {
            LinkInterrupt(&packet);
        }
    }

    packet.Finalize();
}

void Asix88772::Start()
NN_NOEXCEPT
{
    m_DataEvent.Signal();
}

nn::Result Asix88772::UpdateMultiFilter(struct ifnet *pIfnet)
NN_NOEXCEPT
{
    struct ifmultiaddr *ifma;
    uint32_t h = 0;
    uint16_t rxmode;
    uint8_t hashtbl[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_RXCTL_READ, 0, 0, &rxmode));

    if (pIfnet->if_flags & (IFF_ALLMULTI | IFF_PROMISC))
    {
        rxmode |= AXE_RXCMD_ALLMULTI;
        RETURN_IF_FAILURE(AxeCommand(AXE_CMD_RXCTL_WRITE, 0, rxmode, NULL));
        return ResultSuccess();
    }
    rxmode &= ~AXE_RXCMD_ALLMULTI;

    TAILQ_FOREACH(ifma, &pIfnet->if_multiaddrs, ifma_link)
    {
        if (ifma->ifma_addr->sa_family != AF_LINK)
        {
            continue;
        }

        h = ether_crc32_be(
                reinterpret_cast<uint8_t *>(LLADDR(reinterpret_cast<struct sockaddr_dl *>(ifma->ifma_addr))),
                ETHER_ADDR_LEN) >> 26;
        hashtbl[h / 8] |= 1 << (h % 8);
    }

    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_WRITE_MCAST, 0, 0, (void *)&hashtbl));
    RETURN_IF_FAILURE(AxeCommand(AXE_CMD_RXCTL_WRITE, 0, rxmode, NULL));

    return ResultSuccess();
}

nn::Result Asix88772::Initialize()
NN_NOEXCEPT
{
    nn::Result result;

    if ((result = Gmii::Initialize()).IsFailure())
    {
        return result;
    }

    if ((SocketIf::Initialize(
            Gmii::m_Name,
            Gmii::m_Instance,
            Gmii::m_PhyMacAddress)).IsFailure())
    {
        Gmii::Finalize();
        return result;
    }

    m_Running = true;

    m_BulkInThread.Initialize(
            std::bind(&Asix88772::ReadThread, this),
            NN_SYSTEM_THREAD_PRIORITY(socket, EthInterfaceIngress),
            NN_SYSTEM_THREAD_NAME(socket, EthInterfaceIngress));

    m_BulkOutThread.Initialize(
            std::bind(&Asix88772::WriteThread, this),
            NN_SYSTEM_THREAD_PRIORITY(socket, EthInterfaceEgress),
            NN_SYSTEM_THREAD_NAME(socket, EthInterfaceEgress));

    m_InterruptThread.Initialize(
            std::bind(&Asix88772::InterruptThread, this),
            NN_SYSTEM_THREAD_PRIORITY(socket, EthEvents),
            NN_SYSTEM_THREAD_NAME(socket, EthEvents));

    return nn::ResultSuccess();
}

void Asix88772::Finalize()
NN_NOEXCEPT
{
    m_Running = false;
    m_pUsbDevice->CancelTransfers();
    m_DataEvent.Signal();
    m_BulkInThread.Finalize();
    m_BulkOutThread.Finalize();
    m_InterruptThread.Finalize();
    SocketIf::Finalize();
    Gmii::Finalize();
}

Asix88772::Asix88772(UsbDevice* usb)
NN_NOEXCEPT :
    Gmii(UsbInterfaceName, Asix88772::PollRateMs),
    m_LinkState(0),
    m_DataEvent(nn::os::EventClearMode_AutoClear, true),
    m_pUsbDevice(usb)
{

}

Asix88772::~Asix88772()
NN_NOEXCEPT
{

}

}}}}
