﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>

#include "../../../util/eth_Util.h"
#include "eth_Pcv.h"
#include "eth_Pcv.tegra-x2.reg.h"

namespace nn  {
namespace eth {
namespace device {
namespace tx2    {
namespace pcv {

namespace {

const auto SysramPollInterval = nn::TimeSpan::FromMilliSeconds(1);
const auto SysramPollTimeout = nn::TimeSpan::FromSeconds(1);

uintptr_t g_pGpio;      // PIO_CTL0_GPIO5
uintptr_t g_pSysRam;    // SYSRAM
uintptr_t g_pHsp;       // HSP

NN_FORCEINLINE MrqClockCommand MakeMrqClockCommand(MrqClockCommandType type, MrqClockModule module)
{
    return
        (type << MrqClockCommand_OffsetType) |
        (module << MrqClockCommand_OffsetModule);
}

NN_FORCEINLINE MrqClockModule GetMrqClockModule(Module moduleId)
{
    switch (moduleId)
    {
    case Module_EqosClockMasterBus:
        return MrqClockModule_EqosMasterBus;
        break;
    case Module_EqosClockSlaveBus:
        return MrqClockModule_EqosSlaveBus;
        break;
    case Module_EqosClockRx:
        return MrqClockModule_EqosRx;
        break;
    case Module_EqosClockTx:
        return MrqClockModule_EqosTx;
        break;
    case Module_EqosClockPtpRef:
        return MrqClockModule_EqosPtpRef;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void CheckBpmpRxEmpty() NN_NOEXCEPT
{
    const uint32_t writeCount = util::Read32(g_pSysRam, SysramOffset_BpmpRxWriteCount);
    const uint32_t readCount = util::Read32(g_pSysRam, SysramOffset_BpmpRxReadCount);
    NN_ABORT_UNLESS(writeCount == readCount);
}

void CheckBpmpRxSuccess() NN_NOEXCEPT
{
    MrqResponseHeader header;
    util::CopyFromIo(&header, g_pSysRam, SysramOffset_BpmpRxResponse, sizeof(header));
    NN_ABORT_UNLESS(header.error >= 0);
}

void WaitBpmpTxComplete() NN_NOEXCEPT
{
    nn::TimeSpan elapsedTime = 0;
    while (true)
    {
        const uint32_t writeCount = util::Read32(g_pSysRam, SysramOffset_BpmpTxWriteCount);
        const uint32_t readCount = util::Read32(g_pSysRam, SysramOffset_BpmpTxReadCount);
        if (writeCount == readCount)
        {
            return;
        }

        nn::os::SleepThread(SysramPollInterval);
        if ((elapsedTime += SysramPollInterval) > SysramPollTimeout)
        {
            NN_ABORT("Timeout (%s)\n", __FUNCTION__);
        }
    }
}

void WaitBpmpRx() NN_NOEXCEPT
{
    nn::TimeSpan elapsedTime = 0;
    while (true)
    {
        const uint32_t writeCount = util::Read32(g_pSysRam, SysramOffset_BpmpRxWriteCount);
        const uint32_t readCount = util::Read32(g_pSysRam, SysramOffset_BpmpRxReadCount);

        if (writeCount != readCount)
        {
            break;
        }

        nn::os::SleepThread(SysramPollInterval);
        if ((elapsedTime += SysramPollInterval) > SysramPollTimeout)
        {
            NN_ABORT("Timeout (%s)\n", __FUNCTION__);
        }
    }
}

void DoBpmpCall(MrqType type, void* pData, size_t dataSize) NN_NOEXCEPT
{
    CheckBpmpRxEmpty();

    // REQ
    MrqRequestHeader header;
    header.type = type;
    header.flags = MrqFlag_BpmpFlagDoAck | MrqFlag_BpmpFlagRingDoorbell;

    util::CopyToIo(g_pSysRam, SysramOffset_BpmpTxRequest, &header, sizeof(header));
    util::CopyToIo(g_pSysRam, SysramOffset_BpmpTxRequest + sizeof(header), pData, dataSize);

    nn::dd::EnsureMemoryAccess();

    const uint32_t writeCount = util::Read32(g_pSysRam, SysramOffset_BpmpTxWriteCount);
    util::Write32(g_pSysRam, SysramOffset_BpmpTxWriteCount, writeCount + 1);
    util::DummyRead(g_pSysRam, SysramOffset_BpmpTxWriteCount);

    util::Write32(g_pHsp, HspDbOffset_Bpmp, 1);
    util::DummyRead(g_pHsp, HspDbOffset_Bpmp);

    WaitBpmpTxComplete();
    WaitBpmpRx();

    nn::dd::EnsureMemoryAccess();

    CheckBpmpRxSuccess();

    // ACK
    const uint32_t readCount = util::Read32(g_pSysRam, SysramOffset_BpmpRxReadCount);
    util::Write32(g_pSysRam, SysramOffset_BpmpRxReadCount, readCount + 1);
    util::DummyRead(g_pSysRam, SysramOffset_BpmpRxReadCount);

    util::Write32(g_pHsp, HspDbOffset_Bpmp, 1);
    util::DummyRead(g_pHsp, HspDbOffset_Bpmp);
}

void DoMrqResetRequest(MrqResetCommand command, MrqResetModule module) NN_NOEXCEPT
{
    MrqRequestReset req;
    req.command = command;
    req.module = module;

    DoBpmpCall(MrqType_Reset, &req, sizeof(req));
}

void DoMrqClockEnableRequest(MrqClockModule module) NN_NOEXCEPT
{
    MrqRequestClockEnable req;
    req.command = MakeMrqClockCommand(MrqClockCommandType_Enable, module);

    DoBpmpCall(MrqType_Clock, &req, sizeof(req));
}

void DoMrqClockSetRateRequest(MrqClockModule module, ClockHz rate) NN_NOEXCEPT
{
    MrqRequestClockSetRate req;
    req.command = MakeMrqClockCommand(MrqClockCommandType_SetRate, module);
    req.rate = rate;

    DoBpmpCall(MrqType_Clock, &req, sizeof(req));
}

} // namespace

void Initialize() NN_NOEXCEPT
{
    g_pGpio     = util::GetVirtualAddress(GpioCtl0Gpio5PhysAddr, GpioCtl0Gpio5Size);
    g_pSysRam   = util::GetVirtualAddress(SysramPhysAddr, SysramSize);
    g_pHsp      = util::GetVirtualAddress(HspDbPhysAddr, HspDbSize);
}

nn::Result SetReset(Module moduleId, bool asserted) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(moduleId, Module_EqosReset);

    if (asserted)
    {
        util::Write32(g_pGpio, GpioOffset_EqosReset, 0);
        util::DummyRead(g_pGpio, GpioOffset_EqosReset);

        DoMrqResetRequest(MrqResetCommand_Assert, MrqResetModule_Eqos);
    }
    else
    {
        DoMrqResetRequest(MrqResetCommand_Deassert, MrqResetModule_Eqos);

        util::Write32(g_pGpio, GpioOffset_EqosReset, 1);
        util::DummyRead(g_pGpio, GpioOffset_EqosReset);
    }

    return nn::ResultSuccess();
}

nn::Result SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    if (enabled)
    {
        DoMrqClockEnableRequest(GetMrqClockModule(moduleId));
    }
    else
    {
        NN_ABORT("Module %d disable is not implemented.\n", static_cast<int>(moduleId));
    }

    return nn::ResultSuccess();
}

nn::Result SetClockRate(Module moduleId, ClockHz rate) NN_NOEXCEPT
{
    DoMrqClockSetRateRequest(GetMrqClockModule(moduleId), rate);

    return nn::ResultSuccess();
}

}}}}}
