﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
  Copyright (c) 1998 The NetBSD Foundation, Inc.
  All rights reserved.

  This code is derived from software contributed to The NetBSD Foundation
  by Jason R. Thorpe of the Numerical Aerospace Simulation Facility,
  NASA Ames Research Center.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:
  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
  TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

  See relevant portions in IEEE Std 802.3-2012
*/

#pragma once

namespace nn     {
namespace eth    {
namespace device {

enum
{
    MII_BMCR                = 0x00,         /* Basic mode control register (rw) */
    MII_BMSR                = 0x01,         /* Basic mode status register (ro) */
    MII_PHYIDR1             = 0x02,         /* ID register 1 (ro) */
    MII_PHYIDR2             = 0x03,         /* ID register 2 (ro) */
    MII_ANAR                = 0x04,         /* Autonegotiation advertisement (rw) */
    MII_ANLPAR              = 0x05,         /* Autonegotiation link partner abilities (rw) */
    MII_ANER                = 0x06,         /* Autonegotiation expansion (ro) */
    MII_ANNP                = 0x07,         /* Autonegotiation next page */
    MII_ANLPRNP             = 0x08,         /* Autonegotiation link partner rx next page */
    MII_100T2CR             = 0x09,         /* 100base-T2 control register */
    MII_100T2SR             = 0x0a,         /* 100base-T2 status register */
    MII_EXTSR               = 0x0f,         /* Extended status register */
    MII_PHYSR               = 0x11,         /* PHY status register */
};

enum
{
    BMCR_RESET              = 0x8000,       /* reset */
    BMCR_LOOP               = 0x4000,       /* loopback */
    BMCR_SPEED0             = 0x2000,       /* speed selection (LSB) */
    BMCR_AUTOEN             = 0x1000,       /* autonegotiation enable */
    BMCR_PDOWN              = 0x0800,       /* power down */
    BMCR_ISO                = 0x0400,       /* isolate */
    BMCR_STARTNEG           = 0x0200,       /* restart autonegotiation */
    BMCR_FDX                = 0x0100,       /* Set duplex mode */
    BMCR_CTEST              = 0x0080,       /* collision test */
    BMCR_SPEED1             = 0x0040,       /* speed selection (MSB) */
};

enum
{
    BMCR_S10                = 0x0000,       /* 10 Mb/s */
    BMCR_S100               = BMCR_SPEED0,  /* 100 Mb/s */
    BMCR_S1000              = BMCR_SPEED1,  /* 1000 Mb/s */
};

#define BMCR_SPEED(x)    ((x) & (BMCR_SPEED0|BMCR_SPEED1))

enum
{
    BMSR_100T4              = 0x8000,       /* 100 base T4 capable */
    BMSR_100TXFDX           = 0x4000,       /* 100 base Tx full duplex capable */
    BMSR_100TXHDX           = 0x2000,       /* 100 base Tx half duplex capable */
    BMSR_10TFDX             = 0x1000,       /* 10 base T full duplex capable */
    BMSR_10THDX             = 0x0800,       /* 10 base T half duplex capable */
    BMSR_100T2FDX           = 0x0400,       /* 100 base T2 full duplex capable */
    BMSR_100T2HDX           = 0x0200,       /* 100 base T2 half duplex capable */
    BMSR_EXTSTAT            = 0x0100,       /* Extended status in register 15 */
    BMSR_MFPS               = 0x0040,       /* MII Frame Preamble Suppression */
    BMSR_ACOMP              = 0x0020,       /* Autonegotiation complete */
    BMSR_RFAULT             = 0x0010,       /* Link partner fault */
    BMSR_ANEG               = 0x0008,       /* Autonegotiation capable */
    BMSR_LINK               = 0x0004,       /* Link status */
    BMSR_JABBER             = 0x0002,       /* Jabber detected */
    BMSR_EXTCAP             = 0x0001,       /* Extended capability */
    BMSR_DEFCAPMASK         = 0xffffffff,
    BMSR_MEDIAMASK          = (BMSR_100T4 | BMSR_100TXFDX | BMSR_100TXHDX | BMSR_10TFDX | BMSR_10THDX | BMSR_100T2FDX | BMSR_100T2HDX),
};

/*
 * Convert BMSR media capabilities to ANAR bits for autonegotiation.
 * Note the shift chopps off the BMSR_ANEG bit.
 */
#define BMSR_MEDIA_TO_ANAR(x) (((x) & BMSR_MEDIAMASK) >> 6)

enum
{
    IDR2_OUILSB             = 0xfc00,       /* OUI LSB */
    IDR2_MODEL              = 0x03f0,       /* vendor model */
    IDR2_REV                = 0x000f,       /* vendor revision */
};

/* section 28.2.4.1 and 37.2.6.1 */
enum
{
    ANAR_NP                 = 0x8000,       /* Next page (ro) */
    ANAR_ACK                = 0x4000,       /* link partner abilities acknowledged (ro) */
    ANAR_RF                 = 0x2000,       /* remote fault (ro) */
    ANAR_FC                 = 0x0400,       /* local device supports PAUSE */
    ANAR_T4                 = 0x0200,       /* local device supports 100bT4 */
    ANAR_TX_FD              = 0x0100,       /* local device supports 100bTx FD */
    ANAR_TX                 = 0x0080,       /* local device supports 100bTx */
    ANAR_10_FD              = 0x0040,       /* local device supports 10bT FD */
    ANAR_10                 = 0x0020,       /* local device supports 10bT */
    ANAR_CSMA               = 0x0001,       /* protocol selector CSMA/CD */
    ANAR_PAUSE_NONE         = (0 << 10),
    ANAR_PAUSE_SYM          = (1 << 10),
    ANAR_PAUSE_ASYM         = (2 << 10),
    ANAR_PAUSE_TOWARDS      = (3 << 10),
    ANAR_PAUSE_MASK         = (3 << 10),
    ANAR_X_FD               = 0x0020,       /* local device supports 1000BASE-X FD */
    ANAR_X_HD               = 0x0040,       /* local device supports 1000BASE-X HD */
    ANAR_X_PAUSE_NONE       = (0 << 7),
    ANAR_X_PAUSE_SYM        = (1 << 7),
    ANAR_X_PAUSE_ASYM       = (2 << 7),
    ANAR_X_PAUSE_TOWARDS    = (3 << 7),
};

/* section 28.2.4.1 and 37.2.6.1 */
enum
{
    ANLPAR_NP               = 0x8000,       /* Next page (ro) */
    ANLPAR_ACK              = 0x4000,       /* link partner accepted ACK (ro) */
    ANLPAR_RF               = 0x2000,       /* remote fault (ro) */
    ANLPAR_FC               = 0x0400,       /* link partner supports PAUSE */
    ANLPAR_T4               = 0x0200,       /* link partner supports 100bT4 */
    ANLPAR_TX_FD            = 0x0100,       /* link partner supports 100bTx FD */
    ANLPAR_TX               = 0x0080,       /* link partner supports 100bTx */
    ANLPAR_10_FD            = 0x0040,       /* link partner supports 10bT FD */
    ANLPAR_10               = 0x0020,       /* link partner supports 10bT */
    ANLPAR_CSMA             = 0x0001,       /* protocol selector CSMA/CD */
    ANLPAR_PAUSE_MASK       = (3 << 10),
    ANLPAR_PAUSE_NONE       = (0 << 10),
    ANLPAR_PAUSE_SYM        = (1 << 10),
    ANLPAR_PAUSE_ASYM       = (2 << 10),
    ANLPAR_PAUSE_TOWARDS    = (3 << 10),
    ANLPAR_X_FD             = 0x0020,       /* local device supports 1000BASE-X FD */
    ANLPAR_X_HD             = 0x0040,       /* local device supports 1000BASE-X HD */
    ANLPAR_X_PAUSE_MASK     = (3 << 7),
    ANLPAR_X_PAUSE_NONE     = (0 << 7),
    ANLPAR_X_PAUSE_SYM      = (1 << 7),
    ANLPAR_X_PAUSE_ASYM     = (2 << 7),
    ANLPAR_X_PAUSE_TOWARDS  = (3 << 7),
};

/* section 28.2.4.1 and 37.2.6.1 */
enum
{
    ANER_MLF                = 0x0010,       /* multiple link detection fault */
    ANER_LPNP               = 0x0008,       /* link parter next page-able */
    ANER_NP                 = 0x0004,       /* next page-able */
    ANER_PAGE_RX            = 0x0002,       /* Page received */
    ANER_LPAN               = 0x0001,       /* link parter autoneg-able */
};

enum
{
    GTCR_TEST_MASK          = 0xe000,       /* see 802.3ab ss. 40.6.1.1.2 */
    GTCR_MAN_MS             = 0x1000,       /* enable manual master/slave control */
    GTCR_ADV_MS             = 0x0800,       /* 1 = adv. master, 0 = adv. slave */
    GTCR_PORT_TYPE          = 0x0400,       /* 1 = DCE, 0 = DTE (NIC) */
    GTCR_ADV_1000TFDX       = 0x0200,       /* adv. 1000baseT FDX */
    GTCR_ADV_1000THDX       = 0x0100,       /* adv. 1000baseT HDX */
};

enum
{
    GTSR_MAN_MS_FLT         = 0x8000,       /* master/slave config fault */
    GTSR_MS_RES             = 0x4000,       /* result: 1 = master, 0 = slave */
    GTSR_LRS                = 0x2000,       /* local rx status, 1 = ok */
    GTSR_RRS                = 0x1000,       /* remove rx status, 1 = ok */
    GTSR_LP_1000TFDX        = 0x0800,       /* link partner 1000baseT FDX capable */
    GTSR_LP_1000THDX        = 0x0400,       /* link partner 1000baseT HDX capable */
    GTSR_LP_ASM_DIR         = 0x0200,       /* link partner asym. pause dir. capable */
    GTSR_IDLE_ERR           = 0x00ff,       /* IDLE error count */
};

enum
{
    EXTSR_1000XFDX          = 0x8000,       /* 1000X full-duplex capable */
    EXTSR_1000XHDX          = 0x4000,       /* 1000X half-duplex capable */
    EXTSR_1000TFDX          = 0x2000,       /* 1000T full-duplex capable */
    EXTSR_1000THDX          = 0x1000,       /* 1000T half-duplex capable */
    EXTSR_MEDIAMASK         = (EXTSR_1000XFDX | EXTSR_1000XHDX | EXTSR_1000TFDX | EXTSR_1000THDX),
};

enum
{
    PHYSR_SPEED1            = 0x8000,
    PHYSR_SPEED0            = 0x4000,
    PHYSR_DUPLEX            = 0x2000,
    PHYSR_AUTONEG_DONE      = 0x0800,
    PHYSR_LINK              = 0x0400,
    PHYSR_JABBER            = 0x0001,
};

#define MII_OUI(id1, id2)   (((id1) << 6) | ((id2) >> 10))
#define MII_MODEL(id2)      (((id2) & IDR2_MODEL) >> 4)
#define MII_REV(id2)        ((id2) & IDR2_REV)

enum
{
    MII_ANEGTICKS           = 5,            /* seconds */
    MII_ANEGTICKS_GIGE      = 17,           /* seconds */
};

}}}
