﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/eth/eth_Result.h>

#include <siglo/device.h>
#include <net/if_dl.h>

#include "eth_SocketIf.h"

namespace nn     {
namespace eth    {
namespace device {

nn::Result SocketIf::UpdateMtu(int mtu)
NN_NOEXCEPT
{
    m_Mtu = mtu;
    return ResultSuccess();
}

nn::Result SocketIf::UpdateMultiFilter(struct ifnet *pIfnet)
NN_NOEXCEPT
{
    return ResultSuccess();
}

int SocketIf::GetMtu()
NN_NOEXCEPT
{
    return m_Mtu;
}

int  SocketIf::Ioctl(uint32_t command, char* pData)
NN_NOEXCEPT
{
    int result = 0;

    switch (command)
    {
    case SIOCSIFMTU:
        if ((result = ether_ioctl(m_pIfnet, command, pData)) == 0)
        {
            result = UpdateMtu((reinterpret_cast<ifreq *>(pData))->ifr_mtu).IsFailure() ? -1 : 0;
        }
        break;

    case SIOCSIFFLAGS:
        if ((reinterpret_cast<ifreq *>(pData))->ifr_flags & IFF_UP)
        {
            if (!(m_pIfnet->if_drv_flags & IFF_DRV_RUNNING))
            {
                Init();
                m_pIfnet->if_drv_flags &= ~IFF_DRV_OACTIVE;
                m_pIfnet->if_drv_flags |=  IFF_DRV_RUNNING;
            }
        }
        else
        {
            if (m_pIfnet->if_drv_flags & IFF_DRV_RUNNING)
            {
                Stop();
                m_pIfnet->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);
            }
        }
        break;

    case SIOCADDMULTI:
    case SIOCDELMULTI:
        {
            if ((m_pIfnet->if_flags & IFF_MULTICAST) == 0)
            {
                return EOPNOTSUPP;
            }

            result = (UpdateMultiFilter(m_pIfnet)).IsFailure() ? -1 : 0;
            break;
        }

    case SIOCGIFMEDIA:
        {
            struct ifmediareq *pIfmr = reinterpret_cast<struct ifmediareq *>(pData);
            pIfmr->ifm_status = IFM_AVALID | (m_LinkState ? IFM_ACTIVE : 0);
            pIfmr->ifm_active = IFM_ETHER;
            pIfmr->ifm_count  = 0;
            pIfmr->ifm_ulist  = NULL;
            break;
        }

    case SIOCSIFMEDIA:
        break;

    case SIOCSIFCAP:
        break;

    default:
        result = ether_ioctl(m_pIfnet, command, pData);
        break;
    }

    return result;
}

void SocketIf::Init()
NN_NOEXCEPT
{
    return;
}

void SocketIf::Stop()
NN_NOEXCEPT
{
    return;
}

void SocketIf::Start()
NN_NOEXCEPT
{
    return;
}

void SocketIf::StartEntry(struct ifnet *pIfnet)
NN_NOEXCEPT
{
    SocketIf* p = (SocketIf*)(pIfnet->if_softc);
    p->Start();
}

int SocketIf::IoctlEntry(struct ifnet *pIfnet, unsigned long command, char* pData)
NN_NOEXCEPT
{
    SocketIf* p = (SocketIf*)(pIfnet->if_softc);
    return p->Ioctl(command, pData);
}

void SocketIf::InitEntry(void* argument)
NN_NOEXCEPT
{
    SocketIf* socketIf = (SocketIf*)(argument);

    if (socketIf->m_pIfnet->if_drv_flags & IFF_DRV_RUNNING)
    {
        return;
    }

    socketIf->Init();

    socketIf->m_pIfnet->if_drv_flags |= IFF_DRV_RUNNING;
}

void SocketIf::SetLinkState(bool linkStateUp)
NN_NOEXCEPT
{
    if (linkStateUp)
    {
        if_link_state_change(m_pIfnet, LINK_STATE_UP);
        if_up(m_pIfnet);
        m_LinkState = true;
    }
    else
    {
        if_link_state_change(m_pIfnet, LINK_STATE_DOWN);
        if_down(m_pIfnet);
        m_LinkState = false;
    }
}

nn::Result SocketIf::GetFrameFromStack(size_t* pLengthOut, uint8_t* pData, int length)
NN_NOEXCEPT
{
    mbuf* pMbuf;

    *pLengthOut = 0;
    IFQ_DRV_LOCK(&m_pIfnet->if_snd);

    if (!IFQ_DRV_IS_EMPTY(&m_pIfnet->if_snd))
    {
        IFQ_DRV_DEQUEUE(&m_pIfnet->if_snd, pMbuf);

        if (pMbuf != nullptr)
        {
            if (pMbuf->m_pkthdr.len <= length)
            {
                *pLengthOut = pMbuf->m_pkthdr.len;
                m_copydata(pMbuf, 0, pMbuf->m_pkthdr.len, reinterpret_cast<caddr_t>(pData));
            }
            m_freem(pMbuf);
        }
    }

    IFQ_DRV_UNLOCK(&m_pIfnet->if_snd);
    return ResultSuccess();
}

nn::Result SocketIf::SendFrameToStack(uint8_t* pData, int length)
NN_NOEXCEPT
{
    mbuf* pMbuf;

    pMbuf = m_devget(reinterpret_cast<caddr_t>(pData), length, ETHER_ALIGN, m_pIfnet, NULL);

    if (pMbuf != NULL)
    {
        m_pIfnet->if_input(m_pIfnet, pMbuf);
        m_pIfnet->if_ipackets++;
    }

    return ResultSuccess();
}

nn::Result SocketIf::Initialize(char* name, int instance, uint8_t* mac)
NN_NOEXCEPT
{
    nn::Result result;

    if ((m_pIfnet = if_alloc(IFT_ETHER)) == nullptr)
    {
        return nn::eth::ResultOutOfMemory();
    }

    if_initname(m_pIfnet, name, instance);

    m_pIfnet->if_softc              = this;
    m_pIfnet->if_flags              = IFF_BROADCAST | IFF_MULTICAST;
    m_pIfnet->if_capabilities       = 0;
    m_pIfnet->if_hwassist           = 0;
    m_pIfnet->if_capenable          = m_pIfnet->if_capabilities;
    m_pIfnet->if_ioctl              = IoctlEntry;
    m_pIfnet->if_start              = StartEntry;
    m_pIfnet->if_init               = InitEntry;
    m_pIfnet->if_snd.ifq_drv_maxlen = ifqmaxlen;
    m_pIfnet->if_drv_flags          = 0;
    m_Mtu                           = ETHERMTU;

    IFQ_SET_MAXLEN(&m_pIfnet->if_snd, ifqmaxlen);
    IFQ_SET_READY(&m_pIfnet->if_snd);

    ether_ifattach(m_pIfnet, mac);

    return nn::ResultSuccess();
}

nn::Result SocketIf::Finalize()
NN_NOEXCEPT
{
    m_pIfnet->if_flags &= ~IFF_UP;

    ether_ifdetach(m_pIfnet);
    if_free(m_pIfnet);

    return ResultSuccess();
}

}}}
