﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>

#include <nn/eth/eth_EthTypes.h>

#include "eth_Gmii.h"
#include "mii_Registers.h"

namespace nn     {
namespace eth    {
namespace device {

const MiiConfig Gmii::m_MediaConfiguration[] =
{
    MiiConfig(MediaType_AUTO,                   BMCR_AUTOEN | BMCR_STARTNEG, 0,                                0),
    MiiConfig(MediaType_None,                   BMCR_ISO,             ANAR_CSMA,                               0),
    MiiConfig(MediaType_10_T,                   BMCR_S10,             ANAR_CSMA | ANAR_10,                     0),
    MiiConfig(MediaType_10_T_FDX,               BMCR_S10 | BMCR_FDX,  ANAR_CSMA | ANAR_10_FD,                  0),
    MiiConfig(MediaType_10_T_FDX_FLOW,          BMCR_S10 | BMCR_FDX,  ANAR_CSMA | ANAR_10_FD | ANAR_FC,        0),
    MiiConfig(MediaType_100_T4,                 BMCR_S100,            ANAR_CSMA | ANAR_T4,                     0),
    MiiConfig(MediaType_100_TX,                 BMCR_S100,            ANAR_CSMA | ANAR_TX,                     0),
    MiiConfig(MediaType_100_TX_FDX,             BMCR_S100 | BMCR_FDX, ANAR_CSMA | ANAR_TX_FD,                  0),
    MiiConfig(MediaType_100_TX_FDX_FLOW,        BMCR_S100 | BMCR_FDX, ANAR_CSMA | ANAR_TX_FD | ANAR_FC,        0),
    MiiConfig(MediaType_1000_T,                 BMCR_S1000,           ANAR_CSMA,                               GTCR_ADV_1000THDX | GTCR_MAN_MS),
    MiiConfig(MediaType_1000_T_MASTER,          BMCR_S1000,           ANAR_CSMA,                               GTCR_ADV_1000THDX | GTCR_ADV_MS),
    MiiConfig(MediaType_1000_T_FDX,             BMCR_S1000,           ANAR_CSMA | ANAR_FC | ANAR_X_PAUSE_ASYM, GTCR_ADV_1000TFDX | GTCR_MAN_MS),
    MiiConfig(MediaType_1000_T_FDX_FLOW,        BMCR_S1000,           ANAR_CSMA | ANAR_FC | ANAR_X_PAUSE_ASYM, GTCR_ADV_1000TFDX | GTCR_MAN_MS),
    MiiConfig(MediaType_1000_T_FDX_MASTER,      BMCR_S1000,           ANAR_CSMA | ANAR_FC | ANAR_X_PAUSE_ASYM, GTCR_ADV_1000TFDX | GTCR_ADV_MS),
    MiiConfig(MediaType_1000_T_FDX_MASTER_FLOW, BMCR_S1000,           ANAR_CSMA | ANAR_FC | ANAR_X_PAUSE_ASYM, GTCR_ADV_1000TFDX | GTCR_ADV_MS),
};

uint8_t Gmii::m_DeviceInstanceCounter = 0;

nn::Result Gmii::CheckSupportedMedia(MediaType mediaType)
NN_NOEXCEPT
{
    for (uint32_t i = 0; i < m_MediaList.mediaCount; i++)
    {
        if (mediaType == m_MediaList.mediaType[i])
        {
            return ResultSuccess();
        }
    }
    return ResultMediaNotSupported();
}

nn::Result Gmii::GetInterfaceInfo(InterfaceInfo* pInterfaceInfo)
NN_NOEXCEPT
{
    pInterfaceInfo->interfaceModel = m_PhyModel;
    pInterfaceInfo->interfaceRevision = m_PhyRevision;
    pInterfaceInfo->interfaceUniqueIdentifier = m_PhyUniqueIdentifier;
    std::memcpy(pInterfaceInfo->interfaceMacAddress, m_PhyMacAddress, sizeof(pInterfaceInfo->interfaceMacAddress));
    nn::util::Strlcpy(pInterfaceInfo->interfaceName, m_FullName, sizeof(pInterfaceInfo->interfaceName));
    return ResultSuccess();
}

nn::Result Gmii::GetMediaList(MediaList* pMediaList)
NN_NOEXCEPT
{
    *pMediaList = m_MediaList;
    return ResultSuccess();
}

nn::Result Gmii::GetMediaType(MediaType* pRequestedType, MediaType* pCurrentType, uint32_t*  pEventCounter)
NN_NOEXCEPT
{
    *pRequestedType = m_MediaRequestedType;
    *pCurrentType   = m_MediaCurrentType;
    *pEventCounter  = m_EventCounter;
    return ResultSuccess();
}

nn::Result Gmii::UpdateLinkStatus()
NN_NOEXCEPT
{
    MediaType mediaState;

    if (PhyGetMedia(&mediaState).IsSuccess())
    {
        if (!EthLinkUp(mediaState))
        {
            if (m_MediaCurrentType != MediaType_ConfigInProgress &&
                m_MediaCurrentType != MediaType_None)
            {
                m_MediaCurrentType = MediaType_LinkLost;
                m_EventCounter++;
                Signal(ResultSuccess());
            }
        }
        else
        {
            if (m_MediaRequestedType == MediaType_AUTO)
            {
                // adjust PHY settings after auto neg is completed
                // to match new configuration (if needed)
                PhyAdjustMedia(mediaState);
            }
            m_MediaCurrentType = mediaState;
            m_EventCounter++;
            Signal(ResultSuccess());
        }
    }

    return ResultSuccess();
}

nn::Result Gmii::SetMediaType(MediaType mediaType) NN_NOEXCEPT
{
    uint32_t i;
    nn::Result result = ResultMediaNotSupported();

    for (i = 0;
         i < sizeof(m_MediaConfiguration) / sizeof(m_MediaConfiguration[0]);
         i++)
    {
        if (m_MediaConfiguration[i].m_MediaType == mediaType && CheckSupportedMedia(mediaType).IsSuccess())
        {
            uint16_t anar = m_MediaConfiguration[i].m_Anar;
            uint16_t bmcr = m_MediaConfiguration[i].m_Bmcr;
            uint16_t gtcr = m_MediaConfiguration[i].m_Gtcr;

            if (mediaType == MediaType_AUTO)
            {
                anar |= BMSR_MEDIA_TO_ANAR(m_PhyCapabilities) | ANAR_CSMA;
                anar &= ~ANAR_PAUSE_MASK;

                if (m_PhyCapabilities & BMSR_10TFDX ||
                    m_PhyCapabilities & BMSR_100TXFDX)
                {
                    anar |= ANAR_FC;
                }
                if (m_PhyExtraCapabilities & EXTSR_MEDIAMASK &&
                    m_PhyExtraCapabilities & EXTSR_1000THDX)
                {
                    anar |= ANAR_PAUSE_ASYM;
                    gtcr |= GTCR_ADV_1000THDX;
                }
                if (m_PhyExtraCapabilities & EXTSR_MEDIAMASK &&
                    m_PhyExtraCapabilities & EXTSR_1000TFDX)
                {
                    anar |= ANAR_PAUSE_ASYM;
                    gtcr |= GTCR_ADV_1000TFDX;
                }
            }

            m_MediaRequestedType = mediaType;
            m_Ticks              = 0;

            if (m_MediaCurrentType != MediaType_LinkLost)
            {
                m_MediaCurrentType  = MediaType_ConfigInProgress;
            }

            if ((result = PhySetMedia(mediaType)).IsFailure())
            {
                return result;
            }

            if ((result = PhyWriteReg(MII_ANAR, anar)).IsFailure() ||
               ((m_Flags & MIIF_HAVE_GTCR) &&
               (result = PhyWriteReg(MII_100T2CR, gtcr)).IsFailure()) ||
               (result = PhyWriteReg(MII_BMCR, bmcr)).IsFailure())
            {
                return result;
            }

            m_Flags &= ~MIIF_STOPPED;

            break;
        }
    }

    return result;
}

nn::Result Gmii::Reset() NN_NOEXCEPT
{
    return PhyReset();
}

nn::Result Gmii::LinkReset() NN_NOEXCEPT
{
    return PhyLinkReset();
}

nn::Result Gmii::Tick() NN_NOEXCEPT
{
    nn::Result result;

    if (m_MediaCurrentType   == MediaType_ConfigInProgress &&
        m_MediaRequestedType == MediaType_AUTO)
    {
        if (++m_Ticks >= m_AutoNegotiationTicks)
        {
            if ((result = LinkReset()).IsFailure() ||
                (result = SetMediaType(m_MediaRequestedType)).IsFailure())
            {
                return result;
            }
        }
    }

    return ResultSuccess();
}

uint32_t Gmii::AdjustForTickRate(uint32_t rateSeconds)
NN_NOEXCEPT
{
    return (m_TickRateMs < 1000) ?
           rateSeconds * (1000 / m_TickRateMs) :
           rateSeconds;
}

nn::Result Gmii::Initialize() NN_NOEXCEPT
{
    uint16_t   bmsr;
    uint16_t   extstat = 0;
    uint16_t   id1;
    uint16_t   id2;
    nn::Result result;

    if ((result = Reset()).IsFailure())
    {
        NN_SDK_LOG("[eth]: InitializeAdapter - failed to reset adapter, error %d:%d\n",
                    result.GetModule(),
                    result.GetDescription()
        );
        return result;
    }

    if ((result = PhyReadReg(MII_BMSR,    &bmsr)).IsFailure() ||
        (result = PhyReadReg(MII_PHYIDR1, &id1)).IsFailure()  ||
        (result = PhyReadReg(MII_PHYIDR2, &id2)).IsFailure()  ||
        ((bmsr & BMSR_EXTSTAT) && (result = PhyReadReg(MII_EXTSR, &extstat)).IsFailure()))
    {
        NN_SDK_LOG("[eth]: InitializeAdapter - failed to read MII registers, error %d:%d\n",
                    result.GetModule(),
                    result.GetDescription()
        );
        return result;
    }

    m_PhyCapabilities       = bmsr;
    m_PhyExtraCapabilities  = extstat;
    m_PhyUniqueIdentifier   = MII_OUI(id1, id2);
    m_PhyModel              = MII_MODEL(id2);
    m_PhyRevision           = MII_REV(id2);
    m_MediaList.mediaCount  = 0;

    if (m_PhyCapabilities & BMSR_10THDX)
    {
        m_AutoNegotiationTicks = AdjustForTickRate(MII_ANEGTICKS);
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_10_T;
    }

    if (m_PhyCapabilities & BMSR_10TFDX)
    {
        m_AutoNegotiationTicks = AdjustForTickRate(MII_ANEGTICKS);
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_10_T_FDX;
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_10_T_FDX_FLOW;
    }

    if (m_PhyCapabilities & BMSR_100TXHDX)
    {
        m_AutoNegotiationTicks = AdjustForTickRate(MII_ANEGTICKS);
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_100_TX;
    }

    if (m_PhyCapabilities & BMSR_100TXFDX)
    {
        m_AutoNegotiationTicks = AdjustForTickRate(MII_ANEGTICKS);
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_100_TX_FDX;
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_100_TX_FDX_FLOW;
    }

    if (m_PhyCapabilities & BMSR_100T4)
    {
        m_AutoNegotiationTicks = AdjustForTickRate(MII_ANEGTICKS);
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_100_T4;
    }

    if (m_PhyExtraCapabilities & EXTSR_MEDIAMASK &&
        m_PhyExtraCapabilities & EXTSR_1000THDX)
    {
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_1000_T;
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_1000_T_MASTER;
        m_AutoNegotiationTicks = AdjustForTickRate(MII_ANEGTICKS_GIGE);
        m_Flags |= MIIF_HAVE_GTCR;
    }

    if (m_PhyExtraCapabilities & EXTSR_MEDIAMASK &&
        m_PhyExtraCapabilities & EXTSR_1000TFDX)
    {
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_1000_T_FDX;
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_1000_T_FDX_FLOW;
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_1000_T_FDX_MASTER;
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_1000_T_FDX_MASTER_FLOW;
        m_AutoNegotiationTicks = AdjustForTickRate(MII_ANEGTICKS_GIGE);
        m_Flags |= MIIF_HAVE_GTCR;
    }

    if (m_PhyCapabilities & BMSR_ANEG)
    {
        m_MediaList.mediaType[m_MediaList.mediaCount++] = MediaType_AUTO;
    }

    return Interface::Initialize();
}

nn::Result Gmii::Finalize() NN_NOEXCEPT
{
    m_Flags = MIIF_STOPPED;
    PhyWriteReg(MII_BMCR, BMCR_PDOWN);
    return Interface::Finalize();
}

}}}
