﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/dt.h>
#include <nn/dt/dt_DebugApi.h>
#include <nn/dt/dt_ResultPrivate.h>
#include <nn/dt/detail/dt_IDeviceTree.h>
#include <nn/dt/detail/dt_Log.h>
#include <nn/dt/detail/dt_PropertyApi.h>
#include "server/dt_ServiceName.h"

#include <nn/sf/sf_ShimLibraryUtility.h>

#define NN_DETAIL_DT_USE_HIPC

namespace nn { namespace dt {

namespace {
    const int NumberOfClients = 1;
    sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
    sf::ShimLibraryObjectHolder<detail::IDeviceTree> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    nn::os::Mutex g_InitializeMutex(false);
    bool g_Initialized(false);

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_InitializeMutex)> lock(g_InitializeMutex);
    if (g_Initialized)
    {
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::dt::server::DeviceTreeServiceName));
    g_Initialized = true;
}

}

namespace detail {

void InitializeWith(nn::sf::SharedPointer<IDeviceTree>&& manager) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_InitializeMutex)> lock(g_InitializeMutex);
    if (g_Initialized)
    {
        NN_ABORT("This function cannot be called after library initialization.\n");
    }

    g_Holder.InitializeHolderDirectly(std::move(manager));
    g_Initialized = true;
}

}

// Debug API

nn::Result GetNodePath(char* pOutBuffer, size_t bufferSize, const Node* pNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutBuffer == nullptr || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    auto outBuffer = nn::sf::OutBuffer(pOutBuffer, bufferSize);
    NN_RESULT_DO(g_Holder.GetObject()->GetNodePath(outBuffer, *pNode));

    return nn::ResultSuccess();
}

nn::Result DumpNode(const Node* pNode) NN_NOEXCEPT
{
    return DumpNode(pNode, -1);
}

nn::Result DumpNode(const Node* pNode, int depth) NN_NOEXCEPT
{
    Initialize();

    if (pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    return g_Holder.GetObject()->DumpNode(*pNode, depth);
}

// Node API

nn::Result FindNodeByPath(Node* pOutNode, const char* path) NN_NOEXCEPT
{
    Initialize();

    if (pOutNode == nullptr || path == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    auto pathBuffer = nn::sf::InBuffer(path, std::strlen(path) + 1);
    return g_Holder.GetObject()->FindNodeByPath(pOutNode, pathBuffer);
}

nn::Result FindNodeByPHandle(Node* pOutNode, PHandle phandle) NN_NOEXCEPT
{
    Initialize();

    if (pOutNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    return g_Holder.GetObject()->FindNodeByPHandle(pOutNode, phandle);
}

nn::Result GetCompatibleNodeCount(int* pOutCount, const char* compatible) NN_NOEXCEPT
{
    Initialize();

    if (pOutCount == nullptr || compatible == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t count;
    auto compatibleBuffer = nn::sf::InBuffer(compatible, std::strlen(compatible) + 1);
    NN_RESULT_DO(g_Holder.GetObject()->GetCompatibleNodeCount(&count, compatibleBuffer));

    if (count >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = static_cast<int>(count);
    return nn::ResultSuccess();
}

nn::Result ListCompatibleNode(Node* pOutNodeList, int* pOutCount, int maxCount, const char* compatible) NN_NOEXCEPT
{
    Initialize();

    if (pOutNodeList == nullptr || pOutCount == nullptr || maxCount < 0 || compatible == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t count;
    auto outNodeList = nn::sf::OutBuffer(reinterpret_cast<char*>(pOutNodeList), sizeof(Node) * maxCount);
    auto compatibleBuffer = nn::sf::InBuffer(compatible, std::strlen(compatible) + 1);
    NN_RESULT_DO(g_Holder.GetObject()->ListCompatibleNode(outNodeList, &count, maxCount, compatibleBuffer));

    if (count >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = static_cast<int>(count);
    return nn::ResultSuccess();
}

nn::Result GetParentNode(Node* pOutNode, const Node* pCurrentNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutNode == nullptr || pCurrentNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    return g_Holder.GetObject()->GetParentNode(pOutNode, *pCurrentNode);
}

nn::Result GetChildNodeByName(Node* pOutNode, const Node* pCurrentNode, const char* name) NN_NOEXCEPT
{
    Initialize();

    if (pOutNode == nullptr || pCurrentNode == nullptr || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    return g_Holder.GetObject()->GetChildNodeByName(pOutNode, *pCurrentNode, nameBuffer);
}

nn::Result GetChildNodeCount(int* pOutCount, const Node* pCurrentNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutCount == nullptr || pCurrentNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t count;
    NN_RESULT_DO(g_Holder.GetObject()->GetChildNodeCount(&count, *pCurrentNode));

    if (count >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = static_cast<int>(count);
    return nn::ResultSuccess();
}

nn::Result ListChildNode(Node* pOutNodeList, int* pOutCount, int maxCount, const Node* pCurrentNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutNodeList == nullptr || pOutCount == nullptr || maxCount < 0 || pCurrentNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t count;
    auto outNodeList = nn::sf::OutBuffer(reinterpret_cast<char*>(pOutNodeList), sizeof(Node) * maxCount);
    NN_RESULT_DO(g_Holder.GetObject()->ListChildNode(outNodeList, &count, maxCount, *pCurrentNode));

    if (count >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = static_cast<int>(count);

    return nn::ResultSuccess();
}

nn::Result GetFirstChildNode(Node* pOutNode, const Node* pCurrentNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutNode == nullptr || pCurrentNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    return g_Holder.GetObject()->GetFirstChildNode(pOutNode, *pCurrentNode);
}

nn::Result GetNextSiblingNode(Node* pOutNode, const Node* pCurrentNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutNode == nullptr || pCurrentNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    return g_Holder.GetObject()->GetNextSiblingNode(pOutNode, *pCurrentNode);
}

nn::Result GetValueOfAddressCells(uint32_t* pOutValue, const Node* pNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutValue == nullptr || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    return g_Holder.GetObject()->GetValueOfAddressCells(pOutValue, *pNode);
}

nn::Result GetValueOfSizeCells(uint32_t* pOutValue, const Node* pNode) NN_NOEXCEPT
{
    Initialize();

    if (pOutValue == nullptr || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    return g_Holder.GetObject()->GetValueOfSizeCells(pOutValue, *pNode);
}

// Property API

namespace detail {

nn::Result GetPropertyCount(int *pOutCount, const Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutCount == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    size_t size;
    NN_RESULT_DO(GetPropertySize(&size, pNode, name));

    if (size % elementSize != 0)
    {
        return nn::dt::ResultPropertyTypeError();
    }

    size_t count = size / elementSize;
    if (count >= static_cast<size_t>(std::numeric_limits<int>::max()))
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = static_cast<int>(count);

    return nn::ResultSuccess();
}

nn::Result GetProperty(char* pOutValue, const Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutValue == nullptr || pNode == nullptr || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    auto outBuffer = nn::sf::OutBuffer(pOutValue, elementSize);
    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    return g_Holder.GetObject()->GetProperty(outBuffer, *pNode, nameBuffer, elementSize);
}

nn::Result GetProperty(char* pOutValue, const Node* pNode, const char* name, int index, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutValue == nullptr || pNode == nullptr || index < 0 || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    auto outBuffer = nn::sf::OutBuffer(pOutValue, elementSize);
    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    return g_Holder.GetObject()->GetPropertyWithIndex(outBuffer, *pNode, nameBuffer, static_cast<size_t>(index), elementSize);
}

nn::Result GetPropertyArray(char* pOutArray, int count, const Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutArray == nullptr || count < 0 || pNode == nullptr || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t bufferSize = elementSize * count;
    uint64_t actualSize;
    auto outBuffer = nn::sf::OutBuffer(pOutArray, elementSize * count);
    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    NN_RESULT_DO(g_Holder.GetObject()->GetPropertyList(outBuffer, &actualSize, count, *pNode, nameBuffer, elementSize));

    if (bufferSize != actualSize)
    {
        return nn::dt::ResultSizeIncorrect();
    }

    return nn::ResultSuccess();
}

nn::Result GetPropertyList(char* pOutList, int* pOutCount, int maxCount, const Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutList == nullptr || pOutCount == nullptr || maxCount < 0 || pNode == nullptr || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t actualSize;
    auto outBuffer = nn::sf::OutBuffer(pOutList, elementSize * maxCount);
    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    NN_RESULT_DO(g_Holder.GetObject()->GetPropertyList(outBuffer, &actualSize, maxCount, *pNode, nameBuffer, elementSize));

    if (actualSize >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = std::min(maxCount, static_cast<int>(actualSize) / static_cast<int>(elementSize)); // elementSize はヘッダでサイズをチェック済

    return nn::ResultSuccess();
}

nn::Result GetRegisterAddress(char* pOutValue, const Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    return GetRegisterAddressArray(pOutValue, 1, pNode, elementSize);
}

nn::Result GetRegisterAddressArray(char* pOutArray, int count, const Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutArray == nullptr || count < 0 || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t actualCount;
    auto outBuffer = nn::sf::OutBuffer(pOutArray, elementSize * count);
    NN_RESULT_DO(g_Holder.GetObject()->GetRegisterAddressList(outBuffer, &actualCount, count, *pNode, elementSize));

    if (actualCount != count)
    {
        return nn::dt::ResultSizeIncorrect();
    }
    return nn::ResultSuccess();
}

nn::Result GetRegisterAddressList(char* pOutList, int* pOutCount, int maxCount, const Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutList == nullptr || pOutCount == nullptr || maxCount < 0 || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t actualCount;
    auto outBuffer = nn::sf::OutBuffer(pOutList, elementSize * maxCount);
    NN_RESULT_DO(g_Holder.GetObject()->GetRegisterAddressList(outBuffer, &actualCount, maxCount, *pNode, elementSize));

    if (actualCount >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = std::min(maxCount, static_cast<int>(actualCount));

    return nn::ResultSuccess();
}

nn::Result GetRegisterSize(char* pOutValue, const Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    return GetRegisterSizeArray(pOutValue, 1, pNode, elementSize);
}

nn::Result GetRegisterSizeArray(char* pOutArray, int count, const Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutArray == nullptr || count < 0 || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t actualCount;
    auto outBuffer = nn::sf::OutBuffer(pOutArray, elementSize * count);
    NN_RESULT_DO(g_Holder.GetObject()->GetRegisterSizeList(outBuffer, &actualCount, count, *pNode, elementSize));

    if (actualCount != count)
    {
        return nn::dt::ResultSizeIncorrect();
    }
    return nn::ResultSuccess();
}

nn::Result GetRegisterSizeList(char* pOutList, int* pOutCount, int maxCount, const Node* pNode, size_t elementSize) NN_NOEXCEPT
{
    Initialize();

    if (pOutList == nullptr || pOutCount == nullptr || maxCount < 0 || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t actualCount;
    auto outBuffer = nn::sf::OutBuffer(pOutList, elementSize * maxCount);
    NN_RESULT_DO(g_Holder.GetObject()->GetRegisterSizeList(outBuffer, &actualCount, maxCount, *pNode, elementSize));

    if (actualCount >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = std::min(maxCount, static_cast<int>(actualCount));

    return nn::ResultSuccess();
}

} // detail

bool IsPropertyExist(const Node* pNode, const char* name) NN_NOEXCEPT
{
    bool exist;
    auto result = IsPropertyExist(&exist, pNode, name);

    if (result.IsFailure())
    {
        return false;
    }

    return exist;
}

nn::Result IsPropertyExist(bool *pOut, const Node* pNode, const char* name) NN_NOEXCEPT
{
    Initialize();

    if (pOut == nullptr || pNode == nullptr || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    return g_Holder.GetObject()->IsPropertyExist(pOut, *pNode, nameBuffer);
}

nn::Result GetPropertySize(size_t *pOutSize, const Node* pNode, const char* name) NN_NOEXCEPT
{
    Initialize();

    if (pOutSize == nullptr || pNode == nullptr || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t size;
    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    NN_RESULT_DO(g_Holder.GetObject()->GetPropertySize(&size, *pNode, nameBuffer));

    if (size >= std::numeric_limits<size_t>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutSize = static_cast<size_t>(size);

    return nn::ResultSuccess();
}

nn::Result GetPropertyString(size_t* pOutSize, char* pOutBuffer, size_t bufferSize, const Node* pNode, const char* name) NN_NOEXCEPT
{
    Initialize();

    if (pOutSize == nullptr || pOutBuffer == nullptr || pNode == nullptr || name == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t propertySize;
    auto outBuffer = nn::sf::OutBuffer(pOutBuffer, bufferSize);
    auto nameBuffer = nn::sf::InBuffer(name, std::strlen(name) + 1);
    NN_RESULT_DO(g_Holder.GetObject()->GetPropertyList(outBuffer, &propertySize, bufferSize, *pNode, nameBuffer, sizeof(char)));
    if ( propertySize >= std::numeric_limits<size_t>::max() )
    {
        return nn::dt::ResultIntegerOverflow();
    }
    if (bufferSize < propertySize)
    {
        return nn::dt::ResultBufferTooSmall();
    }
    if (pOutBuffer[propertySize - 1] != '\0')
    {
        return nn::dt::ResultPropertyTypeError();
    }
    *pOutSize = static_cast<size_t>(propertySize);

    return nn::ResultSuccess();
}

nn::Result GetInterruptCount(int* pOutCount, const Node* pNode) NN_NOEXCEPT
{
    Initialize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(pOutCount);
    NN_UNUSED(pNode);
    return nn::dt::ResultApiNotSupported();
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)

    if (pOutCount == nullptr || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t count;
    NN_RESULT_DO(g_Holder.GetObject()->GetInterruptCount(&count, *pNode));

    if (count >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = count;
    return nn::ResultSuccess();
#else
    #error "Unknown OS."
#endif
}

nn::Result GetInterrupt(nn::dt::InterruptInfo* pOutList, const Node* pNode) NN_NOEXCEPT
{
    return GetInterruptArray(pOutList, 1, pNode);
}

nn::Result GetInterruptArray(nn::dt::InterruptInfo* pOutArray, int count, const Node* pNode) NN_NOEXCEPT
{
    Initialize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(pOutArray);
    NN_UNUSED(count);
    NN_UNUSED(pNode);
    return nn::dt::ResultApiNotSupported();
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)

    if (pOutArray == nullptr || count < 0 || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t actualCount;
    auto outNodeList = nn::sf::OutBuffer(reinterpret_cast<char*>(pOutArray), sizeof(nn::dt::InterruptInfo) * count);
    NN_RESULT_DO(g_Holder.GetObject()->GetInterruptList(outNodeList, &actualCount, count, *pNode));

    if (actualCount != count)
    {
        return nn::dt::ResultSizeIncorrect();
    }
    return nn::ResultSuccess();
#else
    #error "Unknown OS."
#endif
}

nn::Result GetInterruptList(nn::dt::InterruptInfo* pOutList, int* pOutCount, int maxCount, const Node* pNode) NN_NOEXCEPT
{
    Initialize();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(pOutList);
    NN_UNUSED(pOutCount);
    NN_UNUSED(maxCount);
    NN_UNUSED(pNode);
    return nn::dt::ResultApiNotSupported();
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)

    if (pOutList == nullptr || pOutCount == nullptr || maxCount < 0 || pNode == nullptr)
    {
        return nn::dt::ResultInvalidArgument();
    }

    uint64_t actualCount;
    auto outNodeList = nn::sf::OutBuffer(reinterpret_cast<char*>(pOutList), sizeof(nn::dt::InterruptInfo) * maxCount);
    NN_RESULT_DO(g_Holder.GetObject()->GetInterruptList(outNodeList, &actualCount, maxCount, *pNode));

    if (actualCount >= std::numeric_limits<int>::max())
    {
        return nn::dt::ResultIntegerOverflow();
    }
    *pOutCount = std::min(maxCount, static_cast<int>(actualCount));
    return nn::ResultSuccess();
#else
    #error "Unknown OS."
#endif
}


}}  // namespace nn::dt
