﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "eth_EthernetDriver.h"
#include "eth_Mii.h"

namespace nn { namespace drivers { namespace eth {

EthernetDriver::EthernetDriver() :
    m_IsDriverInit(false), m_ReInit(false),
    m_IsDeviceAvailable(false), m_PcieDrvHandle(0), m_PcieFuncHandle(0),
    m_ChipInfoIndex(-1)
{

}

EthernetDriver::~EthernetDriver()
{

}

Result EthernetDriver::Initialize()
{
    Result result = ResultSuccess();

    nn::os::InitializeSemaphore(&m_AccessSem, 0, 1);

    ETH_ABORT_UNLESS_SUCCESS(nn::pcie::Initialize());

    // Data init
    m_PcieFuncHandle = 0;
    m_PcieDrvHandle = 0;
    m_ChipInfoIndex = -1;
    m_ReInit = false;
    memset(m_MacAddress, 0, sizeof(m_MacAddress));
    m_LastReportedInterfaceState = InterfaceState_Invalid;

    // Configuration
    m_IsPromiscuous = false;

    // Init semaphore, which makes this call block until device availability
    nn::os::InitializeSemaphore(&m_InitSem, 0, 1);

    // Setup multiwaits and holders which are valid at this time
    nn::os::InitializeMultiWait(&m_MultiWait);
    nn::os::CreateSystemEvent(&m_SystemEvent[REEVALUATION_PORT], nn::os::EventClearMode_AutoClear, false);
    nn::os::InitializeMultiWaitHolder(&m_Holder[REEVALUATION_PORT], &m_SystemEvent[REEVALUATION_PORT]);
    nn::os::SetMultiWaitHolderUserData(&m_Holder[REEVALUATION_PORT], REEVALUATION_PORT);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_Holder[REEVALUATION_PORT]);

    // Register this class driver with core PCI
    m_PcieDrvCfg.selectMask = (nn::pcie::FunctionSelect_BaseClassCode | nn::pcie::FunctionSelect_SubClassCode |
                               nn::pcie::FunctionSelect_VendorId | nn::pcie::FunctionSelect_DeviceId);
    m_PcieDrvCfg.baseClassCode = nn::pcie::BaseClassCodeNetwork;
    m_PcieDrvCfg.vendorId = 0x10ec;
    m_PcieDrvCfg.deviceId = 0x8168;


    ETH_ABORT_UNLESS_SUCCESS(RegisterClassDriver(&m_PcieDrvHandle,
                                                 &m_SystemEvent[PCIE_REGISTRATION_PORT],
                                                 &m_PcieDrvCfg,
                                                 nn::os::EventClearMode_AutoClear));
    nn::os::InitializeMultiWaitHolder(&m_Holder[PCIE_REGISTRATION_PORT],  &m_SystemEvent[PCIE_REGISTRATION_PORT]);
    nn::os::SetMultiWaitHolderUserData(&m_Holder[PCIE_REGISTRATION_PORT], PCIE_REGISTRATION_PORT);
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_Holder[PCIE_REGISTRATION_PORT]);

    m_IsDriverInit = true;
    ETH_ABORT_UNLESS_SUCCESS(nn::os::CreateThread(&m_Thread, InterruptThreadEntry, this, m_ThreadStack,
                                                  sizeof(m_ThreadStack), nn::os::HighestSystemThreadPriority));
    nn::os::StartThread(&m_Thread);

    // Lock now allows calls to be made into the driver
    Unlock();

    // Wait a while for device availability, because socket0 does not support asynchronous device attachment
    // and will immediately ask for mac address upon return
    if(!nn::os::TimedAcquireSemaphore(&m_InitSem, nn::TimeSpan::FromSeconds(3)))
    {
        NN_SDK_LOG("EthernetDriver::Initialize() - no device found yet.\n");
    }

    return ResultSuccess();
}

Result EthernetDriver::Finalize()
{
    Result result = ResultSuccess();

    Lock();

    // Driver already finalized?
    if (!m_IsDriverInit)
    {
        return nn::os::ResultBusy();
    }

    // Initiate interrupt thread exit
    m_IsDriverInit = false;
    nn::os::SignalSystemEvent(&m_SystemEvent[REEVALUATION_PORT]);

    // Stop Ethernet device
    StopHardware();

    // Synchronize with interrupt thread exit
    WaitThread(&m_Thread);

    // Release device, if acquired
    if (m_PcieFuncHandle != 0)
    {
        nn::pcie::ReleaseFunction(&m_SystemEvent[PCIE_PM_PORT], m_PcieDrvHandle, m_PcieFuncHandle);
    }

    // Unregister driver
    nn::pcie::UnregisterClassDriver(&m_SystemEvent[PCIE_REGISTRATION_PORT], m_PcieDrvHandle);

    // Done using PCIe driver
    nn::pcie::Finalize();

    // Cleanup OS resources
    nn::os::FinalizeSemaphore(&m_AccessSem);
    nn::os::FinalizeSemaphore(&m_InitSem);
    nn::os::UnlinkMultiWaitHolder(&m_Holder[REEVALUATION_PORT]);
    nn::os::FinalizeMultiWaitHolder(&m_Holder[REEVALUATION_PORT]);
    nn::os::DestroySystemEvent(&m_SystemEvent[REEVALUATION_PORT]);
    nn::os::UnlinkMultiWaitHolder(&m_Holder[PCIE_REGISTRATION_PORT]);
    nn::os::FinalizeMultiWaitHolder(&m_Holder[PCIE_REGISTRATION_PORT]);
    if (m_PcieFuncHandle != 0)
    {
        nn::os::UnlinkMultiWaitHolder(&m_Holder[PCIE_PM_PORT]);
        nn::os::FinalizeMultiWaitHolder(&m_Holder[PCIE_PM_PORT]);
        nn::os::UnlinkMultiWaitHolder(&m_Holder[PCIE_IRQ_PORT]);
        nn::os::FinalizeMultiWaitHolder(&m_Holder[PCIE_IRQ_PORT]);
    }
    nn::os::FinalizeMultiWait(&m_MultiWait);
    nn::os::DestroyThread(&m_Thread);

    m_PcieFuncHandle = 0;
    m_PcieDrvHandle  = 0;

    return ResultSuccess();
}

Result EthernetDriver::Send(const void *pFrame, int32_t size)
{
    Result result = ResultSuccess();

    Lock();

    do
    {
        nn::pcie::BusAddress frameBusAddress = 0;
        uint32_t txIndex = -1;
        struct TxDMADesc *pTxd = NULL;

        // Device available?
        if (!m_IsDeviceAvailable)
        {
            result = nn::os::ResultNotSupported();
            break;
        }

        // Is total size too large?
        if (size > ETHER_MTU_SIZE)
        {
            result = nn::os::ResultInvalidParameter();
            break;
        }

        // Driver finalizing?
        if (!m_IsDriverInit)
        {
            result = nn::os::ResultBusy();
        }

        // Link up?
        if (!m_LinkState.linkUp)
        {
            result = nn::os::ResultBusy();
            break;
        }

        // Descriptor available?
        if (m_TxPendCount >= NUM_TX_DESC)
        {
            result = nn::os::ResultOverflow();
            break;
        }

        // Get next free descriptor
        txIndex = m_TxFreeIndex % NUM_TX_DESC;
        pTxd = m_DmaDesc.data.txDesc.data.list + txIndex;

        // Save transaction information
        m_TxTxnList[txIndex].pFrame = pFrame;
        m_TxTxnList[txIndex].size   = size;

        // Put data into buffer
        void *pBuf = m_DmaTxFrames.list[txIndex].data;
        memcpy(pBuf, pFrame, size);
        nn::dd::FlushDataCache(pBuf, size);

        // Populate descriptor
        pTxd->addr     = m_TxFramesBusAddressList[txIndex];
        pTxd->status   = DESC_OWN_MASK | FIRST_FRAGMENT_MASK | LAST_FRAGMENT_MASK | size | (RING_END_MASK * !((txIndex + 1) % NUM_TX_DESC));
        pTxd->vlan_tag = 0;
        nn::dd::FlushDataCache(pTxd, sizeof(TxDMADesc));

        // Advance
        m_TxFreeIndex =  (m_TxFreeIndex + 1) % NUM_TX_DESC;
        m_TxPendCount++;

        // Begin
        RegWrite8(TX_POLL_OFFSET, NPQ_MASK);
    }while (false);

    Unlock();

    return result;
}

Result EthernetDriver::Send(nnnetOslMbuf *pMbufHead)
{
    Result result = ResultSuccess();

    Lock();

    do
    {
        nnnetOslMbuf *pMbuf = pMbufHead;
        size_t totalSize = 0;
        size_t offset = 0;
        int32_t segmentsInChain = 0;
        uint32_t txIndex = -1;
        struct TxDMADesc *pTxd = NULL;

        // Device available?
        if (!m_IsDeviceAvailable)
        {
            result = nn::os::ResultNotSupported();
            break;
        }

        // Determine how many segments in chain
        while (pMbuf != NULL)
        {
            totalSize += pMbuf->m_len;
            segmentsInChain++;
            pMbuf = pMbuf->m_next;
        }

        // Is total size too large?
        if (totalSize > ETHER_MTU_SIZE)
        {
            result = nn::os::ResultInvalidParameter();
            break;
        }

        // Driver finalizing?
        if (!m_IsDriverInit)
        {
            result = nn::os::ResultBusy();
            break;
        }

        // Link up?
        if (!m_LinkState.linkUp)
        {
            result = nn::os::ResultBusy();
            break;
        }

        // Descriptor available?
        if (m_TxPendCount >= NUM_TX_DESC)
        {
            result = nn::os::ResultOverflow();
            break;
        }

        // Get next free descriptor
        txIndex = m_TxFreeIndex % NUM_TX_DESC;
        pTxd = m_DmaDesc.data.txDesc.data.list + txIndex;
        uint8_t *pTxBuf = m_DmaTxFrames.list[txIndex].data;

        // Save transaction information
        m_TxTxnList[txIndex].pFrame = pMbufHead;
        m_TxTxnList[txIndex].size   = totalSize;

        //Put segment data into buffer
        pMbuf = pMbufHead;
        for (int32_t count = 0; (count < segmentsInChain) && (pMbuf != NULL); count++)
        {
            size_t segmentSize = pMbufHead->m_len;
            memcpy(pTxBuf + offset, nnnetOslMbuf_tod(pMbuf), segmentSize);
            offset += segmentSize;
            pMbuf = pMbuf->m_next;
        }
        nn::dd::FlushDataCache(pTxBuf, totalSize);

        // Populate descriptor
        nn::dd::InvalidateDataCache(pTxd, sizeof(TxDMADesc));
        pTxd->addr     = m_TxFramesBusAddressList[txIndex];
        pTxd->status   = DESC_OWN_MASK | FIRST_FRAGMENT_MASK | LAST_FRAGMENT_MASK | totalSize | (RING_END_MASK * !((txIndex + 1) % NUM_TX_DESC));
        pTxd->vlan_tag = 0;
        nn::dd::FlushDataCache(pTxd, sizeof(TxDMADesc));

        // Advance
        m_TxFreeIndex =  (m_TxFreeIndex + 1) % NUM_TX_DESC;
        m_TxPendCount++;

        // Begin
        RegWrite8(TX_POLL_OFFSET, NPQ_MASK);

    }while (false);

    Unlock();

    return result;
}

Result EthernetDriver::Send(int32_t numSegments, uint8_t **segmentBuffers, size_t *segmentSizes)
{
    Result result = ResultSuccess();

    Lock();

    do
    {
        size_t totalSize = 0;
        size_t offset = 0;
        int32_t segmentsInChain = 0;
        uint32_t txIndex = -1;
        struct TxDMADesc *pTxd = NULL;

        // Device available?
        if ( !m_IsDeviceAvailable)
        {
            result = nn::os::ResultNotSupported();
            break;
        }

        // Determine how many segments in chain
        for (int32_t index = 0; index < numSegments; index++)
        {
            totalSize += segmentSizes[index];
            segmentsInChain++;
        }

        // Is total size too large?
        if (totalSize > ETHER_MTU_SIZE)
        {
            result = nn::os::ResultInvalidParameter();
            break;
        }

        // Driver finalizing?
        if ( !m_IsDriverInit)
        {
            result = nn::os::ResultBusy();
            break;
        }

        // Link up?
        if ( !m_LinkState.linkUp)
        {
            result = nn::os::ResultBusy();
            break;
        }

        // Descriptor available?
        if (m_TxPendCount >= NUM_TX_DESC)
        {
            result = nn::os::ResultOverflow();
            break;
        }

        // Get next free descriptor
        txIndex = m_TxFreeIndex % NUM_TX_DESC;
        pTxd = m_DmaDesc.data.txDesc.data.list + txIndex;
        uint8_t *pTxBuf = m_DmaTxFrames.list[txIndex].data;

        // Save transaction information
        m_TxTxnList[txIndex].pFrame = segmentBuffers;
        m_TxTxnList[txIndex].size   = totalSize;

        //Put segment data into buffer
        for (int32_t count = 0; count < segmentsInChain; count++)
        {
            memcpy(pTxBuf + offset, segmentBuffers[count], segmentSizes[count]);
            offset += segmentSizes[count];
        }
        nn::dd::FlushDataCache(pTxBuf, totalSize);

        // Populate descriptor
        nn::dd::InvalidateDataCache(pTxd, sizeof(TxDMADesc));
        pTxd->addr     = m_TxFramesBusAddressList[txIndex];
        pTxd->status   = DESC_OWN_MASK | FIRST_FRAGMENT_MASK | LAST_FRAGMENT_MASK | totalSize | (RING_END_MASK * !((txIndex + 1) % NUM_TX_DESC));
        pTxd->vlan_tag = 0;
        nn::dd::FlushDataCache(pTxd, sizeof(TxDMADesc));

        // Advance
        m_TxFreeIndex =  (m_TxFreeIndex + 1) % NUM_TX_DESC;
        m_TxPendCount++;

        // Begin
        RegWrite8(TX_POLL_OFFSET, NPQ_MASK);

    }while (false);

    Unlock();

    return result;
}

Result EthernetDriver::GetMacAddress(uint8_t pMacAddress[NN_NET_MAC_ADDRESS_SIZE]) const
{
    Result result = ResultSuccess();
    memcpy(pMacAddress, m_MacAddress, NN_NET_MAC_ADDRESS_SIZE);
    return result;
}


/////////////////////////////////////////////////////////////////////////////

EthernetDriver::ChipParams EthernetDriver::m_ChipInfoList[3] =
{
    {
        "RTL8168G",
        TX_CFG_HWREV_8168G,
        0xff7e5880,
        RX_CONFIG_128_INT_EN_MASK | RX_EARLY_OFF_V2_MASK |
        RX_SINGLE_FETCH_V2_MASK | (TX_DMA_BURST_UNLIMITED_MASK << RX_CONFIG_DMA_SHIFT)
    },
    { "RTL8168E",
      TX_CFG_HWREV_8168E,
      0xff7e5880,
      RX_CONFIG_128_INT_EN_MASK |
      RX_SINGLE_FETCH_V2_MASK | (TX_DMA_BURST_UNLIMITED_MASK << RX_CONFIG_DMA_SHIFT)

    },
    { "RTL8168E_VL",
      TX_CFG_HWREV_8168E_VL,
      0xff7e5880,
      RX_CONFIG_128_INT_EN_MASK | RX_EARLY_OFF_MASK |
      RX_SINGLE_FETCH_V2_MASK | (TX_DMA_BURST_UNLIMITED_MASK << RX_CONFIG_DMA_SHIFT)
    },
};


void EthernetDriver::Lock()
{
    AcquireSemaphore(&m_AccessSem);
}

void EthernetDriver::Unlock()
{
    ReleaseSemaphore(&m_AccessSem);
}

void EthernetDriver::ReadLinkState(LinkState *pLinkState)
{
    uint8_t phyStatusVal;
    memset(pLinkState, 0, sizeof(LinkState));
    phyStatusVal = RegRead8(PHY_STATUS_OFFSET);
    pLinkState->linkUp = (phyStatusVal & LINK_STATUS_MASK) ? true : false;
    pLinkState->autoNegComplete = (BMSR_ANEGCOMPLETE & MdioRead(0x0000, MII_BMSR)) ? true : false;
    if (pLinkState->linkUp)
    {
        pLinkState->duplex = (phyStatusVal & FULL_DUPLEX_MASK) ? LinkDuplex_Full : LinkDuplex_Half;
        pLinkState->txFlowControl = (phyStatusVal & TX_FLOW_CTRL_MASK) ? true : false;
        pLinkState->rxFlowControl = (phyStatusVal & RX_FLOW_CTRL_MASK) ? true : false;
        if (phyStatusVal & SPEED_10BPS_MASK)
        {
            pLinkState->speed = LinkSpeed_10;
        }
        else if (phyStatusVal & SPEED_100BPS_MASK)
        {
            pLinkState->speed = LinkSpeed_100;
        }
        else if (phyStatusVal & SPEED_1000BPS_MASK)
        {
            pLinkState->speed = LinkSpeed_1000;
        }
        else
        {
            pLinkState->speed = LinkSpeed_Unknown;
        }
    }
}

void EthernetDriver::InterruptThread()
{
    uint32_t intLoopCount = 0;
    Result result = ResultSuccess();

    nn::os::MultiWaitHolderType *signaledHolder;
    while ((signaledHolder = nn::os::WaitAny(&m_MultiWait)) != NULL)
    {
        auto portIndex = nn::os::GetMultiWaitHolderUserData(signaledHolder);
        if (nn::os::TryWaitSystemEvent(&m_SystemEvent[portIndex]) == false)
        {
            continue;
        }

        // Time to finalize?
        if(!m_IsDriverInit)
        {
            break;
        }

        if(portIndex == PCIE_REGISTRATION_PORT)
        {
            HandlePcieDriverRegistrationEvent();
            continue;
        }
        if(portIndex != PCIE_IRQ_PORT)
        {
            continue;
        }

        uint16_t intStatus = RegRead16(INTR_STATUS_OFFSET);

        Lock();

        // while there are pending interrupts
        while (intStatus != 0)
        {
            RegWrite16(INTR_STATUS_OFFSET, intStatus);

            if (intStatus & SYS_ERR_INT_MASK)
            {
                m_Statistics.SysErrInt++;
            }
            if (intStatus & PCS_TIMEOUT_INT_MASK)
            {
                ETH_LOG_INFO("PCS_TIMEOUT_INT_MASK detected.\n");
                m_Statistics.PcsTimeoutInt++;
            }
            if (intStatus & RX_FIFO_OVERFLOW_INT_MASK)
            {
                ETH_LOG_INFO("RX FIFO Overflow detected.\n");
                m_Statistics.RxFifoOverflowInt++;
                m_ReInit = true;
            }
            if (intStatus & RX_DESC_UNAVAILABLE_INT_MASK)
            {
                m_Statistics.RxDescUnavailInt++;
            }
            if (intStatus & RX_ERR_INT_MASK)
            {
                m_Statistics.RxErrInt++;
            }
            if (intStatus & RX_OK_INT_MASK)
            {
                m_Statistics.RxOkInt++;
            }
            if (intStatus & TX_OK_INT_MASK)
            {
                m_Statistics.TxOkInt++;
            }
            if (intStatus & TX_ERR_INT_MASK)
            {
                m_Statistics.TxErrInt++;
            }

            // RX completion
            if (intStatus & (RX_OK_INT_MASK | RX_ERR_INT_MASK))
            {
                HandleRxCompletions(false);
            }

            // TX completion
            if (intStatus & (TX_OK_INT_MASK | TX_ERR_INT_MASK))
            {
                HandleTxCompletions(false);
            }

            // Work-arounds for hardware errata
            if (m_ReInit)
            {
                m_ReInit = false;
                ETH_LOG_INFO("Restarting MAC.\n");
                nn::pcie::ReleaseIrq(&m_SystemEvent[PCIE_IRQ_PORT], m_PcieDrvHandle, m_PcieFuncHandle);
                HandleTxCompletions(true);
                ResetHardware();
                StartHardware(true);
            }

            // Link status changes
            if (intStatus & LINK_CHANGE_INT_MASK)
            {
                MonitorLinkState();
            }

            // Legacy INT has to be re-enabled each time since it is level sensitive
            if (IRQ_TYPE == nn::pcie::IrqType_IntX)
            {
                nn::pcie::SetIrqEnable(m_PcieDrvHandle, m_PcieFuncHandle, 0, true);
            }
            intStatus = RegRead16(INTR_STATUS_OFFSET);
        }
        Unlock();
    }
}

void EthernetDriver::HandlePcieDriverRegistrationEvent()
{
    Result result = ResultSuccess();

    // Inspect function to see if it makes sense
    do
    {
        int32_t chipIndex = -1;
        nn::pcie::FunctionState functions[4] = {{0}};
        int32_t functionCount = -1;
        nn::pcie::FunctionState* pMyFunction = NULL;
        if(!(result=nn::pcie::QueryFunctions(functions, &functionCount, m_PcieDrvHandle, sizeof(functions))).IsSuccess() &&
           (functionCount > 0))
        {
            ETH_LOG_WARN("Device not found.\n");
            break;
        }

        // Not interested if we already have a device
        if (m_PcieFuncHandle != 0)
        {
            ETH_LOG_WARN("Ignoring presented function because one has already been acquired.\n");
            result = nn::os::ResultNotSupported();
            break;
        }

        // Just take the first in the list
        m_PcieFuncState = functions[0];
        m_PcieFuncHandle = m_PcieFuncState.functionHandle;

        // get and check BAR 2 details
        if (!(result = nn::pcie::GetBarProfile(&m_Bar2Profile, m_PcieDrvHandle, m_PcieFuncHandle, 2)).IsSuccess())
        {
            result = nn::os::ResultNotSupported();
            ETH_LOG_WARN("BAR2 does not exist.\n");
            break;
        }
        if (!(m_Bar2Profile.flags & nn::pcie::ResourceFlag_Mem))
        {
            result = nn::os::ResultNotSupported();
            ETH_LOG_WARN("BAR2 not MEM.\n");
            break;
        }

        // Identify device
        uint32_t version = RegRead32(TX_CONFIG_OFFSET) & TX_CFG_HWREV_MASK;
        for (int32_t index = 0; index < (sizeof(m_ChipInfoList) / sizeof(m_ChipInfoList[0])); index++)
        {
            if (m_ChipInfoList[index].version == version)
            {
                chipIndex = index;
                break;
            }
        }
        if (chipIndex < 0)
        {
            ETH_LOG_INFO("Device version 0x%08x not supported.\n", version);
            result = nn::os::ResultNotSupported();
            break;
        }

        // All looks okay, take ownership of the function
        if (!(result = nn::pcie::AcquireFunction(&m_SystemEvent[PCIE_PM_PORT],
                                                 m_PcieDrvHandle, m_PcieFuncHandle,
                                                 nn::os::EventClearMode_AutoClear)).IsSuccess())
        {
            ETH_LOG_ERROR("nn::pcie::AcquireFunction() failed.\n");
            break;
        }


        m_ChipInfoIndex = chipIndex;
        ETH_LOG_INFO("PCIe Ethernet Device %s Acquired.\n",
                     m_ChipInfoList[m_ChipInfoIndex].name);
    }while (FALSE);

    if (result.IsSuccess())
    {
        // Start driver
        if ((result = InitHardware()).IsSuccess())
        {
            result = StartHardware(false);
            SetSpeedXmii(LinkAutoNegotiation_Enable, LinkSpeed_1000, LinkDuplex_Full);
            nn::os::InitializeMultiWaitHolder(&m_Holder[PCIE_PM_PORT], &m_SystemEvent[PCIE_PM_PORT]);
            nn::os::SetMultiWaitHolderUserData(&m_Holder[PCIE_PM_PORT], PCIE_PM_PORT);
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_Holder[PCIE_PM_PORT]);
            nn::os::InitializeMultiWaitHolder(&m_Holder[PCIE_IRQ_PORT], &m_SystemEvent[PCIE_IRQ_PORT]);
            nn::os::SetMultiWaitHolderUserData(&m_Holder[PCIE_IRQ_PORT], PCIE_IRQ_PORT);
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_Holder[PCIE_IRQ_PORT]);
            m_IsDeviceAvailable = true;
            nn::os::ReleaseSemaphore(&m_InitSem);
        }
    }
    else
    {
        m_PcieFuncHandle = 0;
    }
}

Result EthernetDriver::InitHardware()
{
    size_t offset;
    uint32_t data32;
    uint16_t data16;
    Result result = ResultSuccess();

    // Put hardware to deterministic stopped state
    StopHardware();

    // Do soft-reset
    ResetHardware();

    // Retrieve serial number
    if (nn::pcie::FindExtendedCapability(&offset, m_PcieDrvHandle, m_PcieFuncHandle,
                                         nn::pcie::ExtendedCapabilityId_DeviceSerialNumber).IsSuccess())
    {
        uint8_t buf[8] = { 0 };
        offset += 4;
        for (int32_t i = 0; i < 8; i++)
        {
            nn::pcie::ReadConfig8(&m_SerialNum[i], m_PcieDrvHandle, m_PcieFuncHandle, offset + i);
        }
        ETH_LOG_INFO("Device Serial Number: %02X%02X%02X%02X%02X%02X%02X%02X\n",
                     m_SerialNum[7], m_SerialNum[6], m_SerialNum[5], m_SerialNum[4],
                     m_SerialNum[3], m_SerialNum[2], m_SerialNum[1], m_SerialNum[0]);
    }

    // Retrieve MAC address
    if (m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168G)
    {
        data32 = EriRead(0xE0, sizeof(data32), ERIAR_EXGMAC_MASK);
        memcpy(&m_MacAddress, &data32, sizeof(data32));
        data16 = (uint16_t)EriRead(0xE4, sizeof(data16), ERIAR_EXGMAC_MASK);
        memcpy(((char *)&m_MacAddress) + sizeof(data32), &data16, sizeof(data16));

        // Program MAC address into receive address register
        uint32_t rar_low = ((uint32_t)m_MacAddress[0] | ((uint32_t)m_MacAddress[1] << 8) |
                            ((uint32_t)m_MacAddress[2] << 16) | ((uint32_t)m_MacAddress[3] << 24));
        uint32_t rar_high = ((uint32_t)m_MacAddress[4] | ((uint32_t)m_MacAddress[5] << 8));
        RegWrite8(CFG9346_OFFSET, CFG9346_UNLOCK_MASK);
        RegWrite32(MAC0_OFFSET, rar_low);
        RegWrite32(MAC4_OFFSET, rar_high);
        RegWrite8(CFG9346_OFFSET, CFG9346_LOCK_MASK);
    }
    else if (m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168E ||
             m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168E_VL)
    {
        uint32_t rar_low = RegRead32(MAC0_OFFSET);
        uint32_t rar_high = RegRead32(MAC4_OFFSET);
        m_MacAddress[0] = rar_low & 0xff;
        m_MacAddress[1] = (rar_low >> 8) & 0xff;
        m_MacAddress[2] = (rar_low >> 16) & 0xff;
        m_MacAddress[3] = (rar_low >> 24) & 0xff;
        m_MacAddress[4] = rar_high & 0xff;
        m_MacAddress[5] = (rar_high >> 8) & 0xff;
    }

    return result;
}

Result EthernetDriver::StartHardware(bool isReInit)
{
    uint32_t val32;
    uint16_t val16;
    bool completed = false;
    Result result = ResultSuccess();

    do
    {
        // Map DMA buffer area
        if (!isReInit)
        {
            ETH_BREAK_UPON_ERROR(nn::pcie::MapDma(NULL, m_PcieDrvHandle, m_PcieFuncHandle, nn::pcie::DmaDirection_FromDevice,
                                                  &m_DmaRxFrames, sizeof(m_DmaRxFrames)));
            ETH_BREAK_UPON_ERROR(nn::pcie::MapDma(NULL, m_PcieDrvHandle, m_PcieFuncHandle, nn::pcie::DmaDirection_ToDevice,
                                                  &m_DmaTxFrames, sizeof(m_DmaTxFrames)));
            ETH_BREAK_UPON_ERROR(nn::pcie::MapDma(NULL, m_PcieDrvHandle, m_PcieFuncHandle, nn::pcie::DmaDirection_BiDirectional,
                                                  &m_DmaDesc, sizeof(m_DmaDesc)));
        }

        // Enable DMA, configure IRQ
        ETH_BREAK_UPON_ERROR(nn::pcie::SetDmaEnable(m_PcieDrvHandle, m_PcieFuncHandle, true));
        ETH_BREAK_UPON_ERROR(nn::pcie::AcquireIrq(&m_SystemEvent[PCIE_IRQ_PORT], m_PcieDrvHandle, m_PcieFuncHandle,
                                                  nn::os::EventClearMode_AutoClear, IRQ_TYPE));

        // Setup statistics counters DMA buffer
        memset(&m_DmaDesc.data.counters.data, 0, sizeof(m_DmaDesc.data.counters.data));
        nn::dd::FlushDataCache(&m_DmaDesc.data.counters.data, sizeof(m_DmaDesc.data.counters.data));
        ETH_BREAK_UPON_ERROR(nn::pcie::GetDmaBusAddress(&m_CountersBusAddress, m_PcieDrvHandle, m_PcieFuncHandle,
                                                        (void *)&m_DmaDesc.data.counters.data,
                                                        sizeof(m_DmaDesc.data.counters.data)));
        RegWrite32(COUNTER_ADDR_HIGH_OFFSET, (uint64_t)m_CountersBusAddress >> 32);
        RegWrite32(COUNTER_ADDR_LOW_OFFSET, (uint64_t)m_CountersBusAddress & (DMA_BIT_MASK(32)));

        // Setup tx descriptor DMA
        m_TxFreeIndex = 0;
        m_TxPendIndex = 0;
        m_TxPendCount = 0;
        memset(&m_DmaDesc.data.txDesc.data, 0, sizeof(m_DmaDesc.data.txDesc.data));
        m_DmaDesc.data.txDesc.data.list[NUM_TX_DESC - 1].status = RING_END_MASK;
        nn::dd::FlushDataCache(m_DmaDesc.data.txDesc.data.list, sizeof(m_DmaDesc.data.txDesc.data.list[0]) * NUM_TX_DESC);
        ETH_BREAK_UPON_ERROR(nn::pcie::GetDmaBusAddress(&m_TxDMADescArrayBusAddress, m_PcieDrvHandle, m_PcieFuncHandle,
                                                        (void *)m_DmaDesc.data.txDesc.data.list,
                                                        sizeof(m_DmaDesc.data.txDesc.data.list[0]) * NUM_TX_DESC));
        RegWrite32(TXDESC_START_ADDR_LOW_OFFSET, ((uint64_t)m_TxDMADescArrayBusAddress & DMA_BIT_MASK(32)));
        RegWrite32(TXDESC_START_ADDR_HIGH_OFFSET, ((uint64_t)m_TxDMADescArrayBusAddress >> 32));
        for (int32_t index = 0; index < NUM_TX_DESC; index++)
        {
            nn::pcie::BusAddress frameBusAddress;
            struct TxDMADesc *pTxd = m_DmaDesc.data.txDesc.data.list + index;
            ETH_BREAK_UPON_ERROR(nn::pcie::GetDmaBusAddress(&frameBusAddress, m_PcieDrvHandle, m_PcieFuncHandle,
                                                            &m_DmaTxFrames.list[index], ETHER_FRAME_BUF_SIZE));
            pTxd->addr = frameBusAddress;
            m_TxFramesBusAddressList[index] = frameBusAddress;
        }
        if (!result.IsSuccess())
        {
            break;
        }

        // Setup rx descriptor DMA
        m_RxIndex = 0;
        memset(&m_DmaDesc.data.rxDesc.data, 0, sizeof(m_DmaDesc.data.rxDesc.data));
        nn::dd::FlushDataCache(m_DmaDesc.data.rxDesc.data.list, sizeof(m_DmaDesc.data.rxDesc.data));
        ETH_BREAK_UPON_ERROR(nn::pcie::GetDmaBusAddress(&m_RxDescArrayBusAddresss,
                                                        m_PcieDrvHandle, m_PcieFuncHandle,
                                                        m_DmaDesc.data.rxDesc.data.list,
                                                        sizeof(m_DmaDesc.data.rxDesc.data.list[0]) * NUM_RX_DESC));
        RegWrite32(RX_DESC_START_ADDR_LOW_OFFSET, ((uint64_t)m_RxDescArrayBusAddresss & DMA_BIT_MASK(32)));
        RegWrite32(RX_DESC_START_ADDR_HIGH_OFFSET, ((uint64_t)m_RxDescArrayBusAddresss >> 32));
        for (int32_t index = 0; index < NUM_RX_DESC; index++)
        {
            nn::pcie::BusAddress frameBusAddress;
            struct RxDMADesc *pRxd = m_DmaDesc.data.rxDesc.data.list + index;
            pRxd->status = static_cast<uint32_t>(DESC_OWN_MASK | ETHER_FRAME_BUF_SIZE);
            ETH_BREAK_UPON_ERROR(nn::pcie::GetDmaBusAddress(&frameBusAddress, m_PcieDrvHandle, m_PcieFuncHandle,
                                                         &m_DmaRxFrames.list[index], ETHER_FRAME_BUF_SIZE));
            pRxd->addr = frameBusAddress;
            m_RxFramesBusAddressList[index] = frameBusAddress;
        }
        if (!result.IsSuccess())
        {
            break;
        }
        m_DmaDesc.data.rxDesc.data.list[NUM_RX_DESC - 1].status |= RING_END_MASK;
        nn::dd::FlushDataCache(m_DmaDesc.data.rxDesc.data.list, sizeof(m_DmaDesc.data.rxDesc.data.list[0]) * NUM_RX_DESC);
        nn::dd::InvalidateDataCache(m_DmaRxFrames.list, sizeof(m_DmaRxFrames.list[0]) * NUM_RX_DESC);

        // Configure
        RegWrite8(CFG9346_OFFSET, CFG9346_UNLOCK_MASK);
        RegWrite8(CHIP_CMD_OFFSET, 0);
        RegWrite16(INTR_MASK_OFFSET, 0xffff);
        RegWrite8(EARLY_TX_THRESH_OFFSET, EARLY_TX_THLD);
        RegWrite16(RX_MAX_SIZE_OFFSET, ETHER_FRAME_BUF_SIZE);

        // Rx interrupt scheduling
        //RegWrite32(TCTR_OFFSET,     0x2600);
        //RegWrite32(TIME_INT0_OFFSET, 0x2600);
        RegWrite32(TIME_INT0_OFFSET, 0x0000);
        RegWrite32(TIME_INT1_OFFSET, 0x0000);
        RegWrite32(TIME_INT2_OFFSET, 0x0000);
        RegWrite32(TIME_INT3_OFFSET, 0x0000);

        // Rx MAC address filtering (hardware)
        SetRxFilter(NULL, 0, m_IsPromiscuous);

        // Tx configuration
        RegWrite32(TX_CONFIG_OFFSET, (TX_DMA_BURST << TX_DMA_SHIFT) |
                   (INTER_FRAME_GAP << TX_INTER_FRAME_GAP_SHIFT));

        RegWrite8(CFG9346_OFFSET, CFG9346_LOCK_MASK);
        (void)RegRead8(CFG9346_OFFSET);

        // ISRs are now expected to come at any time
        ETH_BREAK_UPON_ERROR(nn::pcie::SetIrqEnable(m_PcieDrvHandle, m_PcieFuncHandle, 0, true));

        // Now, enable TX and RX
        RegWrite16(INTR_STATUS_OFFSET, RegRead16(INTR_STATUS_OFFSET));
        RegWrite8(CHIP_CMD_OFFSET, CMD_TX_ENABLE_MASK | CMD_RX_ENABLE_MASK);

    }while (false);

    if (!result.IsSuccess())
    {
        RegWrite8(CHIP_CMD_OFFSET, 0x00);
        nn::pcie::SetIrqEnable(m_PcieDrvHandle, m_PcieFuncHandle, 0, false);
        nn::pcie::SetDmaEnable(m_PcieDrvHandle, m_PcieFuncHandle, false);
    }

    return result;
} //NOLINT(impl/function_size)

Result EthernetDriver::StopHardware()
{
    Result result = ResultSuccess();

    // Filter out receive
    RegWrite32(RX_CONFIG_OFFSET, RegRead32(RX_CONFIG_OFFSET) & ~0x3f);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

    // Disable TX and RX
    RegWrite8(CHIP_CMD_OFFSET, 0x00);

    // No interrupts
    RegWrite16(INTR_MASK_OFFSET, 0x0000);

    // Stop PCI bus from permitting DMA to/from this device function
    nn::pcie::SetDmaEnable(m_PcieDrvHandle, m_PcieFuncHandle, false);

    do
    {
        // Unmap DMA buffer area
        nn::pcie::UnmapDma(m_PcieDrvHandle, m_PcieFuncHandle, &m_DmaRxFrames);
        nn::pcie::UnmapDma(m_PcieDrvHandle, m_PcieFuncHandle, &m_DmaTxFrames);
        nn::pcie::UnmapDma(m_PcieDrvHandle, m_PcieFuncHandle, &m_DmaDesc);
    }while(false);

    return result;
}

void EthernetDriver::MonitorLinkState()
{
    LinkState linkState;
    bool changed = false;

    // Get latest link state from hardware
    ReadLinkState(&linkState);

    // Has link state changed?
    if (memcmp(&linkState, &m_LinkState, sizeof(linkState)) != 0)
    {
        // Transitioning to up
        if (linkState.linkUp)
        {
            ETH_LOG_INFO("Link UP - speed=%d Mbps, duplex=%s, autoNegComplete=%s, tx/rx flow control %s/%s.\n",
                         linkState.speed,
                         ((linkState.duplex == LinkDuplex_Half) ? "HALF" :
                          ((linkState.duplex == LinkDuplex_Full) ? "FULL" : "?")),
                         (linkState.autoNegComplete) ? "yes" : "no",
                         (linkState.txFlowControl) ? "yes" : "no",
                         (linkState.rxFlowControl) ? "yes" : "no");
        }
        else
        {
            ETH_LOG_INFO("Link DOWN.\n");
        }
        changed = true;
    }

    // Save latest link state
    m_LinkState = linkState;

    // Report
    if (changed)
    {
        ReportInterfaceState();
    }
}


void EthernetDriver::ReportInterfaceState()
{
    InterfaceState ifState = InterfaceState_Invalid;
    if (m_IsDeviceAvailable)
    {
        ifState = (m_LinkState.linkUp) ? InterfaceState_LinkUp : InterfaceState_LinkDown;
    }
    else
    {
        ifState = InterfaceState_NoDevice;
    }
    if ((ifState != m_LastReportedInterfaceState) && (m_InterfaceStateCallBack != NULL))
    {
        (*m_InterfaceStateCallBack)(m_LastReportedInterfaceState = ifState);
    }
}

void EthernetDriver::HandleTxCompletions(bool reset)
{
    int32_t count;

    // for all pending tx descriptors
    for (count = 0; count < m_TxPendCount; count++)
    {
        int32_t txIndex = (m_TxPendIndex + count) % NUM_TX_DESC;
        TxTxn *pTxn = m_TxTxnList + txIndex;
        TxDMADesc *pTxd = m_DmaDesc.data.txDesc.data.list + txIndex;


        nn::dd::InvalidateDataCache(pTxd, sizeof(TxDMADesc));
        uint32_t status = pTxd->status;

        // stop looking for completions upon discovery of hardware owned descriptor
        if ((status & DESC_OWN_MASK) && !reset)
        {
            break;
        }

        // user callback
        if (m_SendCompletedCallBack != NULL)
        {
            (*m_SendCompletedCallBack)((void *)pTxn->pFrame);
        }

        // clear transaction fields
        memset(pTxn, 0, sizeof(*pTxn));
    }

    // advance
    m_TxPendIndex = (count + m_TxPendIndex) % NUM_TX_DESC;
    m_TxPendCount -= count;
}

void EthernetDriver::HandleRxCompletions(bool reset)
{
    int32_t count;
    static int32_t maxCount = 0;


    // for all pending tx descriptors
    for (count = 0; count < NUM_RX_DESC; count++)
    {
        bool discard = false;
        int32_t rxIndex = (m_RxIndex + count) % NUM_RX_DESC;
        RxDMADesc *pRxd = m_DmaDesc.data.rxDesc.data.list + rxIndex;

        nn::dd::InvalidateDataCache(pRxd, sizeof(RxDMADesc));
        uint32_t status = pRxd->status;

        // stop looking for completions upon discovery of hardware owned descriptor
        if (status & DESC_OWN_MASK)
        {
            if (reset)
            {
                continue;
            }
            else
            {
                break;
            }
        }

        // handle receive errors
        if (status & RXRES_MASK)
        {
            discard = true;
            m_Statistics.RxErrIntDesc++;
            if (status & (RXRWT_MASK | RXRUNT_MASK))
            {
                m_Statistics.RxLengthErrDesc++;
            }
            if (status & RXCRC_MASK)
            {
                m_Statistics.RxCrcErrDesc++;
            }
            if (status & RXFOVF_MASK)
            {
                m_Statistics.RxFifoErrDesc++;
            }
        }

        // check for MTU error, which may appear as fragmented segment
        if ((status & (FIRST_FRAGMENT_MASK | LAST_FRAGMENT_MASK)) != (FIRST_FRAGMENT_MASK | LAST_FRAGMENT_MASK))
        {
            discard = true;
            m_Statistics.RxFragmentErrDesc++;
        }

        // should it be delivered?
        if (discard)
        {
            m_Statistics.RxDiscarded++;
        }
        else
        {
            uint8_t *pBuf = m_DmaRxFrames.list[rxIndex].data;
            size_t size = (status & 0x00003fff);
            if ((size > ETHER_CRC_SIZE) && (size <= ETHER_FRAME_BUF_SIZE))
            {
                size_t packetSize = size - ETHER_CRC_SIZE;
                if (m_FrameReceivedCallBack != NULL)
                {
                    m_Statistics.RxDelivered++;
                    nn::dd::InvalidateDataCache(pBuf, packetSize);
                    (*m_FrameReceivedCallBack)((const void *)pBuf, packetSize);
                }
            }
            else
            {
                m_Statistics.RxSizeError++;
            }
        }

        // prepare buffer and descriptor to receive again.
        // N-at-a-time since they are not cache line aligned
        if ((rxIndex & (COHERENT_RX_DESC_COUNT - 1)) == (COHERENT_RX_DESC_COUNT - 1))
        {
            int32_t baseIndex = rxIndex - (COHERENT_RX_DESC_COUNT - 1);
            nn::dd::InvalidateDataCache(m_DmaDesc.data.rxDesc.data.list + baseIndex, sizeof(RxDMADesc) * COHERENT_RX_DESC_COUNT);
            for (int32_t x = 0; x < COHERENT_RX_DESC_COUNT; x++)
            {
                int32_t idx = baseIndex + x;
                bool bLast = (idx == (NUM_RX_DESC - 1)) ? true : false;
                RxDMADesc *pD   = m_DmaDesc.data.rxDesc.data.list + idx;
                pD->status   =  DESC_OWN_MASK | ((bLast) ? RING_END_MASK : 0) | ETHER_FRAME_BUF_SIZE;
                pD->vlan_tag = 0;
                pD->addr     = m_RxFramesBusAddressList[idx];
            }
            nn::dd::InvalidateDataCache(m_DmaRxFrames.list + baseIndex,
                                        sizeof(m_DmaRxFrames.list[0]) * COHERENT_RX_DESC_COUNT);
            nn::dd::FlushDataCache(m_DmaDesc.data.rxDesc.data.list + baseIndex,
                                   sizeof(m_DmaDesc.data.rxDesc.data.list[0]) * COHERENT_RX_DESC_COUNT);
        }
    }

    if (count > maxCount)
    {
        //ETH_LOG_INFO("Rx descriptor usage water mark now at %d.\n", count);
        maxCount = count;
    }
    // advance
    m_RxIndex = (m_RxIndex + count) % NUM_RX_DESC;

    if (reset)
    {
        m_RxIndex = 0;
    }
}

Result EthernetDriver::SetRxFilter(uint8_t addresses[MAX_MULTICAST_ADDRESSES][NN_NET_MAC_ADDRESS_SIZE],
                                   int32_t numMultiCastAddr, bool bPromiscuous)
{
    Result result = ResultSuccess();
    if (numMultiCastAddr < MAX_MULTICAST_ADDRESSES)
    {
        uint32_t mcFilter[2];
        uint32_t rxConfigRegVal;
        int32_t rxMode;
        uint32_t tmp = 0;
        if (bPromiscuous)
        {
            rxMode = RX_CONFIG_ACCEPT_BROADCAST_MASK | RX_CONFIG_ACCEPT_MULTICAST_MASK | RX_CONFIG_ACCEPT_MY_PHYS_MASK | RX_CONFIG_ACCEPT_ALL_MASK;
            mcFilter[1] = mcFilter[0] = 0xffffffff;
        }
        else
        {
            rxMode = RX_CONFIG_ACCEPT_BROADCAST_MASK | RX_CONFIG_ACCEPT_MY_PHYS_MASK;
            mcFilter[1] = mcFilter[0] = 0;
            for (int32_t i = 0; i < numMultiCastAddr; i++)
            {
                int32_t bitNr = EtherCRC32BigEndian(addresses[i], NN_NET_MAC_ADDRESS_SIZE) >> 26;
                mcFilter[bitNr >> 5] |= 1 << (bitNr & 31);
                rxMode |= RX_CONFIG_ACCEPT_MULTICAST_MASK;
            }
        }
        tmp = mcFilter[0];
        mcFilter[0] = ETH_ENDIAN_SWAP_U32(mcFilter[1]);
        mcFilter[1] = ETH_ENDIAN_SWAP_U32(tmp);
        rxConfigRegVal = RegRead32(RX_CONFIG_OFFSET);
        rxConfigRegVal &= ~(RX_CONFIG_ACCEPT_BROADCAST_MASK | RX_CONFIG_ACCEPT_MULTICAST_MASK | RX_CONFIG_ACCEPT_ALL_MASK | RX_CONFIG_ACCEPT_MY_PHYS_MASK);
        rxConfigRegVal &= m_ChipInfoList[m_ChipInfoIndex].rxConfigMask;
        rxConfigRegVal |= rxMode;
        rxConfigRegVal |= m_ChipInfoList[m_ChipInfoIndex].rxConfig;

        RegWrite32(RX_CONFIG_OFFSET, rxConfigRegVal);
        RegWrite32(MAR0_OFFSET + 0, mcFilter[0]);
        RegWrite32(MAR0_OFFSET + 4, mcFilter[1]);
    }
    else
    {
        result = nn::os::ResultInvalidParameter();
    }
    return result;
}

uint32_t EthernetDriver::EtherCRC32BigEndian(uint8_t *buf, size_t len)
{
    size_t i;
    int32_t bit;
    uint8_t data, carry;
    uint32_t crc = 0xffffffff;
    static const uint32_t ETHER_CRC_POLY_BE = 0x04c11db6;
    for (i = 0; i < len; i++)
    {
        for (data = *buf++, bit = 0; bit < 8; bit++, data >>= 1)
        {
            carry = ((crc & 0x80000000) ? 1 : 0) ^ (data & 0x01);
            crc <<= 1;
            if (carry)
            {
                crc = (crc ^ ETHER_CRC_POLY_BE) | carry;
            }
        }
    }
    return (crc);
}

void EthernetDriver::SetSpeedXmii(uint8_t autoneg, uint16_t speed, uint8_t duplex)
{
    int32_t auto_nego = 0;
    int32_t giga_ctrl = 0;
    int32_t bmcr_true_force = 0;
    uint32_t flags;

    // Linkstate will be stale after this
    memset(&m_LinkState, 0, sizeof(m_LinkState));

    if ((speed != LinkSpeed_1000) &&
        (speed != LinkSpeed_100) &&
        (speed != LinkSpeed_10))
    {
        speed = LinkSpeed_1000;
        duplex = LinkDuplex_Full;
    }
    auto_nego = MdioRead(0x0000, MII_ADVERTISE);
    auto_nego &= ~(ADVERTISE_10HALF | ADVERTISE_10FULL | ADVERTISE_100HALF | ADVERTISE_100FULL | ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM);
    giga_ctrl = MdioRead(0x0000, MII_CTRL1000);
    giga_ctrl &= ~(ADVERTISE_1000HALF | ADVERTISE_1000FULL);
    if ((autoneg == LinkAutoNegotiation_Enable) || (speed == LinkSpeed_1000))
    {
        if ((speed == LinkSpeed_10) && (duplex == LinkDuplex_Half))
        {
            auto_nego |= ADVERTISE_10HALF;
        }
        else if ((speed == LinkSpeed_10) && (duplex == LinkDuplex_Full))
        {
            auto_nego |= ADVERTISE_10HALF | ADVERTISE_10FULL;
        }
        else if ((speed == LinkSpeed_100) && (duplex == LinkDuplex_Half))
        {
            auto_nego |= ADVERTISE_100HALF | ADVERTISE_10HALF | ADVERTISE_10FULL;
        }
        else if ((speed == LinkSpeed_100) && (duplex == LinkDuplex_Full))
        {
            auto_nego |= ADVERTISE_100HALF | ADVERTISE_100FULL | ADVERTISE_10HALF | ADVERTISE_10FULL;
        }
        else if (speed == LinkSpeed_1000)
        {
            giga_ctrl |= ADVERTISE_1000HALF | ADVERTISE_1000FULL;
            auto_nego |= ADVERTISE_100HALF | ADVERTISE_100FULL | ADVERTISE_10HALF | ADVERTISE_10FULL;
        }
        auto_nego |= ADVERTISE_PAUSE_CAP | ADVERTISE_PAUSE_ASYM; // flow control
        MdioWrite(0x0000, MII_ADVERTISE, auto_nego);
        MdioWrite(0x0000, MII_CTRL1000, giga_ctrl);
        MdioWrite(0x0000, MII_BMCR, BMCR_RESET | BMCR_ANENABLE | BMCR_ANRESTART);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
    }
    else
    {
        if ((speed == LinkSpeed_10) && (duplex == LinkDuplex_Half))
        {
            bmcr_true_force = BMCR_SPEED10;
        }
        else if ((speed == LinkSpeed_10) && (duplex == LinkDuplex_Full))
        {
            bmcr_true_force = BMCR_SPEED10 | BMCR_FULLDPLX;
        }
        else if ((speed == LinkSpeed_100) && (duplex == LinkDuplex_Half))
        {
            bmcr_true_force = BMCR_SPEED100;
        }
        else if ((speed == LinkSpeed_100) && (duplex == LinkDuplex_Full))
        {
            bmcr_true_force = BMCR_SPEED100 | BMCR_FULLDPLX;
        }
        MdioWrite(0x0000, MII_BMCR, bmcr_true_force);
    }
}

uint16_t EthernetDriver::MapPhyOcpAddr(uint16_t PageNum, uint8_t RegNum)
{
    uint16_t OcpPageNum = 0;
    uint8_t OcpRegNum = 0;
    uint16_t OcpPhyAddress = 0;
    if (PageNum == 0)
    {
        OcpPageNum = OCP_STD_PHY_BASE_PAGE + (RegNum / 8);
        OcpRegNum = 0x10 + (RegNum % 8);
    }
    else
    {
        OcpPageNum = PageNum;
        OcpRegNum = RegNum;
    }
    OcpPageNum <<= 4;
    if (OcpRegNum < 16)
    {
        OcpPhyAddress = 0;
    }
    else
    {
        OcpRegNum -= 16;
        OcpRegNum <<= 1;

        OcpPhyAddress = OcpPageNum + OcpRegNum;
    }
    return OcpPhyAddress;
}

void EthernetDriver::MdioWrite(uint16_t PageNum, uint32_t RegAddr, uint32_t value)
{
    int32_t i;
    if (m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168G)
    {
        uint32_t data32;
        uint16_t ocp_addr = MapPhyOcpAddr(PageNum, RegAddr);
        data32 = ocp_addr / 2;
        data32 <<= OCPR_ADDR_REG_SHIFT;
        data32 |= OCPR_WRITE_MASK | value;
        RegWrite32(PHYOCP_OFFSET, data32);
        for (i = 0; i < 100; i++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(25));
            if (!(RegRead32(PHYOCP_OFFSET) & OCPR_FLAG_MASK))
            {
                break;
            }
        }
    }
    else if (m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168E ||
             m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168E_VL)
    {
        RegWrite32(PHYAR_OFFSET, 0x80000000 | (RegAddr & 0x1f) << 16 | (value & 0xffff));
        for (i = 0; i < 100; i++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(25));
            if (!(RegRead32(PHYAR_OFFSET) & 0x80000000))
            {
                break;
            }
        }

        /*
         * According to hardware specs a 20us delay is required after read
         * complete indication, but before sending next command.
         */
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(20));
    }
}

uint32_t EthernetDriver::MdioRead(uint16_t PageNum, uint32_t RegAddr)
{
    int32_t i;
    uint32_t value = 0;
    if (m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168G)
    {
        int32_t data32;
        uint16_t ocp_addr = MapPhyOcpAddr(PageNum, RegAddr);
        int32_t i;
        data32 = ocp_addr / 2;
        data32 <<= OCPR_ADDR_REG_SHIFT;
        RegWrite32(PHYOCP_OFFSET, data32);
        for (i = 0; i < 100; i++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(25));
            if (RegRead32(PHYOCP_OFFSET) & OCPR_FLAG_MASK)
            {
                value = RegRead32(PHYOCP_OFFSET) & OCPDR_DATA_MASK;
                break;
            }
        }
    }
    else if (m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168E ||
             m_ChipInfoList[m_ChipInfoIndex].version == TX_CFG_HWREV_8168E_VL)
    {
        RegWrite32(PHYAR_OFFSET, 0x0 | (RegAddr & 0x1f) << 16);
        for (i = 0; i < 100; i++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(25));
            if (RegRead32(PHYAR_OFFSET) & 0x80000000)
            {
                value = RegRead32(PHYAR_OFFSET) & 0xffff;
                break;
            }
        }

        /*
         * According to hardware specs a 20us delay is required after read
         * complete indication, but before sending next command.
         */
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(20));
    }

    return value;
}

uint32_t EthernetDriver::EriRead(int32_t addr, int32_t len, int32_t type)
{
    int32_t i, val_shift, shift = 0;
    uint32_t value1 = 0, value2 = 0, mask;
    uint32_t eri_cmd;
    if (len > 4 || len <= 0)
    {
        return -1;
    }
    while (len > 0)
    {
        val_shift = addr % ERIAR_ADDR_ALIGN;
        addr = addr & ~0x3;
        eri_cmd = ERIAR_READ_MASK |
            type << ERIAR_TYPE_SHIFT |
            ERIAR_BYTE_EN_MASK << ERIAR_BYTE_EN_SHIFT |
            (addr & 0x0FFF);
        if (addr & 0xF000)
        {
            uint32_t tmp;

            tmp = addr & 0xF000;
            tmp >>= 12;
            eri_cmd |= (tmp << 20) & 0x00F00000;
        }
        RegWrite32(ERIAR_OFFSET, eri_cmd);
        for (i = 0; i < 10; i++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
            if (RegRead32(ERIAR_OFFSET) & ERIAR_FLAG_MASK)
            {
                break;
            }
        }
        if (len == 1)
        {
            mask = (0xFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        else if (len == 2)
        {
            mask = (0xFFFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        else if (len == 3)
        {
            mask = (0xFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        else
        {
            mask = (0xFFFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        value1 = RegRead32(ERIDR_OFFSET) & mask;
        value2 |= (value1 >> val_shift * 8) << shift * 8;
        if (len <= 4 - val_shift)
        {
            len = 0;
        }
        else
        {
            len -= (4 - val_shift);
            shift = 4 - val_shift;
            addr += 4;
        }
    }
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(20));
    return value2;
}

int32_t EthernetDriver::EriWrite(int32_t addr, int32_t len, uint32_t value, int32_t type)
{
    int32_t i, val_shift, shift = 0;
    uint32_t value1 = 0, mask;
    uint32_t eri_cmd;
    if (len > 4 || len <= 0)
    {
        return -1;
    }
    while (len > 0)
    {
        val_shift = addr % ERIAR_ADDR_ALIGN;
        addr = addr & ~0x3;
        if (len == 1)
        {
            mask = (0xFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        else if (len == 2)
        {
            mask = (0xFFFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        else if (len == 3)
        {
            mask = (0xFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        else
        {
            mask = (0xFFFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;
        }
        value1 = EriRead(addr, 4, type) & ~mask;
        value1 |= ((value << val_shift * 8) >> shift * 8);
        RegWrite32(ERIDR_OFFSET, value1);
        eri_cmd = ERIAR_WRITE_MASK |
            type << ERIAR_TYPE_SHIFT |
            ERIAR_BYTE_EN_MASK << ERIAR_BYTE_EN_SHIFT |
            (addr & 0x0FFF);
        if (addr & 0xF000)
        {
            uint32_t tmp;
            tmp = addr & 0xF000;
            tmp >>= 12;
            eri_cmd |= (tmp << 20) & 0x00F00000;
        }
        RegWrite32(ERIAR_OFFSET, eri_cmd);
        for (i = 0; i < 10; i++)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
            if (!(RegRead32(ERIAR_OFFSET) & ERIAR_FLAG_MASK))
            {
                break;
            }
        }
        if (len <= 4 - val_shift)
        {
            len = 0;
        }
        else
        {
            len -= (4 - val_shift);
            shift = 4 - val_shift;
            addr += 4;
        }
    }
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(20));
    return 0;
}

void EthernetDriver::SampleCounters()
{
    RegWrite32(COUNTER_ADDR_LOW_OFFSET, RegRead32(COUNTER_ADDR_LOW_OFFSET) | COUNTER_DUMP_MASK);
    while (RegRead32(COUNTER_ADDR_LOW_OFFSET) & COUNTER_DUMP_MASK)
    {
        // Synchronize with MAC MCU
    }
    nn::dd::InvalidateDataCache(&m_DmaDesc.data.counters.data, sizeof(m_DmaDesc.data.counters.data));
}

void EthernetDriver::ResetHardware()
{
    int32_t i;
    bool completed;

    // Stop reception
    RegWrite32(RX_CONFIG_OFFSET, RegRead32(RX_CONFIG_OFFSET) & ~0x3f);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

    // Soft reset the chip
    RegWrite8(CHIP_CMD_OFFSET, CMD_RESET_MASK);

    // Wait for reset to complete
    for (i = 100,completed = false; i > 0; i--)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        if ((RegRead8(CHIP_CMD_OFFSET) & CMD_RESET_MASK) == 0)
        {
            completed = true;
            break;
        }
    }
    if (!completed)
    {
        ETH_LOG_INFO("EthernetDriver::ResetHardware() internal timeout on FIFO.\n");
    }
}

void EthernetDriver::RegisterInterfaceStateCallBack(InterfaceStateCallBack callback)
{
    Lock();
    m_InterfaceStateCallBack = callback;
    ReportInterfaceState();
    Unlock();
}

void EthernetDriver::UnregisterInterfaceStateCallBack()
{
    m_InterfaceStateCallBack = NULL;
}

void EthernetDriver::RegWrite8(size_t offset, uint8_t val8)
{
    nn::pcie::WriteBarRegion8(m_PcieDrvHandle, m_PcieFuncHandle, BAR, offset, val8);
}

void EthernetDriver::RegWrite16(size_t offset, uint16_t val16)
{
    nn::pcie::WriteBarRegion16(m_PcieDrvHandle, m_PcieFuncHandle, BAR, offset, val16);
}

void EthernetDriver::RegWrite32(size_t offset, uint32_t val32)
{
    nn::pcie::WriteBarRegion32(m_PcieDrvHandle, m_PcieFuncHandle, BAR, offset, val32);
}

uint8_t EthernetDriver::RegRead8(size_t offset)
{
    uint8_t val8 = 0;
    nn::pcie::ReadBarRegion8(&val8, m_PcieDrvHandle, m_PcieFuncHandle, BAR, offset);
    return val8;
}

uint16_t EthernetDriver::RegRead16(size_t offset)
{
    uint16_t val16 = 0;
    nn::pcie::ReadBarRegion16(&val16, m_PcieDrvHandle, m_PcieFuncHandle, BAR, offset);
    return val16;
}

uint32_t EthernetDriver::RegRead32(size_t offset)
{
    uint32_t val32 = 0;
    nn::pcie::ReadBarRegion32(&val32, m_PcieDrvHandle, m_PcieFuncHandle, BAR, offset);
    return val32;
}


} // namespace eth
} // amespace drivers
} // namespace nn
