﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SdkThread.h>
#include <nn/dd.h>
#include <cstring>
#include <mutex>
#include "eth_EthernetDriver.h"
#include <nn/svc/svc_MemoryMapSelect.h>
#include <alloca.h>

namespace nn { namespace drivers { namespace eth {
namespace {

const uint64_t EnetRegionAddr = 0x20000000;
const size_t   EnetRegionSize = 0x1000;
uintptr_t g_EnetBegin;

const uint64_t GioRegionAddr  = 0xE0050000;
const size_t   GioRegionSize  = 0x1000;
uint64_t g_GioBegin;

#define REG_ETH_RX_DATA          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x000))
#define REG_ETH_TX_DATA          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x020))
#define REG_ETH_RX_STATUS        (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x040))
#define REG_ETH_RX_STATUS_PEEK   (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x044))
#define REG_ETH_TX_STATUS        (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x048))
#define REG_ETH_TX_STATUS_PEEK   (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x04c))

#define REG_ETH_ID_REV           (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x050))
#define REG_ETH_IRQ_CFG          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x054))
#define REG_ETH_INT_STS          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x058))
#define REG_ETH_INT_EN           (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x05c))
#define REG_ETH_BYTE_TEST        (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x064))
#define REG_ETH_FIFO_INT         (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x068))
#define REG_ETH_RX_CFG           (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x06c))
#define REG_ETH_TX_CFG           (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x070))
#define REG_ETH_HW_CFG           (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x074))
#define REG_ETH_RX_DP_CTL        (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x078))
#define REG_ETH_RX_FIFO_INF      (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x07c))
#define REG_ETH_TX_FIFO_INF      (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x080))
#define REG_ETH_PMT_CTRL         (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x084))
#define REG_ETH_GPIO_CFG         (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x088))
#define REG_ETH_GPT_CFG          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x08c))
#define REG_ETH_GPT_CNT          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x090))
#define REG_ETH_WORD_SWAP        (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x098))
#define REG_ETH_FREE_RUN         (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x09c))
#define REG_ETH_RX_DROP          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x0a0))
#define REG_ETH_MAC_CSR_CMD      (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x0a4))
#define REG_ETH_MAC_CSR_DATA     (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x0a8))
#define REG_ETH_AFC_CFG          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x0ac))
#define REG_ETH_E2P_CMD          (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x0b0))
#define REG_ETH_E2P_DATA         (*reinterpret_cast<volatile uint32_t *>(g_EnetBegin + 0x0b4))

#define REG_GIO_E0          (*reinterpret_cast<volatile uint32_t *>(g_GioBegin + 0x0004))
#define REG_GIO_IIA         (*reinterpret_cast<volatile uint32_t *>(g_GioBegin + 0x0014))
#define REG_GIO_IEN         (*reinterpret_cast<volatile uint32_t *>(g_GioBegin + 0x0018))
#define REG_GIO_MST         (*reinterpret_cast<volatile uint32_t *>(g_GioBegin + 0x0024))
#define REG_GIO_IIR         (*reinterpret_cast<volatile uint32_t *>(g_GioBegin + 0x0028))
#define REG_GIO_IDT         (*reinterpret_cast<volatile uint32_t *>(g_GioBegin + 0x0040))

const int ETH_ID_REV_ID_SHIFT       = 16;
const uint32_t ETH_ID_REV_ID_MASK   = 0xffff;
const int ETH_ID_REV_REV_SHIFT      = 0;
const uint32_t ETH_ID_REV_REV_MASK  = 0xffff;

const int ETH_FIFO_INT_TX_DATA_LEVEL_SHIFT        = 24;
const int ETH_FIFO_INT_TX_STATUS_LEVEL_SHIFT      = 16;
const int ETH_FIFO_INT_RX_STATUS_LEVEL_SHIFT      = 0;


const int RX_DATA_ALIGN = 16;
const int TX_DATA_ALIGN = 16;

const uint32_t ETH_INT_EN_TDFO     = 1u << 10; // TX Data FIFO Overrun
const uint32_t ETH_INT_EN_TDFA     = 1u << 9;  // TX Data FIFO Available
const uint32_t ETH_INT_EN_TSFF     = 1u << 8;  // TX Status FIFO Full
const uint32_t ETH_INT_EN_TSFL     = 1u << 7;  // TX Status FIFO Level
const uint32_t ETH_INT_EN_RSFF     = 1u << 4;  // RX Status FIFO Full
const uint32_t ETH_INT_EN_RSFL     = 1u << 3;  // RX Status FIFO Level
const uint32_t ETH_INT_EN_GPIO2    = 1u << 2;  // GPIO2_INT_EN
const uint32_t ETH_INT_EN_GPIO1    = 1u << 1;  // GPIO1_INT_EN
const uint32_t ETH_INT_EN_GPIO0    = 1u << 0;  // GPIO0_INT_EN

const uint32_t ETH_INT_STS_RXE     = 1u << 14; // Receiver Error
const uint32_t ETH_INT_STS_TXE     = 1u << 13; // Transmitter Error
const uint32_t ETH_INT_STS_TDFA    = 1u << 9;  // TX Data FIFO Available
const uint32_t ETH_INT_STS_RSFL    = 1u << 3;  // RX Status FIFO Level

const uint32_t MAX_FRAME_BYTES     = 1514;  // FCSを除いた最大1514バイト

// 1フレーム分の受信バッファサイズ。4バイトアラインされた MAX_FRAME_BYTES バイトを前後の16バイト境界に伸長したときの最大サイズ。
const uint32_t RX_BUFFER_BYTES     = 1536;

// FIFO の分配
// ホストからの転送速度を上げるため、RX FIFO に多めに配分してみる。（効果は未検証）
// max frame length 1518 * 7 = 10626
const uint32_t TX_FIFO_SIZE         = 4096;
const uint32_t TX_DATA_FIFO_SIZE    = TX_FIFO_SIZE - 512;
const uint32_t RX_FIFO_SIZE         = 16384 - TX_FIFO_SIZE;
const uint32_t RX_DATA_FIFO_SIZE    = RX_FIFO_SIZE - (RX_FIFO_SIZE / 16);


const int MAC_CSR_MAC_CR    = 1;
const int MAC_CSR_ADDRH     = 2;
const int MAC_CSR_ADDRL     = 3;
const int MAC_CSR_HASHH     = 4;
const int MAC_CSR_HASHL     = 5;
const int MAC_CSR_MII_ACC   = 6;
const int MAC_CSR_MII_DATA  = 7;
const int MAC_CSR_FLOW      = 8;
const int MAC_CSR_VLAN1     = 9;
const int MAC_CSR_VLAN2     = 0xa;
const int MAC_CSR_WUFF      = 0xb;
const int MAC_CSR_WUCSR     = 0xc;

const int ETH_IREQ_PRIORITY = 3;

uint32_t read_MAC_CSR(uint32_t  csr)
{
    REG_ETH_MAC_CSR_CMD = (1u << 31) | (1u << 30) | csr;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(100000));
    } while (REG_ETH_MAC_CSR_CMD & (1u << 31));
    return REG_ETH_MAC_CSR_DATA;
}

void write_MAC_CSR(uint32_t csr, uint32_t value)
{
    REG_ETH_MAC_CSR_DATA = value;
    REG_ETH_MAC_CSR_CMD = (1u << 31) | (0u << 30) | csr;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(100000));
    } while (REG_ETH_MAC_CSR_CMD & (1u << 31));
}


const size_t STACK_SIZE  = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char g_stack[STACK_SIZE];

void InterruptThread(void* p)
{
    EthernetDriver* enet = reinterpret_cast<EthernetDriver*>(p);
    enet->HandleInterrupt();
}

int GetRequiredTxFifoBytes(const void* segmentData, int segmentBytes)
{
    // TODO: 4バイトアライメント以外の FrameData に対応する？（先頭もアラインする）

    // 4 バイトアライメント + TXCommandA&B
    return (segmentBytes + sizeof(uint32_t) - 1) / sizeof(uint32_t) * sizeof(uint32_t) + 8;
}

int GetRequiredTxFifoBytes(nnnetOslMbuf* pMbufHead)
{
    int totalRequiredBytes = 0;

    nnnetOslMbuf* pMbuf = pMbufHead;
    while (true)
    {
        void* segmentData = nnnetOslMbuf_tod(pMbuf);
        uint32_t segmentBytes = pMbufHead->m_len;
        totalRequiredBytes += GetRequiredTxFifoBytes(segmentData, segmentBytes);

        bool isLastSegment = (pMbuf->m_next == NULL);
        if (isLastSegment)
        {
            break;
        }
        pMbuf = pMbuf->m_next;
    }
    return totalRequiredBytes;
}

}

void EthernetDriver::Reset()
{
    write_MAC_CSR(MAC_CSR_MAC_CR, 0);

    REG_ETH_HW_CFG = 0x00180001;

    // ソフトリセットの完了を待つ（bit0 SRST が 0 に戻る）。データシートによると2us前後で完了する。
    while (REG_ETH_HW_CFG & 0x1)
    {
        nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds(1000) );
    }

    // EEPROM 読み取り完了（MACアドレス読み取り完了）を待つ
    while (REG_ETH_E2P_CMD & (1u << 31))
    {
        nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds(1000) );
    }

    REG_ETH_RX_CFG = 0x40000000;
    REG_ETH_HW_CFG = (1u << 20) | ((TX_FIFO_SIZE / 1024u) << 16);
    REG_ETH_TX_CFG = 0x0000c002;

    REG_ETH_INT_EN = ETH_INT_EN_RSFL | ETH_INT_STS_RXE ;
    REG_ETH_FIFO_INT = 0u;

    // 割り込みを有効化
    // IRQ_EN = 1, IRQ_POL = 1 (active high), IRQ_TYPE = 1 (Push-Pull)
    REG_ETH_IRQ_CFG = 0x00000111;

    write_MAC_CSR(MAC_CSR_MAC_CR, (1u << 20) | (1u << 8) | (1u << 3) | (1u << 2));

    REG_ETH_GPIO_CFG = 0x00000702;
}

nn::Result EthernetDriver::Initialize()
{
    g_EnetBegin = nn::dd::QueryIoMappingAddress(EnetRegionAddr, EnetRegionSize);
    g_GioBegin = nn::dd::QueryIoMappingAddress(GioRegionAddr, GioRegionSize);
    NN_ABORT_UNLESS(g_EnetBegin != 0);
    NN_ABORT_UNLESS(g_GioBegin != 0);

    REG_GIO_E0 = 1U << 1; // 方向をinput にする。
    REG_GIO_IIA &= ~(1U << 1); // 一旦割り込みを不許可にする
    REG_GIO_IDT &= ~(0xf << 4); // モードをクリア
    REG_GIO_IDT |= (0x2) << 4; // High Level モードに設定
    REG_GIO_IIR = 1U << 1; // 割り込みフラグをクリア
    REG_GIO_IIA |= 1U << 1; // 割り込みの検知を有効にする
    REG_GIO_IEN = 1U << 1; // 割り込み発行を許可する

    m_TxMutex.Initialize();
    m_ConditionVariable.Initialize();

    m_RequestedFinalize = false;
    m_SendCompletedCallBack = NULL;
    m_FrameReceivedCallBack = NULL;
    m_PacketTag = 0;

    Reset();

    nn::os::CreateThread(&m_InterruptTthread, &InterruptThread, this, g_stack, STACK_SIZE, nn::os::HighestSystemThreadPriority);
    nn::os::StartThread(&m_InterruptTthread);

    // Phy Reset & Auto-negotiation
    StartAutoNegotiation();

    return ResultSuccess();
}

nn::Result EthernetDriver::Finalize()
{
    REG_GIO_IIA &=  ~(1U << 1); // 割り込みを不許可にする
    REG_GIO_IIR = 1U << 1; // 割り込みフラグをクリア
    REG_GIO_IDT &= ~(0xf << 4); // モードをクリア

    m_RequestedFinalize = true;
    nn::os::WaitThread(&m_InterruptTthread);
    nn::os::DestroyThread(&m_InterruptTthread);

    REG_ETH_TX_CFG = 0x00000003;

    // TXEN = 0, RXEN = 0
    write_MAC_CSR(MAC_CSR_MAC_CR, read_MAC_CSR(MAC_CSR_MAC_CR) & ~static_cast<uint32_t>(0x0000000c));

    m_TxMutex.Finalize();
    m_ConditionVariable.Finalize();

    return ResultSuccess();
}

nn::Result EthernetDriver::Send(const void* frameData, int frameBytes)
{
    if (frameBytes > MAX_FRAME_BYTES)
    {
        return nn::os::ResultInvalidParameter();
    }

    {
        std::lock_guard<InternalMutex> lock(m_TxMutex);

        int requiredFifoBytes = GetRequiredTxFifoBytes(frameData, frameBytes);

        while ((REG_ETH_TX_FIFO_INF & 0x0000ffff) < requiredFifoBytes)
        {
            // 64 Byte ブロックの必要数
            uint32_t requiredFifoBlocks = (requiredFifoBytes + 64 - 1) / 64;
            REG_ETH_INT_EN   |= ETH_INT_EN_TDFA;
            REG_ETH_FIFO_INT = (requiredFifoBlocks << ETH_FIFO_INT_TX_DATA_LEVEL_SHIFT);

            m_ConditionVariable.Wait(&m_TxMutex);
        }

        while (REG_ETH_TX_FIFO_INF & 0x00ff0000)
        {
            uint32_t status = REG_ETH_TX_STATUS;
            NN_UNUSED(status);
        }

        REG_ETH_INT_EN   &= ~ETH_INT_EN_TDFA;
        REG_ETH_FIFO_INT = 0;

        REG_ETH_TX_DATA = (0u << 24) | (1u << 13) | (1u << 12) | frameBytes;
        REG_ETH_TX_DATA = (static_cast<uint32_t>(m_PacketTag) << 16) | frameBytes;

        for (int i = 0; i < (frameBytes + sizeof(uint32_t) - 1) / sizeof(uint32_t); i ++)
        {
            REG_ETH_TX_DATA = static_cast<const uint32_t*>(frameData)[i];
        }
        m_PacketTag++;
    }

    SendCompletedCallBack callback = m_SendCompletedCallBack;
    if (callback)
    {
        callback(const_cast<void*>(frameData));
    }

    return ResultSuccess();
}

nn::Result EthernetDriver::Send(nnnetOslMbuf* pMbufHead)
{
    int frameBytes = nnnetOslMbuf_length(pMbufHead, NULL);
    if (frameBytes > MAX_FRAME_BYTES)
    {
        return nn::os::ResultInvalidParameter();
    }

    {
        std::lock_guard<InternalMutex> lock(m_TxMutex);
        int requiredFifoBytes = GetRequiredTxFifoBytes(pMbufHead);

        while ((REG_ETH_TX_FIFO_INF & 0x0000ffff) < requiredFifoBytes)
        {
            // 64 Byte ブロックの必要数
            uint32_t requiredFifoBlocks = (requiredFifoBytes + 64 - 1) / 64;
            REG_ETH_INT_EN   |= ETH_INT_EN_TDFA;
            REG_ETH_FIFO_INT = (requiredFifoBlocks << ETH_FIFO_INT_TX_DATA_LEVEL_SHIFT);

            m_ConditionVariable.Wait(&m_TxMutex);
        }

        while (REG_ETH_TX_FIFO_INF & 0x00ff0000)
        {
            uint32_t status = REG_ETH_TX_STATUS;
            NN_UNUSED(status);
        }

        REG_ETH_INT_EN   &= ~ETH_INT_EN_TDFA;
        REG_ETH_FIFO_INT = 0;

        nnnetOslMbuf* pMbuf = pMbufHead;
        for (;;)
        {
            void* segmentData = nnnetOslMbuf_tod(pMbuf);
            int  segmentBytes = pMbuf->m_len;
            bool isFirstSegment = (pMbuf == pMbufHead);
            bool isLastSegment  = (pMbuf->m_next == NULL);

            REG_ETH_TX_DATA = (0u << 24) | (static_cast<uint32_t>(isFirstSegment) << 13) | (static_cast<uint32_t>(isLastSegment) << 12) | segmentBytes;
            REG_ETH_TX_DATA = (static_cast<uint32_t>(m_PacketTag) << 16) | frameBytes;

            for (int i = 0; i < (segmentBytes + sizeof(uint32_t) - 1) / sizeof(uint32_t); i ++)
            {
                REG_ETH_TX_DATA = static_cast<const uint32_t*>(segmentData)[i];
            }

            if (isLastSegment)
            {
                break;
            }
            pMbuf = pMbuf->m_next;
        }

        m_PacketTag++;
    }

    SendCompletedCallBack callback = m_SendCompletedCallBack;
    if (callback)
    {
        callback(const_cast<nnnetOslMbuf*>(pMbufHead));
    }

    return ResultSuccess();
}

void EthernetDriver::HandleInterrupt()
{
    // 割り込みの設定がうまくいかないので、割り込み関連の処理は一時的にコメントアウト
    /*
    Result result;
    nn::svc::Handle  interruptHandle;
    result = nn::dd::CreateInterruptEvent(&interruptHandle, nn::svc::Interrupt::INTERRUPT_ETH, true);
    NN_ABORT_UNLESS(result.IsSuccess(), "割り込み用のEvent作成に失敗");
    */


    for (;;)
    {
        /*
        nn::dd::WaitEvent(interruptHandle);
        std::lock_guard<InternalMutex> lock(m_TxMutex);
        */
        // 割り込みフラグが立つまでスリープして待つ
        while((REG_GIO_MST & 0x00000002) == 0)
        {
            nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(1000000));
        }
        // 割り込みクリア
        uint32_t interruptStatus = REG_ETH_INT_STS;
        REG_ETH_INT_STS = interruptStatus;

        Receive();
        TxReady(interruptStatus);


        /*
        nn::dd::ClearEvent(interruptHandle);
        */
    }

    /*
    result = nn::dd::DestroyEvent(interruptHandle);
    NN_ABORT_UNLESS(result.IsSuccess(), "割り込み用のEvent削除に失敗");
    */
}

void EthernetDriver::TxReady(uint32_t interruptStatus)
{
    if (interruptStatus & ETH_INT_EN_TDFA)
    {
        m_ConditionVariable.Broadcast();
    }
}

void EthernetDriver::Receive()
{
    uint32_t *buf = static_cast<uint32_t*>(alloca(((MAX_FRAME_BYTES + RX_DATA_ALIGN - 1) / RX_DATA_ALIGN) * RX_DATA_ALIGN));
    while ((REG_ETH_RX_FIFO_INF >> 16) & 0xFF)
    {
        uint32_t status = REG_ETH_RX_STATUS;
        uint32_t frameBytes  = (status >> 16) & 0x3fff;

        if ((status & (1<<15)) || (frameBytes > MAX_FRAME_BYTES))
        {
#if 0
            REG_ETH_RX_DP_CTL = (0x1 << 31);
            do
            {
                nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(1000000));
            } while (REG_ETH_RX_DP_CTL & (0x1 << 31));
#else
            uint32_t saveIntEn = REG_ETH_INT_EN;
            uint32_t saveFifoInt = REG_ETH_FIFO_INT;
            Reset();
            REG_ETH_FIFO_INT = saveFifoInt;
            REG_ETH_INT_EN = saveIntEn;
            nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(1000000));
            m_ConditionVariable.Broadcast();
            return;
#endif
        }
        else
        {
            // RX_DATA_ALIGN バイトアライメント
            uint32_t frameAlignedBytes = ((frameBytes + RX_DATA_ALIGN - 1) / RX_DATA_ALIGN) * RX_DATA_ALIGN;
            uint32_t frameDwords = frameAlignedBytes / sizeof(uint32_t);

            for (int i = 0; i < frameDwords; i ++)
            {
                buf[i] = REG_ETH_RX_DATA;
            }

            FrameReceivedCallBack callback = m_FrameReceivedCallBack;
            if (callback)
            {
                callback(&buf[0], frameBytes);
            }
        }
    }
}

nn::Result EthernetDriver::GetMacAddress(uint8_t pMacAddress[NN_NET_MAC_ADDRESS_SIZE]) const
{
    uint32_t lo = read_MAC_CSR(MAC_CSR_ADDRL);
    uint32_t hi = read_MAC_CSR(MAC_CSR_ADDRH);
    pMacAddress[0] = (lo >>  0) & 0xff;
    pMacAddress[1] = (lo >>  8) & 0xff;
    pMacAddress[2] = (lo >> 16) & 0xff;
    pMacAddress[3] = (lo >> 24) & 0xff;
    pMacAddress[4] = (hi >>  0) & 0xff;
    pMacAddress[5] = (hi >>  8) & 0xff;

    return ResultSuccess();
}

uint16_t EthernetDriver::PhyRead(uint8_t addr)
{
    write_MAC_CSR(MAC_CSR_MII_ACC, (addr << 11) | 0x00000001);
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(100000));
    } while (read_MAC_CSR(MAC_CSR_MII_ACC) & (1u << 0));

    return read_MAC_CSR(MAC_CSR_MII_DATA);
}

void EthernetDriver::PhyWrite(uint8_t addr, uint16_t value)
{
    write_MAC_CSR(MAC_CSR_MII_DATA, value);
    write_MAC_CSR(MAC_CSR_MII_ACC, (addr << 11) | 0x00000003);
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(100000));
    } while (read_MAC_CSR(MAC_CSR_MII_ACC) & (1u << 0));
}

void EthernetDriver::StartAutoNegotiation()
{
    PhyWrite(0, 0x8000);
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1) );

    PhyWrite(4, 0x0100 | 0x0080 | 0x0040 | 0x0020 | 0x0001);
    PhyWrite(0, 0x1200);

    // Auto-Negotiation 完了待ち (最大10秒)
    // Link Up も同時に待つ。
    // Auto-Negotiation 完了直後で Link Up前に送信を開始すると
    // Link Upに失敗することがある。
    for (int i = 0; i < 100; i++)
    {
        uint16_t bmsr = PhyRead(1);
        if ((bmsr & 0x24) == 0x24)
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
    }
}

void EthernetDriver::DumpRegister()
{
}

}}}
