﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/diag/diag_LogObserver.h>
#include "diag_LogImpl.h"
#include <cstdarg>
#include <algorithm>
#include "detail/diag_PrintDebugString.h"

namespace nn { namespace diag {

namespace {

inline void GetUptime(int* hours, int* minutes, int* seconds, int* milliSeconds) NN_NOEXCEPT
{
    const nn::TimeSpan now = nn::os::GetSystemTick().ToTimeSpan();

    const int64_t totalHours = now.GetHours();
    const int64_t totalMinutes = now.GetMinutes();
    const int64_t totalSeconds = now.GetSeconds();
    const int64_t totalMilliSeconds = now.GetMilliSeconds();

    // 除算器のないハードウェアでも高速に動作するように、除算せずに計算する。
    *hours = static_cast<int>(totalHours);
    *minutes = static_cast<int>(totalMinutes - totalHours * 60);
    *seconds = static_cast<int>(totalSeconds - totalMinutes * 60);
    *milliSeconds = static_cast<int>(totalMilliSeconds - totalSeconds * 1000);
}

const size_t DecorationStringLengthMax = 32u;

#if !defined(NN_BUILD_CONFIG_OS_WIN)
const char* EscapeSequencesForSeverity[] =
    {
        "\033[90m",                             // Trace: Dark gray,
        nullptr,                                // Info:  White (terminal default),
        "\033[33m",                             // Warn:  Yellow,
        "\033[31m",                             // Error: Red,
        "\033[41m\033[37m",                     // Fatal: White with Red background
    };
#endif

const char* EscapeSequenceReset = "\033[0m";  // Reset

// 正式なデフォルトログオブザーバが実装されるまでの、暫定実装
void TentativeDefaultLogObserver(
    const LogMetaData& logMetaData,
    const LogBody& logBody,
    void* /*argument*/
) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Windows 版はログが分割されない十分な長さのバッファを利用している。
    // そのため、ログが分割されることはない。
    NN_SDK_ASSERT(logBody.isHead && logBody.isTail);
#endif

    const char* message;
    int messageBytes = 0;

#if defined(NN_BUILD_CONFIG_OS_WIN)
    const size_t PrintfBufferLength = logBody.messageBytes + DecorationStringLengthMax;
    char* printfBuffer = reinterpret_cast<char*>(std::malloc(PrintfBufferLength));
    NN_ABORT_UNLESS(printfBuffer != nullptr);
    NN_UTIL_SCOPE_EXIT { std::free(printfBuffer); };
#else
    // 出力時にどうせ直列化されるのでスタックではなく共有バッファを使う
    NN_FUNCTION_LOCAL_STATIC(nn::os::MutexType, s_PrintfBufferMutex, = NN_OS_MUTEX_INITIALIZER(false));
    nn::os::LockMutex(&s_PrintfBufferMutex);
    NN_UTIL_SCOPE_EXIT { nn::os::UnlockMutex(&s_PrintfBufferMutex); };

    const size_t PrintfBufferLength = detail::DebugPrintfBufferLength + DecorationStringLengthMax;
    NN_FUNCTION_LOCAL_STATIC(char, printfBuffer[PrintfBufferLength]);
#endif

    const char* pEscapeSequence = nullptr;
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // Severity に対応した色を出力するためのエスケープシーケンスを選択
    if (diag::LogSeverity_Trace <= logMetaData.severity && logMetaData.severity <= LogSeverity_Fatal)
    {
        pEscapeSequence = EscapeSequencesForSeverity[logMetaData.severity];
    }
#endif

    // モジュール名が二文字以上なら構造化ログを使っているとみなす
    const bool isStructured = logMetaData.moduleName && std::strlen(logMetaData.moduleName) >= 2;
    if (isStructured || pEscapeSequence)
    {
        // 構造化ログの最初なら uptime とモジュール名を付与
        if (isStructured && logBody.isHead)
        {
            int hours, minutes, seconds, milliSeconds;
            GetUptime(&hours, &minutes, &seconds, &milliSeconds);

            messageBytes += nn::util::SNPrintf(
                &printfBuffer[messageBytes], PrintfBufferLength - messageBytes,
                "%s" "%d:%02d:%02d.%03d" " [%-5s] ",
                pEscapeSequence ? pEscapeSequence : "",
                hours, minutes, seconds, milliSeconds,
                logMetaData.moduleName[0] == '$' ? &logMetaData.moduleName[1] : &logMetaData.moduleName[0]
            );
        }
        else if (pEscapeSequence)
        {
            // 最初でないならエスケープシーケンスのみをセット（他のプロセスの割り込みに備えて毎出力ごとにリセット）
            messageBytes += nn::util::SNPrintf(
                &printfBuffer[messageBytes], PrintfBufferLength - messageBytes,
                "%s", pEscapeSequence
            );
        }

        // リセット用のエスケープシーケンスが途切れないように格納可能サイズを計算
        const size_t maxMessageLength =
            PrintfBufferLength - messageBytes - (pEscapeSequence ? sizeof(EscapeSequenceReset) - 1: 0);

        // 実際に格納するメッセージのサイズを計算
        size_t messageBytesToPut = std::min<size_t>(logBody.messageBytes, maxMessageLength);

        // 色反転エスケープシーケンスのリセットのために最後の改行は特別扱いする
        bool bPendingNewLine = false;
        if (logBody.messageBytes > 0 && logBody.message[logBody.messageBytes - 1] == '\n')
        {
            --messageBytesToPut;
            bPendingNewLine = true;
        }

        // メッセージを追記
        messageBytes += nn::util::SNPrintf(
            &printfBuffer[messageBytes], PrintfBufferLength - messageBytes,
            "%.*s%s%s",
            messageBytesToPut, logBody.message,
            pEscapeSequence ? EscapeSequenceReset : "",
            bPendingNewLine ? "\n" : ""
        );
        message     = printfBuffer;
    }
    else
    {
        // 構造化ログでない出力は以前の処理にフォールバック
        message      = logBody.message;
        messageBytes = static_cast<int>(logBody.messageBytes);
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    if (logMetaData.useDefaultLocaleCharset)
    {
        detail::PrintDebugString(message, messageBytes);
    }
    else
    {
        detail::PrintDebugStringWithConversion(message, messageBytes);
    }
#else
    detail::PrintDebugString(message, messageBytes);
#endif
}

// デフォルトログオブザーバのホルダ
LogObserverHolder g_DefaultLogObserverHolder = {
    TentativeDefaultLogObserver, // logObserver
    NULL,                        // next
    true                         // isRegistered
};

} // anonymous namespace

namespace detail {

// ログオブザーバのリンクリストの先頭
// デフォルトログオブザーバを設定しておく
LogObserverHolder* g_LogObserverListHead = { &g_DefaultLogObserverHolder };

void CallAllLogObserver(
    const LogMetaData& logMetaData,
    const LogBody& logBody) NN_NOEXCEPT
{
    for(LogObserverHolder* holder = g_LogObserverListHead; holder != NULL; holder = holder->next)
    {
        holder->logObserver(logMetaData, logBody, holder->argument);
    }
}

void ReplaceDefaultLogObserver(LogObserver logObserver) NN_NOEXCEPT
{
    UnregisterLogObserver(&g_DefaultLogObserverHolder);
    InitializeLogObserverHolder(&g_DefaultLogObserverHolder, logObserver, NULL);
    RegisterLogObserver(&g_DefaultLogObserverHolder);
}

void ResetDefalutLogObserver() NN_NOEXCEPT
{
    UnregisterLogObserver(&g_DefaultLogObserverHolder);
    InitializeLogObserverHolder(&g_DefaultLogObserverHolder, TentativeDefaultLogObserver, NULL);
    RegisterLogObserver(&g_DefaultLogObserverHolder);
}

} // namespace detail

void InitializeLogObserverHolder(
    LogObserverHolder* logObserverHolder,
    LogObserver logObserver,
    void* argument) NN_NOEXCEPT
{
    logObserverHolder->logObserver = logObserver;
    logObserverHolder->next = NULL;
    logObserverHolder->isRegistered = false;
    logObserverHolder->argument = argument;
}

void RegisterLogObserver(LogObserverHolder* logObserverHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!logObserverHolder->isRegistered, "logObserverHolder is already registered");

    if (detail::g_LogObserverListHead == NULL)
    {
        detail::g_LogObserverListHead = logObserverHolder;
    }
    else
    {
        for (LogObserverHolder* holder = detail::g_LogObserverListHead;
            holder != NULL; holder = holder->next)
        {
            if (holder->next == NULL)
            {
                holder->next = logObserverHolder;
                break;
            }
        }
    }

    logObserverHolder->next = NULL;
    logObserverHolder->isRegistered = true;
}

void UnregisterLogObserver(LogObserverHolder* logObserverHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT(logObserverHolder->isRegistered, "logObserverHolder is not registered");

    if (detail::g_LogObserverListHead == logObserverHolder)
    {
        detail::g_LogObserverListHead = detail::g_LogObserverListHead->next;
    }
    else
    {
        for (LogObserverHolder* holder = detail::g_LogObserverListHead;
            holder != NULL; holder = holder->next)
        {
            if (holder->next == logObserverHolder)
            {
                holder->next = logObserverHolder->next;
                break;
            }
        }
    }

    logObserverHolder->isRegistered = false;
}

}}
