﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

#include <nn/diag/diag_AssertionTypes.h>
#include <nn/diag/diag_AbortTypes.h>
#include <nn/diag/diag_LogTypes.h>
#include <nn/diag/diag_AssertionFailureHandler.h>

#include "detail/diag_Abort.h"

namespace nn { namespace diag {

namespace detail {
    void InvokeAbortObserver(const AbortInfo& abortInfo) NN_NOEXCEPT;
}

namespace {
    AbortReason ToAbortReason(
        AssertionType assertionType) NN_NOEXCEPT
    {
        switch( assertionType )
        {
        case AssertionType_SdkAssert:
            return AbortReason_SdkAssert;
        case AssertionType_SdkRequires:
            return AbortReason_SdkRequires;
        case AssertionType_UserAssert:
            return AbortReason_UserAssert;
        default:
            return AbortReason_Abort;
        }
    }

    AssertionFailureOperation DefaultAssertionFailureHandler(
        const AssertionInfo &assertionInfo) NN_NOEXCEPT
    {
        NN_UNUSED(assertionInfo);
        return AssertionFailureOperation_Abort;
    }

    AssertionFailureHandler g_AssertionFailureHandler =
        &DefaultAssertionFailureHandler;

    void ExecuteAssertionFailureOperation(
        AssertionFailureOperation operation,
        const AssertionInfo& assertionInfo) NN_NOEXCEPT
    {
        switch( operation )
        {
        case AssertionFailureOperation_Continue:
            break;
        case AssertionFailureOperation_Abort:
            {
                const AbortInfo abortInfo = {
                    ToAbortReason(assertionInfo.assertionType),
                    assertionInfo.message,
                    assertionInfo.condition,
                    assertionInfo.functionName,
                    assertionInfo.fileName,
                    assertionInfo.lineNumber
                };
                detail::InvokeAbortObserver(abortInfo);
                detail::Abort(nullptr);
            }
            break;
        default:
            NN_ABORT_UNLESS(false, "Unknown AssertionFailureOperation");
            break;
        };
    }

    void InvokeAssertionFailureHandler(
        const AssertionInfo& assertionInfo) NN_NOEXCEPT
    {
        AssertionFailureOperation operation = g_AssertionFailureHandler(assertionInfo);
        ExecuteAssertionFailureOperation(operation, assertionInfo);
    }
} // anonymous

namespace detail {
    void OnAssertionFailure(
        AssertionType assertionType,
        const char* condition,
        const char* functionName,
        const char* fileName,
        int lineNumber,
        const char* format,
        ...) NN_NOEXCEPT
    {
        std::va_list list;
        va_start(list, format);

        const nn::diag::LogMessage message = {format, &list};
        const AssertionInfo assertionInfo = {
            assertionType,
            &message,
            condition,
            functionName,
            fileName,
            lineNumber
        };
        InvokeAssertionFailureHandler(assertionInfo);

        va_end(list);
    }

    void OnAssertionFailure(
        AssertionType assertionType,
        const char* condition,
        const char* functionName,
        const char* fileName,
        int lineNumber) NN_NOEXCEPT
    {
        OnAssertionFailure(
            assertionType,
            condition,
            functionName,
            fileName,
            lineNumber,
            "");
    }
} // detail

void SetAssertionFailureHandler(
    AssertionFailureHandler assertionFailureHandler) NN_NOEXCEPT
{
    g_AssertionFailureHandler = assertionFailureHandler;
}

}} // nn::diag
