﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/rocrt/rocrt.h>
#include <nn/ro/detail/ro_RoExceptionInfo.h>
#include "diag_SymbolTableParser-os.horizon.h"

namespace nn { namespace diag { namespace detail {

bool SymbolTableParser::Initialize(uintptr_t address) NN_NOEXCEPT
{
    if (address == 0u)
    {
        return false;
    }

    ro::detail::ExceptionInfo exceptionInfo;
    if (!ro::detail::GetExceptionInfo(&exceptionInfo, address))
    {
        return false;
    }

    auto moduleAddress = exceptionInfo.moduleAddress;
    auto pLocation = reinterpret_cast<const rocrt::ModuleHeaderLocation*>(moduleAddress);
    auto pHeader = reinterpret_cast<const rocrt::ModuleHeader*>(moduleAddress + pLocation->headerOffset);
    if (pHeader->signature != rocrt::ModuleHeaderVersion)
    {
        return false;
    }

    auto pDyn = reinterpret_cast<const ro::detail::Elf::Dyn*>(moduleAddress + pHeader->dynamicOffset + pLocation->headerOffset);
    const ro::detail::Elf::Sym* pSymTab = nullptr;
    const char* pStrTab = nullptr;
    int count = 0;
    for (; pDyn->GetTag() != ro::detail::Elf::DT_NULL; pDyn++)
    {
        if (pDyn->GetTag() == ro::detail::Elf::DT_SYMTAB)
        {
            pSymTab = reinterpret_cast<const ro::detail::Elf::Sym*>(moduleAddress + pDyn->GetPtr());
        }
        else if (pDyn->GetTag() == ro::detail::Elf::DT_STRTAB)
        {
            pStrTab = reinterpret_cast<const char*>(moduleAddress + pDyn->GetPtr());
        }
        else if (pDyn->GetTag() == ro::detail::Elf::DT_HASH)
        {
            auto pHash = reinterpret_cast<const uint32_t*>(moduleAddress + pDyn->GetPtr());
            count = static_cast<int>(pHash[1]); // ハッシュエントリの 4 バイト目にシンボル数が格納されている
        }
    }
    if (!(pSymTab != nullptr && pStrTab != nullptr && count > 0))
    {
        return false;
    }

    m_pSymTab = pSymTab;
    m_pStrTab = pStrTab;
    m_ModuleAddress = moduleAddress;
    m_Count = count;
    m_Index = -1;

    return true;
}

bool SymbolTableParser::MoveNext() NN_NOEXCEPT
{
    if (!(m_Index + 1 < m_Count))
    {
        return false;
    }
    m_Index++;
    return true;
}

void SymbolTableParser::MoveTo(int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(index, 0, m_Count);
    m_Index = index;
}

int SymbolTableParser::GetIndex() const NN_NOEXCEPT
{
    return m_Index;
}

int SymbolTableParser::GetCount() const NN_NOEXCEPT
{
    return m_Count;
}

const char* SymbolTableParser::GetName() const NN_NOEXCEPT
{
    return m_pStrTab + m_pSymTab[m_Index].GetName();
}

size_t SymbolTableParser::GetSize() const NN_NOEXCEPT
{
    return m_pSymTab[m_Index].GetSize();
}

uintptr_t SymbolTableParser::GetAddress() const NN_NOEXCEPT
{
    return m_ModuleAddress + m_pSymTab[m_Index].GetValue();
}

unsigned char SymbolTableParser::GetType() const NN_NOEXCEPT
{
    return m_pSymTab[m_Index].GetType();
}

}}} // nn::diag::detail
