﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/rocrt/rocrt.h>
#include <nn/ro/detail/ro_RoExceptionInfo.h>
#include <nn/svc/svc_Base.h>
#include <nn/diag/diag_Module.h>

namespace nn { namespace diag { namespace detail {

bool IsSystemSymbolImpl(uintptr_t address) NN_NOEXCEPT;
uintptr_t GetAddressFromSymbolName(uintptr_t address, const char* symbolName) NN_NOEXCEPT;

namespace {
    // const uintptr_t ModuleVersionOffset = 0; // 未使用
    const uintptr_t ModulePathLengthOffset = 4;
    const uintptr_t ModulePathOffset = 8;
}

uintptr_t GetModuleInfoForHorizon(const char** pOutPath, size_t* pOutLength, size_t* outModuleSize, uintptr_t address) NN_NOEXCEPT
{
    if (address == 0u)
    {
        return 0u;
    }

    ro::detail::ExceptionInfo exceptionInfo;
    if (!ro::detail::GetExceptionInfo(&exceptionInfo, address))
    {
        return 0u;
    }

    // メモリブロックを読み飛ばしながら、.ro セクションの先頭を検索する
    auto baseAddress = exceptionInfo.moduleAddress;
    svc::MemoryInfo memoryInfo;
    svc::PageInfo pageInfo;
    while (baseAddress < exceptionInfo.moduleAddress + exceptionInfo.moduleSize)
    {
        auto result = svc::QueryMemory(&memoryInfo, &pageInfo, baseAddress);
        if (result.IsFailure())
        {
            return 0u;
        }
        if (memoryInfo.permission != svc::MemoryPermission_ReadExecute)
        {
            break;
        }
        baseAddress += memoryInfo.size;
    }

    *pOutLength = *reinterpret_cast<const uint32_t*>(baseAddress + ModulePathLengthOffset);
    *pOutPath = reinterpret_cast<const char*>(baseAddress + ModulePathOffset);
    *outModuleSize = exceptionInfo.moduleSize;

    return exceptionInfo.moduleAddress;
}

uintptr_t GetModulePathImpl(char* outPathBuffer, size_t pathBufferSize, uintptr_t address) NN_NOEXCEPT
{
    const char* path;
    size_t pathLength;
    size_t moduleSize;

    auto baseAddress = GetModuleInfoForHorizon(&path, &pathLength, &moduleSize, address);
    if (baseAddress == 0u)
    {
        return 0u;
    }

    auto copyLength = std::min(pathLength, pathBufferSize - 1);
    std::memcpy(outPathBuffer, path, copyLength);
    outPathBuffer[copyLength] = '\0';

    return baseAddress;
}

int GetAllModuleInfoImpl(size_t* outRequiredBufferSize, void* outBuffer, size_t bufferSize, bool skipSave) NN_NOEXCEPT
{
    int moduleCount = 0;
    size_t requiredBufferSize = 0;
    auto pInfo = reinterpret_cast<ModuleInfo*>(outBuffer);
    auto pPath = reinterpret_cast<char*>(outBuffer) + bufferSize; // パス文字列はバッファの後ろから詰める。

    for (uintptr_t address = 0;;)
    {
        svc::MemoryInfo memoryInfo;
        svc::PageInfo pageInfo;

        auto result = svc::QueryMemory(&memoryInfo, &pageInfo, address);
        if (result.IsFailure())
        {
            NN_SDK_ASSERT(false);
            break;
        }

        if (memoryInfo.permission == svc::MemoryPermission_ReadExecute)
        {
            const char* path;
            size_t pathLength;
            size_t moduleSize;

            const auto baseAddress = GetModuleInfoForHorizon(&path, &pathLength, &moduleSize, address);
            if (baseAddress == 0u)
            {
                NN_SDK_ASSERT(false);
                break;
            }

            requiredBufferSize += sizeof(ModuleInfo) + pathLength + 1;

            if (!skipSave)
            {
                if (requiredBufferSize > bufferSize)
                {
                    break;
                }

                pPath -= pathLength + 1;
                std::memcpy(pPath, path, pathLength);
                pPath[pathLength] = '\0';

                pInfo->path = pPath;
                pInfo->baseAddress = baseAddress;
                pInfo->size = moduleSize;
                pInfo++;
            }

            moduleCount++;
        }

        const uintptr_t nextAddress = memoryInfo.baseAddress + memoryInfo.size;
        if (nextAddress <= address)
        {
            break; // 一周してオーバーフローしたら終わり。
        }
        address = nextAddress;
    }

    *outRequiredBufferSize = requiredBufferSize;
    return moduleCount;
}

bool GetReadOnlyDataSectionRangeImpl(uintptr_t* outStartAddress, uintptr_t* outEndAddress, uintptr_t baseAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outStartAddress);
    NN_SDK_REQUIRES_NOT_NULL(outEndAddress);

    if (IsSystemSymbolImpl(baseAddress))
    {
        return false;
    }

    auto getStartAddress = GetAddressFromSymbolName(baseAddress, "nndetailRoGetRoDataStart");
    if (!getStartAddress)
    {
        return false;
    }

    auto getEndAddress = GetAddressFromSymbolName(baseAddress, "nndetailRoGetRoDataEnd");
    if (!getEndAddress)
    {
        return false;
    }

    using addressGetter = void* (*)();
    *outStartAddress = reinterpret_cast<uintptr_t>((*reinterpret_cast<addressGetter>(getStartAddress))());
    *outEndAddress = reinterpret_cast<uintptr_t>((*reinterpret_cast<addressGetter>(getEndAddress))());

    return true;
}

}}} // nn::diag::detail
