﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cinttypes>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/diag_Backtrace.h>
#include "diag_DumpStackTrace.h"

namespace nn { namespace diag {

bool IsSystemSymbol(uintptr_t address) NN_NOEXCEPT;
void PrintModules(uintptr_t* traces, int traceCount) NN_NOEXCEPT;

namespace detail {

uintptr_t GetNearestExportedSymbol(const char** pOutSymbolName, size_t* outSymbolSize, uintptr_t address) NN_NOEXCEPT;
uintptr_t GetModuleInfoForHorizon(const char** pOutPath, size_t* pOutLength, size_t* outModuleSize, uintptr_t address) NN_NOEXCEPT;
void PrintBacktraceItemImpl(uintptr_t address, bool showDatail) NN_NOEXCEPT;

namespace {
    // システムシンボルは、セキュリティのために詳細表示しない。
    // ただし、ユーザから直接コールされている場合は詳細表示する。
    void PrintBacktrace(uintptr_t* traces, int traceCount) NN_NOEXCEPT
    {
        NN_SDK_LOG("Stack trace:\n");

        if (traceCount > 0)
        {
    #if !defined(NN_DETAIL_DIAG_PRINT_SYSTEM_BACKTRACE)
            auto isUserSymbol = !IsSystemSymbol(traces[0]);
            for (int i = 0; i < traceCount; i++)
            {
                auto isCalledByUser = (i + 1 < traceCount) && !IsSystemSymbol(traces[i + 1]);
                PrintBacktraceItemImpl(traces[i], (isUserSymbol || isCalledByUser));
                isUserSymbol = isCalledByUser; // ユーザが呼び出しているなら、次はユーザーシンボル
            }
    #else
            for (int i = 0; i < traceCount; i++)
            {
                PrintBacktraceItemImpl(traces[i], true);
            }
    #endif
        }

        NN_SDK_LOG("\n");
    }

}

void PrintBacktraceItemImpl(uintptr_t address, bool showDatail) NN_NOEXCEPT
{
    if (showDatail)
    {
        // noreturn 関数の場合は、関数末尾の bl 命令の直後に、次の関数の先頭が来るため、
        // アドレスを 1 引いて、次の関数のシンボルが検索されないようにする。
        const char* symbolName;
        size_t symbolSize;
        auto symbolAddress = GetNearestExportedSymbol(&symbolName, &symbolSize, address - 1);
        if (symbolAddress != 0u)
        {
            // noreturn 関数の場合は、関数末尾の bl 命令の直後に、次の関数の先頭が来るため、
            // 次の関数の先頭アドレスまでは関数内とみなす。
            auto isOutOfFunction = address > symbolAddress + symbolSize;
            NN_UNUSED(isOutOfFunction);
        #if defined(NN_BUILD_CONFIG_ADDRESS_64)
            NN_SDK_LOG("  0x%016" PRIXPTR " %s+0x%" PRIXPTR "%s\n", address, symbolName, address - symbolAddress, isOutOfFunction ? " (too far)" : "");
        #else
            NN_SDK_LOG("  0x%08" PRIXPTR " %s+0x%" PRIXPTR "%s\n", address, symbolName, address - symbolAddress, isOutOfFunction ? " (too far)" : "");
        #endif
            return;
        }
    }
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
    NN_SDK_LOG("  0x%016" PRIXPTR " (unknown)\n", address);
#else
    NN_SDK_LOG("  0x%08" PRIXPTR " (unknown)\n", address);
#endif
}

void TentativeDumpStackTrace() NN_NOEXCEPT
{
    const int traceCountMax = 32;
    uintptr_t traces[traceCountMax];
    auto traceCount = nn::diag::GetBacktrace(traces, traceCountMax);
    PrintBacktrace(traces, traceCount);
    PrintModules(traces, traceCount);
}

} // namespace nn::diag::detail

const int ModuleCountMax = 32;

// バックトレースに含まれるモジュールの一覧を出力する。
void PrintModules(uintptr_t* traces, int traceCount) NN_NOEXCEPT
{
    NN_SDK_LOG("Related modules:\n");

#if defined(NN_BUILD_CONFIG_ADDRESS_64)
    NN_SDK_LOG("  base               size               name/path\n");
#else
    NN_SDK_LOG("  base       size       name/path\n");
#endif

    if (traceCount > 0)
    {
        // モジュールサイズも記憶すれば、一度検索したモジュールを再検索しなくて済むが、
        // メモリ使用量を抑えるために、ベースアドレスだけ記憶しておく。
        uintptr_t printedModuleBaseAddresses[ModuleCountMax];

        int printedModuleCount = 0;

        for (int i = 0; i < traceCount; i++)
        {
            static const char* path;
            size_t pathLength;
            size_t moduleSize;

            auto baseAddress = detail::GetModuleInfoForHorizon(&path, &pathLength, &moduleSize, traces[i]);
            if (baseAddress == 0u)
            {
                continue;
            }

            // モジュールを重複して出力しないようにチェック。
            int j = 0;
            for (; j < printedModuleCount && printedModuleBaseAddresses[j] != baseAddress; j++);
            if (j < printedModuleCount)
            {
                continue;
            }

            if (pathLength > 0)
            {
            #if defined(NN_BUILD_CONFIG_ADDRESS_64)
                NN_SDK_LOG("  0x%016" PRIXPTR " 0x%016zX %.*s\n", baseAddress, moduleSize, pathLength, path);
            #else
                NN_SDK_LOG("  0x%08" PRIXPTR " 0x%08zX %.*s\n", baseAddress, moduleSize, pathLength, path);
            #endif
            }
            else
            {
            #if defined(NN_BUILD_CONFIG_ADDRESS_64)
                NN_SDK_LOG("  0x%016" PRIXPTR " 0x%016zX (empty)\n", baseAddress, moduleSize);
            #else
                NN_SDK_LOG("  0x%08" PRIXPTR " 0x%08zX (empty)\n", baseAddress, moduleSize);
            #endif
            }

            printedModuleBaseAddresses[printedModuleCount++] = baseAddress; // 一度出力したモジュールのベースアドレスを記憶。

            NN_SDK_ASSERT_LESS_EQUAL(printedModuleCount, ModuleCountMax);
        }
    }

    NN_SDK_LOG("\n");
}

void PrintBacktraceItem(uintptr_t addr, bool showDatail) NN_NOEXCEPT
{
    detail::PrintBacktraceItemImpl(addr, showDatail);
}

}} // namespace nn::diag
