﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/ddsf/ddsf_DeviceCodeEntryManager.h>
#include <nn/ddsf/ddsf_IDevice.h>
#include <nn/ddsf/ddsf_Result.h>
#include <nn/ddsf/detail/ddsf_Log.h>

namespace nn { namespace ddsf {

void DeviceCodeEntryManager::Reset() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_EntryListLock)> lock(m_EntryListLock);

    while ( !m_UsedEntryList.empty() )
    {
        auto& e = m_UsedEntryList.front();
        m_UsedEntryList.pop_front();

        NN_SDK_ASSERT(e.IsBuilt());
        e.Destroy();

        m_FreeEntryList.push_back(e);
    }
}

nn::Result DeviceCodeEntryManager::Add(nn::DeviceCode deviceCode, IDevice* pDevice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDevice);
    NN_SDK_REQUIRES(pDevice->IsDriverAttached(), "Driver must be attached to specified IDevice object before associating IDevice with device code\n");
    std::lock_guard<decltype(m_EntryListLock)> lock(m_EntryListLock);

    // リスト探索コストがあるので、アサートでのチェックのみとする
    for ( const auto& e : m_UsedEntryList )
    {
        NN_UNUSED(e); // Release ビルド対策
        NN_SDK_ASSERT(e.IsBuilt());
        NN_SDK_REQUIRES(e.Get().GetDeviceCode() != deviceCode, "Specified device code (0x%x) has already been added\n", deviceCode.GetInternalValue());
    }

    if ( m_FreeEntryList.empty() )
    {
        // 見つかったデバイスコードの数がドライバ側で持っているデバイスエントリのバッファを超えた
        NN_DETAIL_DDSF_ERROR("Cannot add any more device code entry. Dropping.\n");
        NN_RESULT_THROW(nn::ddsf::ResultOutOfResource());
    }
    auto& p = m_FreeEntryList.front();
    m_FreeEntryList.pop_front();

    NN_SDK_ASSERT(!p.IsBuilt());
    p.Build(deviceCode, pDevice);

    m_UsedEntryList.push_back(p);

    NN_RESULT_SUCCESS;
}

bool DeviceCodeEntryManager::Remove(nn::DeviceCode deviceCode) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_EntryListLock)> lock(m_EntryListLock);

    bool wasErased = false;
    for ( auto first = m_UsedEntryList.begin(), end = m_UsedEntryList.end(); first != end; )
    {
        auto&& e = *first;
        ++first; // この後 e をリストから抜く可能性があるので、先にイテレートしておく

        if ( e.Get().GetDeviceCode() == deviceCode )
        {
            m_UsedEntryList.erase(m_UsedEntryList.iterator_to(e));

            NN_SDK_ASSERT(e.IsBuilt());
            e.Destroy();

            m_FreeEntryList.push_back(e);
            wasErased = true;
        }
    }

    return wasErased;
}

nn::Result DeviceCodeEntryManager::FindDeviceCodeEntry(DeviceCodeEntry** ppOutDeviceCodeEntry, nn::DeviceCode code) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutDeviceCodeEntry);
    if ( !ppOutDeviceCodeEntry )
    {
        return nn::ddsf::ResultInvalidArgument();
    }

    bool isFound = false;
    ForEachEntry(
        [&](DeviceCodeEntry* pEntry) NN_NOEXCEPT -> bool
        {
            NN_SDK_ASSERT_NOT_NULL(pEntry);
            if ( pEntry->GetDeviceCode() == code )
            {
                isFound = true;
                *ppOutDeviceCodeEntry = pEntry;
                return false; // STOP foreach
            }
            return true;
        }
    );
    if ( !isFound )
    {
        NN_DETAIL_DDSF_ERROR("Unsupported device code (0x%x) is specified\n", code);
        return nn::ddsf::ResultDeviceCodeNotFound();
    }
    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeEntryManager::FindDeviceCodeEntry(const DeviceCodeEntry** ppOutDeviceCodeEntry, nn::DeviceCode code) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutDeviceCodeEntry);
    if ( !ppOutDeviceCodeEntry )
    {
        return nn::ddsf::ResultInvalidArgument();
    }

    bool isFound = false;
    ForEachEntry(
        [&](const DeviceCodeEntry* pEntry) NN_NOEXCEPT -> bool
        {
            NN_SDK_ASSERT_NOT_NULL(pEntry);
            if ( pEntry->GetDeviceCode() == code )
            {
                isFound = true;
                *ppOutDeviceCodeEntry = pEntry;
                return false; // STOP foreach
            }
            return true;
        }
    );
    if ( !isFound )
    {
        NN_DETAIL_DDSF_ERROR("Unsupported device code (0x%x) is specified\n", code);
        return nn::ddsf::ResultDeviceCodeNotFound();
    }
    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeEntryManager::FindDevice(IDevice** ppOutDevice, nn::DeviceCode code) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(ppOutDevice);
    DeviceCodeEntry* pEntry;
    NN_RESULT_DO(FindDeviceCodeEntry(&pEntry, code));
    if ( ppOutDevice )
    {
        *ppOutDevice = &pEntry->GetDevice();
    }
    NN_RESULT_SUCCESS;
}

nn::Result DeviceCodeEntryManager::FindDevice(const IDevice** ppOutDevice, nn::DeviceCode code) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(ppOutDevice);
    const DeviceCodeEntry* pEntry;
    NN_RESULT_DO(FindDeviceCodeEntry(&pEntry, code));
    if ( ppOutDevice )
    {
        *ppOutDevice = &pEntry->GetDevice();
    }
    NN_RESULT_SUCCESS;
}

}} // nn::ddsf
