﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief Finite state machine driver
 *
 * @details
 */

namespace nn { namespace bsdsocket { namespace dhcpc {

class DhcpcMain;

class Fsm
{
public:
    static const size_t MAX_FSM_NAME_SIZE = 32;
    enum Event
    {
        Event_Invalid = 0,
        Event_Entry,
        Event_Exit,
        Event_Base
    };

    NN_IMPLICIT Fsm()
    {
        m_pMain         = NULL;
        m_State         = 0;
        m_NextState     = 0;
        m_CurrentEvent  = 0;
        memset(m_FsmName, 0, sizeof(m_FsmName));
        m_EventNames    = NULL;
        m_StateNames    = NULL;
        m_NumEvents     = 0;
        m_NumStates     = 0;
    }
    virtual ~Fsm()
    {

    }

    LocalEventPoolType* GetLocalEventPool();
    int32_t GetState();
    int32_t GetCurrentEvent();
    void SetFsmTraceEnable(bool isEnabled);

protected:
    Result Initialize(DhcpcMain *pMain, char *name,
                      const char **eventNames, int32_t numEvents,
                      const char **stateNames, int32_t numStates,
                      int32_t initialState);
    Result Finalize();
    void SetName(const char *name);
    const char* GetName();
    void QueueFsmEvent(LocalEventDataType *pEvent);
    void QueueFsmEvent(int32_t eventId);
    void QueueFsmEvent(int32_t eventId, Result status);
    void QueueFsmEvent(int32_t eventId, Result status, uint32_t data);
    void QueueFsmEvent(int32_t eventId, void *pData);
    void QueueFsmEvent(int32_t eventId, Result status, void *pData);
    Result DispatchFsmEvent(int32_t eventId, void *pData);
    Result DispatchFsmEvent(int32_t eventId, void *pData0, void *pData1);
    Result DispatchFsmEvent(int32_t eventId);
    LocalEventType* CreateTimedFsmEvent(int32_t eventId);
    void DestroyTimedFsmEvent(LocalEventType *pEvent);
    void InitStaticTimedFsmEvent(LocalEventType *pEvent, int32_t eventId);
    void StartTimedFsmEvent(LocalEventType *pEvent, TimeSpan relativeTime);
    void ReStartTimedFsmEvent(LocalEventType *pEvent, TimeSpan relativeTime);
    void StopTimedFsmEvent(LocalEventType *pEvent);
    virtual Result FsmHandler(int32_t state, LocalEventDataType *pEvent) = 0;
    void SetState(int32_t nextState);
    bool IsEventStale(LocalEventDataType *pEventData);
    void Schedule();

private:
    DhcpcMain           *m_pMain;
    int32_t             m_State;
    int32_t             m_NextState;
    int32_t             m_CurrentEvent;
    char                m_FsmName[MAX_FSM_NAME_SIZE];
    const char          **m_EventNames;
    const char          **m_StateNames;
    int32_t             m_NumEvents;
    int32_t             m_NumStates;
    bool                m_FsmTraceEnabled;
    LocalEventPoolType  m_LocalEventPool;
    LocalEventType      m_LocalEventStorage[32];

    Result HandleEvent(LocalEventDataType *pEvent);
    static void HandleEventStatic(void *context, LocalEventType *pEvent)
    {
        Fsm *pFsm = static_cast<Fsm *>(context);
        pFsm->HandleEvent(&pEvent->data);
    }
};

} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn
