﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cfg_PrivateIncludes.h"

namespace nn { namespace bsdsocket { namespace cfg {


typedef nn::sf::UnmanagedServiceObject<ServerInterface, Server> CfgServiceObject;

// Server declaration
std::aligned_storage<sizeof(CfgServiceObject), NN_ALIGNOF(CfgServiceObject)>::type g_CfgServerStorage;
CfgServiceObject *g_pCfgServer = NULL;

void InitializeServer()
{
    g_pCfgServer = new(&g_CfgServerStorage) CfgServiceObject;
    CFG_ABORT_UNLESS_SUCCESS(g_pCfgServer->GetImpl().Initialize());
}

void FinalizeServer()
{
    CFG_ABORT_NULL(g_pCfgServer);
    CFG_ABORT_UNLESS_SUCCESS(g_pCfgServer->GetImpl().Finalize());
    g_pCfgServer->GetImpl(). ~Server();
    g_pCfgServer = NULL;
}

Result Server::Initialize()
{
    Result result = ResultSuccess();

    memset(m_IfTable, 0, sizeof(m_IfTable));

    do
    {
        // Initialize DHCP subsystem
        CFG_BREAK_UPON_ERROR(dhcpc::Initialize());

        CFG_BREAK_UPON_ERROR(RegisterObjectForPort(g_pCfgServer->GetShared(), Limits_MaxSessions, "bsdcfg"));
        Start();

        for (int32_t index = 0; index < Limits_MaxSessionThreads; index++)
        {
            CreateNetworkThread(&m_Threads[index], SessionThreadEntryStatic,
                                this, &m_ThreadStacks[index][0],
                                sizeof(m_ThreadStacks[index]),
                                NN_SYSTEM_THREAD_PRIORITY(socket, Config), -1);
            SetNetworkThreadNamePointer(&m_Threads[index],
                                        NN_SYSTEM_THREAD_NAME(socket, Config));
            StartNetworkThread(&m_Threads[index]);
        }

    } while (false);

    return result;
}

Result Server::Finalize()
{
    Result result = ResultSuccess();

    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);

    // Take down any interfaces which are still up
    for (int32_t ifIndex = 0; ifIndex < Limits_MaxInterfaces; ifIndex++)
    {
        ManagedIf *pMif = m_IfTable + ifIndex;
        if (pMif->allocated)
        {
            pMif->i.Finalize();
            FreeMif(pMif);
        }
    }

    // Stop the server
    RequestStop();

    // Synchronize with worker thread exit
    for (int32_t index = 0; index < Limits_MaxSessionThreads; index++)
    {
        WaitNetworkThread(&m_Threads[index]);
    }

    // Finalize DHCP subsystem
    dhcpc::Finalize();

    return result;
}

Server::ManagedIf* Server::GetMif(const char *name)
{
    ManagedIf *pMif = NULL;
    for (int32_t ifIndex = 0; ifIndex < Limits_MaxInterfaces; ifIndex++)
    {
        ManagedIf *p = m_IfTable + ifIndex;
        if (p->allocated)
        {
            if (strncmp(p->name, name, dhcpc::Config_MaxIfNameSize) == 0)
            {
                pMif = p;
                break;
            }
        }
    }
    return pMif;
}

Server::ManagedIf* Server::AllocMif(const char *ifName)
{
    ManagedIf *pMif = NULL;
    for (int32_t ifIndex = 0; ifIndex < Limits_MaxInterfaces; ifIndex++)
    {
        ManagedIf *p = m_IfTable + ifIndex;
        if ( !p->allocated)
        {
            pMif = p;
            pMif->allocated = true;
            pMif->ifIndex = ifIndex;
            nn::util::Strlcpy(pMif->name, ifName, sizeof(pMif->name));
            new(&pMif->i) Interface();
            break;
        }
    }

    return pMif;
}

void Server::FreeMif(ManagedIf *pMif)
{
    pMif->i. ~Interface();
    pMif->ifIndex = -1;
    pMif->allocated = false;
}


void Server::SessionThreadEntry()
{
    CFG_ABORT_NULL(g_pCfgServer);
    g_pCfgServer->GetImpl().LoopAuto();
}

Result Server::SetIfUp(const nn::sf::InBuffer &ifName, IfSettings ifOptionData)
{
    ManagedIf *pMif = NULL;
    Result result = ResultSuccess();
    const char *pIfName = ifName.GetPointerUnsafe();

    // Critical section
    {
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);

        // Is interface already up?
        if ((pMif = GetMif(pIfName)) != NULL)
        {
            return ResultBusy();
        }

        // Instances exhausted?
        if ((pMif = AllocMif(pIfName)) == NULL)
        {
            return ResultMaximumExceeded();
        }

        if((ifOptionData.metric == 0) && (m_DefaultIfIndex < 0))
        {
            m_DefaultIfIndex = pMif->ifIndex;
        }
        else if (ifOptionData.metric == 0)
        {
            // Gracefully increase metric if there is already a default interface
            ifOptionData.metric++;
            CFG_LOG_INFO("Metric for %s is being increased to 1, since default interface has already been configured.\n",
                         pIfName);
        }
    }

    // This call may block for a long time
    result = pMif->i.Initialize(pMif->ifIndex, pMif->name, &ifOptionData, NULL);

    if ( !result.IsSuccess())
    {
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
        if ((pMif = GetMif(pIfName)) != NULL)
        {
            if (m_DefaultIfIndex == pMif->ifIndex)
            {
                m_DefaultIfIndex = -1;
            }
            FreeMif(pMif);
        }
    }
    /*else
    {
        DumpRoute();
    }*/

    return result;
}

Result Server::SetIfUpWithEvent(const nn::sf::InBuffer &ifName, IfSettings ifOptionData, nn::sf::Out<nn::sf::NativeHandle> handleOut)
{
    ManagedIf *pMif = NULL;
    Result result = ResultSuccess();
    const char *pIfName = ifName.GetPointerUnsafe();

    // Critical section
    {
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);

        // Is interface already up?
        if ((pMif = GetMif(pIfName)) != NULL)
        {
            return ResultBusy();
        }

        // Instances exhausted?
        if ((pMif = AllocMif(pIfName)) == NULL)
        {
            return ResultMaximumExceeded();
        }

        // Update record of default interface
        if((ifOptionData.metric == 0) && (m_DefaultIfIndex < 0))
        {
            m_DefaultIfIndex = pMif->ifIndex;
        }
        else if (ifOptionData.metric == 0)
        {
            // Gracefully increase metric if there is already a default interface
            ifOptionData.metric++;
            CFG_LOG_INFO("Metric for %s is being increased to 1, since default interface has already been configured.\n",
                         pIfName);
        }
    }

    // This call may block for a long time
    result = pMif->i.Initialize(pMif->ifIndex, pMif->name, &ifOptionData, &handleOut);

    if ( !result.IsSuccess())
    {
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
        if ((pMif = GetMif(pIfName)) != NULL)
        {
            if (m_DefaultIfIndex == pMif->ifIndex)
            {
                m_DefaultIfIndex = -1;
            }
            FreeMif(pMif);
        }
    }

    return result;
}

Result Server::CancelIf(const nn::sf::InBuffer &ifName)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    ManagedIf *pMif;
    Result result = ResultIfInvalid();
    const char *pIfName = ifName.GetPointerUnsafe();
    if ((pMif = GetMif(pIfName)) != NULL)
    {
        result = pMif->i.Cancel();
    }
    return result;
}

Result Server::SetIfDown(const nn::sf::InBuffer &ifName, uint32_t options)
{

    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    ManagedIf *pMif;
    Result result = ResultIfInvalid();
    const char *pIfName = ifName.GetPointerUnsafe();
    if ((pMif = GetMif(pIfName)) != NULL)
    {
        result = pMif->i.Finalize(options);
        if (result.IsSuccess())
        {
            if (m_DefaultIfIndex == pMif->ifIndex)
            {
                m_DefaultIfIndex = -1;
            }
            FreeMif(pMif);
        }
    }
    return result;
}


Result Server::GetIfState(const nn::sf::InBuffer &ifName, nn::sf::OutBuffer ifState)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    Result result = ResultSuccess();

    if (ifState.GetSize() == sizeof(IfState))
    {
        ManagedIf *pMif;
        const char *pIfName = ifName.GetPointerUnsafe();
        if ((pMif = GetMif(pIfName)) != NULL)
        {
            IfState *pIfState = reinterpret_cast<IfState *>(ifState.GetPointerUnsafe());
            result = pMif->i.GetState(pIfState);
        }
        else
        {
            result = ResultIfInvalid();
        }
    }
    else
    {
        result = ResultSfSizeError();
    }

    return result;
}

Result Server::DhcpRenew(const nn::sf::InBuffer &ifName)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    ManagedIf *pMif;
    Result result = ResultIfInvalid();
    const char *pIfName = ifName.GetPointerUnsafe();
    if ((pMif = GetMif(pIfName)) != NULL)
    {
        result = pMif->i.DhcpRenew();
    }
    return result;
}

Result Server::AddStaticArpEntry(uint32_t ipAddress,const nn::sf::InBuffer &inputHardwareAddress)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return nn::bsdsocket::cfg::Arp::AddStaticArpEntry(*reinterpret_cast<nn::socket::InAddr *>(&ipAddress), reinterpret_cast<const uint8_t *>(inputHardwareAddress.GetPointerUnsafe()), inputHardwareAddress.GetSize());
}

Result Server::RemoveArpEntry(uint32_t ipAddress)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return nn::bsdsocket::cfg::Arp::RemoveArpEntry(*reinterpret_cast<nn::socket::InAddr *>(&ipAddress));
}

Result Server::LookupArpEntry(nn::sf::OutBuffer outputHardwareAddress, uint32_t ipAddress)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return nn::bsdsocket::cfg::Arp::LookupArpEntry(reinterpret_cast<uint8_t *>(outputHardwareAddress.GetPointerUnsafe()), outputHardwareAddress.GetSize(),
        *reinterpret_cast<nn::socket::InAddr *>(&ipAddress));
}

Result Server::LookupArpEntry2(nn::sf::OutBuffer outputIpAddress,const nn::sf::InBuffer &hardwareAddress)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return nn::bsdsocket::cfg::Arp::LookupArpEntry(reinterpret_cast<nn::socket::InAddr *>(outputIpAddress.GetPointerUnsafe()), outputIpAddress.GetSize(),
        reinterpret_cast<const uint8_t *>(hardwareAddress.GetPointerUnsafe()), hardwareAddress.GetSize());
}

Result Server::ClearArpEntries()
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return nn::bsdsocket::cfg::Arp::ClearArpEntries();
}

Result Server::ClearArpEntries2(const nn::sf::InBuffer &ifName)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return nn::bsdsocket::cfg::Arp::ClearArpEntries(ifName.GetPointerUnsafe());
}

Result Server::PrintArpEntries()
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    return nn::bsdsocket::cfg::Arp::PrintArpEntries();
}

} // namespace cfg
} // namespace bsdsocket
} // namespace nn




