﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cfg_PrivateIncludes.h"
#include "../../socket/resolver/server/resolver_DnsServerHelper.h"
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
namespace nn { namespace bsdsocket { namespace cfg {


Result Interface::Initialize(int32_t ifIndex, const char *ifName,
                             const IfSettings *pOptions, nn::sf::Out<nn::sf::NativeHandle> *pSfHandle)
{
    bool waitForDhcpCompletion = false;
    Result result = ResultSuccess();
    do
    {
        // Lock during configuration, and DHCP initiation
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);

        // Nothing to do if preemptively cancelled
        if (m_State == State_CancelDhcp)
        {
            return ResultCancelled();
        }

        // Nothing to do if already init
        if (m_State != State_Null)
        {
            return ResultBusy();
        }

        m_State = State_InitInProgress;

        // Accept DNS overrides
        memcpy(m_DnsAddrs, pOptions->dnsAddrs, sizeof(m_DnsAddrs));

        // Proceed with initialization
        m_IfIndex            = ifIndex;
        m_IfName             = ifName;
        m_IfSettings         = *pOptions;
        m_IsDhcpInit         = false;
        m_FailureCause       = ResultSuccess();
        nn::os::InitializeEvent(&m_InitializeEvent, false, nn::os::EventClearMode_AutoClear);

        // Resolve MAC address of low level interface
        CFG_BREAK_UPON_ERROR(ResolveHardwareAddress());

        // Tell low level interface that link is assumed to be up
        CFG_BREAK_UPON_ERROR(SetLinkState(true));

        if (pOptions->mode == IfIpAddrMode_Dhcp)
        {
            nn::socket::InAddr emptyAddr;
            emptyAddr.S_addr = 0;
            result = StartDhcp(emptyAddr,emptyAddr);
            waitForDhcpCompletion = true;
        }
        else if (pOptions->mode == IfIpAddrMode_DhcpReboot)
        {
            result = StartDhcp(m_IfSettings.u.modeDhcpReboot.requestedAddr,m_IfSettings.u.modeDhcpReboot.serverAddr);

            waitForDhcpCompletion = true;
        }
        else if (pOptions->mode == IfIpAddrMode_Static)
        {
            CFG_BREAK_UPON_ERROR(SetStaticIp());
        }
        else
        {
            CFG_BREAK_UPON_ERROR(ResultInvalidIfAddressMode());
        }

    }while (false);

    // If DHCP was successfully initiated, wait for completion
    if (waitForDhcpCompletion && result.IsSuccess())
    {
        nn::os::WaitEvent(&m_InitializeEvent);
    }



    // Update state
    {
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
        if (result.IsSuccess())
        {
            switch (m_State)
            {
            case State_InitInProgress:
                m_State = State_Initialized;
                result = m_FailureCause;
                break;
            case State_CancelDhcp:
                m_State = State_CancelDhcp;
                m_FailureCause = ResultCancelled();
                result = ResultCancelled();
                break;
            default:
                result = ResultInvalidInternalState();
                break;
            }
        }
        ShutdownOption shutdownOption = ShutdownOption_None;
        if(result.IsSuccess())
        {
            result = CheckForDuplicateIps(shutdownOption);
        }

        if (result.IsSuccess())
        {

            if (pSfHandle != NULL)
            {
                pSfHandle->Set(nn::sf::NativeHandle(m_OperationEvent.GetReadableHandle(), false));
            }

        }
        else
        {

            m_State = State_Error;
            Finalize(shutdownOption);
        }


    }



    return result;
}
Result Interface::Cancel()
{
    Result result = ResultSuccess();

    {
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
        // Evaluate current state
        switch (m_State)
        {
        case State_InitInProgress:
            // If prior mode of operation used DHCP
            if (m_IfSettings.mode == IfIpAddrMode_Dhcp || m_IfSettings.mode == IfIpAddrMode_DhcpReboot)
            {
                m_State = State_CancelDhcp;
                nn::os::SignalEvent(&m_InitializeEvent);
            }
            break;
        case State_Null:
            // Cancel call was made before Initialize() started
            // (object only created, not initialized yet)
            m_State = State_CancelDhcp;
            break;
        case State_Initialized:
            result = ResultIfIsUp();
            break;
        case State_Error:
        case State_CancelDhcp:
            result = ResultSuccess();
            break;
        default:
            result = ResultInvalidInternalState();
            break;
        }
    }

    return result;
}

Result Interface::Finalize(uint32_t options)
{
    Result result = ResultSuccess();

    // Lock during this whole call
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);

    // Evaluate current state, perhaps exit if nothing to do
    switch (m_State)
    {
    case State_InitInProgress:
    case State_Null:
        return ResultBusy();
        break;
    case State_Initialized:
    case State_Error:
    case State_CancelDhcp:
        // If prior mode of operation used DHCP
        if (m_IfSettings.mode == IfIpAddrMode_Dhcp || m_IfSettings.mode == IfIpAddrMode_DhcpReboot)
        {
            // Finalize DHCP
            StopDhcp(options);
        }
        break;
    default:
        return ResultInvalidInternalState();
        break;
    }

    // Tell low level interface that link is assumed to be down
    SetLinkState(false);

    // Remove any prior mapping
    if (m_Addr.S_addr != 0)
    {
        switch (m_IfSettings.mode)
        {
        case IfIpAddrMode_DhcpReboot:
        case IfIpAddrMode_Dhcp:
            InvalidateDhcpLease();
            break;

        case IfIpAddrMode_Static:
            InvalidateStaticIp();
            break;
        default:
            break;
        }
    }

    m_State = State_Null;
    nn::os::FinalizeEvent(&m_InitializeEvent);
    m_OperationEvent.Signal();

    return result;
}

Result Interface::GetState(IfState *pIfState)
{
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
    Result result = ResultSuccess();
    memset(pIfState, 0, sizeof(IfState));
    pIfState->mode             = m_IfSettings.mode;
    pIfState->metric           = m_IfSettings.metric;
    pIfState->mtu              = m_IfSettings.mtu;
    pIfState->addr             = GetAddr();
    pIfState->subnetMask       = GetNetMask();
    pIfState->gatewayAddr      = GetGatewayAddr();
    pIfState->broadcastAddr    = GetBcastAddr();
    pIfState->serverAddr       = GetServerAddr();
    pIfState->status           = m_FailureCause;
    pIfState->u.modeDhcp.newLeaseCount     = m_NewLeaseCount;
    pIfState->u.modeDhcp.renewedLeaseCount = m_RenewedLeaseCount;
    pIfState->u.modeDhcp.reboundLeaseCount = m_ReboundLeaseCount;
    pIfState->u.modeDhcp.invalidLeaseCount = m_InvalidLeaseCount;
    memcpy(pIfState->dnsAddrs, m_DnsAddrs, sizeof(pIfState->dnsAddrs));
    if (m_IfSettings.mode == IfIpAddrMode_Dhcp || m_IfSettings.mode == IfIpAddrMode_DhcpReboot)
    {
        nn::bsdsocket::dhcpc::GetState(m_IfIndex,
                                       pIfState->u.modeDhcp.currentState,
                                       sizeof(pIfState->u.modeDhcp.currentState));
    }
    return result;
}

Result Interface::DhcpRenew()
{
    Result result = ResultSuccess();

    // Lock during this whole call
    std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);

    // Evaluate current state, perhaps exit if nothing to do
    switch (m_State)
    {
    case State_InitInProgress:
    case State_Null:
    case State_Error:
    case State_CancelDhcp:
        return ResultBusy();
        break;
    case State_Initialized:
        // If prior mode of operation used DHCP
        if (m_IfSettings.mode == IfIpAddrMode_Dhcp || m_IfSettings.mode == IfIpAddrMode_DhcpReboot)
        {
            result = nn::bsdsocket::dhcpc::ForceRenew(m_IfIndex);
        }
        else
        {
            // fails on Mode Static
            result = ResultInvalidIfAddressMode();
        }
        break;
    default:
        return ResultInvalidInternalState();
        break;
    }
    return result;
}

const char* Interface::GetName()
{
    return m_IfName;
}

nn::socket::InAddr Interface::GetAddr()
{
    return m_Addr;
}

nn::socket::InAddr Interface::GetNetMask()
{
    return m_SubnetMask;
}

nn::socket::InAddr Interface::GetBcastAddr()
{
    return m_BroadcastAddr;
}

nn::socket::InAddr Interface::GetGatewayAddr()
{
    return m_GatewayAddr;
}

nn::socket::InAddr Interface::GetServerAddr()
{
    return m_ServerAddr;
}

Result Interface::InitLink(int mibsock)
{
    Result result = ResultSuccess();
    nn::socket::IfReq ifr;

    // Set address
    memset(&ifr, 0, sizeof(ifr));
    nn::util::Strlcpy(ifr.ifr_name, GetName(), sizeof(ifr.ifr_name));
    memcpy(ifr.ifr_addr.sa_data, m_HardwareAddress, 6);
    ifr.ifr_addr.sa_family = nn::socket::Family::Af_Link;
    ifr.ifr_addr.sa_len = dhcpc::Config_MaxHardwareAddressSize;
    if (nn::socket::Ioctl(mibsock, nn::socket::IoctlCommandPrivate::SiocSIfLlAddr, (caddr_t)&ifr, sizeof(ifr)) < 0)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Ioctl( SiocSIfLlAddr ) failed, errno=%d.\n", lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
    }

    return result;
}

Result Interface::InitMtu(int mibsock)
{
    Result result = ResultSuccess();
    nn::socket::IfReq ifr;
    memset(&ifr, 0, sizeof(ifr));
    nn::util::Strlcpy(ifr.ifr_name, GetName(), sizeof(ifr.ifr_name));
    ifr.ifr_mtu = m_IfSettings.mtu;
    if (nn::socket::Ioctl(mibsock, nn::socket::IoctlCommandPrivate::SiocSIfMtu, (caddr_t)&ifr, sizeof(ifr)) < 0)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Ioctl( SiocSIfMtu, mtu=%d ) failed, errno=%d.\n",
                     ifr.ifr_mtu, lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        case nn::socket::Errno::EInval:
            result = ResultMtuSizeError();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
    }
    return result;
}

Result Interface::SetDhcpLease(dhcpc::Lease *pLease)
{
    Result result = ResultSuccess();
    if (pLease->isValid)
    {
        m_NewLeaseCount++;
        m_CurrentDhcpLease = *pLease;
        do
        {
            nn::socket::InAddr address;
            size_t size;
            nn::socket::InAddr dnsAddrs[2];
            memset(dnsAddrs, 0, sizeof(dnsAddrs));
            size_t validDnsAddrs = 0;

            if(dhcpc::GetOptionData(m_IfIndex, dhcpc::DhcpProtOption_Router,
                                     &address, sizeof(address), &size).IsSuccess())
            {
                m_GatewayAddr = address;
            }
            else
            {
                m_GatewayAddr.S_addr = 0;
            }

            if (m_IfSettings.isManualDns == false)
            {
                if (dhcpc::GetOptionData(m_IfIndex, dhcpc::DhcpProtOption_DnsServer,
                    &dnsAddrs, sizeof(dnsAddrs), &size).IsSuccess())
                {
                    m_DnsAddrs[0] = dnsAddrs[0];
                    m_DnsAddrs[1] = dnsAddrs[1];
                }
                else
                {
                    m_DnsAddrs[0].S_addr = 0;
                    m_DnsAddrs[1].S_addr = 0;
                }
            }

            m_Addr             = m_CurrentDhcpLease.addr;
            m_SubnetMask       = m_CurrentDhcpLease.subnetMask;
            m_BroadcastAddr    = m_CurrentDhcpLease.broadcastAddr;
            m_ServerAddr       = m_CurrentDhcpLease.serverAddr;

            CurateDnsAddresses(&validDnsAddrs);

            CFG_LOG_INFO("Lease Acquired:\n");
            CFG_LOG_NOPREFIX("    address     = %s\n", nn::socket::InetNtoa(m_Addr));
            CFG_LOG_NOPREFIX("    subnet      = %s\n", nn::socket::InetNtoa(m_SubnetMask));
            CFG_LOG_NOPREFIX("    gateway     = %s\n", nn::socket::InetNtoa(m_GatewayAddr));
            CFG_LOG_NOPREFIX("    broadcast   = %s\n", nn::socket::InetNtoa(m_BroadcastAddr));
            CFG_LOG_NOPREFIX("    dns[0]      = %s\n", nn::socket::InetNtoa(m_DnsAddrs[0]));
            CFG_LOG_NOPREFIX("    dns[1]      = %s\n", nn::socket::InetNtoa(m_DnsAddrs[1]));
            CFG_LOG_NOPREFIX("    renew time  = %ds\n",(int)pLease->renewalTime.GetSeconds());
            CFG_LOG_NOPREFIX("    rebind time = %ds\n",(int)pLease->rebindTime.GetSeconds());
            CFG_LOG_NOPREFIX("    lease time  = %ds\n",(int)pLease->leaseTime.GetSeconds());

            CFG_BREAK_UPON_ERROR(AddAddress());
            CFG_BREAK_UPON_ERROR(ConfigureRoute(true));
            nn::socket::resolver::SetInaddrDnsAddresses(reinterpret_cast<uint32_t*>(m_DnsAddrs),
                                                        validDnsAddrs);

        } while (false);
    }

    if ( !result.IsSuccess())
    {
        InvalidateDhcpLease();
    }

    return result;
}

Result Interface::InvalidateDhcpLease()
{
    Result result = ResultSuccess();
    do
    {
        if ( !m_CurrentDhcpLease.isValid)
        {
            break;
        }
        m_FailureCause = ResultLeaseInvalid();
        m_InvalidLeaseCount++;
        CFG_BREAK_UPON_ERROR(ConfigureRoute(false));
        CFG_BREAK_UPON_ERROR(DeleteAddress());
    }while (false);

    memset(&m_Addr, 0, sizeof(m_Addr));
    m_SubnetMask = m_BroadcastAddr = m_GatewayAddr = m_Addr;
    memset(&m_CurrentDhcpLease, 0, sizeof(m_CurrentDhcpLease));

    return result;
}

Result Interface::SetStaticIp()
{
    Result result = ResultSuccess();
    do
    {
        size_t validDnsAddrs = 0;

        m_GatewayAddr      = m_IfSettings.u.modeStatic.gatewayAddr;
        m_Addr             = m_IfSettings.u.modeStatic.addr;
        m_SubnetMask       = m_IfSettings.u.modeStatic.subnetMask;
        m_BroadcastAddr    = m_IfSettings.u.modeStatic.broadcastAddr;

        CurateDnsAddresses(&validDnsAddrs);

        CFG_LOG_INFO("Static IP:\n");
        CFG_LOG_NOPREFIX("    address    = %s\n", nn::socket::InetNtoa(m_Addr));
        CFG_LOG_NOPREFIX("    subnet     = %s\n", nn::socket::InetNtoa(m_SubnetMask));
        CFG_LOG_NOPREFIX("    gateway    = %s\n", nn::socket::InetNtoa(m_GatewayAddr));
        CFG_LOG_NOPREFIX("    broadcast  = %s\n", nn::socket::InetNtoa(m_BroadcastAddr));
        CFG_LOG_NOPREFIX("    dns[0]     = %s\n", nn::socket::InetNtoa(m_DnsAddrs[0]));
        CFG_LOG_NOPREFIX("    dns[1]     = %s\n", nn::socket::InetNtoa(m_DnsAddrs[1]));

        CFG_BREAK_UPON_ERROR(AddAddress());
        CFG_BREAK_UPON_ERROR(ConfigureRoute(true));
        nn::socket::resolver::SetInaddrDnsAddresses(reinterpret_cast<uint32_t*>(m_DnsAddrs),
                                                    validDnsAddrs);

    } while (false);

    return result;
}

void Interface::CurateDnsAddresses(size_t* validDnsAddrs)
{
    *validDnsAddrs = 0;
    for (int i = 0; i < 2; ++i)
    {
        // Check if m_DnsAddrs[i] is a loopback address, and clear it if it is.
        if ((nn::socket::InetNtohl(m_DnsAddrs[i].S_addr) & 0x7F000000) == 0x7F000000)
        {
            m_DnsAddrs[i].S_addr = 0;
        }
        // Check if m_DnsAddrs[i] is the broadcast address, and clear it if it is.
        if (nn::socket::InetNtohl(m_DnsAddrs[i].S_addr) == 0xFFFFFFFF)
        {
            m_DnsAddrs[i].S_addr = 0;
        }

        if (m_DnsAddrs[i].S_addr != 0)
        {
            ++*validDnsAddrs;
        }
    }

    if (*validDnsAddrs == 1 && m_DnsAddrs[0].S_addr == 0)
    {
        m_DnsAddrs[0].S_addr = m_DnsAddrs[1].S_addr;
        m_DnsAddrs[1].S_addr = 0;
    }
}

Result Interface::InvalidateStaticIp()
{
    Result result = ResultSuccess();
    do
    {
        ConfigureRoute(false);
        DeleteAddress();
    }while (false);

    memset(&m_Addr, 0, sizeof(m_Addr));
    m_SubnetMask = m_BroadcastAddr = m_GatewayAddr = m_Addr;

    return result;
}

Result Interface::ConfigureRoute(bool add)
{
    int routeFd = -1;
    Result result = ResultSuccess();

    do
    {
        Ipv4::Route rt;
        memset(&rt, 0, sizeof(rt));

        if( (routeFd = nn::socket::SocketExempt(nn::socket::Family::Pf_Route, nn::socket::Type::Sock_Raw | nn::socket::TypePrivate::Sock_CloExec | nn::socket::Type::Sock_NonBlock, nn::socket::Protocol::IpProto_Ip)) < 0 )
        {
            CFG_BREAK_UPON_ERROR(ResultSocketOpenError());
        }

        // Default interface has some special routes applied
        if (m_IfSettings.metric == 0)
        {
            // Setup default route (to gateway)
            if (m_GatewayAddr.S_addr != nn::socket::InetHtonl(nn::socket::InAddr_Any))
            {
                memset(&rt, 0, sizeof(rt));
                rt.net.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
                rt.dest.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
                rt.gate.S_addr = m_GatewayAddr.S_addr;
                CFG_BREAK_UPON_ERROR(Ipv4::IfRoute(routeFd, GetNetMask(), GetAddr(), m_SdlIndex,
                    &rt, (add) ? Ipv4::Operation_Add : Ipv4::Operation_Delete));
            }

        }


    }while (false);

    if (routeFd >= 0)
    {
        nn::socket::Close(routeFd);
    }

    return result;
}

Result Interface::AddAddress()
{
    Result result = ResultSuccess();
    int mibsock = -1;
    do
    {
        if ((mibsock = nn::socket::SocketExempt(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) == -1)
        {
            CFG_BREAK_UPON_ERROR(ResultSocketOpenError());
        }
        CFG_BREAK_UPON_ERROR(InitMtu(mibsock));
        CFG_BREAK_UPON_ERROR(Ipv4::IfAddress(mibsock, GetName(), GetAddr(), GetNetMask(),
                                             GetBcastAddr(), m_IfSettings.metric, Ipv4::Operation_Add));
        CFG_BREAK_UPON_ERROR(Ipv4::VerifyIfAddress(mibsock, GetName(), GetAddr(), GetNetMask(), m_IfSettings.metric));
        CFG_BREAK_UPON_ERROR(InitLink(mibsock));
    }while (false);

    if (mibsock >= 0)
    {
        nn::socket::Close(mibsock);
    }

    return result;
}

Result Interface::DeleteAddress()
{
    Result result = ResultSuccess();
    int mibsock = -1;
    do
    {
        if ((mibsock = nn::socket::SocketExempt(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) == -1)
        {
            CFG_BREAK_UPON_ERROR(ResultSocketOpenError());
        }
        CFG_BREAK_UPON_ERROR(Ipv4::IfAddress(mibsock, GetName(), GetAddr(), GetNetMask(),
                                             GetBcastAddr(), m_IfSettings.metric, Ipv4::Operation_Delete));
    }while (false);

    if (mibsock >= 0)
    {
        nn::socket::Close(mibsock);
    }

    return result;
}

Result Interface::CheckForDuplicateIps(ShutdownOption & shutdownOption)
{
    Result result = ResultSuccess();
    // skip duplicate IP check if wait time set to 0
    if(m_IfSettings.duplicateIpWaitTime == 0)
    {
        return result;
    }

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_IfSettings.duplicateIpWaitTime));
    // check for duplicate IPs
    int fd = nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    if( fd >= 0)
    {

        nn::socket::IfDuplicateIpInfo ifdupinfo;
        memset(&ifdupinfo,0,sizeof(ifdupinfo));
        nn::util::Strlcpy(ifdupinfo.ifdup_name, m_IfName, sizeof(ifdupinfo.ifdup_name));

        if(nn::socket::Ioctl(fd, nn::socket::IoctlCommandPrivate::SiocGIfDupInfo, &ifdupinfo, sizeof(ifdupinfo)) == -1)
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            CFG_LOG_WARN("Ioctl( SiocGIfDupInfo ) failed, errno=%d.\n", lastErrno);
            switch (lastErrno)
            {
            case nn::socket::Errno::ENxIo:
                result = ResultIfRemoved();
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }
        }


        nn::socket::InAddr dupip;
        dupip.S_addr = ifdupinfo.ifdup_addr;

        if(dupip.S_addr != 0)
        {
            CFG_LOG_ERROR("Duplicate IP Detected! IP: %s MAC: %02x:%02x:%02x:%02x:%02x:%02x \n",nn::socket::InetNtoa(dupip)
                ,ifdupinfo.ifdup_mac[0],ifdupinfo.ifdup_mac[1],ifdupinfo.ifdup_mac[2],ifdupinfo.ifdup_mac[3],ifdupinfo.ifdup_mac[4],ifdupinfo.ifdup_mac[5]);
            m_State = State_Error;
            shutdownOption = ShutdownOption_Decline;
            result = ResultDuplicateIp();
        }

    }
    else
    {
        m_State = State_Error;
        result = ResultInvalidInternalState();
    }

    if (fd >=0)
    {
        nn::socket::Close(fd);
    }
    return result;
}

// This callback runs in DHCP context
void Interface::DhcpCallback(dhcpc::IndicationDataType *pIndData)
{
    switch (m_State)
    {
    case State_InitInProgress:
        {
            // No locking needed because client is blocked on Initialize(),
            // and call to Finalize() not valid at this time
            if (pIndData->indication == dhcpc::Indication_InvalidLease) break;
            if (pIndData->indication == dhcpc::Indication_NewLease)
            {
                m_FailureCause = SetDhcpLease(&pIndData->lease);
            }
            else if (pIndData->indication == dhcpc::Indication_Failure)
            {
                m_FailureCause = pIndData->failureCause;
            }
            nn::os::SignalEvent(&m_InitializeEvent);
            break;
        }
    case State_Initialized:
        {
            // No locking needed:
            //   It actually is not necessary for Interface::DhcpCallback() to acquire m_AccessMutex,
            //   due to the nature of the processing done during that callback,
            //   and how that callback interacts with other nn::bsdsocket::cfg API. (SIGLO-61147)
            if (pIndData->indication == dhcpc::Indication_InvalidLease)
            {
                InvalidateDhcpLease();
            }
            else if (pIndData->indication == dhcpc::Indication_NewLease)
            {
                InvalidateDhcpLease();
                m_FailureCause = SetDhcpLease(&pIndData->lease);
            }
            else if (pIndData->indication == dhcpc::Indication_RenewedLease)
            {
                m_RenewedLeaseCount++;
                m_FailureCause = ResultSuccess();
            }
            else if (pIndData->indication == dhcpc::Indication_ReboundLease)
            {
                m_ReboundLeaseCount++;
                m_FailureCause = ResultSuccess();
            }
            else if (pIndData->indication == dhcpc::Indication_Failure)
            {
                InvalidateDhcpLease();
                m_FailureCause = pIndData->failureCause;
            }
            m_OperationEvent.Signal();
            break;
        }
    default:
        break;
    }
}

Result Interface::StartDhcp(nn::socket::InAddr requestedAddr,nn::socket::InAddr serverAddr)
{
    Result result = ResultSuccess();

    do
    {
        dhcpc::InterfaceConfigType ifCfg;

        // Configure callback from DHCP client
        memset(&ifCfg, 0, sizeof(ifCfg));
        ifCfg.callbackContext = this;
        ifCfg.callback = DhcpCallbackStatic;
        snprintf(ifCfg.ifName, sizeof(ifCfg.ifName),
                 "%s", m_IfName);
        memcpy(ifCfg.hardwareAddress, m_HardwareAddress, sizeof(ifCfg.hardwareAddress));

        // Select protocol options
        DHCPC_ADD_PROT_OPTION_MASK(ifCfg.requestMask, dhcpc::DhcpProtOption_Router);

        DHCPC_ADD_PROT_OPTION_MASK(ifCfg.requestMask, dhcpc::DhcpProtOption_DnsServer);

        DHCPC_ADD_PROT_OPTION_MASK(ifCfg.requestMask, dhcpc::DhcpProtOption_Broadcast);

        DHCPC_ADD_PROT_OPTION_MASK(ifCfg.requestMask, dhcpc::DhcpProtOption_SubnetMask);

        // Configure reboot params
        ifCfg.requestedAddr = requestedAddr.S_addr;
        ifCfg.serverAddr = serverAddr.S_addr;
        // Start DHCP
        CFG_BREAK_UPON_ERROR(dhcpc::InitializeClient(m_IfIndex, &ifCfg));


        m_IsDhcpInit = true;
    } while (false);

    return result;
}

Result Interface::StopDhcp(uint32_t options)
{
    Result result = ResultSuccess();
    if (m_IsDhcpInit)
    {
        m_IsDhcpInit = false;
        result = dhcpc::FinalizeClient(m_IfIndex,options);
    }
    return result;
}

Result Interface::ResolveHardwareAddress()
{
    bool isResolved = false;
    Result result = ResultSuccess();
    int fd;
    nn::socket::IfReq ibuf[16];
    nn::socket::IfConf ifc;
    nn::socket::IfReq *ifrp, *ifend;

    /* Fetch the interface configuration */
    fd = nn::socket::SocketExempt(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    if (fd < 0)
    {
        CFG_RETURN_UPON_ERROR(ResultSocketOpenError());
    }
    ifc.ifc_len = sizeof(ibuf);
    ifc.ifc_buf = (caddr_t)ibuf;
    if ((nn::socket::Ioctl(fd, nn::socket::IoctlCommandPrivate::SiocGIfConf, (char *)&ifc, sizeof(ifc)) < 0) ||
        (static_cast<size_t>(ifc.ifc_len) < sizeof(nn::socket::IfReq)))
    {
        result = ResultUnknownStackError();
        goto out;
    }
    /* Search interface configuration list for link layer address. */
    ifrp = ibuf;
    ifend = (nn::socket::IfReq *)((char *)ibuf + ifc.ifc_len);
    while (ifrp < ifend)
    {
        size_t n;

        /* Look for interface */
        if (strncmp(GetName(), ifrp->ifr_name, sizeof(ifrp->ifr_name)) == 0 &&
            ifrp->ifr_addr.sa_family == nn::socket::Family::Af_Link &&
            ((nn::socket::SockAddrDl *)&ifrp->ifr_addr)->sdl_type == dhcpc::Config_MaxHardwareAddressSize)
        {
            memcpy(m_HardwareAddress, nn::socket::LlAddr((nn::socket::SockAddrDl *)&ifrp->ifr_addr),
                   sizeof(m_HardwareAddress));
            m_SdlIndex = ((nn::socket::SockAddrDl *)&ifrp->ifr_addr)->sdl_index;
            isResolved = true;
            break;
        }
        /* Bump interface config pointer */
        n = ifrp->ifr_addr.sa_len + sizeof(ifrp->ifr_name);
        if (n < sizeof(*ifrp)) n = sizeof(*ifrp);
        ifrp = (nn::socket::IfReq *)((char *)ifrp + n);
    }

out:
    nn::socket::Close(fd);

    if (result.IsSuccess() && !isResolved)
    {
        result = ResultIfInvalid();
    }

    return result;
}

Result Interface::SetLinkState(bool isLinkStateUp)
{
    int mibsock = -1;
    Result result = ResultSuccess();
    do
    {
        nn::socket::IfReq ifr;
        memset(&ifr, 0, sizeof(ifr));
        nn::util::Strlcpy(ifr.ifr_name, m_IfName, sizeof(ifr.ifr_name));
        ifr.ifr_flags = (isLinkStateUp) ? nn::socket::IfrFlag::Iff_Up : nn::socket::IfrFlag::Iff_None;
        if ((mibsock = nn::socket::SocketExempt(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) == -1)
        {
            CFG_BREAK_UPON_ERROR(ResultSocketOpenError());
        }

        if (nn::socket::Ioctl(mibsock, nn::socket::IoctlCommandPrivate::SiocSIfFlags, (caddr_t)&ifr, sizeof(ifr)) < 0)
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            CFG_LOG_WARN("Ioctl( SiocSIfFlags ) failed, errno=%d.\n", lastErrno);
            switch (lastErrno)
            {
            case nn::socket::Errno::ENxIo:
                result = ResultIfRemoved();
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }
            break;
        }
    }while (false);

    if (mibsock >= 0)
    {
        nn::socket::Close(mibsock);
    }

    return result;
}

Result Interface::CheckLinkState(bool *pReturnedLinkState)
{
    enum
    {
        IFM_AVALID      = 0x00000001,      /* Active bit valid */
        IFM_ACTIVE      = 0x00000002,      /* Interface attached to working net */
        IFM_NMASK       = 0x000000e0,      /* Network type */
        IFM_ETHER       = 0x00000020,
        IFM_IEEE80211   = 0x00000080,
    };

    int mibsock = -1;
    bool linkUp = false;
    Result result = ResultSuccess();
    do
    {
        nn::socket::IfMediaReq ifmr;
        memset(&ifmr, 0, sizeof(ifmr));
        nn::util::Strlcpy(ifmr.ifm_name, m_IfName, sizeof(ifmr.ifm_name));

        if ((mibsock = nn::socket::SocketExempt(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)) == -1)
        {
            CFG_BREAK_UPON_ERROR(ResultSocketOpenError());
        }

        if (nn::socket::Ioctl(mibsock, nn::socket::IoctlCommandPrivate::SiocGIfMedia, (caddr_t)&ifmr, sizeof(ifmr)) < 0)
        {
            nn::socket::Errno lastErrno = nn::socket::GetLastError();
            CFG_LOG_WARN("Ioctl( SiocGIfMedia ) failed, errno=%d.\n", lastErrno);
            switch (lastErrno)
            {
            case nn::socket::Errno::ENxIo:
                result = ResultIfRemoved();
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }
            break;
        }

        if (ifmr.ifm_status & IFM_AVALID)
        {
            switch (ifmr.ifm_active & IFM_NMASK)
            {
            case IFM_ETHER:
            case IFM_IEEE80211:
                linkUp = (ifmr.ifm_status & IFM_ACTIVE) ? true : false;
                break;
            default:
                break;
            }
        }

    }while (false);

    if (mibsock >= 0)
    {
        nn::socket::Close(mibsock);
    }

    if (pReturnedLinkState != NULL)
    {
        *pReturnedLinkState = linkUp;
    }

    return result;
}

} // namespace cfg
} // namespace bsdsocket
} // namespace nn
