﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Assert.h"
#include "../kern_Kernel.h"
#include "../kern_KSession.h"
#include "../kern_KServerSession.h"
#include "../kern_KClientSession.h"
#include "../kern_KSynchronization.h"
#include "../kern_KProcess.h"
#include "../kern_KHandleTable.h"
#include "../kern_KThread.h"
#include "../kern_DebugString.h"
#include "../kern_PageTableSelect.h"
#include "../kern_KLightSession.h"
#include <nn/svc/ipc/svc_SessionMessage.h>
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include "../kern_KScopedResourceLimitTester.h"

namespace nn { namespace kern { namespace svc {

namespace {
/*!
    @brief サーバーにリクエストを送信します。

    @param[in] session セッションのハンドル

    @return 関数の実行結果を返します。
*/
NN_FORCEINLINE Result SvcSendSyncRequestImpl(uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KClientSession* pSession = handleTable.GetObject<KClientSession>(session);
    if (!pSession)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KClientSession> autoCloserClient(pSession);

    KSession* pParent = pSession->GetParent();

    // 親の Open() は不要と思われるが、念のため残しておく
    pParent->Open();
    KScopedAutoObject<KSession> autoCloser(pParent);

#if defined(NN_KERN_ENABLE_IPC_PROFILE)
    // IPCの回数を計測します。
    GetCurrentThread().IncrementNumIpc();
#endif  // ifdef NN_KERN_ENABLE_IPC_PROFILE

    return pSession->SendSyncRequest(pMessage, bufferSize);
}

NN_FORCEINLINE Result SvcSendSyncRequestWithUserBuffer(uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    if ((pMessage % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((bufferSize % NN_KERN_FINEST_PAGE_SIZE) != 0 || bufferSize == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(pMessage < pMessage + bufferSize))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    Result result;
    KProcessPageTable& pt = GetCurrentProcess().GetPageTable();

    result = pt.LockForIpcUserBuffer(pMessage, bufferSize);
    if (result.IsSuccess())
    {
        NN_KERN_ASSERT(pMessage);
        result = SvcSendSyncRequestImpl(pMessage, bufferSize, session);
        Result resultCheck = pt.UnlockForIpcUserBuffer(pMessage, bufferSize);
        if (resultCheck.IsFailure())
        {
            NN_WARNING(false, "failed to unlock IPC user buffer");
        }
        if (result.IsSuccess())
        {
            result = resultCheck;
        }
    }
    return result;
}

NN_FORCEINLINE Result SvcSendSyncRequest(nn::svc::Handle session)
{
    return SvcSendSyncRequestImpl(0, 0, session);
}

NN_FORCEINLINE Result SvcSendAsyncRequestWithUserBufferImpl(nn::svc::Handle* pHandle, uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    Result result;

    KProcess& process = GetCurrentProcess();
    KHandleTable& handleTable = process.GetHandleTable();

    KScopedResourceLimitTester eventTester(&process, nn::svc::LimitableResource_EventCountMax);
    if (eventTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    KClientSession* pSession = handleTable.GetObject<KClientSession>(session);
    if (!pSession)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KClientSession> autoCloserClient(pSession);

    KSession* pParent = pSession->GetParent();

    // 親の Open() は不要と思われるが、念のため残しておく
    pParent->Open();
    KScopedAutoObject<KSession> autoCloser(pParent);

#if defined(NN_KERN_ENABLE_IPC_PROFILE)
    // IPCの回数を計測します。
    GetCurrentThread().IncrementNumIpc();
#endif  // ifdef NN_KERN_ENABLE_IPC_PROFILE

    KEvent* pEvent = KEvent::Create();
    if (pEvent)
    {
        pEvent->Initialize();
        eventTester.Accepted();

        result = KEvent::Register(pEvent);

        if (result.IsSuccess())
        {
            result = handleTable.Add(pHandle, &pEvent->GetReadableEvent());
            if (result.IsSuccess())
            {
                result = pSession->SendAsyncRequest(&pEvent->GetWritableEvent(), pMessage, bufferSize);
                if (result.IsFailure())
                {
                    handleTable.Remove(*pHandle);
                }
            }
        }
        pEvent->GetWritableEvent().Close();
        pEvent->GetReadableEvent().Close();
    }
    else
    {
        result = nn::svc::ResultOutOfResource();
    }

    return result;
}

NN_FORCEINLINE Result SvcSendAsyncRequestWithUserBuffer(nn::svc::Handle* pEvent, uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    if ((pMessage % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((bufferSize % NN_KERN_FINEST_PAGE_SIZE) != 0 || bufferSize == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(pMessage < pMessage + bufferSize))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    Result result;
    KProcessPageTable& pt = GetCurrentProcess().GetPageTable();

    result = pt.LockForIpcUserBuffer(pMessage, bufferSize);
    if (result.IsSuccess())
    {
        result = SvcSendAsyncRequestWithUserBufferImpl(pEvent, pMessage, bufferSize, session);
        if (result.IsFailure())
        {
            NN_KERN_ASSERT(pMessage);
            Result resultCheck = pt.UnlockForIpcUserBuffer(pMessage, bufferSize);
            if (resultCheck.IsFailure())
            {
                NN_WARNING(false, "failed to unlock IPC user buffer");
            }
        }
    }

    return result;
}

NN_FORCEINLINE Result SvcReplyAndReceiveImpl2(int32_t* pIndex, uintptr_t pMessage, size_t bufferSize, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    Result result;

    if (replyTarget != nn::svc::INVALID_HANDLE_VALUE)
    {
        // セッションを取得
        KServerSession* pSession = GetCurrentProcess().GetHandleTable().GetObject<KServerSession>(replyTarget);
        if (!pSession)
        {
            return nn::svc::ResultInvalidHandle();
        }

        KScopedAutoObject<KServerSession> autoCloser(pSession);
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
        GetCurrentThread().IncrementNumIpcReply();
#endif
        result = pSession->SendReply(pMessage, bufferSize);
        if (result.IsFailure())
        {
            *pIndex = -1;
            return result;
        }
    }

    {
        int64_t timeout;
        if (ns > 0)
        {
            nn::svc::Tick offset(TimeSpan::FromNanoSeconds(ns));
            if (NN_LIKELY(offset > 0))
            {
                timeout = KHardwareTimer::GetTick() + offset + 2;
                if (NN_UNLIKELY(timeout <= 0))
                {
                    timeout = INT64_MAX;
                }
            }
            else
            {
                timeout = INT64_MAX;
            }
        }
        else
        {
            timeout = ns;
        }

        KSynchronizationObject** pObjs = GetCurrentThread().GetSynchronizationObjectBuffer();
        for (;;)
        {
            int32_t index;
            result = Kernel::GetSynchronization().Wait(&index, pObjs, numHandles, timeout);
            if (result <= nn::svc::ResultTimeout())
            {
                return result;
            }
            if (result.IsSuccess())
            {
                KServerSession* pSession = pObjs[index]->DynamicCast<KServerSession*>();
                if (pSession)
                {
#if defined(NN_KERN_ENABLE_IPC_PROFILE)
                    GetCurrentThread().IncrementNumIpcRecv();
#endif
                    result = pSession->ReceiveRequest(pMessage, bufferSize);
                    if (result <= nn::svc::ResultNotFound())
                    {
                        continue;
                    }
                }
            }
            *pIndex = index;
            break;
        }
    }

    return result;
}

NN_FORCEINLINE Result SvcReplyAndReceiveImpl(int32_t* pIndex, uintptr_t pMessage, size_t bufferSize, KUserPointer<const nn::svc::Handle*> userHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    if (!(0 <= numHandles && numHandles <= nn::svc::ArgumentHandleCountMax))
    {
        return nn::svc::ResultOutOfRange();
    }

    KSynchronizationObject** objs = GetCurrentThread().GetSynchronizationObjectBuffer();
    nn::svc::Handle* pHandle = GetCurrentThread().GetHandleBuffer();
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    if (numHandles > 0)
    {
        if (!GetCurrentProcess().GetPageTable().IsInRange(KProcessAddress(GetUnsafePointer(userHandles)), numHandles * sizeof(nn::svc::Handle)))
        {
            return nn::svc::ResultInvalidPointer();
        }

        if (userHandles.CopyArrayTo(pHandle, numHandles).IsFailure())
        {
            return nn::svc::ResultInvalidPointer();
        }

        if (!handleTable.GetObject<KSynchronizationObject>(objs, pHandle, numHandles))
        {
            return nn::svc::ResultInvalidHandle();
        }
    }

    Result result = SvcReplyAndReceiveImpl2(pIndex, pMessage, bufferSize, numHandles, replyTarget, ns);

    for (int j = 0; j < numHandles; ++j)
    {
        objs[j]->Close();
    }

    return result;
}

NN_FORCEINLINE Result SvcReplyAndReceive(int32_t* pIndex, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    return SvcReplyAndReceiveImpl(pIndex, 0, 0, pHandles, numHandles, replyTarget, ns);
}

NN_FORCEINLINE Result SvcReplyAndReceiveWithUserBuffer(int32_t* pIndex, uintptr_t pMessage, size_t bufferSize, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    if ((pMessage % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((bufferSize % NN_KERN_FINEST_PAGE_SIZE) != 0 || bufferSize == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(pMessage < pMessage + bufferSize))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    Result result;
    KProcessPageTable& pt = GetCurrentProcess().GetPageTable();

    result = pt.LockForIpcUserBuffer(pMessage, bufferSize);
    if (result.IsSuccess())
    {
        NN_KERN_ASSERT(pMessage);
        result = SvcReplyAndReceiveImpl(pIndex, pMessage, bufferSize, pHandles, numHandles, replyTarget, ns);
        Result resultCheck = pt.UnlockForIpcUserBuffer(pMessage, bufferSize);
        if (resultCheck.IsFailure())
        {
            NN_WARNING(false, "failed to unlock IPC user buffer");
        }
        if (result.IsSuccess())
        {
            result = resultCheck;
        }
    }
    return result;
}
NN_FORCEINLINE Result SvcSendSyncRequestLight(nn::svc::Handle session, Bit32* pArgs)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KLightClientSession* pSession = handleTable.GetObject<KLightClientSession>(session);
    if (!pSession)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KLightClientSession> autoCloser(pSession);
    return pSession->SendRequest(pArgs);
}
NN_FORCEINLINE Result SvcReplyAndReceiveLight(nn::svc::Handle session, Bit32* pArgs)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KLightServerSession* pSession = handleTable.GetObject<KLightServerSession>(session);
    if (!pSession)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KLightServerSession> autoCloser(pSession);
    return pSession->ReplyAndReceive(pArgs);
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcSendSyncRequest32(nn::svc::Handle session)
{
    Result result = SvcSendSyncRequest(session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceive32(int32_t* pIndex, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    Result result = SvcReplyAndReceive(pIndex, pHandles, numHandles, replyTarget, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendSyncRequestWithUserBuffer32(uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    Result result = SvcSendSyncRequestWithUserBuffer(pMessage, bufferSize, session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendAsyncRequestWithUserBuffer32(nn::svc::Handle* pEvent, uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    Result result = SvcSendAsyncRequestWithUserBuffer(pEvent, pMessage, bufferSize, session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceiveWithUserBuffer32(int32_t* pIndex, uintptr_t pMessage, size_t bufferSize, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    Result result = SvcReplyAndReceiveWithUserBuffer(pIndex, pMessage, bufferSize, pHandles, numHandles, replyTarget, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendSyncRequestLight32(nn::svc::Handle session, Bit32* pArgs)
{
    Result result = SvcSendSyncRequestLight(session, pArgs);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceiveLight32(nn::svc::Handle session, Bit32* pArgs)
{
    Result result = SvcReplyAndReceiveLight(session, pArgs);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcSendSyncRequest64(nn::svc::Handle session)
{
    Result result = SvcSendSyncRequest(session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceive64(int32_t* pIndex, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    Result result = SvcReplyAndReceive(pIndex, pHandles, numHandles, replyTarget, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendSyncRequestWithUserBuffer64(uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    Result result = SvcSendSyncRequestWithUserBuffer(pMessage, bufferSize, session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendAsyncRequestWithUserBuffer64(nn::svc::Handle* pEvent, uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    Result result = SvcSendAsyncRequestWithUserBuffer(pEvent, pMessage, bufferSize, session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceiveWithUserBuffer64(int32_t* pIndex, uintptr_t pMessage, size_t bufferSize, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    Result result = SvcReplyAndReceiveWithUserBuffer(pIndex, pMessage, bufferSize, pHandles, numHandles, replyTarget, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendSyncRequestLight64(nn::svc::Handle session, Bit32* pArgs)
{
    Result result = SvcSendSyncRequestLight(session, pArgs);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceiveLight64(nn::svc::Handle session, Bit32* pArgs)
{
    Result result = SvcReplyAndReceiveLight(session, pArgs);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcSendSyncRequest64From32(nn::svc::Handle session)
{
    Result result = SvcSendSyncRequest(session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceive64From32(int32_t* pIndex, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    Result result = SvcReplyAndReceive(pIndex, pHandles, numHandles, replyTarget, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendSyncRequestWithUserBuffer64From32(uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    Result result = SvcSendSyncRequestWithUserBuffer(pMessage, bufferSize, session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendAsyncRequestWithUserBuffer64From32(nn::svc::Handle* pEvent, uintptr_t pMessage, size_t bufferSize, nn::svc::Handle session)
{
    Result result = SvcSendAsyncRequestWithUserBuffer(pEvent, pMessage, bufferSize, session);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceiveWithUserBuffer64From32(int32_t* pIndex, uintptr_t pMessage, size_t bufferSize, KUserPointer<const nn::svc::Handle*> pHandles, int32_t numHandles, nn::svc::Handle replyTarget, int64_t ns)
{
    Result result = SvcReplyAndReceiveWithUserBuffer(pIndex, pMessage, bufferSize, pHandles, numHandles, replyTarget, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSendSyncRequestLight64From32(nn::svc::Handle session, Bit32* pArgs)
{
    Result result = SvcSendSyncRequestLight(session, pArgs);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReplyAndReceiveLight64From32(nn::svc::Handle session, Bit32* pArgs)
{
    Result result = SvcReplyAndReceiveLight(session, pArgs);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)

}}}
