﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_KProcess.h"
#include "kern_MemoryMap.h"
#include "kern_SvcHandlers.h"

namespace nn { namespace kern { namespace svc {
namespace {
bool IsKernelAddress(uintptr_t addr)
{
    return (NN_KERN_V_ADDR_KERNEL <= addr && addr < NN_KERN_V_ADDR_KERNEL_END);
}

Result SvcArbitrateUnlock(uintptr_t addr)
{
    if (IsKernelAddress(addr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if ((addr % sizeof(Bit32)) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }

    return GetCurrentProcess().SignalToAddress(addr);
}

Result SvcArbitrateLock(nn::svc::Handle handle, uintptr_t addr, Bit32 ownValue)
{
    if (IsKernelAddress(addr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if ((addr % sizeof(Bit32)) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }

    return GetCurrentProcess().WaitForAddress(handle, addr, ownValue);
}

void SvcSignalProcessWideKey(uintptr_t cvKey, int32_t num)
{
    GetCurrentProcess().SignalConditionVariable((cvKey & ~(sizeof(Bit32) - 1)), num);
}

Result SvcWaitProcessWideKeyAtomic(uintptr_t addr, uintptr_t cvKey, Bit32 ownValue, int64_t ns)
{
    if (IsKernelAddress(addr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if ((addr % sizeof(Bit32)) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }

    int64_t timeout;
    if (ns > 0)
    {
        nn::svc::Tick offset(TimeSpan::FromNanoSeconds(ns));
        if (NN_LIKELY(offset > 0))
        {
            timeout = KHardwareTimer::GetTick() + offset + 2;
            if (NN_UNLIKELY(timeout <= 0))
            {
                timeout = INT64_MAX;
            }
        }
        else
        {
            timeout = INT64_MAX;
        }
    }
    else
    {
        timeout = ns;
    }

    return GetCurrentProcess().WaitConditionVariable(addr, (cvKey & ~(sizeof(Bit32) - 1)), ownValue, timeout);
}

}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcArbitrateUnlock32(uintptr_t addr)
{
    Result result = SvcArbitrateUnlock(addr);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcArbitrateLock32(nn::svc::Handle handle, uintptr_t addr, Bit32 ownValue)
{
    Result result = SvcArbitrateLock(handle, addr, ownValue);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcSignalProcessWideKey32(uintptr_t cvKey, int32_t num)
{
    SvcSignalProcessWideKey(cvKey, num);
    ClearSvcOutRegisters();
}
Result SvcWaitProcessWideKeyAtomic32(uintptr_t addr, uintptr_t cvKey, Bit32 ownValue, int64_t ns)
{
    Result result = SvcWaitProcessWideKeyAtomic(addr, cvKey, ownValue, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcArbitrateUnlock64(uintptr_t addr)
{
    Result result = SvcArbitrateUnlock(addr);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcArbitrateLock64(nn::svc::Handle handle, uintptr_t addr, Bit32 ownValue)
{
    Result result = SvcArbitrateLock(handle, addr, ownValue);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcSignalProcessWideKey64(uintptr_t cvKey, int32_t num)
{
    SvcSignalProcessWideKey(cvKey, num);
    ClearSvcOutRegisters();
}
Result SvcWaitProcessWideKeyAtomic64(uintptr_t addr, uintptr_t cvKey, Bit32 ownValue, int64_t ns)
{
    Result result = SvcWaitProcessWideKeyAtomic(addr, cvKey, ownValue, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcArbitrateUnlock64From32(uintptr_t addr)
{
    Result result = SvcArbitrateUnlock(addr);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcArbitrateLock64From32(nn::svc::Handle handle, uintptr_t addr, Bit32 ownValue)
{
    Result result = SvcArbitrateLock(handle, addr, ownValue);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcSignalProcessWideKey64From32(uintptr_t cvKey, int32_t num)
{
    SvcSignalProcessWideKey(cvKey, num);
    ClearSvcOutRegisters();
}
Result SvcWaitProcessWideKeyAtomic64From32(uintptr_t addr, uintptr_t cvKey, Bit32 ownValue, int64_t ns)
{
    Result result = SvcWaitProcessWideKeyAtomic(addr, cvKey, ownValue, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
