﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include "../kern_Platform.h"
#include "../kern_CPUSelect.h"
#include "../kern_Utility.h"
#include "../kern_HardwareTimerSelect.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include <algorithm>

namespace nn { namespace kern { namespace svc {
namespace
{
class CacheOperation
{
public:
    virtual void OperateRange(void* p, size_t size) const = 0;
};

Result DoProcessCache(const CacheOperation& op, KProcessPageTable* pPageTable, uintptr_t addr, size_t size)
{
    uintptr_t pageAlignedBegin = RoundDown(addr, NN_KERN_FINEST_PAGE_SIZE);
    uintptr_t pageAlignedEnd = RoundUp(addr + size, NN_KERN_FINEST_PAGE_SIZE);
    size_t numPages = (pageAlignedEnd - pageAlignedBegin) / NN_KERN_FINEST_PAGE_SIZE;

    KScopedPageGroup pg(pPageTable->GetBlockInfoManager());

    Result result = pPageTable->MakePageGroupAndOpen(&pg.GetPageGroup(), pageAlignedBegin, numPages,
            KMemoryState_FlagsMemory, KMemoryState_FlagsMemory,
            KMemoryPermission_UserRead, KMemoryPermission_UserRead,
            KMemoryAttribute_Uncached, 0);
    if (result.IsSuccess())
    {
        uintptr_t blockAddr = pageAlignedBegin;
        size_t leftSize = size;
        for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter(); it++)
        {
            KVirtualAddress opAddr = it->GetBlockAddr();
            size_t opSize = it->GetNumPages() * NN_KERN_FINEST_PAGE_SIZE;

            if (blockAddr < addr)
            {
                opAddr += addr - blockAddr;
            }
            if (opSize > leftSize)
            {
                opSize = leftSize;
            }
            op.OperateRange(GetUntypedPointer(opAddr), opSize);
            leftSize -= opSize;
            blockAddr += it->GetNumPages() * NN_KERN_FINEST_PAGE_SIZE;
        }
        NN_KERN_ASSERT(leftSize == 0);
        pg.GetPageGroup().Close();
    }

    return result;
}

Result
SvcInvalidateProcessDataCache(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    if (addr != static_cast<uintptr_t>(addr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size != static_cast<size_t>(size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size == 0)
    {
        return ResultSuccess();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    return pProcess->GetPageTable().InvalidateProcessDataCache(addr, size);
}

Result
SvcStoreProcessDataCache( nn::svc::Handle process, uint64_t addr, uint64_t size )
{
    if (addr != static_cast<uintptr_t>(addr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size != static_cast<size_t>(size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size == 0)
    {
        return ResultSuccess();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    KProcessPageTable* pPageTable = &pProcess->GetPageTable();
    if (!pPageTable->IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if (pProcess == &GetCurrentProcess())
    {
        return KCPU::StoreDataCache(reinterpret_cast<void*>(addr), size);
    }
    else
    {
        class StoreCacheOperation: public CacheOperation
        {
        public:
            virtual void OperateRange(void* p, size_t s) const { KCPU::StoreDataCache(p, s); }
        } op;

        return DoProcessCache(op, pPageTable, addr, size);
    }
}

Result
SvcFlushProcessDataCache(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    if (addr != static_cast<uintptr_t>(addr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size != static_cast<size_t>(size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size == 0)
    {
        return ResultSuccess();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(handleTable, process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    KProcessPageTable* pPageTable = &pProcess->GetPageTable();
    if (!pPageTable->IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if (pProcess == &GetCurrentProcess())
    {
        return KCPU::FlushDataCache(reinterpret_cast<void*>(addr), size);
    }
    else
    {
        class FlushCacheOperation: public CacheOperation
        {
        public:
            virtual void OperateRange(void* p, size_t s) const { KCPU::FlushDataCache(p, s); }
        } op;

        return DoProcessCache(op, pPageTable, addr, size);
    }
}

void
SvcFlushEntireDataCache()
{
#if defined NN_BUILD_CONFIG_HARDWARE_NX
    int64_t timeout = KHardwareTimer::GetTick() + nn::svc::Tick(TimeSpan::FromMilliSeconds(1));
#endif

    KCPU::FlushEntireDataCache();

#if defined NN_BUILD_CONFIG_HARDWARE_NX
    while (timeout > KHardwareTimer::GetTick())
    {
        asm volatile("yield":::"memory");
    }
#endif
}

Result
SvcFlushDataCache(uintptr_t addr, size_t size)
{
    if (size == 0)
    {
        return ResultSuccess();
    }
    if (!GetCurrentProcess().GetPageTable().IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    return KCPU::FlushDataCache(reinterpret_cast<void*>(addr), size);
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result
SvcInvalidateProcessDataCache32( nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcInvalidateProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result
SvcStoreProcessDataCache32(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcStoreProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result
SvcFlushProcessDataCache32(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcFlushProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void
SvcFlushEntireDataCache32()
{
    SvcFlushEntireDataCache();
    ClearSvcOutRegistersReturnResult();
}
Result
SvcFlushDataCache32(uintptr_t addr, size_t size)
{
    Result result = SvcFlushDataCache(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result
SvcInvalidateProcessDataCache64(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcInvalidateProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result
SvcStoreProcessDataCache64(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcStoreProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result
SvcFlushProcessDataCache64(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcFlushProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void
SvcFlushEntireDataCache64()
{
    SvcFlushEntireDataCache();
    ClearSvcOutRegistersReturnResult();
}
Result
SvcFlushDataCache64(uintptr_t addr, size_t size)
{
    Result result = SvcFlushDataCache(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result
SvcInvalidateProcessDataCache64From32(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcInvalidateProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result
SvcStoreProcessDataCache64From32(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcStoreProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result
SvcFlushProcessDataCache64From32(nn::svc::Handle process, uint64_t addr, uint64_t size)
{
    Result result = SvcFlushProcessDataCache(process, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void
SvcFlushEntireDataCache64From32()
{
    SvcFlushEntireDataCache();
    ClearSvcOutRegistersReturnResult();
}
Result
SvcFlushDataCache64From32(uintptr_t addr, size_t size)
{
    Result result = SvcFlushDataCache(addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)

}}}
