﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <type_traits>
#include <nn/nn_Common.h>
#include "../kern_MemoryCopySelect.h"
#include "../kern_Result.h"

namespace nn
{
    namespace kern
    {
        namespace svc
        {

            namespace detail
            {

                template <typename T, typename = void>
                class KUserPointerImpl
                {
                protected:
                    typedef typename std::remove_const<typename std::remove_pointer<T>::type>::type RawT;

                protected:
                    T   m_Pointer;

                protected:
                    Result CopyTo(RawT* p)
                    {
                        return CopyToImpl(p, sizeof(RawT));
                    }
                    Result CopyFrom(const RawT* p)
                    {
                        return CopyFromImpl(p, sizeof(RawT));
                    }

                    Result CopyArrayEntryTo(RawT* p, int index)
                    {
                        return CopyFromUser(p, m_Pointer + index, sizeof(RawT));
                    }
                    Result CopyArrayEntryFrom(const RawT* p, int index)
                    {
                        return CopyToUser(m_Pointer + index, p, sizeof(RawT));
                    }

                    Result CopyArrayTo(RawT* p, int num)
                    {
                        return CopyToImpl(p, sizeof(RawT) * num);
                    }
                    Result CopyArrayFrom(const RawT* p, int num)
                    {
                        return CopyFromImpl(p, sizeof(RawT) * num);
                    }

                    bool IsNull() const { return m_Pointer == nullptr; }
                    bool IsNotNull() const { return m_Pointer != nullptr; }

                    uintptr_t GetAddress()
                    {
                        return reinterpret_cast<uintptr_t>(m_Pointer);
                    }

                private:
                    Result CopyToImpl(void* p, size_t size)
                    {
                        return CopyFromUser(p, m_Pointer, size);
                    }
                    Result CopyFromImpl(const void* p, size_t size)
                    {
                        return CopyToUser(m_Pointer, p, size);
                    }
                    static Result CopyFromUser(void* pKernel, const void* pUser, size_t size)
                    {
                        bool isSuccess = CopyMemoryFromUser(pKernel, pUser, size);
                        if (!isSuccess)
                        {
                            return nn::svc::ResultInvalidPointer();
                        }
                        return ResultSuccess();
                    }
                    static Result CopyToUser(void* pUser, const void* pKernel, size_t size)
                    {
                        bool isSuccess = CopyMemoryToUser(pUser, pKernel, size);
                        if (!isSuccess)
                        {
                            return nn::svc::ResultInvalidPointer();
                        }
                        return ResultSuccess();
                    }
                };

                template <typename T>
                class KUserPointerImpl<T,
                      typename std::enable_if<(
                              std::alignment_of<typename std::remove_pointer<T>::type>::value >= sizeof(uint32_t)) &&
                          ((sizeof(typename std::remove_pointer<T>::type) % sizeof(uint32_t)) == 0)>::type>
                {
                protected:
                    typedef typename std::remove_const<typename std::remove_pointer<T>::type>::type RawT;

                protected:
                    T   m_Pointer;

                protected:
                    Result CopyTo(RawT* p)
                    {
                        return CopyToImpl(p, sizeof(RawT));
                    }
                    Result CopyFrom(const RawT* p)
                    {
                        return CopyFromImpl(p, sizeof(RawT));
                    }

                    Result CopyArrayEntryTo(RawT* p, int index)
                    {
                        return CopyFromUser(p, m_Pointer + index, sizeof(RawT));
                    }
                    Result CopyArrayEntryFrom(const RawT* p, int index)
                    {
                        return CopyToUser(m_Pointer + index, p, sizeof(RawT));
                    }

                    Result CopyArrayTo(RawT* p, int num)
                    {
                        return CopyToImpl(p, sizeof(RawT) * num);
                    }
                    Result CopyArrayFrom(const RawT* p, int num)
                    {
                        return CopyFromImpl(p, sizeof(RawT) * num);
                    }

                    bool IsNull() const { return m_Pointer == nullptr; }
                    bool IsNotNull() const { return m_Pointer != nullptr; }

                    uintptr_t GetAddress()
                    {
                        return reinterpret_cast<uintptr_t>(m_Pointer);
                    }

                private:
                    Result CopyToImpl(void* p, size_t size)
                    {
                        return CopyFromUser(p, m_Pointer, size);
                    }
                    Result CopyFromImpl(const void* p, size_t size)
                    {
                        return CopyToUser(m_Pointer, p, size);
                    }
                    static Result CopyFromUser(void* pKernel, const void* pUser, size_t size)
                    {
                        bool isSuccess = CopyMemoryFromUserAlign4(pKernel, pUser, size);
                        if (!isSuccess)
                        {
                            return nn::svc::ResultInvalidPointer();
                        }
                        return ResultSuccess();
                    }
                    static Result CopyToUser(void* pUser, const void* pKernel, size_t size)
                    {
                        bool isSuccess = CopyMemoryToUserAlign4(pUser, pKernel, size);
                        if (!isSuccess)
                        {
                            return nn::svc::ResultInvalidPointer();
                        }
                        return ResultSuccess();
                    }
                };

                template <>
                class KUserPointerImpl<const char*>
                {
                protected:
                    const char* m_Pointer;

                protected:
                    Result CopyStringTo(char* p, int num)
                    {
                        return CopyStringToImpl(p, sizeof(char) * num);
                    }
                    Result CopyArrayEntryTo(char* p, int index)
                    {
                        return CopyFromUser(p, m_Pointer + index, sizeof(char));
                    }

                    bool IsNull() const { return m_Pointer == nullptr; }
                    bool IsNotNull() const { return m_Pointer != nullptr; }

                private:
                    Result CopyStringToImpl(char* p, size_t size)
                    {
                        int32_t ret = CopyStringFromUser(p, m_Pointer, size);
                        if (ret > 0)
                        {
                            return ResultSuccess();
                        }
                        else
                        {
                            return nn::svc::ResultInvalidPointer();
                        }
                    }
                    static Result CopyFromUser(void* pKernel, const void* pUser, size_t size)
                    {
                        bool isSuccess = CopyMemoryFromUser(pKernel, pUser, size);
                        if (!isSuccess)
                        {
                            return nn::svc::ResultInvalidPointer();
                        }
                        return ResultSuccess();
                    }
                };
            }


            template <typename T, typename U = void>
            class KUserPointer;

            template <typename T>
            uintptr_t GetAddressOf(const KUserPointer<T>& x,
                    typename std::enable_if<std::is_void<typename KUserPointer<T>::RawT>::value>::type* = 0);
            template <typename T>
            T GetUnsafePointer(const KUserPointer<T>& x);


            template <typename T>
            class KUserPointer<T, typename std::enable_if<
                    std::is_pointer<T>::value
                    && std::is_const<typename std::remove_pointer<T>::type>::value
                >::type>
                : public detail::KUserPointerImpl<T>
            {
            public:
                using detail::KUserPointerImpl<T>::CopyTo;
                //using detail::KUserPointerImpl<T>::CopyFrom;
                using detail::KUserPointerImpl<T>::CopyArrayEntryTo;
                //using detail::KUserPointerImpl<T>::CopyArrayEntryFrom;
                using detail::KUserPointerImpl<T>::CopyArrayTo;
                //using detail::KUserPointerImpl<T>::CopyArrayFrom;
                using detail::KUserPointerImpl<T>::IsNull;
                using detail::KUserPointerImpl<T>::IsNotNull;

                template <typename U>
                friend uintptr_t GetAddressOf(const KUserPointer<U>& x,
                        typename std::enable_if<std::is_void<typename KUserPointer<U>::RawT>::value>::type*);
                template <typename U>
                friend U GetUnsafePointer(const KUserPointer<U>& x);
            };

            template <>
            class KUserPointer<const char*>
                : public detail::KUserPointerImpl<const char*>
            {
            public:
                using detail::KUserPointerImpl<const char*>::CopyArrayEntryTo;
                using detail::KUserPointerImpl<const char*>::CopyStringTo;
                using detail::KUserPointerImpl<const char*>::IsNull;
                using detail::KUserPointerImpl<const char*>::IsNotNull;
                template <typename U>
                friend U GetUnsafePointer(const KUserPointer<U>& x);
            };


            template <typename T>
            class KUserPointer<T, typename std::enable_if<
                    std::is_pointer<T>::value
                    && !std::is_const<typename std::remove_pointer<T>::type>::value
                >::type>
                : public detail::KUserPointerImpl<T>
            {
            public:
                //using detail::KUserPointerImpl<T>::CopyTo;
                using detail::KUserPointerImpl<T>::CopyFrom;
                //using detail::KUserPointerImpl<T>::CopyArrayEntryTo;
                using detail::KUserPointerImpl<T>::CopyArrayEntryFrom;
                //using detail::KUserPointerImpl<T>::CopyArrayTo;
                using detail::KUserPointerImpl<T>::CopyArrayFrom;
                using detail::KUserPointerImpl<T>::IsNull;
                using detail::KUserPointerImpl<T>::IsNotNull;

                template <typename U>
                friend uintptr_t GetAddressOf(const KUserPointer<U>& x,
                        typename std::enable_if<std::is_void<typename KUserPointer<U>::RawT>::value>::type*);
                template <typename U>
                friend U GetUnsafePointer(const KUserPointer<U>& x);
            };

            template <typename T>
            uintptr_t GetAddressOf(const KUserPointer<T>& x,
                    typename std::enable_if<std::is_void<typename KUserPointer<T>::RawT>::value>::type*)
            {
                return reinterpret_cast<uintptr_t>(x.m_Pointer);
            }
            template <typename T>
            T GetUnsafePointer(const KUserPointer<T>& x)
            {
                return x.m_Pointer;
            }

        }
    }
}

