﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include "kern_Platform.h"
namespace nn { namespace kern {

#if defined NN_KERN_ENABLE_KERNEL_TRACE

const int TraceDataTypeGroupNum = 4;

/******************************************************************
* Trace Control Header
* Summary:
* Used keep track of information of the Trace Control Buffer
* which can be parsed by the windows tool
*
* Size:
* 32 bytes
*
* Variables:
* magic:         Stores a magic number to easily identify where the trace controller starts.
* version:       Keeps current version of the kernel payload tracing data.
* length:        Length of the trace control header
* flags:         Flags reserved for possible future use.
* isKernel64bit: Tells us if the kernel traced is 64 bit.
* dataTypeMask:  Stores a bit fields of what events we want to store in the trace buffer.
*********************************************************************/
struct TraceControlHeader
{
    uint32_t m_Magic;
    uint32_t m_Version;
    uint16_t m_Length;
    uint8_t  m_Flags;
    uint8_t  m_IsKernel64bit;
    NN_PADDING4;
    uint64_t m_DataTypeMask[TraceDataTypeGroupNum];
};

enum TraceDataType
{
    TraceDataType_SyscallEntry32                = 0x000,
    TraceDataType_SyscallEntry32_1              = 0x001,
    TraceDataType_SyscallEntry32_2              = 0x002,

    TraceDataType_SyscallExit32                 = 0x010,
    TraceDataType_SyscallExit32_1               = 0x011,
    TraceDataType_SyscallExit32_2               = 0x012,

    TraceDataType_SyscallEntry64From32          = 0x020,
    TraceDataType_SyscallEntry64From32_1        = 0x021,
    TraceDataType_SyscallEntry64From32_2        = 0x022,
    TraceDataType_SyscallEntry64From32_3        = 0x023,

    TraceDataType_SyscallExit64From32           = 0x030,
    TraceDataType_SyscallExit64From32_1         = 0x031,
    TraceDataType_SyscallExit64From32_2         = 0x032,
    TraceDataType_SyscallExit64From32_3         = 0x033,

    TraceDataType_SyscallEntry64                = 0x040,
    TraceDataType_SyscallEntry64_1              = 0x041,
    TraceDataType_SyscallEntry64_2              = 0x042,
    TraceDataType_SyscallEntry64_3              = 0x043,

    TraceDataType_SyscallExit64                 = 0x050,
    TraceDataType_SyscallExit64_1               = 0x051,
    TraceDataType_SyscallExit64_2               = 0x052,
    TraceDataType_SyscallExit64_3               = 0x053,

    TraceDataType_ExceptionEntry32              = 0x100,
    TraceDataType_ExceptionExit32               = 0x110,

    TraceDataType_ExceptionEntry64              = 0x120,
    TraceDataType_ExceptionEntry64_1            = 0x121,
    TraceDataType_ExceptionEntry64_2            = 0x122,

    TraceDataType_ExceptionExit64               = 0x130,
    TraceDataType_UserExceptionEntry32          = 0x140,
    TraceDataType_UserExceptionExit32           = 0x150,

    TraceDataType_UserExceptionEntry64          = 0x160,
    TraceDataType_UserExceptionEntry64_1        = 0x161,
    TraceDataType_UserExceptionEntry64_2        = 0x162,

    TraceDataType_UserExceptionExit64           = 0x170,
    TraceDataType_UserExceptionEntry64From32    = 0x180,
    TraceDataType_UserExceptionExit64From32     = 0x190,

    TraceDataType_ThreadSwitch                  = 0x200,
};

/******************************************************************
* Space
* Summary:
* Helps Transport Control header to keep trace of current placement
* in the Trace Control Buffer
*
* Size:
* 32 bit: 8 bytes
* 64 bit: 16 bytes
* Variables:
* sequence: The latest sequence number.
* latest:   Pointer to the latest trace record.
*********************************************************************/
struct TraceSpace
{
    uint32_t m_Sequence;   // first for comparing one exchanging both on ARM
    uint32_t m_Latest;     // pointer to latest trace record
    bool operator!= (const TraceSpace& other) //required for Interlocked::CompareAndSwapWeak
    {
        return (m_Sequence != other.m_Sequence) || (m_Latest != other.m_Latest);
    }
};

/******************************************************************
* Transport Control
* Summary:
* Helps place the trace record into the next available location
* in the Trace Buffer
*
* Size:
* 32 bit: 32 bytes
* 64 bit: 48 bytes
*
* Variables:
* header:   Helps clearly identify the start of the Transport header
* version:  Tells us what version of transport control we are using.
* unitSize: The size of a block of memory for a trace event. Currently set to 64 bytes.
* space:    Keeps track of where the next trace should. Also helps keeps memory allocation of the trace buffer thread safe.
* start:    Start of the trace buffer memory.
* end:      End of the trace buffer memory.
*********************************************************************/
struct TraceTransportControl
{
    uint64_t     m_Header;
    uint32_t     m_Version;
    uint16_t     m_UnitSize;
    NN_PADDING2;
    TraceSpace   m_Space;
    uintptr_t    m_Start;
    uintptr_t    m_End;
};

/******************************************************************
* Trace Record Common Header
* Summary:
* Information that is found in every Trace Record
*
* Size:
* 32 bytes
*
* Variables:
* m_Type:      Stores the type of trace record (ex. SycallEntry32, ExceptionEntry64, etc.).  The different types are kept in an enum table called TraceDataType.
* m_Cid:       Stores Core id of trace.
* m_Tid:       Stores Thread id of trace.
* m_Pid:       Stores Process id of trace.
* m_Timestamp: Stores the time in ticks the trace occurred.
*********************************************************************/
struct TraceRecordCommonHeader
{
    uint32_t m_Sequence;
    uint16_t m_Type;
    uint8_t  m_Cid;
    NN_PADDING1;
    uint64_t m_Tid;
    uint64_t m_Pid;
    uint64_t m_Timestamp;
};

/******************************************************************
* Trace Record
* Summary:
* Generalized Trace Record that has a 32 byte payload
*
* Size:
* 64 bytes
*
* Variables:
* m_Data: Generalized data
*********************************************************************/
struct TraceRecord
{
    TraceRecordCommonHeader m_Header;
    uint64_t m_Data[4];
};

/******************************************************************
* Trace
* Summary:
* Takes given data and places it in the Trace Control Buffer
*
* Input:
* traceDataType: Describes what type of trace is being passed
* arg1-7:        arguments passed in to be placed into the trace control buffer
*******************************************************************/
void Trace(uint16_t traceDataType, uint64_t arg1, uint64_t arg2, uint64_t arg3, uint64_t arg4);

/******************************************************************
* InitializeTrace
* Summary:
* Helps with setting up Trace Control Header and Trace Control Buffer.
*******************************************************************/
void InitializeTrace();

/******************************************************************
* KernelTraceStart
* Summary:
* Starts recording traces
*******************************************************************/
void KernelTraceStart();

/******************************************************************
* KernelTraceStop
* Summary:
* Stops recording traces
*******************************************************************/
void KernelTraceStop();

#define NN_KERN_TRACE(a,b,c,d,e) ::nn::kern::Trace(a, b, c, d, e)

#define NN_KERN_TRACE_START()    ::nn::kern::KernelTraceStart()

#define NN_KERN_TRACE_STOP()     ::nn::kern::KernelTraceStop()

#define NN_KERN_TRACE_THREAD_SWITCH(b) \
    NN_KERN_TRACE(TraceDataType_ThreadSwitch, b->GetId(), b->GetPriority(), 0, 0)

#define NN_KERN_TRACE_SYSTEM_CALL_ENTRY_64(a,b)                               \
    do                                                                        \
    {                                                                         \
        NN_KERN_TRACE(TraceDataType_SyscallEntry64_1, a,    b[0], b[1], b[2]);\
        NN_KERN_TRACE(TraceDataType_SyscallEntry64_2, b[3], b[4], b[5], b[6]);\
        NN_KERN_TRACE(TraceDataType_SyscallEntry64_3, b[7], 0, 0, 0);         \
    }while(NN_STATIC_CONDITION(0))

#define NN_KERN_TRACE_SYSTEM_CALL_ENTRY_64_FROM_32(a,b)                             \
    do                                                                              \
    {                                                                               \
        NN_KERN_TRACE(TraceDataType_SyscallEntry64From32_1, a,    b[0], b[1], b[2]);\
        NN_KERN_TRACE(TraceDataType_SyscallEntry64From32_2, b[3], b[4], b[5], b[6]);\
        NN_KERN_TRACE(TraceDataType_SyscallEntry64From32_3, b[7], 0, 0, 0);         \
    }while(NN_STATIC_CONDITION(0))

#define NN_KERN_TRACE_SYSTEM_CALL_ENTRY(a,reg)                 \
    do                                                         \
    {                                                          \
        if (GetCurrentProcessPointer()->Is64Bit())   \
        {                                                      \
            NN_KERN_TRACE_SYSTEM_CALL_ENTRY_64(a, reg);        \
        }                                                      \
        else                                                   \
        {                                                      \
            NN_KERN_TRACE_SYSTEM_CALL_ENTRY_64_FROM_32(a, reg);\
        }                                                      \
     }while(NN_STATIC_CONDITION(0))

#define NN_KERN_TRACE_SYSTEM_CALL_EXIT_64(a,b)                                \
     do                                                                       \
     {                                                                        \
         NN_KERN_TRACE(TraceDataType_SyscallExit64_1, a,    b[0], b[1], b[2]);\
         NN_KERN_TRACE(TraceDataType_SyscallExit64_2, b[3], b[4], b[5], b[6]);\
         NN_KERN_TRACE(TraceDataType_SyscallExit64_3, b[7],0, 0, 0);          \
     }while(NN_STATIC_CONDITION(0))

#define NN_KERN_TRACE_SYSTEM_CALL_EXIT_64_FROM_32(a,b)                             \
    do                                                                             \
    {                                                                              \
        NN_KERN_TRACE(TraceDataType_SyscallExit64From32_1, a,    b[0], b[1], b[2]);\
        NN_KERN_TRACE(TraceDataType_SyscallExit64From32_2, b[3], b[4], b[5], b[6]);\
        NN_KERN_TRACE(TraceDataType_SyscallExit64From32_3, b[7], 0, 0, 0);         \
    }while(NN_STATIC_CONDITION(0))

#define NN_KERN_TRACE_SYSTEM_CALL_EXIT(a,reg)                 \
    do                                                        \
    {                                                         \
        if (GetCurrentProcessPointer()->Is64Bit())  \
        {                                                     \
            NN_KERN_TRACE_SYSTEM_CALL_EXIT_64(a, reg);        \
        }                                                     \
        else                                                  \
        {                                                     \
            NN_KERN_TRACE_SYSTEM_CALL_EXIT_64_FROM_32(a, reg);\
        }                                                     \
    }while(NN_STATIC_CONDITION(0))


#else
#define NN_KERN_TRACE(a,b,c,d,e)                         static_cast<void>(0)

#define NN_KERN_TRACE_START()                            static_cast<void>(0)
#define NN_KERN_TRACE_STOP()                             static_cast<void>(0)

#define NN_KERN_TRACE_THREAD_SWITCH(b)                   static_cast<void>(0)

#define NN_KERN_TRACE_SYSTEM_CALL_ENTRY_64(a, b)         static_cast<void>(0)
#define NN_KERN_TRACE_SYSTEM_CALL_ENTRY_64_FROM_32(a, b) static_cast<void>(0)
#define NN_KERN_TRACE_SYSTEM_CALL_ENTRY(a, reg)          static_cast<void>(0)

#define NN_KERN_TRACE_SYSTEM_CALL_EXIT_64(a, b)          static_cast<void>(0)
#define NN_KERN_TRACE_SYSTEM_CALL_EXIT_64_FROM_32(a, b)  static_cast<void>(0)
#define NN_KERN_TRACE_SYSTEM_CALL_EXIT(a, reg)           static_cast<void>(0)
#endif //NN_KERN_ENABLE_KERNEL_TRACE
}}
