﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Kernel.h"
#include "kern_KThread.h"
#include "kern_KProcess.h"
#include "kern_KTrace.h"
#include "kern_SystemControl.h"
#include "kern_DebugSelect.h"
#include "kern_KMemoryLayout.h"
#include "kern_InterlockedSelect.h"
#include "kern_KDumpObject.h"
#include "kern_KScopedSchedulingLockAndSleep.h"
#include <cstring>
#include <algorithm>

namespace nn { namespace kern {

#if defined NN_KERN_ENABLE_KTRACE
struct KTraceHeader
{
    char sig[8];
    uint32_t dataOffset;
    uint32_t size;
    uint32_t index;
    uint32_t num;
};

struct KTraceData
{
    Bit8 cid;
    Bit8 type;
    Bit16 pid;
    Bit32 tid;
    Bit64 tick;
    Bit64 args[6];
};

bool KTrace::g_Enable = false;
KSimpleLock KTrace::g_Lock;
KVirtualAddress KTrace::g_BufferAddress = Null<KVirtualAddress>();
Bit64 KTrace::g_TypeMask;
Bit64 KTrace::g_ProcessMask[2];
size_t KTrace::g_Size;

void KTrace::Initialize(KVirtualAddress addr, size_t size)
{
    if (KTargetSystem::IsDevelopmentHardware())
    {
        size_t offset = (sizeof(KTraceHeader) + sizeof(KTraceData) - 1) / sizeof(KTraceData) * sizeof(KTraceData);
        if (size > offset)
        {
            std::memset(GetUntypedPointer(addr), 0, size);
            KTraceHeader* pHeader = GetTypedPointer<KTraceHeader>(addr);
            std::strncpy(pHeader->sig, "KTRACE0", sizeof(pHeader->sig));
            pHeader->dataOffset = offset;
            pHeader->size = size;
            pHeader->index = 0;
            pHeader->num = (size - offset) / sizeof(KTraceData);
            g_TypeMask = ~0ull;
            for (size_t i = 0; i < sizeof(g_ProcessMask) / sizeof(g_ProcessMask[0]); i++)
            {
                g_ProcessMask[i] = ~0ull;
            }

            g_Size = size;
            g_BufferAddress = addr;
        }

#if defined NN_KERN_START_KTRACE_ON_STARTUP
        KTrace::Start();
#endif
    }
}

void KTrace::Start()
{
    if (g_BufferAddress != Null<KVirtualAddress>())
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&g_Lock);
        KTraceHeader* pHeader = GetTypedPointer<KTraceHeader>(g_BufferAddress);
        pHeader->index = 0;
        KTraceData* pRec = GetTypedPointer<KTraceData>(g_BufferAddress + pHeader->dataOffset + (pHeader->num - 1) * sizeof(KTraceData));
        std::memset(pRec, 0, sizeof(*pRec));
        g_Enable = true;
    }
}

void KTrace::Stop()
{
    if (g_BufferAddress != Null<KVirtualAddress>())
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&g_Lock);
        g_Enable = false;
    }
}

void KTrace::Info()
{
    if (g_BufferAddress != Null<KVirtualAddress>())
    {
        NN_KERN_RELEASE_LOG("Trace Buffer %p (phys = %p) size = %p\n",
                GetAsInteger(g_BufferAddress),
                GetAsInteger(KMemoryLayout::ToLinearPhysicalAddress(g_BufferAddress)),
                g_Size);
        NN_KERN_RELEASE_LOG("    TypeMask =     %016llx\n", g_TypeMask);
        NN_KERN_RELEASE_LOG("    ProcessMask0 = %016llx\n", g_ProcessMask[0]);
        NN_KERN_RELEASE_LOG("    ProcessMask1 = %016llx\n", g_ProcessMask[1]);
    }
}


void KTrace::SetTypeFilter(Bit64 mask)
{
    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_Lock);
    g_TypeMask = mask;
}

void KTrace::SetProcessFilter(uint32_t index, Bit64 mask)
{
    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_Lock);
    if (index < sizeof(g_ProcessMask) / sizeof(g_ProcessMask[0]))
    {
        g_ProcessMask[index] = mask;
    }
}

void KTrace::Record(Bit8 type, Bit64 arg0, Bit64 arg1, Bit64 arg2, Bit64 arg3, Bit64 arg4, Bit64 arg5)
{
    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_Lock);
    if (g_Enable && (g_TypeMask & (1ull << (type % 64))))
    {
        KThread& thread = GetCurrentThread();
        KProcess* pProcess = thread.GetParentPointer();
        uint32_t pindex = (pProcess? pProcess->GetIndex(): 127);
        if (g_ProcessMask[pindex / 64] & (1ull << (pindex % 64)))
        {
            KTraceHeader* pHeader = GetTypedPointer<KTraceHeader>(g_BufferAddress);
            uint32_t index = pHeader->index;
            KTraceData* pRec = GetTypedPointer<KTraceData>(g_BufferAddress + pHeader->dataOffset + index * sizeof(KTraceData));

            pRec->cid = GetCurrentCpuNo();
            pRec->tid = thread.GetId();
            pRec->pid = (pProcess? pProcess->GetId(): ~0);
            pRec->tick =  KHardwareTimer::GetTick();
            pRec->type = type;
            pRec->args[0] = arg0;
            pRec->args[1] = arg1;
            pRec->args[2] = arg2;
            pRec->args[3] = arg3;
            pRec->args[4] = arg4;
            pRec->args[5] = arg5;

            index = index + 1;
            if (index >= pHeader->num)
            {
                index = 0;
            }
            pHeader->index = index;
        }
    }
}

void KTrace::GetCurrentBacktrace(uintptr_t* pOut, size_t num)
{
    KDebug::GetCurrentBacktrace(pOut, num);
}

#ifdef NN_KERN_ENABLE_KTRACE_DUMP

namespace {

static const int KTraceDumpDelayS = 20;
static const uint32_t KTraceDumpCountLimit = 8192;

#ifdef NN_KERN_FOR_DEVELOPMENT
// NN_KERN_ASSERT calls NN_KERNEL_PANIC which takes the KTrace lock
#define NN_KERN_KTRACE_ASSERT(exp)                                                  \
    do                                                                              \
    {                                                                               \
        bool isValid = exp;                                                         \
        if (! isValid)                                                              \
        {                                                                           \
            NN_KERN_RELEASE_LOG("Failed assertion at %s:%d\n", __FILE__, __LINE__); \
            NN_KERN_RELEASE_LOG("%s\n", #exp);                                      \
            for (;;) { /* Never resume */ }                                         \
        }                                                                           \
    } while (0)
#else
#define NN_KERN_KTRACE_ASSERT(exp)
#endif

const char* GetTraceTypeString(nn::Bit8 type)
{
#define CASE(x) case KTrace::KTraceType_ ## x: return #x
    switch (type)
    {
        case 0: return "Empty";
        CASE(Sched);
        CASE(ThreadStateChanged);
        CASE(SvcEnter0);
        CASE(SvcEnter1);
        CASE(SvcLeave0);
        CASE(SvcLeave1);
        CASE(Irq);
        CASE(IpcSend);
        CASE(IpcReceive);
        CASE(IpcReply);
        CASE(SchedChanged);
        CASE(Backtrace0);
        CASE(Backtrace1);
        CASE(CoreMigration);
        CASE(User0);
    default:
        return nullptr;
    }
#undef CASE

    return nullptr;
}

class KStopHandler: public KInterruptHandler
{
public:
    KStopHandler()
    {}

    virtual KInterruptTask* OnInterrupt(int32_t /*interruptRequestNo*/)
    {
        for (;;)
        {
            // Never return
        }

        return nullptr;
    }
};

void SuspendSystem()
{
    // Will only be called by Dump(), which is called at most once by DumpOnce() or DelayedDumpOnce()
    // As a consequence, this function cannot be called twice.
    int set = ((1 << KCPU::NUM_CORE) - 1) & ~(1 << KCPU::GetCurrentCoreNo());
    Kernel::GetInterruptManager().SendIpi(set, KInterruptName::INTR_CORE_ID_STOP);
}

void DumpAdditionalInfo()
{
    KDumpObject::DumpProcess();
    KDumpObject::DumpPort();
    KDumpObject::DumpHandle();
    KDumpObject::DumpThread();
    KDumpObject::DumpMemory();
}

uint64_t MultiplyDivide(uint64_t x, uint32_t m, uint32_t d)
{
    uint64_t x1 = x / d;
    uint64_t x2 = x % d;
    return x1 * m + x2 * m / d;
}

void FormatAndPrintNumber(int64_t x, uint32_t d)
{
    if (x == LLONG_MIN)
    {
        // -LLONG_MIN == LLONG_MIN < 0, but we only want positive values
        NN_KERN_RELEASE_LOG("LLONG_MIN/%u", d);
        return;
    }

    bool isNegative = x < 0;
    if (isNegative)
    {
        x = -x;
    }

    uint64_t integerPart = uint64_t(x) / d;
    const uint32_t fractionalPartMultiplier = 1000000;
    uint64_t fractionalPart = MultiplyDivide(uint64_t(x), fractionalPartMultiplier, d) - integerPart * fractionalPartMultiplier;
    NN_UNUSED(integerPart);
    NN_UNUSED(fractionalPart);

    if (integerPart == 0)
    {
        NN_KERN_RELEASE_LOG(isNegative ? "       -0" : "        0");
    }
    else
    {
        NN_KERN_RELEASE_LOG("%9lld", static_cast<long long> (isNegative ? - integerPart : integerPart));
    }

    NN_KERN_RELEASE_LOG(".%06llu", fractionalPart);
}

static KStopHandler s_StopHandler;
static InterlockedVariable<bool> s_HasAlreadyDumpedTraces;

} // end of anonymous namespace

void KTrace::BindStopInterruptHandler(int32_t coreNo)
{
    Kernel::GetInterruptManager().BindHandler(
        &s_StopHandler,
        KInterruptName::INTR_CORE_ID_STOP,
        coreNo,
        KInterruptController::PriorityLevel_SchInterrupt,
        false,
        false);
}

void KTrace::DelayedDumpOnceWithAdditionalInfo()
{
    NN_KERN_ASSERT( ! KScheduler::IsSchedulerLocked() );

    const int64_t timeout = KHardwareTimer::GetTick() + nn::svc::Tick(TimeSpan::FromSeconds(KTraceDumpDelayS));
    NN_KERN_ASSERT(timeout > 0);

    KHardwareTimer* pTimer;
    while (KHardwareTimer::GetTick() < timeout)
    {
        {
            KScopedSchedulingLockAndSleep sleep(&pTimer, &GetCurrentThread(), timeout);
            GetCurrentThread().SetState(KThread::STATE_WAIT);
        }

        pTimer->CancelTask(&GetCurrentThread());
    }

    KTrace::DumpOnceWithAdditionalInfo();
}

void KTrace::DumpOnceWithAdditionalInfo()
{
    if (s_HasAlreadyDumpedTraces.CompareAndSwap(false, true))
    {
        return;
    }

    DumpAdditionalInfo();
    Dump();
}

void KTrace::DumpOnce()
{
    if (s_HasAlreadyDumpedTraces.CompareAndSwap(false, true))
    {
        return;
    }

    Dump();
}

void KTrace::Dump()
{
    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_Lock);

    SuspendSystem();

    NN_KERN_KTRACE_ASSERT(g_BufferAddress != Null<KVirtualAddress>());

    KTraceHeader* pHeader = GetTypedPointer<KTraceHeader> (g_BufferAddress);

#ifdef NN_KERN_KTRACE_ENABLE_DUMP_TRACE_COUNT_LIMIT
    uint32_t traceToDumpCount = std::min(KTraceDumpCountLimit, pHeader->num);
#else
    NN_UNUSED(KTraceDumpCountLimit);
    uint32_t traceToDumpCount = pHeader->num;
#endif

    const KTraceData* pTraces = GetTypedPointer<KTraceData> (g_BufferAddress + pHeader->dataOffset);
    NN_KERN_KTRACE_ASSERT(pHeader->num > 0);
    NN_KERN_KTRACE_ASSERT(pHeader->num >= traceToDumpCount);

    uint64_t lastTick = pTraces[(pHeader->index - 1 + pHeader->num) % pHeader->num].tick;
    for (uint32_t traceOffset = pHeader->num - 1; traceOffset != pHeader->num - traceToDumpCount - 1; traceOffset--)
    {
        uint32_t traceIndex = (pHeader->index + traceOffset) % pHeader->num;
        const KTraceData& trace = pTraces[traceIndex];

        static_assert(nn::svc::Tick::TICKS_PER_SECOND <= UINT_MAX, "");
        NN_KERN_RELEASE_LOG("%6x ", pHeader->num - 1 - traceOffset);
        FormatAndPrintNumber(trace.tick - lastTick, nn::svc::Tick::TICKS_PER_SECOND);
        NN_KERN_RELEASE_LOG(" [C%1d|P%4hd|T%5d] ", trace.cid, trace.pid, trace.tid);

        const char* pTraceTypeString = GetTraceTypeString(trace.type);
        if (pTraceTypeString != nullptr)
        {
            NN_KERN_RELEASE_LOG("%s(", pTraceTypeString);
        }
        else
        {
            NN_KERN_RELEASE_LOG("UnknownTraceType%d(", trace.type);
        }

        static const int ArgCount = sizeof(trace.args) / sizeof(trace.args[0]);
        for (int argIndex = 0; argIndex < ArgCount; argIndex++)
        {
            Bit64 value = trace.args[argIndex];
            int64_t signedValue = int64_t(value);
            NN_UNUSED(signedValue);

            NN_KERN_RELEASE_LOG(
                (-256 < signedValue && signedValue < 256) ? "%lld%s" : "0x%llx%s",
                        value,
                        argIndex != ArgCount - 1 ? ", " : ")\n");
        }
    }

    for (;;)
    {
        // Never return
    }
}

#endif

#endif

}}
