﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_DebugSelect.h"
#include "kern_Assert.h"
#include "kern_CPUSelect.h"
#include "kern_HardwareTimerSelect.h"
#include "kern_KMemoryManager.h"
#include "kern_KScheduler.h"
#include "kern_KSynchronization.h"
#include "kern_KThread.h"
#include "kern_Kernel.h"
#include "kern_MemoryMap.h"
#include "kern_PageTableSelect.h"
#include "kern_Utility.h"
#include "kern_WorkThread.h"
#include "kern_KScopedSchedulingLockAndSleep.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_DebugString.h"
#include "kern_InterlockedSelect.h"
#include "kern_KLockWithPriorityInheritance.h"
#include "kern_UserContextSelect.h"
#include "kern_InterruptControllerSelect.h"
#include "kern_KTrace.h"
#include <cstring>


namespace nn { namespace kern {
namespace {
bool IsKernelKey(KProcessAddress key)
{
    return (KProcessAddress(NN_KERN_V_ADDR_KERNEL) <= key && key <= KProcessAddress(NN_KERN_V_ADDR_KERNEL_END - 1));
}
}

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KThread);

InterlockedVariable<int64_t>    KThread::s_NumCreated; //!< スレッドIDの生成用

/*!
    @brief      スレッドを初期化します

    @param[in]  f               スレッドのエントリーポイント
    @param[in]  param           ユーザー定義の任意ポインタ
    @param[in]  svStackBottom   SVC モードでのスレッドスタック下限
    @param[in]  userStackBottom USR モードでのスレッドスタック下限
    @param[in]  prio            スレッドの優先度
    @param[in]  coreNo          割り当てる CPU コア
    @param[in]  pParent         スレッドの所有プロセス
    @param[in]  type            スレッドのタイプ

    @return     スレッドが正しく生成できたら成功を返します

*/
Result KThread::Initialize(
    ThreadFunc          f,
    uintptr_t           param,
    void*               svStackBottom,
    KProcessAddress     userStackBottom,
    int32_t             prio,
    int32_t             coreNo,
    KProcess*           pParent,
    KThread::Type       type )
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT( (pParent != NULL) || (type != TYPE_USER));
    NN_KERN_ASSERT((nn::svc::HighestThreadPriority <= prio && prio <= nn::svc::LowestThreadPriority) || (type == TYPE_MAIN));
    NN_KERN_ASSERT(0 <= coreNo && coreNo < KCPU::NUM_CORE);
    NN_KERN_NULL_ASSERT(svStackBottom);

    m_ThreadLocalRegionAddr = 0;

#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
    const uintptr_t nsvStackBottom = reinterpret_cast<uintptr_t>(svStackBottom);
#endif

    // スレッドタイプごとに、パラメーターのとりうる範囲が変わります
    switch( type )
    {
    case TYPE_USER:
        {
            // ユーザーが通常作成するスレッドです

            // svcスタックの境界チェック、アドレスチェックを行います
            NN_KERN_ALIGN_ASSERT( nsvStackBottom, NN_KERN_FINEST_PAGE_SIZE );
            NN_KERN_ASSERT( NN_KERN_V_ADDR_KERNEL <= nsvStackBottom );
            NN_KERN_ASSERT( nsvStackBottom <= NN_KERN_V_ADDR_KERNEL_END );

            // スレッドの関連づくコアを指定します
            NN_KERN_ASSERT( pParent != NULL );
            NN_KERN_ASSERT(coreNo != nn::svc::IdealCoreUseProcessValue);
            NN_KERN_ASSERT((pParent->GetCoreMask() | (1ull << coreNo)) == pParent->GetCoreMask());
            NN_KERN_ASSERT((pParent->GetPriorityMask() | (1ull << prio)) == pParent->GetPriorityMask());
            m_IdealProcessor = coreNo;
            m_AffinityMask.SetAffinity(coreNo);
        }
        break;

    case TYPE_REALTIME:
        // リアルタイムスレッドは自コアの割り込み処理補助の目的で使用します
        NN_KERN_ASSERT( coreNo == GetCurrentCpuNo() );

    case TYPE_KERNEL:
        {
            // svcスタックの境界チェック、アドレスチェックを行います
            NN_KERN_ALIGN_ASSERT( nsvStackBottom, NN_KERN_FINEST_PAGE_SIZE );
            NN_KERN_ASSERT( NN_KERN_V_ADDR_KERNEL <= nsvStackBottom );
            NN_KERN_ASSERT( nsvStackBottom <= NN_KERN_V_ADDR_KERNEL_END );
            NN_KERN_ASSERT( userStackBottom == 0 );

             // 指定どおりに Affinity Mask を設定します。
            m_IdealProcessor     = coreNo;
            m_AffinityMask.SetAffinity(coreNo);
        }
        break;

    case TYPE_MAIN:
        {
            // 最初のスレッドです
            NN_KERN_ASSERT( param == 0 );

            // svcスタックの境界チェック、アドレスチェックを行います
            NN_KERN_ALIGN_ASSERT( nsvStackBottom, NN_KERN_FINEST_PAGE_SIZE );
            NN_KERN_ASSERT( NN_KERN_V_ADDR_KERNEL <= nsvStackBottom );
            NN_KERN_ASSERT( nsvStackBottom <= NN_KERN_V_ADDR_KERNEL_END );
            NN_KERN_ASSERT( userStackBottom == 0 );
            NN_KERN_ASSERT( coreNo == GetCurrentCpuNo() );

             // 指定どおりに Affinity Mask を設定します。
            m_IdealProcessor     = coreNo;
            m_AffinityMask.SetAffinity(coreNo);
        }
        break;

    default:
        NN_KERNEL_PANIC("unknown thread type=%d", type);
        break;
    }


    m_ThreadLocalRegionAddr = 0;
    m_pThreadLocalRegionHeapAddr = nullptr;
    m_pParent               = NULL;
    m_pConditionVariable    = NULL;


    // メンバー変数を初期化します
    m_State                 = (type == TYPE_MAIN)? STATE_RUNNABLE: STATE_INITIALIZED;

    m_IsSignaled            = false;
    m_IpcCancelled          = false;
    m_TerminateRequested    = false;
    m_WaitCanceled          = false;
    m_CanCancel             = false;
    m_WaitAddressArbiter    = false;

    m_pKernelStackEnd      = svStackBottom;

    m_Priority              = prio;
    m_BasePriority          = prio;
    // CHECK: -1 → KThread::PROCESSOR_NOT_REGISTEREDの使用を推奨します
    m_RunningProcessor      = m_IdealProcessor;
    m_SyncedObj             = NULL;
    m_WaitResult            = nn::svc::ResultInternalNoSynchedObject();
    m_pWaitingLock          = NULL;

    m_QueueEntry.Initialize();
    m_SleepingQueue         = NULL;

    m_SuspendRequested      = 0;
    m_DebugCreateThread     = false;
    m_pLightSessionData     = NULL;
    m_pLockOwner            = NULL;
    m_PriorityInheritanceNest = 0;
    m_CoreMigrarionDisableCount = 0;
    m_SchedCount            = -1;
    m_LeaveTime             = 0;
    m_NumKernelWaiter       = 0;

    m_EntryAddress          = reinterpret_cast<uintptr_t>(f);
    m_ReleaseHint           = false;

#ifdef NN_KERN_ENABLE_SVC_PROFILE
    for (int svcNo = 0; svcNo < NN_ARRAY_SIZE(m_NumSvc); svcNo++)
    {
        m_NumSvc[svcNo] = 0;
    }
#endif
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
    m_RunnableStart         = (type == TYPE_MAIN)? KHardwareTimer::GetTick(): -1;
    m_RunnableTime          = 0;
    m_NumSwitch             = 0;
    m_NumFpuSwitch          = 0;
    m_CurrentProcessor      = -1;
    m_NumCoreSwitch         = 0;
#endif
#ifdef NN_KERN_ENABLE_IPC_PROFILE
    m_NumIpc                = 0;
    m_NumIpcRecv            = 0;
    m_NumIpcReply           = 0;
#endif

#ifdef NN_KERN_ENABLE_PERFORMANCE_COUNTER
    for (int i = 0; i < KCPU::NUM_PERFORMANCE_COUNTER; i++)
    {
        m_PerformanceCounter[i] = 0;
    }
#endif

    m_CpuTime               = 0;
#if defined(NN_KERN_CORE_UTILIZATION)
    for (int i = 0; i < KCPU::NUM_CORE; i++)
    {
        m_CoreTime[i] = 0;
    }
#endif

    if (type != TYPE_MAIN)
    {
        FillKernelStack();
    }
    std::memset(&GetParamsOnStack(), 0x0, sizeof(ParamsOnStack));

    // TORIAEZU: きちんとした分岐 etc. にする
    // スレッド固有領域を生成します
    if( type == TYPE_USER )
    {
        Result result = pParent->CreateThreadLocalRegion(&m_ThreadLocalRegionAddr);
        if(result.IsFailure())
        {
            // TODO: 途中まで確保した領域の開放。
            NN_WARNING(false, "unable to create thread local region");
            return nn::svc::ResultOutOfMemory();
        }

        m_pThreadLocalRegionHeapAddr = pParent->GetThreadLocalRegionPointer(m_ThreadLocalRegionAddr);
        ::std::memset(m_pThreadLocalRegionHeapAddr, 0, NN_SVC_MEMORY_THREAD_LOCAL_REGION_SIZE);
    }

    if( pParent != NULL )
    {
        m_pParent = pParent;
        pParent->Open();
        pParent->IncrementThreadCount();
    }

    // スレッドのレジスターコンテキストを初期化します
    // TODO: エラーチェック
    m_Context.Initialize(
        m_EntryAddress,
        reinterpret_cast<uintptr_t>(GetStackBottom()),
        GetAsInteger(userStackBottom),
        param,
        (type == TYPE_USER),
        pParent? pParent->Is64Bit(): (sizeof(void*) == 8),
        (type == TYPE_MAIN));

    ParamsOnStack& pos = GetParamsOnStack();
    if( m_pParent != NULL )
    {
        m_pParent->CopySvcPermisionBitsTo(pos.svcPermission);
    }
    pos.pContext    = &m_Context;
    pos.disableCount= 1;
    SetInExceptionHandler();


    // スレッドに対してユニークな(通しの)IDを割り当てます
    m_ThreadId = s_NumCreated++;
    m_IsInitialized = true;

    if( pParent != NULL )
    {
        pParent->RegisterThread(this);
        if (pParent->IsSuspended())
        {
            SuspendRequest(SuspendType_Process);
        }
    }

    return ResultSuccess();
}   // NOLINT(readability/fn_size)

/*!
    @brief      スレッドの終了処理。

*/
void KThread::Finalize()
{
    NN_KERN_THIS_ASSERT();

    if( m_pParent != NULL )
    {
        // The thread should be removed from the process thread list first.
        // This avoids finalization during TerminateChildren() call.
        m_pParent->UnregisterThread(this);
    }

    // スレッドの参照カウントが 0 になった際に呼ばれます。
    // このとき、GetCurrentThread() != this であることを想定できます。

    // m_ThreadLocalRegionAddr が 0 の場合は Initialize で失敗している
    if( m_ThreadLocalRegionAddr != 0 )
    {
        Result result = m_pParent->DeleteThreadLocalRegion(m_ThreadLocalRegionAddr);

        if( result.IsFailure() )
        {
            NN_KERNEL_PANIC("failed to delete thread local region. result=%08x", result.GetInnerValueForDebug());
        }
    }

    NN_KERN_ASSERT(m_pLockOwner == NULL);
    // Lockを持ったまま終了し待ちスレッドがいる場合
    // 待ちスレッドは起床し ResultInvalidState() を返す
    {
        KScopedSchedulingLock locker;
        NN_KERN_ABORT_UNLESS(m_NumKernelWaiter == 0);
        WaiterList::iterator it = m_WaiterList.begin();
        while (it != m_WaiterList.end())
        {
            NN_KERN_ASSERT(!IsKernelKey(it->GetAddressKey()));
            it->SetLockOwner(NULL);
            it->SetSyncedObject(NULL, nn::svc::ResultInvalidState());
            it->WakeUp();
            it = m_WaiterList.erase(it);
        }
    }

    m_Context.Finalize();

    // svcスタックを開放します
    if( m_pKernelStackEnd != NULL )
    {
        CleanupSuperVisorStack(m_pKernelStackEnd);
    }

    if( m_pParent != NULL )
    {
        m_pParent->DecrementThreadCount();
    }

    KObjectAdaptor<KThread, KSynchronizationObject>::Finalize();
}

/*!
    @brief      スレッドを"実行"します。

    スケジューラー側からスレッドの初期化処理をします。
    STATE_RUNNABLE であればスケジューラーの実行待機キュー入りします。

*/
Result KThread::Run()
{
    NN_KERN_THIS_ASSERT();
    for (;;)
    {
        KScopedSchedulingLock locker;

        if( IsTerminateRequested() )
        {
            return nn::svc::ResultTerminateRequested();
        }

        if (GetCurrentThread().IsTerminateRequested())
        {
            return nn::svc::ResultTerminateRequested();
        }

        if(GetState() != STATE_INITIALIZED)
        {
            return nn::svc::ResultInvalidState();
        }

        // 自分自身が停止要求されていれば、他のスレッドをRunしない。
        if (GetCurrentThread().IsSuspended())
        {
            GetCurrentThread().Suspend();
            continue;
        }
        if (IsUserThread())
        {
            if (IsSuspended())
            {
                Suspend();
            }
        }

        SetState(STATE_RUNNABLE);
        break;
    }
    return ResultSuccess();
}

/*!
    @brief      スレッドはシグナル状態か。

    スレッドのシグナル状態、とは終了処理中であることを意味します。
    こう定義することで、KSynchronization::Wait でスレッドの終了検出に利用できます。

*/
bool KThread::IsSignaled() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    return m_IsSignaled;
}

/*!
    @brief      スレッドのシャットダウンを開始します。

*/
void KThread::ShutdownStep1()
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    if (m_pParent)
    {
       m_pParent->ReleaseUserException(this);
    }

    SetState(STATE_TERMINATED);

    if (m_pParent)
    {
       m_pParent->ClearRunningThread(this);
    }

    // シグナルを立てることで、スレッドの終了処理に入ったことを示します。
    // KSynchronization::Wait のブロッキングが解除されます。
    m_IsSignaled = true;
    NotifyAvailable();

    // コンテキストの終了処理
    KContext::OnThreadTerminating(this);
#ifdef NN_KERN_ENABLE_NX_PROFILER
    KScheduler::ClearPrevThread(this);
#endif
    RegisterDpc(DPC_FUNC_TERMINATED);
}

// Exit では終了通知
void KThread::Exit()
{
    NN_KERN_THIS_ASSERT();

    NN_KERN_ASSERT(this == &GetCurrentThread());

    // デバッガへスレッド終了を通知する
    // デバッガアタッチ時はここで一旦停止する。
    KDebug::OnExitThread(this);
    if (m_pParent)
    {
        m_pParent->ReleaseLimit(nn::svc::LimitableResource_ThreadCountMax, 0, 1);
        m_ReleaseHint = true;
    }

    {
        KScopedSchedulingLock locker;

        m_SuspendRequested &= ~STATE_SUSPENDED_MASK;

        ShutdownStep1();

        // TODO: タスクスレッドの同期を取る

        // カレントスレッドを自分で削除できないため、
        // タスクスレッドに対してこのスレッドの破棄を依頼する。
        KWorkThreadManager::AddTask(KWorkThreadManager::LOW_PRIO_WORK_THREAD, this);

    } // ここでスケジューラが呼ばれる

    // 必ずスケジューリングされ、戻ってくることはない。
    NN_KERNEL_PANIC("dead thread moving.");
}

KThread::ThreadState KThread::RequestTerminate()
{
    NN_KERN_THIS_ASSERT();

    // TORIAEZU: 自スレッドの Terminate 禁止。
    NN_KERN_ASSERT(this != &GetCurrentThread());

    {
        KScopedSchedulingLock lock;

        // 多重に終了リクエストされたら最初の一回目のみ通します
        // この操作で terminateRequested が有効になります。
        if( m_TerminateRequested.CompareAndSwap(false, true) == false )
        {
            if(GetState() == STATE_INITIALIZED)
            {
                ThreadState oldState = m_State;
                NN_UNUSED(oldState);
                m_State = STATE_TERMINATED;
                NN_KERN_KTRACE_THREAD_STATE_CHANGED(this, oldState, m_State);
                return STATE_TERMINATED;
            }
            RegisterDpc(DPC_FUNC_TERMINATE);

            if (IsSuspended())
            {
                m_SuspendRequested &= ~STATE_SUSPENDED_MASK;
                Continue();
            }

            if( GetBasePriority() >= nn::svc::LibraryThreadPriorityHighest )
            {
                SetPriority(nn::svc::LibraryThreadPriorityHighest - 1);
            }

            // 他のコアの実行状態を確認します
            if (GetState() == STATE_RUNNABLE)
            {
#if NN_KERN_MULTI_CORE
                KCPU::DataSynchronizationBarrier();
                Bit64 cpuSet = m_AffinityMask.GetAffinityMask();
                cpuSet &= ~(1ull << GetCurrentCpuNo());
                if (cpuSet)
                {
                    Kernel::GetInterruptManager().SendIpi(cpuSet, KInterruptName::INTR_CORE_ID_TERMINATE);
                }
#endif
            }

            SetSyncedObject(NULL, nn::svc::ResultTerminateRequested());
            WakeUp();
        }

        return GetState();
    }
}

//! スレッドの強制停止
void KThread::Terminate()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(this != &GetCurrentThread());

    ThreadState stateAfterRequest = RequestTerminate();

    if (stateAfterRequest != STATE_TERMINATED)
    {
        // スレッドが終了するのを待つ
        int32_t index;
        KSynchronizationObject* objs[] = { this };
        Kernel::GetSynchronization().Wait(&index, objs, 1, nn::svc::WAIT_INFINITE);
    }
}

/*!
    @brief      スレッドを指定時間休眠させます

    @param[in]  time    timeoutTick 起床時間(tick)

*/
Result KThread::Sleep(int64_t timeout)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT( ! KScheduler::IsSchedulerLocked() );
    NN_KERN_ASSERT(&GetCurrentThread() == this);
    NN_KERN_ASSERT( timeout > 0 );
    KHardwareTimer* pTimer;

    {
        KScopedSchedulingLockAndSleep sleep(&pTimer, this, timeout);

        // スレッドの強制停止処理が始まりました。
        if (IsTerminateRequested())
        {
            // 状態を、"実行可能"にします。
            sleep.Cancel();
            return nn::svc::ResultTerminateRequested();
        }

        // 抜ける瞬間に sleep のデストラクタ処理でこのスレッドは眠り状態に入ります。
        // 指定時間経過後、 KThread::OnTimer() がよばれて目覚めます
        SetState(STATE_WAIT);
    }

    pTimer->CancelTask(this);

    return ResultSuccess();
}

Result KThread::OnFirstSchedule()
{
    NN_KERN_THIS_ASSERT();
    //NN_LOG_DEBUG("KThread::OnFirstSchedule() : \n");

    return ResultSuccess();
}

/*!
    @brief      スレッドの実行状態を設定します

    "STATE_RUNNABLE" に変更すると、このスレッドはスケジューラーのランキューに格納されます。
    "STATE_RUNNABLE" 以外に変更すると、このスレッドはスケジューラーの管理から手放されます。

    @param[in]  state     スレッド実行状態

*/
void KThread::SetState(ThreadState state)
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    // スレッドの実行状態を設定します。
    // m_State は他のフラグと一部共用しているため
    // マスクを施しつつ処理します
    ThreadState oldState = m_State;

    m_State = static_cast<ThreadState>((oldState & ~STATE_MASK) | (state & STATE_MASK));

    if (m_State != oldState)
    {
        NN_KERN_KTRACE_THREAD_STATE_CHANGED(this, oldState, m_State);
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
        if (m_State == STATE_RUNNABLE)
        {
            NN_KERN_ASSERT(m_RunnableStart == -1);
            m_RunnableStart = KHardwareTimer::GetTick();
        }
        else
        {
            m_RunnableTime += (KHardwareTimer::GetTick() - m_RunnableStart);
            m_RunnableStart = -1;
        }
#endif
        KScheduler::OnThreadStateChanged(this, oldState);
    }
}

/*!
    @brief      スレッドの優先度を設定します

    @param[in]  priority     スレッド優先度 (KThread:PRIORITY_*)

*/
void KThread::SetPriority(int32_t priority)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(nn::svc::HighestThreadPriority <= priority && priority <= nn::svc::LowestThreadPriority);

    KScopedSchedulingLock locker;
    m_BasePriority = priority;

    // スレッドの継承つき優先度を更新します
    RestorePriority(this);
}

/*!
    @brief      カレントスレッドの優先度をIDLEに設定します

    カーネル起動スレッドをIdleスレッド化するために呼ばれます。

*/
Result KThread::SetPriorityForIdleThread()
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    const int32_t previousPriority = m_Priority;

    // スレッド優先度を最低に設定します
    m_BasePriority  = IdleThreadPriority;
    m_Priority      = IdleThreadPriority;

    KScheduler::OnThreadPriorityChanged(this, previousPriority);

    return ResultSuccess();
}

/*!
    @brief      スレッド優先度を再設定します。

    スレッドの優先度は、同期オブジェクト(Mutex)のLock経由でサスペンドさせている
    スレッドの優先度を考慮(継承)して決定します。

*/
void KThread::RestorePriority(KThread* pThread)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    for (;;)
    {
        int32_t applyPriority = pThread->m_BasePriority;

        if (pThread->HasWaiterThread())
        {
            if (pThread->m_WaiterList.front().GetPriority() < pThread->m_BasePriority)
            {
                applyPriority = pThread->m_WaiterList.front().GetPriority();
            }
        }

        // 優先度継承後もプライオリティが一緒の場合、以降の処理は不要。
        if(applyPriority == pThread->m_Priority)
        {
            return;
        }

        if (pThread->m_pConditionVariable)
        {
            pThread->m_pConditionVariable->BeforePriorityUpdate(pThread);
        }

        // プライオリティの付け替え
        int32_t previousPriority = pThread->m_Priority;
        pThread->m_Priority = applyPriority;

        if (pThread->m_pConditionVariable)
        {
            pThread->m_pConditionVariable->AfterPriorityUpdate(pThread);
        }

        // スケジューラーに優先度が変化したことを伝えます
        KScheduler::OnThreadPriorityChanged(pThread, previousPriority);

        KThread* pOwner = pThread->m_pLockOwner;
        if (!pOwner)
        {
            return;
        }

        pOwner->RemoveWaiterImpl(pThread);
        pOwner->AddWaiterImpl(pThread);
        pThread = pOwner;
    }
}

//! 親プロセスのIDを取得します。
Bit64 KThread::GetProcessId() const
{
    NN_KERN_THIS_ASSERT();

    return (m_pParent != NULL) ? m_pParent->GetId(): ~0ull;
}

//! 成功を取得します
Result KThread::GetProcessor()
{
    NN_KERN_THIS_ASSERT();

    // CHECK: 戻り値をご確認ください
    return ResultSuccess();
}

//! 使用されていません
Result KThread::EnableFpu()
{
    NN_KERN_THIS_ASSERT();

    return ResultSuccess();
}

void KThread::WakeUp()
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock lock;

    if( GetState() == STATE_WAIT )
    {
        if( m_SleepingQueue != NULL )
        {
            m_SleepingQueue->WakupThread( this );
        }
        else
        {
            SetState(STATE_RUNNABLE);
        }
    }
}

//! タイマー割り込み
void KThread::OnTimer()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    // 指定時刻になったら呼ばれます。
    // このスレッドをSleepからたたき起こす処理に用います
    WakeUp();
}

void KThread::ShutdownStep2()
{
    NN_KERN_THIS_ASSERT();
    // このとき、GetCurrentThread() != this であることを想定できます。

    if (m_pParent != nullptr)
    {
        for (int i = 0; i < KCPU::NUM_CORE; ++i)
        {
            KThread* pCurrentThread;
            do
            {
                pCurrentThread = Kernel::GetCurrentSet(i).pCurrentThread;
                KCPU::DataMemoryBarrier();
            }
            while (pCurrentThread == this);
        }
    }

    Close();
}

/*!
    @brief     ワーカータスクのエントリーポイント

*/
void KThread::DoTask()
{
    // KThread::Exit にて登録された終了処理タスクが起動しました
    // このとき、GetCurrentThread() != this であることを想定できます。
    ShutdownStep2();
}

KThread* KThread::GetThreadById(Bit64 threadId)
{
    ListAccessor a;

    class IdObject: public KObjectContainer::ObjectList::value_type
    {
    public:
        explicit IdObject(Bit64 id): m_Id(id) {}
        virtual Bit64 GetId() const { return m_Id; }
    private:
        Bit64 m_Id;
    };
    IdObject id(threadId);

    const KObjectContainer::ObjectListIterator it = a.Find(id);

    if (it == a.GetEnd())
    {
        return nullptr;
    }

    KThread* p = static_cast<KThread*>(&*it);
    if (NN_LIKELY(p->Open()))
    {
        NN_KERN_ASSERT(p->GetId() == threadId);
        return p;
    }

    return nullptr;
}

#ifdef NN_KERN_ENABLE_SVC_PROFILE
void KThread::IncrementNumSvc(int svcNo)
{
    ++m_NumSvc[svcNo];
    if (m_pParent != nullptr)
    {
        m_pParent->IncrementNumSvc(svcNo);
    }
}
#endif
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
void KThread::IncrementNumSwitch()
{
    ++m_NumSwitch;
    if (m_pParent != nullptr)
    {
        m_pParent->IncrementNumThreadSwitch();
    }
    Kernel::IncrementNumThreadSwitch();
}
void KThread::IncrementNumFpuSwitch()
{
    ++m_NumFpuSwitch;
    if (m_pParent != nullptr)
    {
        m_pParent->IncrementNumFpuSwitch();
    }
    Kernel::IncrementNumFpuSwitch();
}
int64_t KThread::GetRunnableTime() const
{
    KScopedSchedulingLock locker;
    if (m_RunnableStart < 0)
    {
        return m_RunnableTime;
    }
    else
    {
        return m_RunnableTime + (KHardwareTimer::GetTick() - m_RunnableStart);
    }
}

#endif
#ifdef NN_KERN_ENABLE_IPC_PROFILE
void KThread::IncrementNumIpc()
{
    ++m_NumIpc;
    if (m_pParent != nullptr)
    {
        m_pParent->IncrementNumIpc();
    }
    Kernel::IncrementNumIpc();
}
void KThread::IncrementNumIpcRecv()
{
    ++m_NumIpcRecv;
    if (m_pParent != nullptr)
    {
        m_pParent->IncrementNumIpcRecv();
    }
}
void KThread::IncrementNumIpcReply()
{
    ++m_NumIpcReply;
    if (m_pParent != nullptr)
    {
        m_pParent->IncrementNumIpcReply();
    }
}
#endif

void KThread::Suspend()
{
    NN_KERN_THIS_ASSERT();

    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(!IsTerminateRequested());
    NN_KERN_ASSERT(!!m_SuspendRequested);

    ThreadState oldState = m_State;

    m_State = static_cast<ThreadState>(m_SuspendRequested | (oldState & STATE_MASK));
    NN_KERN_KTRACE_THREAD_STATE_CHANGED(this, oldState, m_State);

    KScheduler::OnThreadStateChanged(this, oldState);
}

void KThread::Continue()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    ThreadState oldState = m_State;

    m_State = static_cast<ThreadState>(oldState & STATE_MASK);
    NN_KERN_KTRACE_THREAD_STATE_CHANGED(this, oldState, m_State);

    KScheduler::OnThreadStateChanged(this, oldState);
}

void KThread::TrySuspend()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(IsSuspended());

    if (IsTerminateRequested())
    {
        return;
    }

    if (m_NumKernelWaiter > 0)
    {
        return;
    }
    NN_KERN_ABORT_UNLESS(m_NumKernelWaiter == 0);

    Suspend();
}

void KThread::SuspendRequest(SuspendType type)
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    if (IsTerminateRequested())
    {
        return;
    }
    m_SuspendRequested |= (1u << (STATE_SUSPENDED_SHIFT + type));
    TrySuspend();
}

void KThread::Resume(SuspendType type)
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    if (IsTerminateRequested())
    {
        return;
    }

    m_SuspendRequested &= ~(1u << (STATE_SUSPENDED_SHIFT + type));
    if (!IsSuspended())
    {
        Continue();
    }
}

void KThread::WaitCancel()
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    if (GetState() == STATE_WAIT && m_CanCancel)
    {
        if (m_SleepingQueue)
        {
            // Light IPC
            m_SleepingQueue->WakupThread(this);
            m_WaitCanceled = true;
        }
        else
        {
            SetSyncedObject(NULL, nn::svc::ResultCancelled());
            SetState(KThread::STATE_RUNNABLE);
            m_WaitCanceled = false;
        }
    }
    else
    {
        m_WaitCanceled = true;
    }
}

void KThread::DisableCoreMigration()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(&GetCurrentThread() == this);

    KScopedSchedulingLock locker;
    NN_KERN_MIN_ASSERT(m_CoreMigrarionDisableCount, 0);
    if (m_CoreMigrarionDisableCount++ == 0)
    {
        m_OrgIdealProcessor = m_IdealProcessor;
        m_OrgAffinityMask = m_AffinityMask;

        int32_t runningProcessor = m_RunningProcessor;

        m_IdealProcessor = runningProcessor;
        m_AffinityMask.SetAffinityMask(1ull << runningProcessor);

        if (m_OrgAffinityMask.GetAffinityMask() != m_AffinityMask.GetAffinityMask())
        {
            KScheduler::OnThreadCoreMaskChanged(this, m_OrgAffinityMask, runningProcessor);
        }
    }
}

void KThread::EnableCoreMigration()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(&GetCurrentThread() == this);

    KScopedSchedulingLock locker;
    NN_KERN_MIN_ASSERT(m_CoreMigrarionDisableCount, 1);
    if (--m_CoreMigrarionDisableCount == 0)
    {
        KAffinityMask prevAffnityMask = m_AffinityMask;

        m_IdealProcessor = m_OrgIdealProcessor;
        m_AffinityMask = m_OrgAffinityMask;

        if (prevAffnityMask.GetAffinityMask() != m_AffinityMask.GetAffinityMask())
        {
            int32_t runningProcessor = m_RunningProcessor;

            if (!m_AffinityMask.GetAffinity(m_RunningProcessor))
            {
                if (m_IdealProcessor >= 0)
                {
                    m_RunningProcessor = m_IdealProcessor;
                }
                else
                {
                    m_RunningProcessor = 63 - __builtin_clzll(m_AffinityMask.GetAffinityMask());
                }
            }
            KScheduler::OnThreadCoreMaskChanged(this, prevAffnityMask, runningProcessor);
        }
    }
}

Result KThread::GetCoreMask(int32_t* pIdealCore, Bit64* pAffinityMask)
{
    NN_KERN_THIS_ASSERT();
    {
        KScopedSchedulingLock locker;
        NN_KERN_MIN_ASSERT(m_CoreMigrarionDisableCount, 0);
        if (m_CoreMigrarionDisableCount == 0)
        {
            *pAffinityMask = m_AffinityMask.GetAffinityMask();
            *pIdealCore = m_IdealProcessor;
        }
        else
        {
            *pAffinityMask = m_OrgAffinityMask.GetAffinityMask();
            *pIdealCore = m_OrgIdealProcessor;
        }
    }
    return ResultSuccess();
}

Result KThread::SetCoreMask(int32_t idealCore, Bit64 affinityMask)
{
    NN_KERN_THIS_ASSERT();
    // User Threadのみサポート
    NN_KERN_ASSERT(m_pParent);
    NN_KERN_ASSERT(affinityMask);
    {
        KScopedSchedulingLock locker;
        NN_KERN_MIN_ASSERT(m_CoreMigrarionDisableCount, 0);
        if (idealCore == nn::svc::IdealCoreNoUpdate)
        {
            if (m_CoreMigrarionDisableCount == 0)
            {
                idealCore = m_IdealProcessor;
            }
            else
            {
                idealCore = m_OrgIdealProcessor;
            }

            if (((1ull << idealCore) & affinityMask) == 0)
            {
                return nn::svc::ResultInvalidCombination();
            }
        }

        if (m_CoreMigrarionDisableCount == 0)
        {
            KAffinityMask prevAffnityMask = m_AffinityMask;

            m_IdealProcessor = idealCore;
            m_AffinityMask.SetAffinityMask(affinityMask);

            if (prevAffnityMask.GetAffinityMask() != m_AffinityMask.GetAffinityMask())
            {
                int32_t runningProcessor = m_RunningProcessor;

                if (runningProcessor >= 0)
                {
                    if (!m_AffinityMask.GetAffinity(m_RunningProcessor))
                    {
                        if (m_IdealProcessor >= 0)
                        {
                            m_RunningProcessor = m_IdealProcessor;
                        }
                        else
                        {
                            m_RunningProcessor = 63 - __builtin_clzll(m_AffinityMask.GetAffinityMask());
                        }
                    }
                }
                KScheduler::OnThreadCoreMaskChanged(this, prevAffnityMask, runningProcessor);
            }
        }
        else
        {
            m_OrgIdealProcessor = idealCore;
            m_OrgAffinityMask.SetAffinityMask(affinityMask);
        }
    }
    return ResultSuccess();
}

Result KThread::GetThreadList(int32_t* pNumThreads, svc::KUserPointer<Bit64*> pThreadIds, int32_t arraySize)
{
    KThread::ListAccessor accessor;
    const KObjectContainer::ObjectListIterator end = accessor.GetEnd();

    int32_t index = 0;

    for (KObjectContainer::ObjectListIterator it = accessor.GetBegin(); it != end; it++)
    {
        KThread* p = static_cast<KThread*>(&*it);

        if( index < arraySize )
        {
            Bit64 id = p->GetId();
            Result result = pThreadIds.CopyArrayEntryFrom(&id, index);
            if (result.IsFailure())
            {
                return result;
            }
        }

        ++index;
    }

    *pNumThreads = index;

    return ResultSuccess();
}

void KThread::AddWaiterImpl(KThread* pThread)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    WaiterList::iterator it = m_WaiterList.begin();
    for (; it != m_WaiterList.end(); it++)
    {
        if (it->GetPriority() > pThread->GetPriority())
        {
            break;
        }
    }
    if (IsKernelKey(pThread->GetAddressKey()))
    {
        NN_KERN_ABORT_UNLESS(m_NumKernelWaiter++ >= 0);
    }
    m_WaiterList.insert(it, *pThread);
    pThread->SetLockOwner(this);
}

void KThread::AddWaiter(KThread* pThread)
{
    NN_KERN_THIS_ASSERT();
    AddWaiterImpl(pThread);
    RestorePriority(this);
}

void KThread::RemoveWaiterImpl(KThread* pThread)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    if (IsKernelKey(pThread->GetAddressKey()))
    {
        NN_KERN_ABORT_UNLESS(m_NumKernelWaiter-- > 0);
    }
    WaiterList::iterator it = m_WaiterList.iterator_to(*pThread);
    m_WaiterList.erase(it);
    pThread->SetLockOwner(NULL);
}

void KThread::RemoveWaiter(KThread* pThread)
{
    NN_KERN_THIS_ASSERT();
    RemoveWaiterImpl(pThread);
    RestorePriority(this);
}

KThread* KThread::RemoveWaiterByKey(int32_t* pNumWaiter, KProcessAddress key)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    int32_t numWaiter = 0;
    KThread* pNextOwner = NULL;
    WaiterList::iterator it = m_WaiterList.begin();
    while (it != m_WaiterList.end())
    {
        if (it->GetAddressKey() == key)
        {
            KThread* pThread = &*it;

            if (IsKernelKey(pThread->GetAddressKey()))
            {
                NN_KERN_ABORT_UNLESS(m_NumKernelWaiter-- > 0);
            }
            it = m_WaiterList.erase(it);

            if (pNextOwner == NULL)
            {
                pNextOwner = pThread;
                pNextOwner->SetLockOwner(NULL);
            }
            else
            {
                pNextOwner->AddWaiterImpl(pThread);
            }
            ++numWaiter;
        }
        else
        {
            ++it;
        }
    }

    if (pNextOwner)
    {
        RestorePriority(this);
        RestorePriority(pNextOwner);
    }
    *pNumWaiter = numWaiter;

    return pNextOwner;
}

void KThread::FillKernelStack()
{
#if defined NN_KERN_CHECK_KERNEL_STACK
    std::memset(reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(m_pKernelStackEnd) - NN_KERN_THREAD_SVC_STACK_SIZE), 0xcc, NN_KERN_THREAD_SVC_STACK_SIZE - sizeof(ParamsOnStack));
#endif
}

size_t KThread::GetKernelStackUsage() const
{
    NN_KERN_NULL_ASSERT(m_pKernelStackEnd);
#if defined NN_KERN_CHECK_KERNEL_STACK
    size_t i;
    const Bit8* pStack = reinterpret_cast<const Bit8*>(reinterpret_cast<uintptr_t>(m_pKernelStackEnd) - NN_KERN_THREAD_SVC_STACK_SIZE);

    for (i = 0; i < NN_KERN_THREAD_SVC_STACK_SIZE; i++)
    {
        if (pStack[i] != 0xcc)
        {
            break;
        }
    }
    return NN_KERN_THREAD_SVC_STACK_SIZE - i;
#else

    return 0;
#endif
}

Result KThread::SetActivity(nn::svc::ThreadActivity activity)
{
    KScopedSchedulingLock locker;
    switch (GetState())
    {
    case KThread::STATE_WAIT:
    case KThread::STATE_RUNNABLE:
        break;
    default:
        return nn::svc::ResultInvalidState();
    }
    if (activity == nn::svc::ThreadActivity_Paused)
    {
        if (IsSuspendRequested(SuspendType_Thread))
        {
            return nn::svc::ResultInvalidState();
        }
        SuspendRequest(SuspendType_Thread);
    }
    else
    {
        NN_KERN_ASSERT(activity == nn::svc::ThreadActivity_Runnable);
        if (!IsSuspendRequested(SuspendType_Thread))
        {
            return nn::svc::ResultInvalidState();
        }
        Resume(SuspendType_Thread);
    }
    return ResultSuccess();
}

Result KThread::GetThreadContext3(nn::svc::ThreadContext* pContext)
{
retry:
    {
        KScopedSchedulingLock locker;
        if (!IsSuspendRequested(SuspendType_Thread))
        {
            return nn::svc::ResultInvalidState();
        }
        // カーネル内部のロック競合回避のため実行中となることがある。その場合は既にカーネル内で実行中。
        if (GetStateRaw() != KThread::STATE_RUNNABLE)
        {
            // 停止完了保証
            for (int i = 0; i < KCPU::NUM_CORE; ++i)
            {
                KThread* pCurrentThread = Kernel::GetCurrentSet(i).pCurrentThread;
                KCPU::DataMemoryBarrier();
                if (pCurrentThread == this)
                {
                    goto retry;
                }
            }
        }

        GetUserContext(pContext, this);
    }
    return ResultSuccess();
}

Result KThread::GetUserStackInfo(size_t* pUsageSize, size_t* pStackSize) const
{
    // uspは以前の例外で保存された古い値。
    return m_pParent->GetPageTable().GetStackUsage(pUsageSize, pStackSize, GetUserStackPointer(this));
}


void KThread::PostFinalize(uintptr_t arg)
{
    KProcess* pOwner = reinterpret_cast<KProcess*>(arg & ~0x1ul);
    bool releaseHint = (arg & 1);
    if (pOwner)
    {
        pOwner->ReleaseLimit(nn::svc::LimitableResource_ThreadCountMax, 1, (releaseHint? 0: 1));
        pOwner->Close();
    }
    else
    {
        Kernel::GetSystemResourceLimit().ReleaseLimit(nn::svc::LimitableResource_ThreadCountMax, 1, (releaseHint? 0: 1));
    }
}

KContext* KThread::GetContextPointer()
{
    NN_KERN_THIS_ASSERT();
    return &m_Context;
}

}}

