﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "kern_KAutoObject.h"
#include "kern_KLinkedList.h"

namespace nn { namespace kern {

class KThread;

/*! @file
    @brief      同期オブジェクトクラスの定義です。

*/
class KSynchronizationObject : public KAutoObjectWithList
{
public:
    typedef KLinkedList<KThread> ThreadList;
    typedef ThreadList::Iterator ThreadIterator;

private:
    ThreadList      m_waitList;

protected:
    explicit        KSynchronizationObject();
    virtual         ~KSynchronizationObject();

    /*!
        @brief      この同期オブジェクトが Finalize される直前にコールバックされます。

    */
    virtual void    OnFinalizeSynchronization() {}

    /*!
        @brief      この同期オブジェクトがシグナル状態になった際に呼び出されます。

                    派生クラスの Signal イベントから呼ばれるべきです。
    */
    void            NotifyAvailable();


    /*!
        @brief      この同期オブジェクトが閉じられる際に呼び出されます。

    */
    void            NotifyAbort(Result reason);

public:
    /*!
        @brief      同期オブジェクトの使用権を取得できるかを判定します。

        この関数は派生クラスでオーバーライトします。
        必ずスケジューラーをロックした状態で呼び出す必要があります。

        @param[in]  thread   所有権取得を試みるスレッド

        @return     使用権が取得できるのであれば true
    */
    virtual bool    IsSignaled() const = 0;

    virtual void    Finalize();
    virtual void    DumpWaiter();

    /*!
        @brief      待機スレッドリストに追加します。

        必ずスケジューラーをロックした状態で呼び出す必要があります。

        @param[in]  thread   待機リストに追加するスレッド

        @return     必ず true
    */
    ThreadIterator RegisterWaitingThread(KThread *thread);

    /*!
        @brief      待機スレッドリストから指定のスレッドを削除します。

        必ずスケジューラーをロックした状態で呼び出す必要があります。

        @param[in]  itr   待機リストのイテレーター

        @return     削除したスレッドの次のイテレーター
    */
    ThreadIterator  UnregisterWaitingThread(ThreadIterator itr);

    /*!
        @brief      待機スレッドリストの先頭を示すイテレーターを取得します。

        必ずスケジューラーをロックした状態で呼び出す必要があります。

        @return     先頭の要素を示すイテレーター
    */
    ThreadIterator  GetWaitQueueBeginIterator();

    /*!
        @brief      待機スレッドリストの終了を示すイテレーターを取得します。

        必ずスケジューラーをロックした状態で呼び出す必要があります。

        @return     終了を示すイテレーター
    */
    ThreadIterator  GetWaitQueueEndIterator();

    //! KAutoObjectのプリセット関数セット定義です。クラスの末尾に記述する必要があります
    NN_AUTOOBJECT_DERIVED_FUNCSET(KSynchronizationObject, KAutoObject)
};

}}

