﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>
#include <cstring>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KSharedMemory.h"
#include "kern_KProcess.h"
#include "kern_PageTableSelect.h"
#include "kern_KMemoryManager.h"
#include "kern_KScopedResourceLimitTester.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KSharedMemory);

KSharedMemory::KSharedMemory()
: m_pResourceLimit(nullptr)
, m_Pid(0xffffffffffffffffull)
, m_OwnerProtection(nn::svc::MemoryPermission_None)
, m_OtherProtection(nn::svc::MemoryPermission_None)
, m_IsInitialized(false)
{
    NN_KERN_THIS_ASSERT();
}

KSharedMemory::~KSharedMemory()
{
    NN_KERN_THIS_ASSERT();
}

Result KSharedMemory::Initialize(
    KProcess*       pOwner,
    size_t          size,
    nn::svc::MemoryPermission ownerProtection,
    nn::svc::MemoryPermission otherProtection)
{
    NN_KERN_THIS_ASSERT();

    m_Pid             = pOwner->GetId();
    m_OwnerProtection = ownerProtection;
    m_OtherProtection = otherProtection;

    Result result;
    size_t numPages = KMemoryManager::SizeToPage(size);
    NN_KERN_ASSERT(numPages);

    KResourceLimit* pResourceLimit = pOwner->GetResourceLimit();

    KScopedResourceLimitTester tester(pResourceLimit, nn::svc::LimitableResource_PhysicalMemoryMax, size);
    if (tester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    m_Memory.Initialize(&Kernel::GetBlockInfoManager());

    result = Kernel::GetKernelHeapManager().Allocate(&m_Memory, numPages, pOwner->GetAllocateOption());
    if (result.IsFailure())
    {
        m_Memory.Finalize();
        return result;
    }

    tester.Accepted();

    m_pResourceLimit = pResourceLimit;
    m_pResourceLimit->Open();
    m_Memory.Open();
    m_IsInitialized = true;

    for (KPageGroup::BlockInfoList::const_iterator it = m_Memory.GetBlockBeginIter(); it != m_Memory.GetBlockEndIter(); it++)
    {
        std::memset(GetUntypedPointer(it->GetBlockAddr()), 0, it->GetNumPages() * NN_KERN_FINEST_PAGE_SIZE);
    }

    return ResultSuccess();
}

void KSharedMemory::Finalize()
{
    NN_KERN_THIS_ASSERT();

    size_t numPages = 0;

    numPages = m_Memory.GetTotalNumPages();

    // 解放
    m_Memory.Close();
    m_Memory.Finalize();

    const size_t usedSize = numPages * NN_KERN_FINEST_PAGE_SIZE;
    m_pResourceLimit->ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, usedSize);
    m_pResourceLimit->Close();

    KObjectAdaptor<KSharedMemory, KAutoObjectWithList>::Finalize();
}

Result KSharedMemory::Map(
    KProcessPageTable*  pTable,
    KProcessAddress     addr,
    size_t              size,
    KProcess*           pProcess,
    nn::svc::MemoryPermission myPermission)
{
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    const nn::svc::MemoryPermission testPermission = (pProcess->GetId() == m_Pid) ? m_OwnerProtection: m_OtherProtection;

    if (testPermission == nn::svc::MemoryPermission_DontCare)
    {
        NN_KERN_ASSERT(myPermission == nn::svc::MemoryPermission_Read || myPermission == nn::svc::MemoryPermission_ReadWrite);
    }
    else if (myPermission != testPermission)
    {
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    return pTable->MapPageGroup(addr, m_Memory, KMemoryState_Shared, ConvertToKMemoryPermission(myPermission));
}

Result KSharedMemory::Unmap(KProcessPageTable* pTable, KProcessAddress addr, size_t size, KProcess* pProcess)
{
    NN_UNUSED(pProcess);
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    return pTable->UnmapPageGroup(addr, m_Memory, KMemoryState_Shared);
}

size_t KSharedMemory::GetSize() const
{
    return m_Memory.GetTotalNumPages()  * NN_KERN_FINEST_PAGE_SIZE;
}

}}

