﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_InterlockedSelect.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KProcess.h"
#include "kern_DebugString.h"
#include "kern_KScheduler.h"
#include "kern_KTrace.h"
#include "kern_Trace.h"
#include "kern_SystemControl.h"
#include "kern_KScopedSchedulingLock.h"
#include <cstring>

namespace nn { namespace kern {
namespace {
void IncrementSchedCount(KThread* pThread)
{
    if (pThread->GetParentPointer() != nullptr)
    {
        pThread->GetParentPointer()->IncrementSchedCount();
    }
}

}
KSpinLockMutex KScheduler::s_SchedulerMutex;
KPriorityQueue KScheduler::s_PriorityQueue;
bool KScheduler::s_Update;

KScheduler::KScheduler():
    m_InInterruptHandler(false),
    m_IsActive(false),
    m_MyCore(0),
    m_pPrevThread(nullptr),
    m_PrevSwitch(0),
    m_pIdleThread(nullptr)
{
    m_SchedulingCounter.m_SchedulingRequired = true;
    m_SchedulingCounter.m_InterruptTaskThreadIsRunnable = false;
    m_SchedulingCounter.m_IdleStack = nullptr;
    m_SchedulingCounter.m_pHighestThread = nullptr;
    m_SchedulingCounter.m_IdleCount = 0;
    m_SchedulingCounter.m_EnableIdleCount = false;
}

void KScheduler::Initialize(KThread* pIdleThread)
{
    m_MyCore = GetCurrentCpuNo();
    m_pIdleThread = pIdleThread;
    m_SchedulingCounter.m_IdleStack = pIdleThread->GetStackBottom();

    {
        KScopedSchedulingLock locker;
        s_PriorityQueue.Enqueue(&GetCurrentThread());
        s_Update = true;
    }

    Kernel::GetInterruptManager().BindHandler(
            this, KInterruptName::INTR_CORE_ID_SCHEDULER,
            m_MyCore, KInterruptController::PriorityLevel_SchInterrupt, false, false);
}

void KScheduler::Activate()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 1);

    m_SchedulingCounter.m_EnableIdleCount = KTargetSystem::IsDevelopmentHardware();
    m_IsActive = true;
    RescheduleIfNeeded();
}

void KScheduler::OnThreadStateChanged(KThread* pThread, KThread::ThreadState previousState)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    const KThread::ThreadState currentState = pThread->GetStateRaw();

    if (currentState == previousState)
    {
        return;
    }

    if (previousState == KThread::STATE_RUNNABLE)
    {
        s_PriorityQueue.Remove(pThread);

        IncrementSchedCount(pThread);
        s_Update = true;
    }
    else if (currentState == KThread::STATE_RUNNABLE)
    {
        s_PriorityQueue.Enqueue(pThread);

        IncrementSchedCount(pThread);
        s_Update = true;
    }
}

void KScheduler::OnThreadPriorityChanged(KThread* pThread, int32_t previousPriority)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    if (pThread->GetStateRaw() == KThread::STATE_RUNNABLE)
    {
        s_PriorityQueue.PriorityChange(pThread, previousPriority, &GetCurrentThread() == pThread);

        IncrementSchedCount(pThread);
        s_Update = true;
    }
}

void KScheduler::OnThreadCoreMaskChanged(KThread* pThread, const KAffinityMask& prevAffinity, int prevRunning)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    if (pThread->GetStateRaw() == KThread::STATE_RUNNABLE)
    {
        s_PriorityQueue.AffinityMaskChange(pThread, prevAffinity, prevRunning);

        IncrementSchedCount(pThread);
        s_Update = true;
    }
}

void KScheduler::RescheduleOther() const
{
#if NN_KERN_MULTI_CORE
    KCPU::DataSynchronizationBarrier();
    Bit64 cpuSet = 0;
    for (int32_t coreNo = 0; coreNo < KCPU::NUM_CORE; ++coreNo)
    {
        if (coreNo != m_MyCore)
        {
            if (Kernel::GetScheduler(coreNo).m_SchedulingCounter.m_SchedulingRequired)
            {
                cpuSet |= (1ull << coreNo);
            }
        }
    }
    if (cpuSet != 0)
    {
        Kernel::GetInterruptManager().SendIpi(cpuSet, KInterruptName::INTR_CORE_ID_SCHEDULER);
    }
#endif
}

void KScheduler::EnableScheduling()
{
    NN_KERN_MIN_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 1);

    if (GetCurrentThread().GetDisableDispatchCount() > 1)
    {
        GetCurrentThread().EnableDispatch();
        return;
    }

    KScheduler& sch = GetCoreScheduler();
    sch.RescheduleIfNeeded();
}

void KScheduler::EnableSchedulingAndScheduleAll()
{
    NN_KERN_MIN_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 1);

    if (GetCurrentThread().GetDisableDispatchCount() > 1)
    {
        GetCurrentThread().EnableDispatch();
        return;
    }

    KScheduler& sch = GetCoreScheduler();
    sch.RescheduleOther();
    sch.RescheduleIfNeeded();
}

void KScheduler::ClearPrevThread(KThread* pThread)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    for (int32_t coreNo = 0; coreNo < KCPU::NUM_CORE; ++coreNo)
    {
        KScheduler& sch = Kernel::GetScheduler(coreNo);
        __sync_bool_compare_and_swap(&sch.m_pPrevThread, pThread, nullptr);
    }
}

void KScheduler::YieldWithoutCoreMigration()
{
    NN_KERN_ASSERT(CanSchedule());
    NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);
    NN_KERN_ASSERT(GetCurrentProcessPointer() != nullptr);

    KThread& currentThread = GetCurrentThread();
    KProcess& currentProcess = GetCurrentProcess();

    if (currentThread.GetYieldSchedCount() == currentProcess.GetSchedCount())
    {
        return;
    }

    GetCurrentThread().DisableDispatch();

    {
        KScopedSchedulingLock locker;

        const KThread::ThreadState currentState = currentThread.GetStateRaw();
        if (currentState == KThread::STATE_RUNNABLE)
        {
            KThread* pThread = s_PriorityQueue.MoveToEnd(&currentThread);
            IncrementSchedCount(&currentThread);
            if (pThread != &currentThread)
            {
                s_Update = true;
            }
            else
            {
                currentThread.SetYieldSchedCount(currentProcess.GetSchedCount());
            }
        }
    }

    {
        KDisableInterrupt di;
        GetCurrentThread().EnableDispatch();
        NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);

        GetCoreScheduler().ForceScheduling();
    }
}

void KScheduler::YieldWithCoreMigration()
{
    NN_KERN_ASSERT(CanSchedule());
    NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);
    NN_KERN_ASSERT(GetCurrentProcessPointer() != nullptr);

    KThread& currentThread = GetCurrentThread();
    KProcess& currentProcess = GetCurrentProcess();

    if (currentThread.GetYieldSchedCount() == currentProcess.GetSchedCount())
    {
        return;
    }

    GetCurrentThread().DisableDispatch();

    {
        KScopedSchedulingLock locker;

        const KThread::ThreadState currentState = currentThread.GetStateRaw();
        if (currentState == KThread::STATE_RUNNABLE)
        {
            int coreNo = currentThread.GetRunningProcessor();
            KThread* pNext = s_PriorityQueue.MoveToEnd(&currentThread);
            IncrementSchedCount(&currentThread);
            bool recheck = false;
            KThread* pBack = s_PriorityQueue.GetFrontBack(coreNo);
            while (pBack != nullptr)
            {
                int runningCore = pBack->GetRunningProcessor();
                KThread* pRunningThread = (runningCore >= 0)? Kernel::GetScheduler(runningCore).m_SchedulingCounter.m_pHighestThread: nullptr;
                if (pRunningThread != pBack)
                {
                    if (pBack->GetPriority() > currentThread.GetPriority())
                    {
                        pBack = nullptr;
                        break;
                    }
                    else if (pBack->GetPriority() == currentThread.GetPriority())
                    {
                        if (pNext != &currentThread && pNext->GetLeaveTime() < pBack->GetLeaveTime())
                        {
                            pBack = nullptr;
                            break;
                        }
                    }

                    if (pRunningThread == nullptr || pRunningThread->GetPriority() > CoreMigrationPriorityThreshold)
                    {
                        pBack->SetRunningProcessor(coreNo);
                        s_PriorityQueue.RunningCoreChangeToTop(pBack, runningCore);
                        NN_KERN_KTRACE_CORE_MIGRATION(pBack->GetId(), runningCore, coreNo, __LINE__);
                        IncrementSchedCount(pBack);
                        break;
                    }
                    else
                    {
                        recheck = true;
                    }
                }

                pBack = s_PriorityQueue.GetNextBack(coreNo, pBack);
            }

            if (pBack != nullptr || pNext != &currentThread)
            {
                s_Update = true;
            }
            else if (!recheck)
            {
                currentThread.SetYieldSchedCount(currentProcess.GetSchedCount());
            }
        }
    }

    {
        KDisableInterrupt di;
        GetCurrentThread().EnableDispatch();
        NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);

        GetCoreScheduler().ForceScheduling();
    }
}


void KScheduler::SwitchToAnyThread()
{
    NN_KERN_ASSERT(CanSchedule());
    NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);
    NN_KERN_ASSERT(GetCurrentProcessPointer() != nullptr);

    KThread& currentThread = GetCurrentThread();
    KProcess& currentProcess = GetCurrentProcess();

    if (currentThread.GetYieldSchedCount() == currentProcess.GetSchedCount())
    {
        return;
    }

    GetCurrentThread().DisableDispatch();

    {
        KScopedSchedulingLock locker;

        const KThread::ThreadState currentState = currentThread.GetStateRaw();
        if (currentState == KThread::STATE_RUNNABLE)
        {
            int coreNo = currentThread.GetRunningProcessor();
            int newCoreNo = -1;
            currentThread.SetRunningProcessor(newCoreNo);
            s_PriorityQueue.RunningCoreChange(&currentThread, coreNo);
            NN_KERN_KTRACE_CORE_MIGRATION(currentThread.GetId(), coreNo, newCoreNo, __LINE__);
            IncrementSchedCount(&currentThread);

            if (s_PriorityQueue.GetFront(coreNo) == nullptr)
            {
                KThread* pThread = s_PriorityQueue.GetFrontBack(coreNo);
                while (pThread != nullptr)
                {
                    int runningCore = pThread->GetRunningProcessor();
                    KThread* pRunningThread = (runningCore >= 0)? s_PriorityQueue.GetFront(runningCore): nullptr;
                    if (pRunningThread != pThread)
                    {
                        if (pRunningThread != nullptr && pRunningThread->GetPriority() <= CoreMigrationPriorityThreshold)
                        {
                            break;
                        }
                        pThread->SetRunningProcessor(coreNo);
                        s_PriorityQueue.RunningCoreChange(pThread, runningCore);
                        NN_KERN_KTRACE_CORE_MIGRATION(pThread->GetId(), runningCore, coreNo, __LINE__);
                        IncrementSchedCount(pThread);
                        break;
                    }
                    pThread = s_PriorityQueue.GetNextBack(coreNo, pThread);
                }

                if (pThread != &currentThread)
                {
                    s_Update = true;
                }
                else
                {
                    currentThread.SetYieldSchedCount(currentProcess.GetSchedCount());
                }
            }
            else
            {
                s_Update = true;
            }
        }
    }

    {
        KDisableInterrupt di;
        GetCurrentThread().EnableDispatch();
        NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);

        GetCoreScheduler().ForceScheduling();
    }
}


void KScheduler::RotateReadyQueue(int priority, int coreNo)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    KThread* pTop = s_PriorityQueue.GetFront(priority, coreNo);
    KThread* pNext = nullptr;
    if (pTop != nullptr)
    {
        pNext = s_PriorityQueue.MoveToEnd(pTop);
        if (pNext != pTop)
        {
            IncrementSchedCount(pTop);
            IncrementSchedCount(pNext);
        }
    }

    KThread* pBack = s_PriorityQueue.GetFrontBack(priority, coreNo);
    while (pBack != nullptr)
    {
        int runningCore = pBack->GetRunningProcessor();
        KThread* pRunningThread = (runningCore >= 0)? s_PriorityQueue.GetFront(runningCore): nullptr;
        if (pRunningThread != pBack)
        {
            if (pNext != pTop && pNext != nullptr && pNext->GetLeaveTime() < pBack->GetLeaveTime())
            {
                pBack = nullptr;
                break;
            }

            if (pRunningThread == nullptr || pRunningThread->GetPriority() > CoreMigrationPriorityThreshold)
            {
                pBack->SetRunningProcessor(coreNo);
                s_PriorityQueue.RunningCoreChangeToTop(pBack, runningCore);
                NN_KERN_KTRACE_CORE_MIGRATION(pBack->GetId(), runningCore, coreNo, __LINE__);
                IncrementSchedCount(pBack);
                break;
            }
        }

        pBack = s_PriorityQueue.GetNextSamePriority(coreNo, pBack);
    }

    {
        KThread* pHighestThread = s_PriorityQueue.GetFront(coreNo);
        if (pHighestThread == &GetCurrentThread())
        {
            pHighestThread = s_PriorityQueue.GetNext(coreNo, pHighestThread);
        }

        if (pHighestThread != nullptr && pHighestThread->GetPriority() >= priority)
        {
            pBack = s_PriorityQueue.GetFrontBack(coreNo);
            while (pBack != nullptr)
            {
                if (pBack->GetPriority() >= pHighestThread->GetPriority())
                {
                    break;
                }

                int runningCore = pBack->GetRunningProcessor();
                KThread* pRunningThread = (runningCore >= 0)? s_PriorityQueue.GetFront(runningCore): nullptr;
                if (pRunningThread != pBack)
                {
                    if (pRunningThread == nullptr || pRunningThread->GetPriority() > CoreMigrationPriorityThreshold)
                    {
                        pBack->SetRunningProcessor(coreNo);
                        s_PriorityQueue.RunningCoreChangeToTop(pBack, runningCore);
                        NN_KERN_KTRACE_CORE_MIGRATION(pBack->GetId(), runningCore, coreNo, __LINE__);
                        IncrementSchedCount(pBack);
                        break;
                    }
                }

                pBack = s_PriorityQueue.GetNextBack(coreNo, pBack);
            }
        }
    }

    s_Update = true;
}

void KScheduler::UpdateHighestThread(KThread* pThread)
{
    KThread* pHighestThread = m_SchedulingCounter.m_pHighestThread;
    if (NN_LIKELY(pHighestThread != pThread))
    {
        if (NN_LIKELY(pHighestThread != nullptr))
        {
            IncrementSchedCount(pHighestThread);
            pHighestThread->SetLeaveTime(KHardwareTimer::GetTick());
        }
        if (m_SchedulingCounter.m_EnableIdleCount)
        {
            if (pThread != nullptr)
            {
                KProcess* pProcess = pThread->GetParentPointer();
                if (pProcess)
                {
                    pProcess->SetRunningThread(m_MyCore, pThread, m_SchedulingCounter.m_IdleCount);
                }
            }
            else
            {
                m_SchedulingCounter.m_IdleCount++;
            }
        }
        NN_KERN_KTRACE_SCHE_CHANGED(m_MyCore,
                pHighestThread == nullptr? m_pIdleThread: pHighestThread,
                pThread == nullptr                             ? m_pIdleThread: pThread);
        m_SchedulingCounter.m_pHighestThread = pThread;
        KCPU::DataMemoryBarrier();
        m_SchedulingCounter.m_SchedulingRequired = true;
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
        Kernel::IncrementNumSchedulerUpdate(m_MyCore);
#endif
    }
}

/* スケジューリングロックを抜ける直前に、各コアで実行するべきスレッドを更新する */
void KScheduler::UpdateHighestThreadsImpl()
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    s_Update = false;
    Bit64 idleCores = 0;
    KThread* pTopThreads[KCPU::NUM_CORE];
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
    Kernel::IncrementNumCallSchedulerUpdate();
#endif
    /*
     * 最高優先度のスレッドがそのコアでのみ実行可能なら、そのスレッドを実行する。
     */
    for (int coreNo = 0; coreNo < KCPU::NUM_CORE; coreNo++)
    {
        KThread* pThread = s_PriorityQueue.GetFront(coreNo);
        if (pThread == nullptr)
        {
            idleCores |= (1ull << coreNo);
        }

        Kernel::GetScheduler(coreNo).UpdateHighestThread(pThread);
        pTopThreads[coreNo] = pThread;
    }

    while (idleCores)
    {
        int coreNo = __builtin_ctzll(idleCores);
        KThread* pThread = s_PriorityQueue.GetFrontBack(coreNo);
        if (pThread != nullptr)
        {
            int migrateCandidate[KCPU::NUM_CORE];
            int numCandidate = 0;
            do
            {
                int runningCore = pThread->GetRunningProcessor();
                KThread* pRunningThread = (runningCore >= 0)? pTopThreads[runningCore]: nullptr;
                if (pRunningThread != pThread)
                {
                    if (pRunningThread != nullptr && pRunningThread->GetPriority() <= CoreMigrationPriorityThreshold)
                    {
                        break;
                    }
                    pThread->SetRunningProcessor(coreNo);
                    s_PriorityQueue.RunningCoreChange(pThread, runningCore);
                    NN_KERN_KTRACE_CORE_MIGRATION(pThread->GetId(), runningCore, coreNo, __LINE__);
                    pTopThreads[coreNo] = pThread;

                    Kernel::GetScheduler(coreNo).UpdateHighestThread(pTopThreads[coreNo]);
                    break;
                }

                // 全コアで一致する前に上でヒットするはず。(自コアはヒットしない)
                NN_KERN_ASSERT(numCandidate < KCPU::NUM_CORE - 1);
                migrateCandidate[numCandidate++] = runningCore;
                pThread = s_PriorityQueue.GetNextBack(coreNo, pThread);
            }
            while (pThread != nullptr);

            if (pThread == nullptr)
            {
                for (int i = 0; i < numCandidate; i++)
                {
                    int runningCore = migrateCandidate[i];
                    pThread = pTopThreads[runningCore];
                    KThread* pNext = s_PriorityQueue.GetNext(runningCore, pThread);
                    if (pNext != nullptr)
                    {
                        pTopThreads[runningCore] = pNext;
                        Kernel::GetScheduler(runningCore).UpdateHighestThread(pNext);

                        pThread->SetRunningProcessor(coreNo);
                        s_PriorityQueue.RunningCoreChange(pThread, runningCore);
                        NN_KERN_KTRACE_CORE_MIGRATION(pThread->GetId(), runningCore, coreNo, __LINE__);
                        pTopThreads[coreNo] = pThread;
                        Kernel::GetScheduler(coreNo).UpdateHighestThread(pThread);
                        break;
                    }
                }
            }
        }

        idleCores &= ~(1ull << coreNo);
    }

#if defined NN_SDK_BUILD_DEVELOP || defined NN_SDK_BUILD_DEBUG
    for (int i = 0; i < KCPU::NUM_CORE; i++)
    {
        for (int j = 0; j < i; j++)
        {
            NN_KERN_ASSERT(
                    Kernel::GetScheduler(i).m_SchedulingCounter.m_pHighestThread == nullptr ||
                    Kernel::GetScheduler(j).m_SchedulingCounter.m_pHighestThread == nullptr ||
                    Kernel::GetScheduler(i).m_SchedulingCounter.m_pHighestThread != Kernel::GetScheduler(j).m_SchedulingCounter.m_pHighestThread);
        }
    }
#endif
}

void KScheduler::InterruptTaskThreadToRunnable()
{
    NN_KERN_EQUAL_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 1);
    KInterruptTaskManager& taskManager = GetCoreInterruptTaskManager();
    KThread* pTaskThread = taskManager.GetThread();

    if (NN_LIKELY(pTaskThread->GetState() == KThread::STATE_WAIT))
    {
        pTaskThread->SetState(KThread::STATE_RUNNABLE);
    }
}


bool KScheduler::SwitchThread(KThread* pNextThread)
{
    // pCurrentThread のコンテキストロックは既に外れているため、別コアで実行中の可能性がある。
    // pCurrentThread のメモリにアクセス可能であることの保証は KThread::ShutdownStep2() で
    // どのコアも pCurrentThread を使用していないのを確認している。
    // pNextThread は ロック済み
    KThread* pCurrentThread = &GetCurrentThread();
    KProcess* pCurrentProcess = GetCurrentProcessPointer();

    if (pNextThread == nullptr)
    {
        pNextThread = m_pIdleThread;
    }

    if (pNextThread == pCurrentThread)
    {
        return false;
    }
    NN_KERN_EQUAL_ASSERT(pNextThread->GetDisableDispatchCount(), 1);

#ifdef NN_KERN_LOG_CPU_UTILIZATION
    int64_t prevTick = m_PrevSwitch;
    const int64_t newTick = KHardwareTimer::GetTick();
    pCurrentThread->AddCpuTime(m_MyCore, newTick - prevTick);
    if (pCurrentProcess != nullptr)
    {
        pCurrentProcess->AddCpuTime(newTick - prevTick);
    }
    m_PrevSwitch = newTick;
#endif
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    for (int i = 0; i < KCPU::NUM_PERFORMANCE_COUNTER; i++)
    {
        uint64_t delta =  Kernel::UpdatePerformanceCounter(i);
        pCurrentThread->AddPerformanceCounter(i, delta);
    }
#endif
#ifdef NN_KERN_ENABLE_NX_PROFILER
    if (pCurrentProcess != nullptr)
    {
        if (NN_LIKELY(!pCurrentThread->IsTerminateRequested()) && NN_LIKELY(pCurrentThread->GetRunningProcessor() == m_MyCore))
        {
            m_pPrevThread = pCurrentThread;
        }
        else
        {
            m_pPrevThread = nullptr;
        }
    }
    else
    {
        if (pCurrentThread == m_pIdleThread)
        {
            m_pPrevThread = nullptr;
        }
    }
#endif
    NN_KERN_KTRACE_SCHED(pNextThread);
    NN_KERN_TRACE_THREAD_SWITCH(pNextThread);

#if defined(NN_KERN_ENABLE_SWITCH_PROFILE)
    pNextThread->IncrementNumSwitch();
    pNextThread->SwitchCurrentProcessor(m_MyCore);
#endif

    KProcess* pNextProcess = pNextThread->GetParentPointer();

    if (pNextProcess != pCurrentProcess)
    {
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
        Kernel::IncrementNumProcessSwitch();
#endif
        KProcess::Switch(pCurrentProcess, pNextProcess);
    }

    KCPU::DataMemoryBarrier();
    SetCurrentThread(pNextThread);
    KCPU::DataMemoryBarrier();
    // pCurrentThread の 終了待ちは各コアの CurrentThread をみてスピン待ちしている。
    // これ以降、pCurrentThread と pCurrentProcess の存在は保証されない

    const KProcessAddress tlr = pNextThread->GetThreadLocalRegionAddr();
    KCPU::SwitchThreadLocalRegion(GetAsInteger(tlr));

    return true;
}

}}

