﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>

#include <nn/nn_Common.h>
#include <nn/svc/svc_Types.h>
#include <nn/svc/svc_MemoryConfigSelect.h>
#include <nn/svc/svc_MemoryMapSelect.h>
#include <nn/util/util_IntrusiveList.h>
#include "kern_Assert.h"
#include "kern_PageTableSelect.h"
#include "kern_KObjectAdaptor.h"
#include "kern_KHandleTable.h"
#include "kern_KUseSlabAllocator.h"
#include "kern_KAffinityMask.h"
#include "kern_KCapability.h"
#include "kern_CPUSelect.h"
#include "kern_WorkThread.h"
#include "kern_KLockWithPriorityInheritance.h"
#include "kern_KAddressArbiter.h"
#include "kern_KLightMutex.h"
#include "kern_KThread.h"
#include "svc/kern_KUserPointer.h"
#include "kern_Kernel.h"
#include "kern_KResourceLimit.h"
#include "kern_KSharedMemoryInfo.h"
#include "kern_KMemoryManager.h"
#include "kern_KRbTree.h"
#include "kern_KWaitObject.h"

namespace nn { namespace kern {

class KThread;
class KProcess;
class KEventInfo;

#if NN_KERN_HAS_MMU
class KThreadLocalPage :
    public KUseSlabAllocator<KThreadLocalPage>,
    public IntrusiveRbTreeBaseNode<KThreadLocalPage>
{
public:
    static const int NUM_REGIONS_PER_PAGE = NN_KERN_FINEST_PAGE_SIZE / NN_SVC_MEMORY_THREAD_LOCAL_REGION_SIZE;

private:
    KProcessAddress m_VirtualAddress;
    KProcess*   m_pBindedProcess;
    bool        m_IsTLRFree[NUM_REGIONS_PER_PAGE];

public:
    explicit KThreadLocalPage() : m_VirtualAddress(Null<KProcessAddress>()), m_pBindedProcess(nullptr)
    {
        for (size_t i = 0; i < NUM_REGIONS_PER_PAGE; i++)
        {
            m_IsTLRFree[i] = true;
        }
    }
    explicit KThreadLocalPage(KProcessAddress addr) : m_VirtualAddress(addr), m_pBindedProcess(nullptr)
    {
        for (size_t i = 0; i < NUM_REGIONS_PER_PAGE; i++)
        {
            m_IsTLRFree[i] = true;
        }
    }

    Result  Initialize(KProcess* pProcess);
    Result  Finalize();
    void*   GetPointer() const;

    KProcessAddress ReserveTLR();
    void            ReleaseTLR(KProcessAddress addr);
    KProcessAddress GetAddress() const { return m_VirtualAddress; }
    bool            IsAllUsed() const
    {
        for (int i = 0; i < NUM_REGIONS_PER_PAGE; i++)
        {
            if (m_IsTLRFree[i])
            {
                return false;
            }
        }

        return true;
    }
    bool            IsAllFree() const
    {
        for (int i = 0; i < NUM_REGIONS_PER_PAGE; i++)
        {
            if (!m_IsTLRFree[i])
            {
                return false;
            }
        }

        return true;
    }
};

struct KThreadLocalPageCompare
{
    int operator() (const KThreadLocalPage& a, const KThreadLocalPage& b)
    {
        KProcessAddress addr = a.GetAddress();

        if (addr < b.GetAddress())
        {
            return -1;
        }
        if (addr == b.GetAddress())
        {
            return 0;
        }
        return 1;
    }
};
#endif  // if NN_KERN_HAS_MMU

class KProcess :
    public KObjectAdaptor<KProcess, KSynchronizationObject>,
    public KWorkerTask
{
public:
    enum State
    {
        State_Initializing,
        State_PreAttached,
        State_Running,
        State_WaitAttach,
        State_Attached,
        State_Terminating,
        State_Terminated,
        State_Breaked,
    };
    typedef nn::util::IntrusiveList<KThread, nn::util::IntrusiveListMemberNodeTraits<KThread, &KThread::m_Node>> ThreadList;
    static const size_t ProcessAlign = 0x00200000;

private:

#if NN_KERN_HAS_MMU
    typedef IntrusiveRbTree<KThreadLocalPage, IntrusiveRbTreeBaseNodeTraits<KThreadLocalPage>, KThreadLocalPageCompare> TLPList;
    typedef TLPList::iterator TLPIterator;
#endif  // if NN_KERN_HAS_MMU
    typedef nn::util::IntrusiveList<KSharedMemoryInfo, nn::util::IntrusiveListBaseNodeTraits<KSharedMemoryInfo>> SharedMemoryInfoList;

    // read/write
    KProcessPageTable       m_PageTable;
    InterlockedVariable<size_t>    m_UsingSupervisorMemorySize;
#if NN_KERN_HAS_MMU
    TLPList                 m_TLPListAllUsed;
    TLPList                 m_TLPListPartUsed;
#endif  // if NN_KERN_HAS_MMU

    // read, rare write
    int32_t                 m_IdealProcessor;
    void*                   m_pAttachedObject;
    KResourceLimit*         m_pResourceLimit;
    KVirtualAddress         m_ExtraResourceAddr;
    size_t                  m_ExtraResourcePageCount;
    size_t                  m_ReleaseMemorySizeHint;

    State                   m_State;
    KLightMutex             m_Mutex;
    KLightMutex             m_ListMutex;
    KLockWithPriorityInheritance m_LockWithPriorityInheritance;
    KAddressArbiter         m_AddressArbiter;
    Bit64                   m_Random[4];
    bool                    m_Signaled;
    bool                    m_IsInitialized;
    bool                    m_IsApplication;
    char                    m_Name[13];
    InterlockedVariable<uint16_t> m_NumThreads;
    uint16_t                m_MaxNumThreads;
    Bit32                   m_Flags;
    KMemoryManager::Region  m_Region;
    int64_t                 m_SchedCount;
    // readonly
    KCapability             m_Capability;
    nn::svc::ProgramId      m_ProgramId;
    Bit64                   m_ProcessId;
    int64_t                 m_CreationTime;
    KProcessAddress         m_EntryAddress;
    size_t                  m_MemorySize;
    size_t                  m_MainStackSize;
    size_t                  m_MaxMemoryPerProcess;
    Bit32                   m_Version;
    // huge size
    KHandleTable            m_HandleTable;
    KProcessAddress         m_ProcessLocalRegionAddr;
    KThreadQueue            m_ExceptionThreadQueue;
    KThread*                m_pExceptionThread;
    ThreadList              m_ThreadList;
    SharedMemoryInfoList    m_SharedMemoryList;
    bool                    m_Suspended;

    bool                    m_JitDebug;
    nn::svc::DebugEvent     m_JitEventType;
    nn::svc::DebugException m_JitExceptionType;
    uintptr_t               m_JitParam2;
    uintptr_t               m_JitParam3;
    uintptr_t               m_JitParam4;
    uintptr_t               m_JitParam5;
    Bit64                   m_JitThreadId;
    KWaitObject             m_WaitObject;
    KThread*                m_pRunningThread[KCPU::NUM_CORE];
    uint64_t                m_RunningThreadIdleCount[KCPU::NUM_CORE];

    InterlockedVariable<int32_t>   m_NumCreatedThread;
    InterlockedVariable<int64_t>   m_CpuTime;
    InterlockedVariable<int64_t>   m_NumProcessSwitch;
    InterlockedVariable<int64_t>   m_NumThreadSwitch;
    InterlockedVariable<int64_t>   m_NumFpuSwitch;
    InterlockedVariable<int64_t>   m_NumSystemCall;
    InterlockedVariable<int64_t>   m_NumIpc;
    InterlockedVariable<int64_t>   m_NumIpcReply;
    InterlockedVariable<int64_t>   m_NumIpcRecv;
#ifdef NN_KERN_ENABLE_SVC_PROFILE
    InterlockedVariable<uint32_t>  m_NumSvc[128];
#endif
    static InterlockedVariable<int64_t>    s_ProcessId;
    static InterlockedVariable<int64_t>    s_InitialProcessId;

    KUnusedPageManager              m_UnusedPageManager;
    KMemoryBlockResourceManager     m_MemoryBlockResourceManager;
    KBlockInfoManager               m_BlockInfoManager;
    KPageTableManager               m_PageTableManager;
public:
    enum
    {
        InitialProcessIdMin = 1,
        InitialProcessIdMax = NN_KERN_SLAB_OBJ_NUM_PROCESS,
        ProcessIdMin = InitialProcessIdMax + 1,
        ProcessIdMax = 0xffffffffffffffffull,
    };
    static const int PROCESS_FLAG_MAP_VRAM = 0x1u << 0;

    KProcess() : m_IsInitialized(false) {}
    virtual ~KProcess(){}

    Result Initialize(const nn::svc::CreateProcessParameter& params, const KPageGroup& pg, const Bit32 flags[], int32_t numFlags, KResourceLimit* pResourceLimit, KMemoryManager::Region region);
    Result Initialize(const nn::svc::CreateProcessParameter& params, svc::KUserPointer<const Bit32*> flags, int32_t numFlags, KResourceLimit* pResourceLimit, KMemoryManager::Region region);
    void Exit();
    bool   EnterJitDebug(nn::svc::DebugEvent eventtype, nn::svc::DebugException exception_type, uintptr_t param2 = 0, uintptr_t param3 = 0, uintptr_t param4 = 0, uintptr_t param5 = 0);
    void   ClearJitInfo();
    KEventInfo* GetJitInfo();
    Result Terminate();
    bool   IsTerminated();
    bool   IsApplication() const { return m_IsApplication; }
    KResourceLimit* GetResourceLimit() const { return m_pResourceLimit; }
    bool TestLimit(nn::svc::LimitableResource resource, int64_t value)
    {
        if (m_pResourceLimit)
        {
            return m_pResourceLimit->TestLimit(resource, value);
        }
        else
        {
            return true;
        }
    }
    bool TestLimit(nn::svc::LimitableResource resource, int64_t value, int64_t timeout)
    {
        if (m_pResourceLimit)
        {
            return m_pResourceLimit->TestLimit(resource, value, timeout);
        }
        else
        {
            return true;
        }
    }
    void ReleaseLimit(nn::svc::LimitableResource resource, int64_t value)
    {
        if (m_pResourceLimit)
        {
            m_pResourceLimit->ReleaseLimit(resource, value);
        }
    }

    void ReleaseLimit(nn::svc::LimitableResource resource, int64_t value, int64_t hint)
    {
        if (m_pResourceLimit)
        {
            m_pResourceLimit->ReleaseLimit(resource, value, hint);
        }
    }

    Result Run(int32_t mainThreadPriority, size_t mainThreadStackSize);

    void RegisterThread(KThread* pThread);
    void UnregisterThread(KThread* pThread);
    ThreadList& GetThreadList()
    {
        NN_KERN_THIS_ASSERT();
        return m_ThreadList;
    }

    bool IsSuspended() const
    {
        NN_KERN_THIS_ASSERT();
        return m_Suspended;
    }

    void SetSuspend(bool isSuspend)
    {
        NN_KERN_THIS_ASSERT();
        m_Suspended = isSuspend;
    }

    virtual Bit64 GetId() const
    {
        NN_KERN_THIS_ASSERT();
        return m_ProcessId;
    }

    KLightMutex& GetStateMutex()
    {
        NN_KERN_THIS_ASSERT();
        return m_Mutex;
    }

    KLightMutex& GetListMutex()
    {
        NN_KERN_THIS_ASSERT();
        return m_ListMutex;
    }

    KProcess::State GetState() const;
    KProcess::State GetStateNoCheck() const
    {
        NN_KERN_THIS_ASSERT();
        return m_State;
    }

    int64_t GetSchedCount() const
    {
        NN_KERN_THIS_ASSERT();
        return m_SchedCount;
    }

    void IncrementSchedCount()
    {
        NN_KERN_THIS_ASSERT();
        ++m_SchedCount;
    }

    int64_t GetCreationTime() const
    {
        NN_KERN_THIS_ASSERT();
        return m_CreationTime;
    }

    KProcessPageTable& GetPageTable()
    {
        NN_KERN_THIS_ASSERT();
        return m_PageTable;
    }

    const KProcessPageTable& GetPageTable() const
    {
        NN_KERN_THIS_ASSERT();
        return m_PageTable;
    }

    KHandleTable& GetHandleTable()
    {
        NN_KERN_THIS_ASSERT();
        return m_HandleTable;
    }

    KProcessAddress GetEntryPoint() const
    {
        NN_KERN_THIS_ASSERT();
        return m_EntryAddress;
    }

    size_t GetMemorySize() const
    {
        NN_KERN_THIS_ASSERT();
        return m_MemorySize;
    }

    size_t GetMainStackSize() const
    {
        NN_KERN_THIS_ASSERT();
        return m_MainStackSize;
    }

    bool Is64Bit() const
    {
        NN_KERN_THIS_ASSERT();
        return (m_Flags & nn::svc::CreateProcessParameterFlag_64Bit);
    }

    Bit32 GetCreateProcessParameterFlag() const
    {
        return m_Flags;
    }

    Bit64 GetCoreMask() const
    {
        NN_KERN_THIS_ASSERT();
        return m_Capability.GetCoreMask();
    }

    Bit64 GetPriorityMask() const
    {
        NN_KERN_THIS_ASSERT();
        return m_Capability.GetPriorityMask();
    }

    Result AddSharedMemory(KSharedMemory* pSharedMemory, KProcessAddress addr, size_t size);
    void DeleteSharedMemory(KSharedMemory* pSharedMemory, KProcessAddress addr, size_t size);

    Result CreateThreadLocalRegion(KProcessAddress* pOut);
    Result DeleteThreadLocalRegion(KProcessAddress addr);
    void* GetThreadLocalRegionPointer(KProcessAddress addr);

    KProcessAddress GetProcessLocalRegionAddr() const { return m_ProcessLocalRegionAddr; }

    Bit32 GetAllocateOption() const { return m_PageTable.GetAllocateOption(); }

    static void Switch(KProcess* pCurrentProcess, KProcess* pNextProcess)
    {
        NN_UNUSED(pCurrentProcess);
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
        if (pCurrentProcess)
        {
            pCurrentProcess->IncrementNumSwitch();
        }
#endif
        SetCurrentProcess(pNextProcess);
#if NN_KERN_MULTI_PROCESS
        if (pNextProcess)
        {
            pNextProcess->GetPageTable().Activate(pNextProcess->GetId());
        }
        else
        {
            Kernel::GetKernelPageTable().Activate();
        }
#endif  // if NN_KERN_HAS_MULTI_PROCESS
    }

    bool IsAttachedByDebugger()
    {
        NN_KERN_THIS_ASSERT();
        return m_pAttachedObject;
    }
    void* GetDebugObject()
    {
        NN_KERN_THIS_ASSERT();
        return m_pAttachedObject;
    }
    KProcess::State SetDebugObject(void* pDebug);
    void ClearDebugObject(KProcess::State state);

    void SetIdealProcessor(int32_t number)
    {
        NN_KERN_THIS_ASSERT();
        m_IdealProcessor = number;
    }

    int32_t GetIdealProcessor() const
    {
        NN_KERN_THIS_ASSERT();
        return m_IdealProcessor;
    }
    const char* GetName() const
    {
        NN_KERN_THIS_ASSERT();
        return m_Name;
    }
    nn::svc::ProgramId GetProgramId() const
    {
        NN_KERN_THIS_ASSERT();
        return m_ProgramId;
    }
    Bit64   GetRandomNumber(size_t index) const
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ABORT_UNLESS(index < sizeof(m_Random) / sizeof(m_Random[0]));
        return m_Random[index];
    }
    void AddUsingSupervisorMemorySize(size_t size)
    {
        m_UsingSupervisorMemorySize += size;
    }
    void SubtractUsingSupervisorMemorySize(size_t size)
    {
        NN_KERN_ASSERT( size <= m_UsingSupervisorMemorySize );
        m_UsingSupervisorMemorySize -= size;
    }
    size_t GetUsingSupervisorMemorySize() const
    {
        return m_UsingSupervisorMemorySize;
    }

    size_t GetUsingUserPhysicalMemorySize() const;
    size_t GetAssingedUserPhysicalMemorySize() const;

    static KProcess* GetProcessFromId(Bit64 processId);
    static Result GetProcessList(int32_t* pNumProcesses, svc::KUserPointer<Bit64*> pProcessIds, int32_t arraySize);
    Result GetThreadList(int32_t* pNumThreads, svc::KUserPointer<Bit64*> pThreadIds, int32_t arraySize);
    bool IsPermitIsPermittedInterrupt(int32_t irqNo) const
    {
        return m_Capability.IsPermittedInterruptNumber(irqNo);
    }

    bool   IsDebuggerAttached();

    bool   EnterUserException();
    bool   LeaveUserException();
    bool   ReleaseUserException(KThread* pThread);

    void CopySvcPermisionBitsTo(void* p)
    {
        m_Capability.CopySvcPermisionBitsTo(p);
    }

    bool    IsPermittedDebug() const     { return m_Capability.IsPermittedDebug(); }
    bool    CanForceDebug() const        { return m_Capability.CanForceDebug(); }
    bool    CanShareDeviceMemory() const { return false;}
    Bit32   GetIntentedKernelMajorVersion() const { return m_Capability.GetIntentedKernelMajorVersion(); }
    Bit32   GetIntentedKernelMinorVersion() const { return m_Capability.GetIntentedKernelMinorVersion(); }

    bool    CheckThreadPriority(int32_t priority) const;
    void    IncrementThreadCount();
    void    DecrementThreadCount();

    size_t GetExtraResourceSize()      const { return m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE; }
    size_t GetExtraResourceUsage()     const
    {
        if (m_ExtraResourcePageCount == 0)
        {
            return 0;
        }
        return m_UnusedPageManager.GetUsed() * NN_KERN_FINEST_PAGE_SIZE;
    }

    KThread* GetRunningThread(int32_t coreNo) const { return m_pRunningThread[coreNo]; }
    uint64_t GetRunningThreadIdleCount(int32_t coreNo) const { return m_RunningThreadIdleCount[coreNo]; }
    void SetRunningThread(int32_t coreNo, KThread* pThread, uint64_t idleCount)
    {
        m_pRunningThread[coreNo] = pThread;
        m_RunningThreadIdleCount[coreNo] = idleCount;
    }
    void ClearRunningThread(KThread* pThread)
    {
        for (size_t i = 0; i < NN_ARRAY_SIZE(m_pRunningThread); i++)
        {
            if (m_pRunningThread[i] == pThread)
            {
                m_pRunningThread[i] = nullptr;
            }
        }
    }

    const KUnusedPageManager& GetUnusedPageManager() const { return m_UnusedPageManager; }
    const KMemoryBlockResourceManager& GetMemoryBlockResourceManager() const { return m_MemoryBlockResourceManager; }
    const KBlockInfoManager& GetBlockInfoManager() const { return m_BlockInfoManager; }
    const KPageTableManager& GetPageTableManager() const { return m_PageTableManager; }

    int32_t GetMaxNumThreads()      const { return m_MaxNumThreads; }
    int32_t GetNumCreatedThread()   const { return m_NumCreatedThread; }
    int64_t GetCpuTime()            const { return m_CpuTime; }
    int64_t GetNumProcessSwitch()   const { return m_NumProcessSwitch; }
    int64_t GetNumThreadSwitch()    const { return m_NumThreadSwitch; }
    int64_t GetNumFpuSwitch()       const { return m_NumFpuSwitch; }
    int64_t GetNumSystemCall()      const { return m_NumSystemCall; }
    int64_t GetNumIpc()             const { return m_NumIpc; }
    int64_t GetNumIpcRecv()         const { return m_NumIpcRecv; }
    int64_t GetNumIpcReply()        const { return m_NumIpcReply; }
    void AddCpuTime(int64_t add)            { m_CpuTime       += add; }
    void IncrementNumCreatedThread()        { ++m_NumCreatedThread; }
    void IncrementNumThreadSwitch()         { ++m_NumThreadSwitch; }
    void IncrementNumFpuSwitch()            { ++m_NumFpuSwitch; }
    void IncrementNumSystemCall()           { ++m_NumSystemCall; }
    void IncrementNumIpc()                  { ++m_NumIpc; }
    void IncrementNumIpcRecv()              { ++m_NumIpcRecv; }
    void IncrementNumIpcReply()             { ++m_NumIpcReply; }

#ifdef NN_KERN_ENABLE_SVC_PROFILE
    void IncrementNumSvc(int svcNo) { ++m_NumSvc[svcNo]; }
    uint32_t GetNumSystemCall(int id)
    {
        uint32_t old = m_NumSvc[id].Read();
        for (;;)
        {
            uint32_t v = m_NumSvc[id].CompareAndSwap(old, 0);
            if (v == old)
            {
                break;
            }
            old = v;
        }
        return old;
    }
#endif
    void IncrementNumSwitch();

    KWaitObject* GetWaitObjectPointer() { return &m_WaitObject; }

    Result SignalToAddress(KProcessAddress addr)
    {
        return m_LockWithPriorityInheritance.SignalToAddress(addr);
    }
    Result WaitForAddress(nn::svc::Handle handle, KProcessAddress addr, Bit32 ownValue)
    {
        return m_LockWithPriorityInheritance.WaitForAddress(handle, addr, ownValue);
    }
    void SignalConditionVariable(uintptr_t cvKey, int32_t num)
    {
        m_LockWithPriorityInheritance.SignalConditionVariable(cvKey, num);
    }
    Result WaitConditionVariable(KProcessAddress addr, uintptr_t cvKey, Bit32 ownValue, int64_t ns)
    {
        return m_LockWithPriorityInheritance.WaitConditionVariable(addr, cvKey, ownValue, ns);
    }

    Result WaitAddressArbiter(uintptr_t addr, nn::svc::ArbitrationType type, int32_t value, int64_t ns)
    {
        return m_AddressArbiter.WaitForAddress(addr, type, value, ns);
    }

    Result SignalAddressArbiter(uintptr_t addr, nn::svc::SignalType type, int32_t value, int32_t num)
    {
        return m_AddressArbiter.SignalToAddress(addr, type, value, num);
    }

    void Synchronization(int64_t ns);

    //-----------------------------------------------------------------
    // override
    virtual bool IsInitialized() const { return m_IsInitialized; }

    // override KAutoObject
    virtual void    Finalize();
    static void     PostFinalize(uintptr_t arg) { NN_UNUSED(arg); }

    // override KSynchronizationObject
    virtual bool    IsSignaled() const;

    // override KWorkerTask
    virtual void    DoTask();

    Result  Reset();

    void SetBreaked()
    {
        if (m_State == State_Attached)
        {
            ChangeState(State_Breaked);
        }
    }
    void SetAttached()
    {
        if (m_State == State_Breaked)
        {
            ChangeState(State_Attached);
        }
    }

    Result SetActivity(nn::svc::ProcessActivity activity);
private:
    void ChangeState(State state)
    {
        if (m_State != state)
        {
            m_State = state;
            m_Signaled = true;
            NotifyAvailable();
        }
    }
    Result Initialize(const nn::svc::CreateProcessParameter& params);
    void ShutdownStep1();
    void ShutdownStep2();

    //! KAutoObjectのプリセット関数セット定義です。クラスの末尾に記述する必要があります
    NN_AUTOOBJECT_DERIVED_FUNCSET(KProcess, KSynchronizationObject)
};

}}

