﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/util/util_ScopeExit.h>

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_DebugSelect.h"
#include "kern_KProcess.h"
#include "kern_KThread.h"
#include "kern_KMemoryManager.h"
#include "kern_KEvent.h"
#include "kern_KSharedMemory.h"
#include "kern_KScheduler.h"
#include "kern_KPageTableBase.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_Kernel.h"
#include "kern_Utility.h"
#include "svc/kern_KUserPointer.h"
#include "kern_SystemControl.h"
#include "kern_KScopedResourceLimitTester.h"
#include "kern_KPageBuffer.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KProcess);

InterlockedVariable<int64_t>    KProcess::s_ProcessId(KProcess::ProcessIdMin);
InterlockedVariable<int64_t>    KProcess::s_InitialProcessId(KProcess::InitialProcessIdMin);

namespace
{
void TerminateChildren(KProcess* pTarget, const KThread* pException)
{
    // Request all threads to begin the terminate procedure while holding the scheduler lock
    {
        KScopedLightLock procListLocker(&pTarget->GetListMutex());
        KScopedSchedulingLock locker;

        KProcess::ThreadList::iterator end = pTarget->GetThreadList().end();
        for (KProcess::ThreadList::iterator it = pTarget->GetThreadList().begin(); it != end; it++)
        {
            KThread* p = &*it;

            if (p != pException)
            {
                if (p->GetState() != KThread::STATE_TERMINATED)
                {
                    // KThread::Open() and KThread::Close() are not called.
                    // Concurrent finalization cannot happen: in case of concurrent call to
                    // KThread::Finalize(), it will block on KProcess::UnregisterThread().
                    p->RequestTerminate();
                }
            }
        }

    }

    // Now, wait for each thread to reach the TERMINATED state
    for(;;)
    {
        KThread* pToBeTerminate = nullptr;

        {
            KScopedLightLock procListLocker(&pTarget->GetListMutex());

            KProcess::ThreadList::iterator end = pTarget->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pTarget->GetThreadList().begin(); it != end; ++it)
            {
                KThread* p = &*it;

                if (p != pException)
                {
                    if (p->GetState() != KThread::STATE_TERMINATED)
                    {
                        if (NN_LIKELY(p->Open()))
                        {
                            pToBeTerminate = p;
                            break;
                        }
                    }
                }
            }
        }

        if (pToBeTerminate == nullptr)
        {
            break;
        }

        pToBeTerminate->Terminate();
        pToBeTerminate->Close();
    }
}

}


/*===========================================================================*
 * public メンバ関数
 *===========================================================================*/

Result KProcess::Initialize(const nn::svc::CreateProcessParameter& params, svc::KUserPointer<const Bit32*> flags, int32_t numFlags, KResourceLimit* pResourceLimit, KMemoryManager::Region region)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_POINTER_ASSERT(pResourceLimit);

    Result result;
    size_t memoryNumPages = params.memoryNumPages;
    size_t extraResourcePageCount = params.extraResourcePageCount;

    m_Region = region;
    m_pResourceLimit = pResourceLimit;

    size_t requiredMemoryForExtraResources = KSystemControl::ComputeMemoryRequirementForSecureMemory(extraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, region);
    KScopedResourceLimitTester tester(this, nn::svc::LimitableResource_PhysicalMemoryMax, memoryNumPages * NN_KERN_FINEST_PAGE_SIZE + requiredMemoryForExtraResources);
    if (tester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    KMemoryBlockResourceManager* pMemoryBlockResourceManager;
    KBlockInfoManager* pBlockInfoManager;
    KPageTableManager* pPageTableManager;

    m_ExtraResourceAddr = Null<KVirtualAddress>();
    m_ExtraResourcePageCount = params.extraResourcePageCount;
    if (m_ExtraResourcePageCount != 0)
    {
        result = KSystemControl::AllocateSecureMemory(&m_ExtraResourceAddr, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, region);
        if (result.IsFailure())
        {
            m_ExtraResourcePageCount = 0;
            return result;
        }
        size_t refCounterSize = RoundUp(KPageTableManager::CalcReferenceCounterBufferSize(m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE), NN_KERN_FINEST_PAGE_SIZE);

        m_UnusedPageManager.Initialize(m_ExtraResourceAddr + refCounterSize, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE - refCounterSize);
        m_PageTableManager.Initialize(&m_UnusedPageManager, GetTypedPointer<uint16_t>(m_ExtraResourceAddr));
        m_MemoryBlockResourceManager.Initialize(&m_UnusedPageManager);
        pMemoryBlockResourceManager = &m_MemoryBlockResourceManager;
        m_BlockInfoManager.Initialize(&m_UnusedPageManager);
        pBlockInfoManager = &m_BlockInfoManager;
        pPageTableManager = &m_PageTableManager;
    }
    else
    {
        pPageTableManager = &Kernel::GetPageTableManager();
        pMemoryBlockResourceManager = ((params.flags & nn::svc::CreateProcessParameterFlag_IsApplication)? &Kernel::GetAppMemoryBlockManager(): &Kernel::GetSysMemoryBlockManager());
        pBlockInfoManager = &Kernel::GetBlockInfoManager();
    }

    bool fromBack = ((params.flags & nn::svc::CreateProcessParameterFlag_EnableAslr) == 0);

    result = m_PageTable.Initialize(m_ProcessId,
            static_cast<nn::svc::CreateProcessParameterFlag>(params.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask),
            (params.flags & nn::svc::CreateProcessParameterFlag_EnableAslr),
            fromBack,
            region,
            params.memoryAddress,
            memoryNumPages * NN_KERN_FINEST_PAGE_SIZE,
            pMemoryBlockResourceManager,
            pBlockInfoManager,
            pPageTableManager
            );
    if (result.IsFailure())
    {
        if (m_ExtraResourceAddr != Null<KVirtualAddress>())
        {
            NN_KERN_ABORT_UNLESS(m_MemoryBlockResourceManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_BlockInfoManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_PageTableManager.GetUsed() == 0);
            KSystemControl::FreeSecureMemory(m_ExtraResourceAddr, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, region);
            m_ExtraResourceAddr = Null<KVirtualAddress>();
            m_ExtraResourcePageCount = 0;
        }
        return result;
    }
    if (!m_PageTable.IsInRange(params.memoryAddress, memoryNumPages * NN_KERN_FINEST_PAGE_SIZE, KMemoryState_Code))
    {
        m_PageTable.Finalize();
        if (m_ExtraResourceAddr != Null<KVirtualAddress>())
        {
            NN_KERN_ABORT_UNLESS(m_MemoryBlockResourceManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_BlockInfoManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_PageTableManager.GetUsed() == 0);
            KSystemControl::FreeSecureMemory(m_ExtraResourceAddr, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, region);
            m_ExtraResourceAddr = Null<KVirtualAddress>();
            m_ExtraResourcePageCount = 0;
        }
        return nn::svc::ResultInvalidRegion();
    }

    result = m_PageTable.MapPages(params.memoryAddress, memoryNumPages, KMemoryState_Code, KMemoryPermission_KernelRead);
    if (result.IsFailure())
    {
        m_PageTable.Finalize();
        if (m_ExtraResourceAddr != Null<KVirtualAddress>())
        {
            NN_KERN_ABORT_UNLESS(m_MemoryBlockResourceManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_BlockInfoManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_PageTableManager.GetUsed() == 0);
            KSystemControl::FreeSecureMemory(m_ExtraResourceAddr, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, region);
            m_ExtraResourceAddr = Null<KVirtualAddress>();
            m_ExtraResourcePageCount = 0;
        }
        return result;
    }

    result = m_Capability.Initialize(flags, numFlags, &m_PageTable);
    if (result.IsFailure())
    {
        m_PageTable.Finalize();
        if (m_ExtraResourceAddr != Null<KVirtualAddress>())
        {
            NN_KERN_ABORT_UNLESS(m_MemoryBlockResourceManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_BlockInfoManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_PageTableManager.GetUsed() == 0);
            KSystemControl::FreeSecureMemory(m_ExtraResourceAddr, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, region);
            m_ExtraResourceAddr = Null<KVirtualAddress>();
            m_ExtraResourcePageCount = 0;
        }
        return result;
    }

    m_ProcessId        = s_ProcessId++;
    NN_KERN_ABORT_UNLESS(ProcessIdMin <= m_ProcessId);
    NN_KERN_ABORT_UNLESS(m_ProcessId < ProcessIdMax);

    result = Initialize(params);
    if (result.IsFailure())
    {
        m_PageTable.Finalize();
        if (m_ExtraResourceAddr != Null<KVirtualAddress>())
        {
            NN_KERN_ABORT_UNLESS(m_MemoryBlockResourceManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_BlockInfoManager.GetUsed() == 0);
            NN_KERN_ABORT_UNLESS(m_PageTableManager.GetUsed() == 0);
            KSystemControl::FreeSecureMemory(m_ExtraResourceAddr, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, region);
            m_ExtraResourceAddr = Null<KVirtualAddress>();
            m_ExtraResourcePageCount = 0;
        }
        return result;
    }

    tester.Accepted();

    pResourceLimit->Open();

    return result;
}

Result KProcess::Initialize(const nn::svc::CreateProcessParameter& params, const KPageGroup& pg, const Bit32 flags[], int32_t numFlags, KResourceLimit* pResourceLimit, KMemoryManager::Region region)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_POINTER_ASSERT(pResourceLimit);

    size_t memoryNumPages = params.memoryNumPages;
    NN_KERN_ABORT_UNLESS((memoryNumPages * NN_KERN_FINEST_PAGE_SIZE) / NN_KERN_FINEST_PAGE_SIZE == memoryNumPages);

    m_Region = region;
    m_pResourceLimit = pResourceLimit;
    m_ExtraResourceAddr = Null<KVirtualAddress>();
    m_ExtraResourcePageCount = 0;
    bool fromBack = ((params.flags & nn::svc::CreateProcessParameterFlag_EnableAslr) == 0);
    Result result = m_PageTable.Initialize(m_ProcessId,
            static_cast<nn::svc::CreateProcessParameterFlag>(params.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask),
            (params.flags & nn::svc::CreateProcessParameterFlag_EnableAslr),
            fromBack,
            region,
            params.memoryAddress,
            memoryNumPages * NN_KERN_FINEST_PAGE_SIZE,
            (params.flags & nn::svc::CreateProcessParameterFlag_IsApplication)? &Kernel::GetAppMemoryBlockManager(): &Kernel::GetSysMemoryBlockManager(),
            &Kernel::GetBlockInfoManager(),
            &Kernel::GetPageTableManager()
            );
    if (result.IsFailure())
    {
        return result;
    }

    if (!m_PageTable.IsInRange(params.memoryAddress, memoryNumPages * NN_KERN_FINEST_PAGE_SIZE, KMemoryState_Code))
    {
        m_PageTable.Finalize();
        return nn::svc::ResultInvalidRegion();
    }

    result = m_PageTable.MapPageGroup(params.memoryAddress, pg, KMemoryState_Code, KMemoryPermission_KernelRead);
    if (result.IsFailure())
    {
        m_PageTable.Finalize();
        return result;
    }

    result = m_Capability.Initialize(flags, numFlags, &m_PageTable);
    if (result.IsFailure())
    {
        m_PageTable.Finalize();
        return result;
    }

    m_ProcessId        = s_InitialProcessId++;
    NN_KERN_ABORT_UNLESS(InitialProcessIdMin <= m_ProcessId);
    NN_KERN_ABORT_UNLESS(m_ProcessId <= InitialProcessIdMax);

    result = Initialize(params);
    if (result.IsFailure())
    {
        m_PageTable.Finalize();
        return result;
    }

    pResourceLimit->Open();

    return result;
}

Result KProcess::Initialize(const nn::svc::CreateProcessParameter& params)
{
    Result result;

    size_t memoryNumPages = params.memoryNumPages;
    if (m_Capability.GetIntentedKernelMajorVersion() > NN_SVC_VERSION_MAJOR ||
            (m_Capability.GetIntentedKernelMajorVersion() == NN_SVC_VERSION_MAJOR && m_Capability.GetIntentedKernelMinorVersion() > NN_SVC_VERSION_MINOR))
    {
        NN_WARNING(false, "Invalid intended kernel version(%d.%d). Current Kernel Version %d.%d",
                m_Capability.GetIntentedKernelMajorVersion(),
                m_Capability.GetIntentedKernelMinorVersion(),
                NN_SVC_VERSION_MAJOR, NN_SVC_VERSION_MINOR);
        return nn::svc::ResultInvalidCombination();
    }

    if (m_Capability.GetIntentedKernelMajorVersion() < NN_SVC_REQUIRED_VERSION_MAJOR ||
            (m_Capability.GetIntentedKernelMajorVersion() == NN_SVC_REQUIRED_VERSION_MAJOR && m_Capability.GetIntentedKernelMinorVersion() < NN_SVC_REQUIRED_VERSION_MINOR))
    {
        NN_WARNING(false, "Invalid intended kernel version(%d.%d). Required Kernel Version %d.%d",
                m_Capability.GetIntentedKernelMajorVersion(),
                m_Capability.GetIntentedKernelMinorVersion(),
                NN_SVC_REQUIRED_VERSION_MAJOR, NN_SVC_REQUIRED_VERSION_MINOR);
        return nn::svc::ResultInvalidCombination();
    }

    result = CreateThreadLocalRegion(&m_ProcessLocalRegionAddr);
    if (result.IsFailure())
    {
        return result;
    }
    ::std::memset(GetThreadLocalRegionPointer(m_ProcessLocalRegionAddr), 0, NN_SVC_MEMORY_THREAD_LOCAL_REGION_SIZE);

    static_assert( sizeof(params.name) < sizeof(m_Name), "" );
    ::std::memcpy(m_Name, params.name, sizeof(params.name));
    m_Name[sizeof(m_Name) - 1]  = '\0';

    m_State         = State_Initializing;
    m_MainStackSize = 0;
    m_CreationTime  = KHardwareTimer::GetTick();
    m_UsingSupervisorMemorySize = 0;

    m_IdealProcessor   = 0;
    m_Flags            = params.flags;

    m_Version          = params.version;
    m_ProgramId        = params.programId;
    m_EntryAddress     = params.memoryAddress;
    m_MemorySize       = memoryNumPages * NN_KERN_FINEST_PAGE_SIZE;
    m_IsApplication    = !!(params.flags & nn::svc::CreateProcessParameterFlag_IsApplication);
    m_JitDebug         = false;

    for (size_t i = 0; i < NN_ARRAY_SIZE(m_pRunningThread); i++)
    {
        m_pRunningThread[i] = nullptr;
    }
    for (size_t i = 0; i < NN_ARRAY_SIZE(m_RunningThreadIdleCount); i++)
    {
        m_RunningThreadIdleCount[i] = 0;
    }

    nn::svc::CreateProcessParameterFlag addressSapceType = static_cast<nn::svc::CreateProcessParameterFlag>(params.flags & nn::svc::CreateProcessParameterFlag_AddressSpaceMask);
    switch (addressSapceType)
    {
    case nn::svc::CreateProcessParameterFlag_AddressSpace64Bit:
    case nn::svc::CreateProcessParameterFlag_AddressSpace64BitOld:
    case nn::svc::CreateProcessParameterFlag_AddressSpace32Bit:
        {
            m_MaxMemoryPerProcess = m_PageTable.GetHeapRegionSize();
        }
        break;
    case nn::svc::CreateProcessParameterFlag_AddressSpace32BitNoReserved:
        {
            m_MaxMemoryPerProcess = m_PageTable.GetHeapRegionSize() + m_PageTable.GetReservedRegionSize();
        }
        break;
    default:
        {
            NN_KERN_ABORT();
        }
        break;
    }

    KSystemControl::GenerateRandom(m_Random, sizeof(m_Random) / sizeof(m_Random[0]));

    m_NumThreads       = 0;
    m_MaxNumThreads    = 0;

    m_NumCreatedThread = 0;
    m_NumProcessSwitch = 0;
    m_NumThreadSwitch  = 0;
    m_NumFpuSwitch     = 0;
    m_NumSystemCall    = 0;
    m_NumIpc           = 0;
    m_Signaled         = false;
    m_pAttachedObject  = NULL;
    m_pExceptionThread = NULL;
    m_Suspended        = false;
    m_ReleaseMemorySizeHint = 0;
    m_SchedCount            = 0;

#ifdef NN_KERN_ENABLE_SVC_PROFILE
    for (int svcNo = 0; svcNo < NN_ARRAY_SIZE(m_NumSvc); svcNo++)
    {
        m_NumSvc[svcNo] = 0;
    }
#endif

    m_IsInitialized = true;

    return ResultSuccess();
}

Result KProcess::Run(int32_t mainThreadPriority, size_t mainThreadStackSize)
{
    NN_KERN_THIS_ASSERT();
    Result result;

    KScopedLightLock locker(&m_Mutex);

    State state = m_State;
    if (state != State_Initializing && state != State_PreAttached)
    {
        return nn::svc::ResultInvalidState();
    }

    KScopedResourceLimitTester threadTester(this, nn::svc::LimitableResource_ThreadCountMax);
    if (threadTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    NN_KERN_ABORT_UNLESS(m_MainStackSize == 0);
    mainThreadStackSize = RoundUp(mainThreadStackSize, NN_KERN_FINEST_PAGE_SIZE);
    if (mainThreadStackSize + m_MemorySize > m_MaxMemoryPerProcess || mainThreadStackSize + m_MemorySize < m_MemorySize)
    {
        return nn::svc::ResultOutOfMemory();
    }

    KScopedResourceLimitTester tester(this, nn::svc::LimitableResource_PhysicalMemoryMax, mainThreadStackSize);
    if (tester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

#if NN_KERN_DEFERED_ARRANGE
    // プロセス初期スレッド用のスタックを用意

    KProcessAddress stackBottom(0);

    if (mainThreadStackSize)
    {
        KProcessAddress stackTop;

        result = m_PageTable.MapPages(
                &stackTop,
                mainThreadStackSize / NN_KERN_FINEST_PAGE_SIZE,
                KMemoryState_Stack,
                KMemoryPermission_UserReadWrite);
        if(result.IsFailure())
        {
            NN_WARNING(false, "failed to allocate initial stack");
            return result;
        }
        stackBottom = stackTop + mainThreadStackSize;
        m_MainStackSize += mainThreadStackSize;
    }
#else
    NN_UNUSED(mainThreadStackSize);
    const KProcessAddress stackBottom  = NULL;
#endif
    tester.Accepted();
    bool freeUserStack = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (freeUserStack && m_MainStackSize)
        {
            NN_KERN_ABORT_IF_FAILED(m_PageTable.UnmapPages(stackBottom - m_MainStackSize, m_MainStackSize / NN_KERN_FINEST_PAGE_SIZE, KMemoryState_Stack));
            if (m_pResourceLimit)
            {
                m_pResourceLimit->ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, m_MainStackSize);
            }
            m_MainStackSize = 0;
        }
    };

    result = m_PageTable.SetMaxHeapSize(m_MaxMemoryPerProcess - (m_MainStackSize + m_MemorySize));
    if(result.IsFailure())
    {
        return result;
    }

    result = m_HandleTable.Initialize(m_Capability.GetHandleTableSize());
    if(result.IsFailure())
    {
        return result;
    }

    // スレッドを生成。
    KThread* pFirstThread = KThread::Create();
    if(!pFirstThread)
    {
        m_HandleTable.Finalize();
        return nn::svc::ResultOutOfResource();
    }
    KScopedAutoObject<KThread> autoCloser(pFirstThread);

    result = InitializeThreadForUser(
        pFirstThread,
        reinterpret_cast<ThreadFunc>(GetUntypedPointer(GetEntryPoint())),
        0,
        stackBottom,
        mainThreadPriority,
        m_IdealProcessor,
        this );
    if (result.IsFailure())
    {
        m_HandleTable.Finalize();
        return result;
    }

    threadTester.Accepted();

    KThread::Register(pFirstThread);

    nn::svc::Handle threadHandle;
    result = GetHandleTable().Add(&threadHandle, pFirstThread);
    if (result.IsFailure())
    {
        m_HandleTable.Finalize();
        return result;
    }

    pFirstThread->GetContext()->SetFirstArgument(0, static_cast<nnHandle>(threadHandle).value);

    ChangeState(((state == State_Initializing)? State_Running: State_Attached));
    result = pFirstThread->Run();
    if (result.IsFailure())
    {
        ChangeState(state);
        m_HandleTable.Finalize();
        return result;
    }

    // Runされたスレッドに対するOpen(ExitThread に対応するため)
    pFirstThread->Open();

    NN_LOG("KProcess::Run  pid=%lld name=%-8s tid=%lld AMask=0x%llx Ideal=%d Run=%d\n",
        m_ProcessId, GetName(), pFirstThread->GetId(),
        pFirstThread->GetAffinityMask().GetAffinityMask(), pFirstThread->GetIdealProcessor(),
        pFirstThread->GetRunningProcessor()
    );
    freeUserStack = false;

    return ResultSuccess();
}

bool KProcess::IsSignaled() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    return m_Signaled;
}

Result KProcess::Reset()
{
    NN_KERN_THIS_ASSERT();

    KScopedLightLock locker(&m_Mutex);
    KScopedSchedulingLock lock;

    if (m_State == State_Terminated || !m_Signaled)
    {
        return nn::svc::ResultInvalidState();
    }

    m_Signaled = false;
    return ResultSuccess();
}

bool KProcess::EnterJitDebug(nn::svc::DebugEvent eventtype, nn::svc::DebugException exception_type, uintptr_t param2, uintptr_t param3, uintptr_t param4, uintptr_t param5)
{
    NN_KERN_ASSERT(&GetCurrentThread().GetParent() == this);
    if (!(m_Flags & nn::svc::CreateProcessParameterFlag_EnableJitDebug))
    {
        return false;
    }

    NN_KERN_ASSERT(m_State != State_Initializing);
    NN_KERN_ASSERT(m_State != State_PreAttached);
    NN_KERN_ASSERT(m_State != State_Terminated);

    for (;;)
    {
        KScopedLightLock procLocker(&m_Mutex);
        KScopedLightLock procListLocker(&m_ListMutex);
        KScopedSchedulingLock locker;

        // Terminating Attached を含む
        if (IsDebuggerAttached())
        {
            return true;
        }
        if (GetCurrentThread().IsTerminateRequested())
        {
            return false;
        }
        NN_KERN_ASSERT(m_State != State_Attached);
        NN_KERN_ASSERT(m_State != State_Breaked);
        if (m_State != State_Running && m_State != State_WaitAttach)
        {
            NN_KERN_ASSERT(m_State == State_Terminating);
            return false;
        }

        if (GetCurrentThread().IsSuspended())
        {
            continue;
        }

        ThreadList::iterator end = m_ThreadList.end();
        for (ThreadList::iterator it = m_ThreadList.begin(); it != end; it++)
        {
            it->SuspendRequest(KThread::SuspendType_Debug);
        }

        ChangeState(State_WaitAttach);
        m_JitDebug = true;
        m_JitEventType = eventtype;
        m_JitExceptionType = exception_type;
        m_JitParam2 = param2;
        m_JitParam3 = param3;
        m_JitParam4 = param4;
        m_JitParam5 = param5;
        m_JitThreadId = GetCurrentThread().GetId();
        break;
    }

    {
        KScopedSchedulingLock lock;

        if (m_State == State_Running || m_State == State_WaitAttach || m_State == State_Attached || m_State == State_Breaked)
        {
            return true;
        }
    }

    return false;
}

KEventInfo* KProcess::GetJitInfo()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    KEventInfo* pInfo = nullptr;
    if (m_JitDebug)
    {
        pInfo = KDebug::CreateDebugEvent(m_JitEventType, m_JitExceptionType, m_JitParam2, m_JitParam3, m_JitParam4, m_JitParam5, m_JitThreadId);
    }
    return pInfo;
}

void KProcess::ClearJitInfo()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    m_JitDebug = false;
}

void KProcess::ShutdownStep1()
{
    KThread& currentThread = GetCurrentThread();

    // カレント以外のスレッドを終了する
    TerminateChildren(this, &currentThread);

    // ハンドルテーブルを解放する
    m_HandleTable.Finalize();
}

void KProcess::ShutdownStep2()
{
    NN_KERN_ASSERT(!KScheduler::IsSchedulerLocked());

    if (m_pResourceLimit)
    {
        m_ReleaseMemorySizeHint = GetUsingUserPhysicalMemorySize();
        m_pResourceLimit->ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, 0, m_ReleaseMemorySizeHint);
    }

    {
        KScopedSchedulingLock locker;
        ChangeState(State_Terminated);
    }

    Close();
}

void KProcess::DoTask()
{
    // 最後のスレッドを Terminate
    TerminateChildren(this, NULL);

    KDebug::OnExitProcess(this);
    ShutdownStep2();
    // Close されているので以降 this を参照してはいけない
}

void KProcess::Exit()
{
    NN_KERN_THIS_ASSERT();

    KThread& currentThread = GetCurrentThread();

    bool bNotAlreadyTerminating = false;

    {
        KScopedLightLock locker(&m_Mutex);
        KScopedSchedulingLock schLocker;
        NN_KERN_ASSERT(m_State != State_Initializing);
        NN_KERN_ASSERT(m_State != State_PreAttached);
        NN_KERN_ASSERT(m_State != State_WaitAttach);
        NN_KERN_ASSERT(m_State != State_Terminated);
        if (m_State == State_Running || m_State == State_Attached || m_State == State_Breaked)
        {
            ChangeState(State_Terminating);
            bNotAlreadyTerminating = true;
        }
    }

    if (bNotAlreadyTerminating)
    {
        ShutdownStep1();

        NN_LOG("KProcess::Exit  pid=%lld name=%-8s\n", m_ProcessId, GetName());

        // タスクスレッドに自身の破棄を依頼する。
        KWorkThreadManager::AddTask(KWorkThreadManager::LOW_PRIO_WORK_THREAD, this);
    }

    currentThread.Exit();
    NN_KERNEL_PANIC("dead thread moving.");
}

Result KProcess::Terminate()
{
    NN_KERN_THIS_ASSERT();

    bool bNotAlreadyTerminating = false;

    {
        KScopedLightLock locker(&m_Mutex);
        if (m_State == State_Initializing || m_State == State_PreAttached)
        {
            return nn::svc::ResultInvalidState();
        }

        KScopedSchedulingLock schLocker;
        if (m_State == State_Running || m_State == State_WaitAttach || m_State == State_Attached || m_State == State_Breaked)
        {
            ChangeState(State_Terminating);
            bNotAlreadyTerminating = true;
        }
    }

    if (bNotAlreadyTerminating)
    {
        ShutdownStep1();
        NN_LOG("KProcess::Terminate  pid=%lld name=%-8s\n", m_ProcessId, GetName());
        KDebug::OnTerminateProcess(this);
        ShutdownStep2();
    }

    return ResultSuccess();
}

Result KProcess::AddSharedMemory(KSharedMemory* pSharedMemory, KProcessAddress addr, size_t size)
{
    NN_UNUSED(addr);
    NN_UNUSED(size);
    KScopedLightLock locker(&m_Mutex);
    KSharedMemoryInfo* pInfo = nullptr;
    SharedMemoryInfoList::iterator it = m_SharedMemoryList.begin();
    while (it != m_SharedMemoryList.end())
    {
        if (it->GetSharedMemory() == pSharedMemory)
        {
            pInfo = &*it;
            break;
        }
        it++;
    }

    if (!pInfo)
    {
        pInfo = KSharedMemoryInfo::Allocate();
        pInfo->Initialize(pSharedMemory);
        m_SharedMemoryList.push_back(*pInfo);
    }

    if (!pInfo)
    {
        return nn::svc::ResultOutOfResource();
    }

    pSharedMemory->Open();
    pInfo->Open();

    return ResultSuccess();
}

void KProcess::DeleteSharedMemory(KSharedMemory* pSharedMemory, KProcessAddress addr, size_t size)
{
    NN_UNUSED(addr);
    NN_UNUSED(size);
    KScopedLightLock locker(&m_Mutex);
    KSharedMemoryInfo* pInfo = nullptr;
    SharedMemoryInfoList::iterator it = m_SharedMemoryList.begin();
    while (it != m_SharedMemoryList.end())
    {
        if (it->GetSharedMemory() == pSharedMemory)
        {
            pInfo = &*it;
            break;
        }
        it++;
    }
    NN_KERN_ABORT_UNLESS(pInfo);
    if (pInfo->Close())
    {
        m_SharedMemoryList.erase(it);
        KSharedMemoryInfo::Free(pInfo);
    }
    pSharedMemory->Close();
}

void KProcess::Finalize()
{
    NN_KERN_THIS_ASSERT();

    {
        // this がカレントになっているコアはないはず
        for( int i = 0; i < KCPU::NUM_CORE; ++i )
        {
            NN_KERN_ASSERT( Kernel::GetCurrentSet(i).pCurrentProcess != this );
        }
    }

    DeleteThreadLocalRegion(m_ProcessLocalRegionAddr);

    size_t usingMemorySize = GetUsingUserPhysicalMemorySize();

    // メモリを解放する
    m_PageTable.Finalize();

    if (m_ExtraResourceAddr != Null<KVirtualAddress>())
    {
        NN_KERN_ABORT_UNLESS(m_MemoryBlockResourceManager.GetUsed() == 0);
        NN_KERN_ABORT_UNLESS(m_BlockInfoManager.GetUsed() == 0);
        NN_KERN_ABORT_UNLESS(m_PageTableManager.GetUsed() == 0);
        NN_LOG(
                "ExtraResource pid=%lld name=%-8s:\n"
                "  Total       %5lu / %5lu\n"
                "  MemoryBlock %5lu / %5lu\n"
                "  PageTable   %5lu / %5lu\n"
                "  BlockInfo   %5lu / %5lu\n",
                m_ProcessId, GetName()
                , m_UnusedPageManager.GetPeak(), m_UnusedPageManager.GetNum()
                , m_MemoryBlockResourceManager.GetPeak(), m_MemoryBlockResourceManager.GetNum()
                , m_PageTableManager.GetPeak(), m_PageTableManager.GetNum()
                , m_BlockInfoManager.GetPeak(), m_BlockInfoManager.GetNum()
                );
        KSystemControl::FreeSecureMemory(m_ExtraResourceAddr, m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, m_Region);
        m_ExtraResourceAddr = Null<KVirtualAddress>();
        m_ExtraResourcePageCount = 0;
    }

    if (m_pResourceLimit)
    {
        NN_KERN_ABORT_UNLESS(usingMemorySize >= m_ReleaseMemorySizeHint);
        m_pResourceLimit->ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, usingMemorySize, usingMemorySize - m_ReleaseMemorySizeHint);
        m_pResourceLimit->Close();
    }

    // SharedMemoryを解放する
    // ここに来た時点で、KProcess(this) や m_PageTable を参照するスレッドは存在しない。
    // マップ状態が途中で変更されることはない。
    // アンマップ済みなので、参照カウントを減らすだけでよい
    SharedMemoryInfoList::iterator it = m_SharedMemoryList.begin();
    while (it != m_SharedMemoryList.end())
    {
        KSharedMemoryInfo* pInfo = &*it;

        KSharedMemory* pShared = pInfo->GetSharedMemory();

        while (!pInfo->Close())
        {
            pShared->Close();
        }
        pShared->Close();

        it = m_SharedMemoryList.erase(it);
        KSharedMemoryInfo::Free(pInfo);
    }

    NN_KERN_ABORT_UNLESS(m_TLPListPartUsed.empty());
    NN_KERN_ABORT_UNLESS(m_TLPListAllUsed.empty());

    NN_LOG("KProcess::Finalize  pid=%lld name=%-8s\n", m_ProcessId, GetName());

    KObjectAdaptor<KProcess, KSynchronizationObject>::Finalize();
}

bool KProcess::IsTerminated()
{
    NN_KERN_THIS_ASSERT();
    return m_State == State_Terminated;
}

Result KProcess::CreateThreadLocalRegion(KProcessAddress* pOut)
{
#if NN_KERN_HAS_MMU
    NN_KERN_THIS_ASSERT();

    KThreadLocalPage* tlp = NULL;
    KProcessAddress tlr = 0;

    // 既に確保しているページがあまってないか探す
    {
        KScopedSchedulingLock locker;
        TLPIterator it = m_TLPListPartUsed.begin();
        if (it != m_TLPListPartUsed.end())
        {
            tlr = it->ReserveTLR();
            NN_KERN_ABORT_UNLESS(tlr != Null<KProcessAddress>());
            if (it->IsAllUsed())
            {
                tlp = &*it;
                m_TLPListPartUsed.erase(it);
                m_TLPListAllUsed.insert(*tlp);
            }

            *pOut = tlr;
            return ResultSuccess();
        }
    }

    // 管理領域を確保
    tlp = KThreadLocalPage::Allocate();
    if(tlp == nullptr)
    {
        return nn::svc::ResultOutOfMemory();
    }

    Result result = tlp->Initialize(this);
    if(result.IsFailure())
    {
        KThreadLocalPage::Free(tlp);
        NN_WARNING(false, "failed to initialize thread local page.");
        return result;
    }

    // ページから領域を確保
    //   1 つのページには 4 つの領域を確保できる
    KProcessAddress tlrAddr = tlp->ReserveTLR();
    NN_KERN_ABORT_UNLESS(tlrAddr != Null<KProcessAddress>());

    *pOut = tlrAddr;

    // 既に確保したページとして登録
    {
        KScopedSchedulingLock locker;
        if (tlp->IsAllUsed())
        {
            m_TLPListAllUsed.insert(*tlp);
        }
        else
        {
            m_TLPListPartUsed.insert(*tlp);
        }
    }
#else   // if NN_KERN_HAS_MMU
    *pOut = NULL;
#endif  // if NN_KERN_HAS_MMU else

    return ResultSuccess();
}

Result KProcess::DeleteThreadLocalRegion(KProcessAddress addr)
{
#if NN_KERN_HAS_MMU
    NN_KERN_THIS_ASSERT();
    KThreadLocalPage* pToBeRelease = NULL;

    {
        KScopedSchedulingLock locker;

        TLPIterator it = m_TLPListPartUsed.find(KThreadLocalPage(RoundDown(GetAsInteger(addr), NN_KERN_FINEST_PAGE_SIZE)));
        if (it == m_TLPListPartUsed.end())
        {
            it = m_TLPListAllUsed.find(KThreadLocalPage(RoundDown(GetAsInteger(addr), NN_KERN_FINEST_PAGE_SIZE)));
            if (it == m_TLPListAllUsed.end())
            {
                NN_WARNING(false, "unable to delete the thread local region.");
                return nn::svc::ResultInvalidAddress();
            }
            it->ReleaseTLR(addr);

            KThreadLocalPage* tlp = &*it;
            m_TLPListAllUsed.erase(it);
            if (tlp->IsAllFree())
            {
                pToBeRelease = tlp;
            }
            else
            {
                m_TLPListPartUsed.insert(*tlp);
            }
        }
        else
        {
            it->ReleaseTLR(addr);
            KThreadLocalPage* tlp = &*it;
            if (tlp->IsAllFree())
            {
                m_TLPListPartUsed.erase(it);
                pToBeRelease = tlp;
            }
        }
    }

    if (pToBeRelease != nullptr)
    {
        // ページを解放
        pToBeRelease->Finalize();

        // 管理領域を開放
        KThreadLocalPage::Free(pToBeRelease);
    }

    return ResultSuccess();
#else   // if NN_KERN_HAS_MMU
    NN_UNUSED(addr);
    return ResultSuccess();
#endif  // if NN_KERN_HAS_MMU else
}

void* KProcess::GetThreadLocalRegionPointer(KProcessAddress addr)
{
#if NN_KERN_HAS_MMU
    NN_KERN_THIS_ASSERT();
    KThreadLocalPage* pToBeRelease = nullptr;

    {
        KScopedSchedulingLock locker;

        TLPIterator it = m_TLPListPartUsed.find(KThreadLocalPage(RoundDown(GetAsInteger(addr), NN_KERN_FINEST_PAGE_SIZE)));
        if (it == m_TLPListPartUsed.end())
        {
            it = m_TLPListAllUsed.find(KThreadLocalPage(RoundDown(GetAsInteger(addr), NN_KERN_FINEST_PAGE_SIZE)));
            if (it == m_TLPListAllUsed.end())
            {
                return nullptr;
            }
            pToBeRelease = &*it;
        }
        else
        {
            pToBeRelease = &*it;
        }
    }
    return reinterpret_cast<char*>(pToBeRelease->GetPointer()) + (GetAsInteger(addr) & (NN_KERN_FINEST_PAGE_SIZE - 1));

#else   // if NN_KERN_HAS_MMU
    NN_UNUSED(addr);
    return nullptr;
#endif  // if NN_KERN_HAS_MMU else
}


KProcess* KProcess::GetProcessFromId(Bit64 processId)
{
    ListAccessor accessor;
    const KObjectContainer::ObjectListIterator end = accessor.GetEnd();

    KProcess* pProcess;

    for (KObjectContainer::ObjectListIterator it = accessor.GetBegin(); it != end; it++)
    {
        pProcess = static_cast<KProcess*>(&*it);

        if (pProcess->GetId() == processId)
        {
            if (NN_LIKELY(pProcess->Open()))
            {
                return pProcess;
            }
        }
    }

    return NULL;
}

Result KProcess::GetProcessList(int32_t* pNumProcesses, svc::KUserPointer<Bit64*> pProcessIds, int32_t arraySize)
{
    ListAccessor accessor;
    const KObjectContainer::ObjectListIterator end = accessor.GetEnd();

    int32_t index = 0;

    //arraySize分までIDを返す
    for (KObjectContainer::ObjectListIterator it = accessor.GetBegin(); it != end; it++)
    {
        if (index < arraySize)
        {
            KProcess* pProcess = static_cast<KProcess*>(&*it);
            Bit64 id = pProcess->GetId();
            Result result = pProcessIds.CopyArrayEntryFrom(&id, index);
            if (result.IsFailure())
            {
                return result;
            }
        }

        ++index;
    }

    //プロセス数を返す
    *pNumProcesses = index;

    return ResultSuccess();
}

Result KProcess::GetThreadList(int32_t* pNumThreads, svc::KUserPointer<Bit64*> pThreadIds, int32_t arraySize)
{
    NN_KERN_THIS_ASSERT();
    KScopedLightLock procListLocker(&m_ListMutex);

    int32_t index = 0;

    //arraySize分までIDを返す
    KProcess::ThreadList::iterator end = m_ThreadList.end();
    for (KProcess::ThreadList::iterator it = m_ThreadList.begin(); it != end; ++it)
    {
        KThread* p = &*it;

        if( index < arraySize )
        {
            Bit64 id = p->GetId();
            Result result = pThreadIds.CopyArrayEntryFrom(&id, index);
            if (result.IsFailure())
            {
                return result;
            }
        }

        ++index;
    }

    *pNumThreads = index;

    return ResultSuccess();
}

KProcess::State KProcess::SetDebugObject(void* pDebug)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(pDebug);

    State oldState = m_State;
    m_pAttachedObject = pDebug;

    NN_KERN_ASSERT(m_State == State_Initializing || m_State == State_Running || m_State == State_WaitAttach);
    if (m_State == State_Initializing)
    {
        ChangeState(State_PreAttached);
    }
    else
    {
        ChangeState(State_Breaked);
    }

    return oldState;
}

void KProcess::ClearDebugObject(KProcess::State state)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    m_pAttachedObject = nullptr;
    NN_KERN_ASSERT(m_State == State_PreAttached || m_State == State_Attached || m_State == State_Breaked || m_State == State_Terminating || m_State == State_Terminated);
    if (m_State == State_PreAttached)
    {
        ChangeState(State_Initializing);
    }
    else if (m_State == State_Attached || m_State == State_Breaked)
    {
        if (state == State_Initializing)
        {
            state = State_Running;
        }
        ChangeState(state);
    }
}


bool KProcess::IsDebuggerAttached()
{
    NN_KERN_THIS_ASSERT();
    return m_pAttachedObject;
}

bool KProcess::CheckThreadPriority(int32_t priority) const
{
    NN_KERN_THIS_ASSERT();
    return ((1ull << priority) & m_Capability.GetPriorityMask());
}

void KProcess::IncrementThreadCount()
{
    NN_KERN_ASSERT( m_NumThreads >= 0 );
    const int numThread = ++m_NumThreads;
    NN_UNUSED(numThread);

    ++m_NumCreatedThread;

    if( numThread > m_MaxNumThreads )
    {
        m_MaxNumThreads = numThread;
    }
}

void KProcess::DecrementThreadCount()
{
    NN_KERN_ASSERT(m_NumThreads >= 1);
    const uint16_t afterDec = --m_NumThreads;

    if (afterDec == 0)
    {
        Terminate();
    }
}


void KProcess::IncrementNumSwitch()
{
    ++m_NumProcessSwitch;
    Kernel::IncrementNumProcessSwitch();
}

bool KProcess::EnterUserException()
{
    KThread* pCurrentThread = &GetCurrentThread();
    NN_KERN_ASSERT(this == pCurrentThread->GetParentPointer());
    if (m_pExceptionThread != pCurrentThread)
    {
        for (;;)
        {
            KScopedSchedulingLock lock;
            if (pCurrentThread->IsTerminateRequested())
            {
                return false;
            }

            if (m_pExceptionThread == NULL)
            {
                m_pExceptionThread = pCurrentThread;
                return true;
            }
            m_ExceptionThreadQueue.SleepThread(pCurrentThread);
        }
    }
    else
    {
        return false;
    }
}

bool KProcess::LeaveUserException()
{
    return ReleaseUserException(&GetCurrentThread());
}

bool KProcess::ReleaseUserException(KThread* pThread)
{
    KScopedSchedulingLock lock;

    if (m_pExceptionThread == pThread)
    {
        m_pExceptionThread = NULL;
        if (m_ExceptionThreadQueue.IsNotEmpty())
        {
            m_ExceptionThreadQueue.WakupFrontThread();
        }
        return true;
    }
    else
    {
        return false;
    }
}

void KProcess::RegisterThread(KThread* pThread)
{
    KScopedLightLock locker(&m_ListMutex);
    m_ThreadList.push_back(*pThread);
}

void KProcess::UnregisterThread(KThread* pThread)
{
    KScopedLightLock locker(&m_ListMutex);
    ThreadList::iterator it = m_ThreadList.iterator_to(*pThread);
    m_ThreadList.erase(it);
}

KProcess::State KProcess::GetState() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(m_Mutex.IsLockedByMe());
    return m_State;
}

size_t KProcess::GetUsingUserPhysicalMemorySize() const
{
    return m_PageTable.GetNormalMemorySize() + m_MemorySize + m_MainStackSize + KSystemControl::ComputeMemoryRequirementForSecureMemory(m_ExtraResourcePageCount * NN_KERN_FINEST_PAGE_SIZE, m_Region);
}

size_t KProcess::GetAssingedUserPhysicalMemorySize() const
{
    // アプリのみ正しく機能する
    size_t freeMemorySize = m_pResourceLimit->GetCurrentFreeValue(nn::svc::LimitableResource_PhysicalMemoryMax);
    size_t usingMemorySize = GetUsingUserPhysicalMemorySize();

    if (usingMemorySize + freeMemorySize > m_MaxMemoryPerProcess)
    {
        // 通常このシーケンスに入る
        return m_MaxMemoryPerProcess;
    }
    else
    {
        // 物理メモリサイズが小さい場合
        return usingMemorySize + freeMemorySize;
    }
}

Result KProcess::SetActivity(nn::svc::ProcessActivity activity)
{
    KScopedLightLock procLocker(&m_Mutex);
    KScopedLightLock procListLocker(&m_ListMutex);
    KScopedSchedulingLock locker;

    if (m_State == State_Terminating || m_State == State_Terminated)
    {
        return nn::svc::ResultInvalidState();
    }

    if (activity == nn::svc::ProcessActivity_Paused)
    {
        if (IsSuspended())
        {
            return nn::svc::ResultInvalidState();
        }
        ThreadList::iterator end = m_ThreadList.end();
        for (ThreadList::iterator it = m_ThreadList.begin(); it != end; it++)
        {
            it->SuspendRequest(KThread::SuspendType_Process);
        }
        SetSuspend(true);
    }
    else
    {
        NN_KERN_ASSERT(activity == nn::svc::ProcessActivity_Runnable);
        if (!IsSuspended())
        {
            return nn::svc::ResultInvalidState();
        }
        ThreadList::iterator end = m_ThreadList.end();
        for (ThreadList::iterator it = m_ThreadList.begin(); it != end; it++)
        {
            it->Resume(KThread::SuspendType_Process);
        }
        SetSuspend(false);
    }

    return ResultSuccess();
}

/*===========================================================================*
 * KThreadLocalPage
 *===========================================================================*/

#if NN_KERN_HAS_MMU
Result KThreadLocalPage::Initialize(KProcess* pProcess)
{
    NN_KERN_THIS_ASSERT();
    m_pBindedProcess = pProcess;

    KProcessAddress tlpAddr;
    Result result;

    KPageBuffer* pPageBuffer = KPageBuffer::Allocate();
    if (!pPageBuffer)
    {
        return nn::svc::ResultOutOfMemory();
    }

    result = m_pBindedProcess->GetPageTable().MapPages(
            &tlpAddr,
            1,
            NN_KERN_FINEST_PAGE_SIZE,
            pPageBuffer->GetPhysicalAddress(),
            KMemoryState_ThreadLocal,
            KMemoryPermission_UserReadWrite);

    if (result.IsFailure())
    {
        KPageBuffer::Free(pPageBuffer);
        NN_WARNING(false, "failed to allocate thread local page.");
        return result;
    }
    m_VirtualAddress = tlpAddr;

    return result;
}

Result KThreadLocalPage::Finalize()
{
    NN_KERN_THIS_ASSERT();
    KPhysicalAddress physAddr;
    NN_KERN_ABORT_UNLESS(m_pBindedProcess->GetPageTable().GetPhysicalAddress(&physAddr, GetAddress()));
    Result result = m_pBindedProcess->GetPageTable().UnmapPages(
                        GetAddress(),
                        1,
                        KMemoryState_ThreadLocal);

    if(result.IsFailure())
    {
        NN_WARNING(false, "failed to free thread local page");
        return result;
    }

    KPageBuffer::Free(KPageBuffer::FromPhysicalAddress(physAddr));

    return result;
}

void* KThreadLocalPage::GetPointer() const
{
    NN_KERN_THIS_ASSERT();
    KPhysicalAddress physAddr;
    NN_KERN_ABORT_UNLESS(m_pBindedProcess->GetPageTable().GetPhysicalAddress(&physAddr, GetAddress()));
    return reinterpret_cast<void*>(KPageBuffer::FromPhysicalAddress(physAddr));
}

KProcessAddress KThreadLocalPage::ReserveTLR()
{
    NN_KERN_THIS_ASSERT();
    for (int i = 0; i <NUM_REGIONS_PER_PAGE; i++)
    {
        if (m_IsTLRFree[i])
        {
            m_IsTLRFree[i] = false;
            return GetAddress() + i * NN_SVC_MEMORY_THREAD_LOCAL_REGION_SIZE;
        }
    }

    return Null<KProcessAddress>();
}

void KThreadLocalPage::ReleaseTLR(KProcessAddress addr)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(addr % NN_SVC_MEMORY_THREAD_LOCAL_REGION_SIZE == 0);
    NN_KERN_ASSERT(GetAddress() <= addr && addr < GetAddress() + NN_SVC_MEMORY_PAGE_SIZE);

    int index = (addr - GetAddress()) / NN_SVC_MEMORY_THREAD_LOCAL_REGION_SIZE;

    m_IsTLRFree[index] = true;
}

#endif  // if NN_KERN_HAS_MMU

}}
