﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_KPageTableBase.h"
#include "kern_KMemoryManager.h"
#include "kern_SystemControl.h"
#include <cstring>
#include "kern_MemoryCopySelect.h"
#include "kern_Kernel.h"
#include "kern_KUnsafeMemory.h"
#include "kern_KScopedResourceLimitTester.h"
#include "kern_Utility.h"
#include "kern_KPageBuffer.h"
#include <algorithm>
#include <nn/util/util_ScopeExit.h>

#define DUMP_MEM_BLOCK() // DumpMemoryBlocksLocked()

namespace nn { namespace kern {

Result KPageTableBase::InitializeForProcess(nn::svc::CreateProcessParameterFlag addressSapceType, bool enableAslr, bool fromBack, void* pTable, KProcessAddress begin, KProcessAddress end, KMemoryManager::Region region, KProcessAddress codeRegionAddr, size_t codeRegionSize, KMemoryBlockResourceManager* pMemoryBlockManager, KBlockInfoManager* pBlockInfoManager)
{
    NN_KERN_ASSERT(
            (addressSapceType & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit ||
            (addressSapceType & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace64BitOld ||
            (addressSapceType & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace32Bit ||
            (addressSapceType & nn::svc::CreateProcessParameterFlag_AddressSpaceMask) == nn::svc::CreateProcessParameterFlag_AddressSpace32BitNoReserved);
    NN_KERN_ABORT_UNLESS(begin <= codeRegionAddr && codeRegionAddr < codeRegionAddr + codeRegionSize && codeRegionAddr + codeRegionSize - 1 <= end - 1);

    size_t reservedRegionSize;
    size_t heapRegionSize;
    size_t allocateStackRegionSize = 0;
    size_t allocateKernelMapSmallRegionSize = 0;
    KProcessAddress codeMapRegionBegin;
    switch (addressSapceType & nn::svc::CreateProcessParameterFlag_AddressSpaceMask)
    {
#if defined (NN_BUILD_CONFIG_ABI_LP64)
    case nn::svc::CreateProcessParameterFlag_AddressSpace64Bit:
        {
            m_AddressSpaceSize = 39;
            reservedRegionSize = NN_SVC_ADDR_MEMORY_REGION_RESERVED39_SIZE;
            heapRegionSize = NN_SVC_ADDR_MEMORY_REGION_HEAP39_SIZE;
            KProcessAddress codeRegionEnd = RoundUp(codeRegionAddr + codeRegionSize, RegionAlign);
            codeRegionAddr = RoundDown(codeRegionAddr, RegionAlign);
            codeRegionSize = codeRegionEnd - codeRegionAddr;
            codeMapRegionBegin = NN_SVC_ADDR_MAP39_BEGIN;
            allocateStackRegionSize = NN_SVC_ADDR_MEMORY_REGION_STACK39_SIZE;
            m_StackRegionBegin = 0;
            m_StackRegionEnd = 0;
            allocateKernelMapSmallRegionSize = NN_SVC_ADDR_MEMORY_REGION_SMALL39_SIZE;
            m_KernelMapSmallRegionBegin = 0;
            m_KernelMapSmallRegionEnd = 0;
        }
        break;
    case nn::svc::CreateProcessParameterFlag_AddressSpace64BitOld:
        {
            m_AddressSpaceSize = 36;
            reservedRegionSize = NN_SVC_ADDR_MEMORY_REGION_RESERVED36_SIZE;
            heapRegionSize = NN_SVC_ADDR_MEMORY_REGION_HEAP36_SIZE;
            codeRegionAddr = NN_SVC_ADDR_SMALL_MAP36_BEGIN;
            codeRegionSize = NN_SVC_ADDR_SMALL_MAP36_SIZE;
            codeMapRegionBegin = NN_SVC_ADDR_SMALL_MAP36_BEGIN;
            m_StackRegionBegin = NN_SVC_ADDR_SMALL_MAP36_BEGIN;
            m_StackRegionEnd = NN_SVC_ADDR_SMALL_MAP36_END;
            m_KernelMapSmallRegionBegin = NN_SVC_ADDR_SMALL_MAP36_BEGIN;
            m_KernelMapSmallRegionEnd = NN_SVC_ADDR_SMALL_MAP36_END;
        }
        break;
#endif
    case nn::svc::CreateProcessParameterFlag_AddressSpace32Bit:
        {
            m_AddressSpaceSize = 32;
            reservedRegionSize = NN_SVC_ADDR_MEMORY_REGION_RESERVED32_SIZE;
            heapRegionSize = NN_SVC_ADDR_MEMORY_REGION_HEAP32_SIZE;
            codeRegionAddr = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            codeRegionSize = NN_SVC_ADDR_SMALL_MAP32_SIZE;
            codeMapRegionBegin = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            m_StackRegionBegin = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            m_StackRegionEnd = NN_SVC_ADDR_SMALL_MAP32_END;
            m_KernelMapSmallRegionBegin = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            m_KernelMapSmallRegionEnd = NN_SVC_ADDR_SMALL_MAP32_END;
        }
        break;
    case nn::svc::CreateProcessParameterFlag_AddressSpace32BitNoReserved:
        {
            m_AddressSpaceSize = 32;
            reservedRegionSize = 0;
            heapRegionSize = NN_SVC_ADDR_MEMORY_REGION_HEAP32_SIZE + NN_SVC_ADDR_MEMORY_REGION_RESERVED32_SIZE;
            codeRegionAddr = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            codeRegionSize = NN_SVC_ADDR_SMALL_MAP32_SIZE;
            codeMapRegionBegin = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            m_StackRegionBegin = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            m_StackRegionEnd = NN_SVC_ADDR_SMALL_MAP32_END;
            m_KernelMapSmallRegionBegin = NN_SVC_ADDR_SMALL_MAP32_BEGIN;
            m_KernelMapSmallRegionEnd = NN_SVC_ADDR_SMALL_MAP32_END;
        }
        break;
    default:
        NN_KERN_ABORT();
        break;
    }

    m_EnableAslr        = enableAslr;
    m_AddressBegin      = begin;
    m_AddressEnd        = end;
    m_IsKernel          = false;
    m_pMemoryBlockManager = pMemoryBlockManager;
    m_pBlockInfoManager = pBlockInfoManager;

    KProcessAddress largeRegion;
    size_t largeRegionSize;
    if ((codeRegionAddr - codeMapRegionBegin) >= (end - (codeRegionAddr + codeRegionSize)))
    {
        largeRegion = codeMapRegionBegin;
        largeRegionSize = (codeRegionAddr - codeMapRegionBegin);
    }
    else
    {
        largeRegion = (codeRegionAddr + codeRegionSize);
        largeRegionSize = (end - (codeRegionAddr + codeRegionSize));
    }

    if (largeRegionSize < (reservedRegionSize + heapRegionSize + allocateStackRegionSize + allocateKernelMapSmallRegionSize))
    {
        return nn::svc::ResultOutOfMemory();
    }

    // ランダムにサイズを決めることのできる大きさ
    size_t leftSize = largeRegionSize - (reservedRegionSize + heapRegionSize + allocateStackRegionSize + allocateKernelMapSmallRegionSize);

    size_t tmp0 = 0;
    size_t tmp1 = 0;
    size_t tmp2 = 0;
    size_t tmp3 = 0;

    if (enableAslr)
    {
        tmp0 = KSystemControl::GetRandomValue(0, leftSize / RegionAlign) * RegionAlign;
        tmp1 = KSystemControl::GetRandomValue(0, leftSize / RegionAlign) * RegionAlign;
        tmp2 = KSystemControl::GetRandomValue(0, leftSize / RegionAlign) * RegionAlign;
        tmp3 = KSystemControl::GetRandomValue(0, leftSize / RegionAlign) * RegionAlign;
    }

    m_ReservedRegionBegin   = largeRegion + tmp0;
    m_ReservedRegionEnd     = m_ReservedRegionBegin + reservedRegionSize;
    m_HeapRegionBegin       = largeRegion + tmp1;
    m_HeapRegionEnd         = m_HeapRegionBegin + heapRegionSize;

    if (tmp0 <= tmp1)
    {
        m_HeapRegionBegin += reservedRegionSize;
        m_HeapRegionEnd   += reservedRegionSize;
    }
    else
    {
        m_ReservedRegionBegin += heapRegionSize;
        m_ReservedRegionEnd   += heapRegionSize;
    }

    if (allocateStackRegionSize)
    {
        m_StackRegionBegin = largeRegion + tmp2;
        m_StackRegionEnd   = m_StackRegionBegin + allocateStackRegionSize;
        if (tmp0 < tmp2)
        {
            m_StackRegionBegin += reservedRegionSize;
            m_StackRegionEnd   += reservedRegionSize;
        }
        else
        {
            m_ReservedRegionBegin += allocateStackRegionSize;
            m_ReservedRegionEnd   += allocateStackRegionSize;
        }

        if (tmp1 < tmp2)
        {
            m_StackRegionBegin += heapRegionSize;
            m_StackRegionEnd   += heapRegionSize;
        }
        else
        {
            m_HeapRegionBegin += allocateStackRegionSize;
            m_HeapRegionEnd   += allocateStackRegionSize;
        }
    }

    if (allocateKernelMapSmallRegionSize)
    {
        m_KernelMapSmallRegionBegin = largeRegion + tmp3;
        m_KernelMapSmallRegionEnd   = m_KernelMapSmallRegionBegin + allocateKernelMapSmallRegionSize;
        if (tmp0 < tmp3)
        {
            m_KernelMapSmallRegionBegin += reservedRegionSize;
            m_KernelMapSmallRegionEnd   += reservedRegionSize;
        }
        else
        {
            m_ReservedRegionBegin += allocateKernelMapSmallRegionSize;
            m_ReservedRegionEnd   += allocateKernelMapSmallRegionSize;
        }

        if (tmp1 < tmp3)
        {
            m_KernelMapSmallRegionBegin += heapRegionSize;
            m_KernelMapSmallRegionEnd   += heapRegionSize;
        }
        else
        {
            m_HeapRegionBegin += allocateKernelMapSmallRegionSize;
            m_HeapRegionEnd   += allocateKernelMapSmallRegionSize;
        }

        if (allocateStackRegionSize)
        {
            if (tmp2 < tmp3)
            {
                m_KernelMapSmallRegionBegin += allocateStackRegionSize;
                m_KernelMapSmallRegionEnd   += allocateStackRegionSize;
            }
            else
            {
                m_StackRegionBegin += allocateKernelMapSmallRegionSize;
                m_StackRegionEnd   += allocateKernelMapSmallRegionSize;
            }
        }
    }


    m_HeapEnd               = m_HeapRegionBegin;
    m_MaxHeapSize           = 0;
    m_MapPhysicalMemorySize = 0;
    m_AllocateFillMemoryPattern = KTargetSystem::IsNonZeroFillMemoryEnabled()? FillMemoryPattern_Allocate: FillMemoryPattern_Zero;
    m_IpcFillMemoryPattern = KTargetSystem::IsNonZeroFillMemoryEnabled()? FillMemoryPattern_Ipc: FillMemoryPattern_Zero;
    m_StackFillMemoryPattern = KTargetSystem::IsNonZeroFillMemoryEnabled()? FillMemoryPattern_Stack: FillMemoryPattern_Zero;

    KMemoryManager::From from = (fromBack? KMemoryManager::From_Back: KMemoryManager::From_Front);
    m_AllocateOption        = KMemoryManager::MakeAllocateOption(region, from);

    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_ReservedRegionBegin && m_ReservedRegionBegin <= m_AddressEnd);
    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_ReservedRegionEnd && m_ReservedRegionEnd <= m_AddressEnd);
    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_HeapRegionBegin && m_HeapRegionBegin <= m_AddressEnd);
    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_HeapRegionEnd && m_HeapRegionEnd <= m_AddressEnd);
    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_StackRegionBegin && m_StackRegionBegin <= m_AddressEnd);
    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_StackRegionEnd && m_StackRegionEnd <= m_AddressEnd);
    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_KernelMapSmallRegionBegin && m_KernelMapSmallRegionBegin <= m_AddressEnd);
    NN_KERN_ABORT_UNLESS(m_AddressBegin <= m_KernelMapSmallRegionEnd && m_KernelMapSmallRegionEnd <= m_AddressEnd);

    NN_KERN_ABORT_UNLESS(m_ReservedRegionEnd - 1 < m_HeapRegionBegin  || m_HeapRegionEnd  - 1 < m_ReservedRegionBegin);
    NN_KERN_ABORT_UNLESS(m_ReservedRegionEnd - 1 < m_StackRegionBegin || m_StackRegionEnd - 1 < m_ReservedRegionBegin);
    NN_KERN_ABORT_UNLESS(m_HeapRegionEnd     - 1 < m_StackRegionBegin || m_StackRegionEnd - 1 < m_HeapRegionBegin);
    NN_KERN_ABORT_UNLESS(m_ReservedRegionEnd - 1 < m_KernelMapSmallRegionBegin || m_KernelMapSmallRegionEnd - 1 < m_ReservedRegionBegin);
    NN_KERN_ABORT_UNLESS(m_HeapRegionEnd     - 1 < m_KernelMapSmallRegionBegin || m_KernelMapSmallRegionEnd - 1 < m_HeapRegionBegin);

    m_Table.InitializeForProcess(pTable, begin, end);

    return m_MemoryBlocks.Initialize(GetAsInteger(m_AddressBegin), GetAsInteger(m_AddressEnd), m_pMemoryBlockManager);
}

Result KPageTableBase::InitializeForKernel(bool is64Bit, void* pTable, KProcessAddress begin, KProcessAddress end)
{
    m_AddressSpaceSize  = (is64Bit? 64: 32);
    m_EnableAslr        = false;
    m_AddressBegin      = begin;
    m_AddressEnd        = end;
    m_IsKernel          = true;

    m_KernelMapSmallRegionBegin = 0;
    m_KernelMapSmallRegionEnd = 0;
    m_StackRegionBegin  = 0;
    m_StackRegionEnd    = 0;
    m_HeapRegionBegin   = 0;
    m_HeapRegionEnd     = 0;
    m_ReservedRegionBegin = 0;
    m_ReservedRegionEnd   = 0;
    m_HeapEnd           = 0;
    m_MaxHeapSize       = 0;
    m_MapPhysicalMemorySize = 0;
    m_AllocateFillMemoryPattern =  FillMemoryPattern_Zero;
    m_IpcFillMemoryPattern = FillMemoryPattern_Zero;
    m_StackFillMemoryPattern = FillMemoryPattern_Zero;
    m_pMemoryBlockManager = &Kernel::GetSysMemoryBlockManager();
    m_pBlockInfoManager = &Kernel::GetBlockInfoManager();
    m_AllocateOption    = KMemoryManager::MakeAllocateOption(KMemoryManager::Region_SecureSystem, KMemoryManager::From_Front);

    m_Table.InitializeForKernel(pTable, begin, end);

    return m_MemoryBlocks.Initialize(GetAsInteger(m_AddressBegin), GetAsInteger(m_AddressEnd), m_pMemoryBlockManager);
}

void KPageTableBase::Finalize()
{
    m_MemoryBlocks.Finalize(GetMemoryBlockResourceManager());
    KCPU::InvalidateEntireInstructionCache();
}

Result KPageTableBase::SetMaxHeapSize(size_t size)
{
    KScopedLightLock locker(&m_Mutex);
    NN_KERN_ASSERT(!m_IsKernel);

    m_MaxHeapSize = size;

    return ResultSuccess();
}

Result KPageTableBase::SetHeapSize(KProcessAddress* pOut, size_t size)
{
    KScopedLightLock mapPhyslocker(&m_MapPhysMutex);

    Result result;
    size_t allocSize = 0;
    KProcessAddress curAddr;
    {
        KScopedLightLock locker(&m_Mutex);

        if (m_IsKernel)
        {
            return nn::svc::ResultOutOfMemory();
        }

        if (size > static_cast<size_t>(m_HeapRegionEnd - m_HeapRegionBegin))
        {
            return nn::svc::ResultOutOfMemory();
        }
        if (size > m_MaxHeapSize)
        {
            return nn::svc::ResultOutOfMemory();
        }

        if (static_cast<size_t>(m_HeapEnd - m_HeapRegionBegin) > size)
        {
            // free
            KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
            if (result.IsFailure())
            {
                return result;
            }

            KScopedPageTableUpdater updateData(this);

            result = CheckMemoryState(m_HeapRegionBegin + size, (m_HeapEnd - m_HeapRegionBegin) - size,
                    KMemoryState_All, KMemoryState_Normal,
                    KMemoryPermission_All, KMemoryPermission_UserReadWrite,
                    KMemoryAttribute_All, KMemoryAttribute_None);
            if (result.IsFailure())
            {
                return result;
            }

            size_t numPages = ((m_HeapEnd - m_HeapRegionBegin) - size) / PageSize;
            const KPageProperty property = { KMemoryPermission_None, false, false};
            result = OperateImpl(updateData.GetDataPtr(), m_HeapRegionBegin + size, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_Unmap);
            if (result.IsFailure())
            {
                return result;
            }

            GetCurrentProcess().ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, numPages * PageSize);

            m_MemoryBlocks.Update(&updateBuffer, m_HeapRegionBegin + size, numPages, KMemoryState_Free, property.permission, KMemoryAttribute_None);

            m_HeapEnd = m_HeapRegionBegin + size;
            *pOut = m_HeapRegionBegin;

            return ResultSuccess();
        }
        else if (static_cast<size_t>(m_HeapEnd - m_HeapRegionBegin) == size)
        {
            *pOut = m_HeapRegionBegin;

            return ResultSuccess();
        }
        else
        {
            // alloc
            NN_KERN_ASSERT(static_cast<size_t>(m_HeapEnd - m_HeapRegionBegin) < size);
            allocSize = size - (m_HeapEnd - m_HeapRegionBegin);
            curAddr = m_HeapEnd;
        }
    }

    // メモリのアロケーション処理
    //     m_MapPhysMutex:  locked
    //     m_Mutex:         unlocked

    KScopedResourceLimitTester tester(GetCurrentProcessPointer(), nn::svc::LimitableResource_PhysicalMemoryMax, allocSize);
    if (tester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    KScopedPageGroup pg(GetBlockInfoManager());
    result = Kernel::GetKernelHeapManager().Allocate(&pg.GetPageGroup(), allocSize / PageSize, GetAllocateOption());
    if (result.IsFailure())
    {
        NN_WARNING(false, "failed to allocate memory, numPages = %d", allocSize / PageSize);
        return result;
    }
    pg.GetPageGroup().Open();
    NN_UTIL_SCOPE_EXIT { pg.GetPageGroup().Close(); };

    for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter(); it++)
    {
        ::std::memset(GetUntypedPointer(it->GetBlockAddr()), m_AllocateFillMemoryPattern, it->GetNumPages() * PageSize);
    }

    {
        KScopedLightLock locker(&m_Mutex);

        KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }

        KScopedPageTableUpdater updateData(this);

        NN_KERN_ABORT_UNLESS(curAddr == m_HeapEnd);

        result = CheckMemoryState(m_HeapEnd, allocSize, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0);
        if (result.IsFailure())
        {
            return result;
        }

        size_t numPages = allocSize / PageSize;
        const KPageProperty property = { KMemoryPermission_UserReadWrite, false, false};
        result = OperateImpl(updateData.GetDataPtr(), m_HeapEnd, numPages, pg.GetPageGroup(), property, PageTableOperation_MapRelocate);
        if (result.IsFailure())
        {
            return result;
        }

        tester.Accepted();

        m_MemoryBlocks.Update(&updateBuffer, m_HeapEnd, allocSize / PageSize, KMemoryState_Normal, property.permission, KMemoryAttribute_None);

        m_HeapEnd = m_HeapRegionBegin + size;
        *pOut = m_HeapRegionBegin;

        return ResultSuccess();
    }
}


Result KPageTableBase::MapMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
{
    KScopedLightLock locker(&m_Mutex);
    Result result;

    KMemoryState fromState;
    result = CheckMemoryState(&fromState, nullptr, nullptr, fromAddr, size,
            KMemoryState_FlagsAlias, KMemoryState_FlagsAlias,
            KMemoryPermission_All,   KMemoryPermission_UserReadWrite,
            KMemoryAttribute_All,    KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    result = CheckMemoryState(toAddr, size, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer0(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer1(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    {
        KScopedPageGroup pg(GetBlockInfoManager());
        size_t numPages = size / PageSize;

        result = MakePageGroupImpl(&pg.GetPageGroup(), fromAddr, numPages);
        if (result.IsFailure())
        {
            return result;
        }

        KScopedPageTableUpdater updateData(this);

        // Locked
        const KPageProperty aliasedProperty = { KMemoryPermission_KernelRead, false, false};
        result = OperateImpl(updateData.GetDataPtr(), fromAddr, numPages, Null<KPhysicalAddress>(), false, aliasedProperty, PageTableOperation_ChangeProperties);
        if (result.IsFailure())
        {
            return result;
        }

        // ALIAS でマップ
        const KPageProperty aliasProperty = { KMemoryPermission_UserReadWrite, false, false};
        result = MapPageGroupImpl(updateData.GetDataPtr(), toAddr, pg.GetPageGroup(), aliasProperty);
        if (result.IsFailure())
        {
            const KPageProperty prevProperty = { KMemoryPermission_UserReadWrite, false, false};
            Result resultCheck = OperateImpl(updateData.GetDataPtr(), fromAddr, numPages, Null<KPhysicalAddress>(), false, prevProperty, PageTableOperation_ChangeProperties);
            NN_KERN_ABORT_IF_FAILED(resultCheck);

            return result;
        }

        m_MemoryBlocks.Update(&updateBuffer0, fromAddr, numPages, fromState, aliasedProperty.permission, KMemoryAttribute_Locked);
        m_MemoryBlocks.Update(&updateBuffer1, toAddr,   numPages, KMemoryState_Stack,   aliasProperty.permission,   KMemoryAttribute_None);
    }

    return ResultSuccess();
}

Result KPageTableBase::UnmapMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
{
    KScopedLightLock locker(&m_Mutex);
    Result result;

    KMemoryState fromState;
    result = CheckMemoryState(&fromState, nullptr, nullptr, fromAddr, size,
            KMemoryState_FlagsAlias, KMemoryState_FlagsAlias,
            KMemoryPermission_All,   KMemoryPermission_KernelRead,
            KMemoryAttribute_All,    KMemoryAttribute_Locked);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryPermission toPermission;
    result = CheckMemoryState(nullptr, &toPermission, nullptr, toAddr, size,
            KMemoryState_All, KMemoryState_Stack,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer0(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }
    KMemoryBlockManagerUpdateBuffer updateBuffer1(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    {
        size_t numPages = size / PageSize;
        KScopedPageGroup pgOrg(GetBlockInfoManager());
        KScopedPageGroup pgAlias(GetBlockInfoManager());

        Result result1 = MakePageGroupImpl(&pgOrg.GetPageGroup(),   fromAddr, numPages);
        Result result2 = MakePageGroupImpl(&pgAlias.GetPageGroup(), toAddr, numPages);

        if (result1.IsFailure() || result2.IsFailure())
        {
            return result1.IsFailure() ? result1: result2;
        }

        if (!pgAlias.GetPageGroup().IsSamePages(pgOrg.GetPageGroup()))
        {
            return nn::svc::ResultInvalidRegion();
        }

        KScopedPageTableUpdater updateData(this);

        const KPageProperty aliasProperty = { KMemoryPermission_None, false, false};
        result = OperateImpl(updateData.GetDataPtr(), toAddr, numPages, Null<KPhysicalAddress>(), false, aliasProperty, PageTableOperation_Unmap);
        if (result.IsFailure())
        {
            return result;
        }

        const KPageProperty aliasedProperty = { KMemoryPermission_UserReadWrite, false, false};
        result = OperateImpl(updateData.GetDataPtr(), fromAddr, numPages, Null<KPhysicalAddress>(), false, aliasedProperty, PageTableOperation_ChangeProperties);
        if (result.IsFailure())
        {
            const KPageProperty prevProperty = { toPermission, false, false};
            Result resultCheck = MapPageGroupImpl(updateData.GetDataPtr(), toAddr, pgAlias.GetPageGroup(), prevProperty);
            NN_KERN_ABORT_IF_FAILED(resultCheck);

            return result;
        }

        m_MemoryBlocks.Update(&updateBuffer0, fromAddr, numPages, fromState,         aliasedProperty.permission, KMemoryAttribute_None);
        m_MemoryBlocks.Update(&updateBuffer1, toAddr,   numPages, KMemoryState_Free, aliasProperty.permission,   KMemoryAttribute_None);
    }

    return result;
}

Result KPageTableBase::SetMemoryPermission(KProcessAddress addr, size_t size, nn::svc::MemoryPermission permission)
{
    size_t numPages = size / PageSize;

    Result result;
    KScopedLightLock locker(&m_Mutex);

    KMemoryState oldState;
    KMemoryPermission oldPermission;
    result = CheckMemoryState(&oldState, &oldPermission, nullptr, addr, size,
            KMemoryState_FlagsProtect, KMemoryState_FlagsProtect,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryPermission newPerm = ConvertToKMemoryPermission(permission);
    if (oldPermission == newPerm)
    {
        return ResultSuccess();
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { newPerm, false, false};
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangeProperties);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, oldState, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}


Result KPageTableBase::SetProcessMemoryPermission(KProcessAddress addr, size_t size, nn::svc::MemoryPermission permission)
{
    size_t numPages = size / PageSize;

    Result result;
    KScopedLightLock locker(&m_Mutex);

    KMemoryState oldState;
    KMemoryPermission oldPermission;
    result = CheckMemoryState(&oldState, &oldPermission, nullptr, addr, size,
            KMemoryState_FlagsCode, KMemoryState_FlagsCode,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageGroup pg(GetBlockInfoManager());

    KMemoryPermission newPerm = ConvertToKMemoryPermission(permission);
    KMemoryState newState = oldState;
    if ((newPerm & KMemoryPermission_UserWrite) == KMemoryPermission_UserWrite)
    {
        switch (oldState)
        {
        case KMemoryState_Code:
            {
                newState = KMemoryState_CodeData;
            }
            break;
        case KMemoryState_AliasCode:
            {
                newState = KMemoryState_AliasCodeData;
            }
            break;
        default:
            NN_KERNEL_PANIC("");
            break;
        }
    }
    if ((newPerm & KMemoryPermission_UserExecute) == KMemoryPermission_UserExecute)
    {
        result = MakePageGroupImpl(&pg.GetPageGroup(), GetAsInteger(addr), numPages);
        if (result.IsFailure())
        {
            return result;
        }
    }

    if (oldState == newState && oldPermission == newPerm)
    {
        return ResultSuccess();
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { newPerm, false, false};

    PageTableOperation operation = PageTableOperation_ChangeProperties;
    if ((oldPermission & KMemoryPermission_UserExecute) == KMemoryPermission_UserExecute)
    {
        operation = PageTableOperation_ChangePropertiesWithBreakBeforeMake;
    }

    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, operation);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, newState, property.permission, KMemoryAttribute_None);

    if ((newPerm & KMemoryPermission_UserExecute) == KMemoryPermission_UserExecute)
    {
        for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter(); it++)
        {
            KCPU::StoreDataCache(GetUntypedPointer(it->GetBlockAddr()), it->GetNumPages() * NN_KERN_FINEST_PAGE_SIZE);
        }
        KCPU::InvalidateEntireInstructionCache();
    }

    return ResultSuccess();
}

Result KPageTableBase::SetMemoryAttribute(KProcessAddress addr, size_t size, Bit32 mask, Bit32 attribute)
{
    size_t numPages = size / PageSize;
    NN_KERN_ASSERT((mask | KMemoryAttribute_SetMask) == KMemoryAttribute_SetMask);

    Result result;
    KScopedLightLock locker(&m_Mutex);

    KMemoryState oldState;
    KMemoryPermission oldPermission;
    KMemoryAttribute oldAttribute;
    // キャッシュ属性の連続性をチェックしない
    result = CheckMemoryState(&oldState, &oldPermission, &oldAttribute, addr, size,
            KMemoryState_FlagsUncache, KMemoryState_FlagsUncache,
            0, 0,
            ~(KMemoryAttribute_SetMask | KMemoryAttribute_DeviceShared), 0, (KMemoryAttribute_SetMask | KMemoryAttribute_DeviceShared));
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryAttribute newAttribute = static_cast<KMemoryAttribute>(((oldAttribute & ~mask) | (mask & attribute)));

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { oldPermission, false, (newAttribute & KMemoryAttribute_Uncached) != 0};
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangePropertiesWithBreakBeforeMake);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, oldState, property.permission, newAttribute);

    return ResultSuccess();
}

Result KPageTableBase::MapPageGroup(KProcessAddress addr, const KPageGroup& pg, KMemoryState state, KMemoryPermission permission)
{
    NN_KERN_ASSERT(!m_Mutex.IsLockedByMe());
    Result result;

    const size_t numPages = pg.GetTotalNumPages();

    if (!IsInRange(addr, numPages * PageSize, state))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    result = CheckMemoryState(addr, PageSize * numPages, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { permission, state == KMemoryState_Io, false };
    result = MapPageGroupImpl(updateData.GetDataPtr(), addr, pg, property);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, state, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}

Result KPageTableBase::UnmapPageGroup(KProcessAddress addr, const KPageGroup& pg, KMemoryState state)
{
    Result result;
    NN_KERN_ASSERT(!m_Mutex.IsLockedByMe());

    const size_t numPages = pg.GetTotalNumPages();

    if (!IsPagesInRange(addr, numPages))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    // マッピングされている物理メモリが同一であることをテスト
    {
        KScopedPageGroup pgTest(GetBlockInfoManager());

        result = MakePageGroupImpl(&pgTest.GetPageGroup(), GetAsInteger(addr), numPages);
        if (result.IsFailure())
        {
            return result;
        }

        if (!pgTest.GetPageGroup().IsSamePages(pg))
        {
            return nn::svc::ResultInvalidRegion();
        }
    }

    result = CheckMemoryState(addr, PageSize * numPages,
            KMemoryState_All, state,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { KMemoryPermission_None, false, false};
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_Unmap);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, KMemoryState_Free, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}

Result KPageTableBase::MakePageGroupImpl(KPageGroup* pOut, KProcessAddress addr, size_t numPages) const
{
    NN_KERN_ASSERT(m_Mutex.IsLockedByMe());
    TraverseContext ctx;
    TraverseData data;
    KPhysicalAddress curAddr;
    size_t curSize;
    size_t sumSize;
    bool dataIsValid;
    Result result;

    dataIsValid = m_Table.TraverseBegin(&data, &ctx, addr);
    if (!dataIsValid)
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    curAddr = data.address;
    curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
    sumSize = curSize;

    while (sumSize < numPages * PageSize)
    {
        dataIsValid = m_Table.TraverseNext(&data, &ctx);

        if (!dataIsValid)
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if (data.address != (curAddr + curSize))
        {
            if (!IsHeapPhysicalAddress(curAddr))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }

            result = pOut->AddBlock(GetHeapVirtualAddress(curAddr), curSize / PageSize);
            if (result.IsFailure())
            {
                return result;
            }
            curAddr = data.address;
            curSize = data.size;
        }
        else
        {
            curSize += data.size;
        }
        sumSize += data.size;
    }

    if (sumSize > numPages * PageSize)
    {
        curSize -= sumSize - numPages * PageSize;
    }
    if (!IsHeapPhysicalAddress(curAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    result = pOut->AddBlock(GetHeapVirtualAddress(curAddr), curSize / PageSize);
    if (result.IsFailure())
    {
        return result;
    }

    return ResultSuccess();
}

Result KPageTableBase::MakePageGroupAndOpen(
        KPageGroup* pOut, KProcessAddress addr, size_t numPages,
        Bit32 stateMask, Bit32 state,
        Bit32 permissionMask, Bit32 permission,
        Bit32 attributeMask, Bit32 attribute) const
{
    if (!IsPagesInRange(addr, numPages))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);
    NN_KERN_ASSERT(pOut->GetTotalNumPages() == 0);
    Result result;

    result = CheckMemoryState(
            addr, numPages * NN_KERN_FINEST_PAGE_SIZE,
            stateMask | KMemoryState_FlagsMemory,
            state | KMemoryState_FlagsMemory,
            permissionMask, permission,
            attributeMask, attribute);
    if (result.IsFailure())
    {
        return result;
    }

    result = MakePageGroupImpl(pOut, addr, numPages);
    if (result.IsSuccess())
    {
        pOut->Open();
    }

    return result;
}

Result KPageTableBase::MakePageGroupContiguousAndOpen(
        KPageGroup* pOut, KProcessAddress addr, size_t numPages,
        Bit32 stateMask, Bit32 state,
        Bit32 permissionMask, Bit32 permission,
        Bit32 attributeMask, Bit32 attribute) const
{
    if (!IsPagesInRange(addr, numPages))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);
    NN_KERN_ASSERT(pOut->GetTotalNumPages() == 0);
    Result result;

    result = CheckMemoryStateContiguous(
            addr, numPages * NN_KERN_FINEST_PAGE_SIZE,
            stateMask | KMemoryState_FlagsMemory,
            state | KMemoryState_FlagsMemory,
            permissionMask, permission,
            attributeMask, attribute);
    if (result.IsFailure())
    {
        return result;
    }

    result = MakePageGroupImpl(pOut, addr, numPages);
    if (result.IsSuccess())
    {
        pOut->Open();
    }

    return result;
}

Result KPageTableBase::MapCodeMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
{
    if (!IsInRange(toAddr, size, KMemoryState_AliasCode))
    {
        return nn::svc::ResultInvalidRegion();
    }

    KScopedLightLock locker(&m_Mutex);
    Result result;

    KMemoryState fromState;
    KMemoryPermission fromPermission;
    result = CheckMemoryState(&fromState, &fromPermission, nullptr, fromAddr, size,
            KMemoryState_All, KMemoryState_Normal,
            KMemoryPermission_All, KMemoryPermission_UserReadWrite,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    result = CheckMemoryState(toAddr, size, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer0(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }
    KMemoryBlockManagerUpdateBuffer updateBuffer1(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }


    {
        KScopedPageGroup pg(GetBlockInfoManager());
        size_t numPages = size / PageSize;

        result = MakePageGroupImpl(&pg.GetPageGroup(), fromAddr, numPages);
        if (result.IsFailure())
        {
            return result;
        }

        KScopedPageTableUpdater updateData(this);

        const KPageProperty aliasedProperty = { KMemoryPermission_KernelRead, false, false};
        result = OperateImpl(updateData.GetDataPtr(), fromAddr, numPages, Null<KPhysicalAddress>(), false, aliasedProperty, PageTableOperation_ChangeProperties);
        if (result.IsFailure())
        {
            return result;
        }

        const KPageProperty aliasProperty = { KMemoryPermission_KernelRead, false, false};
        result = MapPageGroupImpl(updateData.GetDataPtr(), toAddr, pg.GetPageGroup(), aliasProperty);
        if (result.IsFailure())
        {
            const KPageProperty prevProperty = { fromPermission, false, false};
            Result resultCheck = OperateImpl(updateData.GetDataPtr(), fromAddr, numPages, Null<KPhysicalAddress>(), false, prevProperty, PageTableOperation_ChangeProperties);
            NN_KERN_ABORT_IF_FAILED(resultCheck);

            return result;
        }

        m_MemoryBlocks.Update(&updateBuffer0, fromAddr, numPages, fromState,              aliasedProperty.permission, KMemoryAttribute_Locked);
        m_MemoryBlocks.Update(&updateBuffer1, toAddr,   numPages, KMemoryState_AliasCode, aliasProperty.permission,   KMemoryAttribute_None);
    }

    return result;
}

Result KPageTableBase::UnmapCodeMemory(KProcessAddress toAddr, KProcessAddress fromAddr, size_t size)
{
    KScopedLightLock locker(&m_Mutex);
    Result result;

    result = CheckMemoryState(fromAddr, size,
            KMemoryState_All, KMemoryState_Normal,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_Locked);
    if (result.IsFailure())
    {
        return result;
    }

    // toAddrの先頭のメモリ状態のチェック
    KMemoryState toState;
    result = CheckMemoryState(
            &toState, nullptr, nullptr,
            toAddr, PageSize,
            KMemoryState_FlagsDll, KMemoryState_FlagsDll,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }
    result = CheckMemoryStateContiguous(
            toAddr, size,
            KMemoryState_All, toState,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    {
        size_t numPages = size / PageSize;
        KScopedPageGroup pgOrg(GetBlockInfoManager());
        KScopedPageGroup pgAlias(GetBlockInfoManager());

        Result result1 = MakePageGroupImpl(&pgOrg.GetPageGroup(),   fromAddr, numPages);
        Result result2 = MakePageGroupImpl(&pgAlias.GetPageGroup(), toAddr, numPages);

        if (result1.IsFailure() || result2.IsFailure())
        {
            return result1.IsFailure() ? result1: result2;
        }

        if (!pgAlias.GetPageGroup().IsSamePages(pgOrg.GetPageGroup()))
        {
            return nn::svc::ResultInvalidRegion();
        }

        KMemoryBlockManagerUpdateBuffer updateBuffer0(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }
        KMemoryBlockManagerUpdateBuffer updateBuffer1(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }

        KScopedPageTableUpdater updateData(this);

        const KPageProperty aliasProperty = { KMemoryPermission_None, false, false};
        result = OperateImpl(updateData.GetDataPtr(), toAddr, numPages, Null<KPhysicalAddress>(), false, aliasProperty, PageTableOperation_Unmap);
        if (result.IsFailure())
        {
            return result;
        }

        const KPageProperty aliasedProperty = { KMemoryPermission_UserReadWrite, false, false};
        result = OperateImpl(updateData.GetDataPtr(), fromAddr, numPages, Null<KPhysicalAddress>(), false, aliasedProperty, PageTableOperation_ChangeProperties);
        if (result.IsFailure())
        {
            KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(toAddr);
            KPageGroup::BlockInfoList::const_iterator pgIt = pgAlias.GetPageGroup().GetBlockBeginIter();
            KPhysicalAddress pgPhysAddr = GetHeapPhysicalAddress(pgIt->GetBlockAddr());
            size_t pgSize = pgIt->GetNumPages() * PageSize;
            for (;;)
            {
                KMemoryInfo mi = it->GetMemoryInfo();
                const KPageProperty prevProperty = { mi.permission, false, false};

                KProcessAddress mapAddr = std::max(mi.baseAddress, GetAsInteger(toAddr));
                size_t mapSize = std::min(GetAsInteger(toAddr + size), mi.baseAddress + mi.size) - GetAsInteger(mapAddr);
                NN_KERN_ABORT_UNLESS(mapSize != 0);

                while (mapSize)
                {
                    if (pgSize == 0)
                    {
                        NN_KERN_ABORT_UNLESS(pgIt != pgAlias.GetPageGroup().GetBlockEndIter());
                        pgIt++;
                        pgPhysAddr = GetHeapPhysicalAddress(pgIt->GetBlockAddr());
                        pgSize = pgIt->GetNumPages() * PageSize;
                    }

                    size_t fragmentSize = std::min(pgSize, mapSize);
                    Result resultCheck = OperateImpl(updateData.GetDataPtr(), mapAddr, fragmentSize / PageSize, pgPhysAddr, true, prevProperty, PageTableOperation_Map);
                    NN_KERN_ABORT_IF_FAILED(resultCheck);

                    mapAddr += fragmentSize;
                    mapSize -= fragmentSize;

                    pgPhysAddr += fragmentSize;
                    pgSize     -= fragmentSize;
                }
                if (toAddr + size - 1 <= mi.baseAddress + mi.size - 1)
                {
                    NN_KERN_ABORT_UNLESS(++pgIt == pgAlias.GetPageGroup().GetBlockEndIter());
                    break;
                }
                it++;
            }

            return result;
        }

        m_MemoryBlocks.Update(&updateBuffer0, toAddr, numPages,   KMemoryState_Free,   aliasProperty.permission,   KMemoryAttribute_None);
        m_MemoryBlocks.Update(&updateBuffer1, fromAddr, numPages, KMemoryState_Normal, aliasedProperty.permission, KMemoryAttribute_None);
    }

    if (toState == KMemoryState_AliasCode)
    {
        KCPU::InvalidateEntireInstructionCache();
    }

    return result;
}

Result KPageTableBase::MapPageGroup(
            KProcessAddress*    pAddr,
            const KPageGroup&   pg,
            KProcessAddress     regionBegin,
            size_t              regionNumPages,
            KMemoryState        state,
            KMemoryPermission   permission)
{
    const size_t numPages = pg.GetTotalNumPages();
    if (!IsInRange(regionBegin, regionNumPages * PageSize, state))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (numPages >= regionNumPages)
    {
        return nn::svc::ResultOutOfMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    KProcessAddress addr = FindFreeAreaInRegion(regionBegin, regionNumPages, numPages, PageSize, 0, GetGuardSize() / PageSize);
    if (addr == Null<KProcessAddress>())
    {
        return nn::svc::ResultOutOfMemory();
    }
    NN_KERN_ASSERT(IsInRange(addr, numPages * PageSize, state));

    NN_KERN_ASSERT(CheckMemoryState(addr, (numPages + 1) * NN_KERN_FINEST_PAGE_SIZE, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0).IsSuccess());

    Result result = ResultSuccess();
    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { permission, state == KMemoryState_Io, false };
    result = MapPageGroupImpl(updateData.GetDataPtr(), addr, pg, property);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, state, property.permission, KMemoryAttribute_None);

    *pAddr = addr;

    return ResultSuccess();
}

Result KPageTableBase::MapPages(KProcessAddress* pAddr, size_t numPages, size_t align, KPhysicalAddress physAddr, bool paIsValid, KProcessAddress regionBegin, size_t regionNumPages, KMemoryState state, KMemoryPermission permission)
{
    NN_KERN_ASSERT(align % PageSize == 0);
    NN_KERN_ASSERT(align >= PageSize);

    if (!IsInRange(regionBegin, regionNumPages * PageSize, state))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (numPages >= regionNumPages)
    {
        return nn::svc::ResultOutOfMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    KProcessAddress addr = FindFreeAreaInRegion(regionBegin, regionNumPages, numPages, align, 0, GetGuardSize() / PageSize);
    if (addr == Null<KProcessAddress>())
    {
        return nn::svc::ResultOutOfMemory();
    }
    NN_KERN_ASSERT(IsInRange(addr, numPages * PageSize, state));

    NN_KERN_ASSERT((addr & (align - 1)) == 0);
    NN_KERN_ASSERT(CheckMemoryState(addr, (numPages + 1) * NN_KERN_FINEST_PAGE_SIZE, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0).IsSuccess());

    Result result = ResultSuccess();
    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { permission, false, false };
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, physAddr, paIsValid, property, paIsValid?PageTableOperation_Map:PageTableOperation_AllocateAndMap);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, state, property.permission, KMemoryAttribute_None);

    *pAddr = addr;

    return ResultSuccess();
}

Result KPageTableBase::MapPages(KProcessAddress addr, size_t numPages, KMemoryState state, KMemoryPermission permission)
{
    Result result;
    if (!IsInRange(addr, numPages * PageSize, state))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    if (CheckMemoryState(addr, numPages * NN_KERN_FINEST_PAGE_SIZE, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0).IsFailure())
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { permission, false, false };
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_AllocateAndMap);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, state, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}

Result KPageTableBase::UnmapPages(KProcessAddress addr, size_t numPages, KMemoryState state)
{
    if (!IsPagesInRange(addr, numPages))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    Result result;
    result = CheckMemoryState(addr, numPages * NN_KERN_FINEST_PAGE_SIZE,
            KMemoryState_All, state,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { KMemoryPermission_None, false, false };
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_Unmap);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, KMemoryState_Free, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}


Result KPageTableBase::QueryPhysicalAddress(nn::svc::PhysicalMemoryInfo* pBlockInfo, KProcessAddress addr)
{
    KScopedLightLock locker(&m_Mutex);

    addr = addr & ~(PageSize - 1);

    KMemoryInfo mi;
    nn::svc::PageInfo pi;
    Result result = QueryInfoImpl(&mi, &pi, GetAsInteger(addr));
    if (result.IsFailure())
    {
        return result;
    }

    result = CheckMemoryState(mi,
            KMemoryState_FlagsQueryPhys, KMemoryState_FlagsQueryPhys,
            KMemoryPermission_UserReadExecute, KMemoryPermission_UserRead,
            0, 0);
    if (result.IsFailure())
    {
        return result;
    }

    KPhysicalAddress physAddr;
    size_t physSize;
    KProcessAddress vaddr = mi.baseAddress;
    KProcessAddress end = mi.baseAddress + mi.size;

    {
        TraverseContext ctx;
        TraverseData data;
        bool dataIsValid;

        dataIsValid = m_Table.TraverseBegin(&data, &ctx, vaddr);
        if (!dataIsValid)
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }

        physAddr = data.address;
        physSize = data.size - (GetAsInteger(data.address) & (data.size - 1));

        for (;;)
        {
            dataIsValid = m_Table.TraverseNext(&data, &ctx);
            if (!dataIsValid)
            {
                break;
            }
            if (data.address != (physAddr + physSize))
            {
                if (vaddr <= addr && addr <= vaddr + physSize - 1)
                {
                    break;
                }
                vaddr += physSize;
                physAddr = data.address;
                physSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
            }
            else
            {
                physSize += data.size;
            }
            if (end < vaddr + physSize)
            {
                break;
            }
        }
        NN_KERN_ASSERT(vaddr <= addr && addr <= vaddr + physSize - 1);
        if (end < vaddr + physSize)
        {
            physSize = end - vaddr;
        }
    }

    pBlockInfo->physicalAddress = GetAsInteger(physAddr);
    pBlockInfo->virtualAddress = GetAsInteger(vaddr);
    pBlockInfo->size = physSize;

    return ResultSuccess();
}

Result KPageTableBase::QueryIoMapping(KProcessAddress* pAddr, KPhysicalAddress ioAddr, size_t ioSize)
{
    Result result;
    KProcessAddress ioAddressBegin = GetMapRegionBegin(KMemoryState_Io);
    KProcessAddress ioAddressEnd = ioAddressBegin + GetMapRegionSize(KMemoryState_Io);

    KScopedLightLock locker(&m_Mutex);

    TraverseContext     context;
    TraverseData        current;
    bool                currentIsValid;
    TraverseData        next;
    bool                nextIsValid;
    const size_t        spaceSize = ioAddressEnd - ioAddressBegin;
    size_t              sizeSum  = 0;

    current.address = 0;
    current.size    = 0;
    currentIsValid  = false;

    nextIsValid = m_Table.TraverseBegin(&next, &context, ioAddressBegin);
    next.size = (next.size - ((next.size - 1) & GetAsInteger(ioAddressBegin)));

    for (;;)
    {
        if( (nextIsValid && currentIsValid && next.address == current.address + current.size)
                || ( !currentIsValid && !nextIsValid ) )
        {
            current.size += next.size;
        }
        else
        {
            if (currentIsValid &&
                    current.address <= ioAddr && ioAddr < ioAddr + ioSize && ioAddr + ioSize <= current.address + current.size)
            {
                // current.address に対応する virtual address は ioAddressBegin + sizeSum
                KProcessAddress mappedAddr = (ioAddressBegin + sizeSum) + (ioAddr - current.address);

                result = CheckMemoryState(mappedAddr, ioSize,
                        KMemoryState_All, KMemoryState_Io,
                        KMemoryPermission_UserRead, KMemoryPermission_UserRead,
                        0, 0);
                if (result.IsSuccess())
                {
                    *pAddr = mappedAddr;
                    return ResultSuccess();
                }
                // エラーの場合は念のため続行する
            }

            sizeSum += current.size;
            current = next;
            currentIsValid = nextIsValid;
        }

        if (sizeSum + current.size >= spaceSize)
        {
            break;
        }

        nextIsValid = m_Table.TraverseNext(&next, &context);
    }

    if (currentIsValid && current.address <= ioAddr && ioAddr < ioAddr + ioSize && ioAddr + ioSize <= current.address + current.size)
    {
        // current.address に対応する virtual address は ioAddressBegin + sizeSum
        KProcessAddress mappedAddr = (ioAddressBegin + sizeSum) + (ioAddr - current.address);

        result = CheckMemoryState(mappedAddr, ioSize,
                        KMemoryState_All, KMemoryState_Io,
                        KMemoryPermission_UserRead, KMemoryPermission_UserRead,
                        0, 0);
        if (result.IsSuccess())
        {
            *pAddr = mappedAddr;
            return ResultSuccess();
        }
    }

    return nn::svc::ResultNotFound();
}


Result KPageTableBase::MapIo(KPhysicalAddress physAddr, size_t size, KMemoryPermission permission)
{
    NN_KERN_ASSERT(physAddr % PageSize == 0);
    NN_KERN_ASSERT(size % PageSize == 0);
    NN_KERN_ASSERT(size);
    KProcessAddress ioAddressBegin = GetMapRegionBegin(KMemoryState_Io);
    KProcessAddress ioAddressEnd = ioAddressBegin + GetMapRegionSize(KMemoryState_Io);
    size_t numPages = size / PageSize;

    size_t ioRegionNumPages = (ioAddressEnd - ioAddressBegin) / PageSize;

    if (physAddr > physAddr + size)
    {
        return nn::svc::ResultInvalidAddress();
    }

    if (!((GetAsInteger(physAddr) + size  - 1 < NN_KERN_P_ADDR_MAIN_MEMORY) || (NN_KERN_P_ADDR_MAIN_MEMORY_END - 1 < GetAsInteger(physAddr))))
    {
        return nn::svc::ResultInvalidAddress();
    }

    if (!KSystemControl::IsIoMappablePhysicalAddress(physAddr, size))
    {
        return nn::svc::ResultInvalidAddress();
    }

    KScopedLightLock locker(&m_Mutex);

    size_t physSizeAlign = (size & (-size));
    size_t physAddrAlign = ((GetAsInteger(physAddr)) & (-GetAsInteger(physAddr)));
    size_t physAlign     = ((physAddrAlign > physSizeAlign || physAddrAlign == 0)? physSizeAlign: physAddrAlign);
    if (physAlign > MapPhysicalMaxAlign)
    {
        physAlign = MapPhysicalMaxAlign;
    }

    KProcessAddress addr = Null<KProcessAddress>();
    for (int szc = GetPageSizeCountMax(); szc >= 0; szc--)
    {
        size_t align = GetPageSize(szc);
        if (align > physAlign)
        {
            continue;
        }
        addr = FindFreeAreaInRegion(ioAddressBegin, ioRegionNumPages, numPages, align, 0, GetGuardSize() / PageSize);
        if (addr != Null<KProcessAddress>())
        {
            break;
        }
    }
    if (addr == Null<KProcessAddress>())
    {
        return nn::svc::ResultOutOfMemory();
    }
    NN_KERN_ASSERT(IsInRange(addr, numPages * PageSize, KMemoryState_Io));

    Result result = ResultSuccess();
    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    NN_KERN_ASSERT(CheckMemoryState(addr, size, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0).IsSuccess());

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { permission, true, false };
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, physAddr, true, property, PageTableOperation_Map);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, KMemoryState_Io, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}

Result KPageTableBase::MapStatic(KPhysicalAddress physAddr, size_t size, KMemoryPermission permission)
{
    NN_KERN_ASSERT(physAddr % PageSize == 0);
    NN_KERN_ASSERT(size % PageSize == 0);
    NN_KERN_ASSERT(size);
    KProcessAddress staticAddressBegin = GetMapRegionBegin(KMemoryState_Static);
    KProcessAddress staticAddressEnd = staticAddressBegin + GetMapRegionSize(KMemoryState_Static);
    size_t numPages = size / PageSize;
    size_t staticRegionNumPages = (staticAddressEnd - staticAddressBegin) / PageSize;

    if (physAddr > physAddr + size)
    {
        return nn::svc::ResultInvalidAddress();
    }

    if (!((physAddr + size  - 1 < KMemoryLayout::GetKernelRegionPhysicalBegin()) || (KMemoryLayout::GetKernelRegionPhysicalEnd() - 1 < physAddr)))
    {
        return nn::svc::ResultInvalidAddress();
    }

    if (!((physAddr + size  - 1 < KMemoryLayout::GetHeapRegionPhysicalBegin())   || (KMemoryLayout::GetHeapRegionPhysicalEnd()   - 1 < physAddr)))
    {
        return nn::svc::ResultInvalidAddress();
    }

    if (!KSystemControl::IsStaticMappablePhysicalAddress(physAddr, size))
    {
        return nn::svc::ResultInvalidAddress();
    }

    KScopedLightLock locker(&m_Mutex);

    size_t physSizeAlign = (size & (-size));
    size_t physAddrAlign = ((GetAsInteger(physAddr)) & (-GetAsInteger(physAddr)));
    size_t physAlign     = ((physAddrAlign > physSizeAlign || physAddrAlign == 0)? physSizeAlign: physAddrAlign);
    if (physAlign > MapPhysicalMaxAlign)
    {
        physAlign = MapPhysicalMaxAlign;
    }

    KProcessAddress addr = Null<KProcessAddress>();
    for (int szc = GetPageSizeCountMax(); szc >= 0; szc--)
    {
        size_t align = GetPageSize(szc);
        if (align > physAlign)
        {
            continue;
        }
        addr = FindFreeAreaInRegion(staticAddressBegin, staticRegionNumPages, numPages, align, 0, GetGuardSize() / PageSize);
        if (addr != Null<KProcessAddress>())
        {
            break;
        }
    }
    if (addr == Null<KProcessAddress>())
    {
        return nn::svc::ResultOutOfMemory();
    }
    NN_KERN_ASSERT(IsInRange(addr, numPages * PageSize, KMemoryState_Static));

    Result result = ResultSuccess();
    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { permission, false, false };
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, physAddr, true, property, PageTableOperation_Map);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, KMemoryState_Static, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}

Result KPageTableBase::QueryInfo(KMemoryInfo* pBlock, nn::svc::PageInfo* pPage, KProcessAddress addr) const
{
    if (!IsInRange(addr, 1))
    {
        pBlock->baseAddress = GetAsInteger(m_AddressEnd);
        pBlock->size        = 0 - GetAsInteger(m_AddressEnd);
        pBlock->state       = static_cast<KMemoryState>(nn::svc::MemoryState_Inaccessible);
        pBlock->permission  = KMemoryPermission_None;
        pBlock->attribute   = KMemoryAttribute_None;
        pBlock->ipcLockCount = 0;
        pBlock->deviceSharedCount = 0;
        pPage->flags = 0;
        return ResultSuccess();
    }

    KScopedLightLock locker(&m_Mutex);
    return QueryInfoImpl(pBlock, pPage, addr);
}

Result KPageTableBase::QueryInfoImpl(KMemoryInfo* pBlock, nn::svc::PageInfo* pPage, KProcessAddress addr) const
{
    NN_KERN_ASSERT(m_Mutex.IsLockedByMe());
    const KMemoryBlock* pMemory = m_MemoryBlocks.FindByAddress(addr);

    if(pBlock && pMemory)
    {
        *pBlock = pMemory->GetMemoryInfo();
        pPage->flags = 0;
        return ResultSuccess();
    }
    else
    {
        NN_WARNING(false, "invalid address = %p, begin = %p, end = %p",
                     addr, GetAsInteger(m_AddressBegin), GetAsInteger(m_AddressEnd));
        return nn::svc::ResultInvalidCurrentMemory();
    }
}

Result  KPageTableBase::CheckMemoryState(const KMemoryInfo& mi, Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute) const
{
    if ((mi.state & stateMask) != state)
    {
        DUMP_MEM_BLOCK();
        //NN_LOG("%s:%d %p %x %x %x\n",__func__,__LINE__, mi.baseAddress, mi.state, stateMask, state);
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if ((mi.permission & permissionMask) != permission)
    {
        DUMP_MEM_BLOCK();
        //NN_LOG("%s:%d %p %x %x %x\n",__func__,__LINE__, mi.baseAddress, mi.permission, permissionMask, permission);
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if ((mi.attribute & attributeMask) != attribute)
    {
        DUMP_MEM_BLOCK();
        //NN_LOG("%s:%d %p %x %x %x\n",__func__,__LINE__, mi.baseAddress, mi.attribute, attributeMask, attribute);
        return nn::svc::ResultInvalidCurrentMemory();
    }

    return ResultSuccess();
}

Result KPageTableBase::CheckMemoryState(
        KMemoryState* pState, KMemoryPermission* pPermission, KMemoryAttribute* pAttribute,
        KProcessAddress addr, size_t size,
        Bit32 stateMask, Bit32 state,
        Bit32 permissionMask, Bit32 permission,
        Bit32 attributeMask, Bit32 attribute, Bit32 ignoreMask) const
{
    NN_KERN_ASSERT(m_Mutex.IsLockedByMe());

    KMemoryInfo mi;
    KProcessAddress lastAddr = addr + size - 1;

    KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(addr);
    mi = it->GetMemoryInfo();

    KMemoryState firstState = mi.state;
    KMemoryPermission firstPermission = mi.permission;
    KMemoryAttribute firstAttribute = mi.attribute;
    for (;;)
    {
        // 連続性チェック
        if (firstState != mi.state)
        {
            DUMP_MEM_BLOCK();
            //NN_LOG("%s:%d %p %p %p\n",__func__,__LINE__, addr, size, mi.baseAddress);
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if (firstPermission != mi.permission)
        {
            DUMP_MEM_BLOCK();
            //NN_LOG("%s:%d %p %p %p\n",__func__,__LINE__, addr, size, mi.baseAddress);
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if ((firstAttribute | ignoreMask) != (mi.attribute | ignoreMask))
        {
            DUMP_MEM_BLOCK();
            //NN_LOG("%s:%d %p %p %p\n",__func__,__LINE__, addr, size, mi.baseAddress);
            return nn::svc::ResultInvalidCurrentMemory();
        }

        Result result = CheckMemoryState(mi, stateMask, state, permissionMask, permission, attributeMask, attribute);
        if (result.IsFailure())
        {
            return result;
        }

        if (lastAddr <= (mi.baseAddress + mi.size - 1))
        {
            break;
        }
        it++;
        NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
        mi = it->GetMemoryInfo();
    }
    if (pState)
    {
        *pState = firstState;
    }
    if (pPermission)
    {
        *pPermission = firstPermission;
    }
    if (pAttribute)
    {
        *pAttribute = static_cast<KMemoryAttribute>(firstAttribute & ~ignoreMask);
    }

    return ResultSuccess();
}

Result KPageTableBase::CheckMemoryStateContiguous(
        KProcessAddress addr, size_t size,
        Bit32 stateMask, Bit32 state,
        Bit32 permissionMask, Bit32 permission,
        Bit32 attributeMask, Bit32 attribute) const
{
    NN_KERN_ASSERT(m_Mutex.IsLockedByMe());

    KMemoryInfo mi;
    KProcessAddress lastAddr = addr + size - 1;

    KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(addr);
    mi = it->GetMemoryInfo();

    for (;;)
    {
        Result result = CheckMemoryState(mi, stateMask, state, permissionMask, permission, attributeMask, attribute);
        if (result.IsFailure())
        {
            return result;
        }

        if (lastAddr <= (mi.baseAddress + mi.size - 1))
        {
            break;
        }
        it++;
        NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
        mi = it->GetMemoryInfo();
    }

    return ResultSuccess();
}


Result KPageTableBase::InvalidateProcessDataCache(KProcessAddress addr, size_t size)
{
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    if (CheckMemoryStateContiguous(addr, size,
                KMemoryState_FlagsMemory, KMemoryState_FlagsMemory,
                KMemoryPermission_UserReadWrite, KMemoryPermission_UserReadWrite,
                KMemoryAttribute_Uncached, 0).IsFailure())
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    TraverseContext ctx;
    TraverseData data;
    KPhysicalAddress curAddr;
    size_t curSize;
    size_t sumSize;
    bool dataIsValid;

    dataIsValid = m_Table.TraverseBegin(&data, &ctx, addr);
    if (!dataIsValid)
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    curAddr = data.address;
    curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
    sumSize = curSize;

    while (sumSize < size)
    {
        dataIsValid = m_Table.TraverseNext(&data, &ctx);

        if (!dataIsValid)
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if (data.address != (curAddr + curSize))
        {
            if (IsLinearMapPhysicalAddress(curAddr))
            {
                if (curSize > 0)
                {
                    KCPU::InvalidateDataCache(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize);
                }
            }
            else
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }

            curAddr = data.address;
            curSize = data.size;
        }
        else
        {
            curSize += data.size;
        }
        sumSize += data.size;
    }

    if (sumSize > size)
    {
        curSize -= sumSize - size;
    }
    if (IsLinearMapPhysicalAddress(curAddr))
    {
        if (curSize > 0)
        {
            KCPU::InvalidateDataCache(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize);
        }
    }
    else
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    return ResultSuccess();
}

Result KPageTableBase::ReadDebugMemory(void* buf, KProcessAddress addr, size_t size)
{
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    if (CheckMemoryStateContiguous(addr, size, 0, 0, KMemoryPermission_UserRead, KMemoryPermission_UserRead, 0, 0).IsFailure())
    {
        if (CheckMemoryStateContiguous(addr, size, KMemoryState_FlagsDebug, KMemoryState_FlagsDebug, 0, 0, 0, 0).IsFailure())
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
    }

    TraverseContext ctx;
    TraverseData data;
    KPhysicalAddress curAddr;
    size_t curSize;
    size_t sumSize;
    bool dataIsValid;

    dataIsValid = m_Table.TraverseBegin(&data, &ctx, addr);
    if (!dataIsValid)
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    curAddr = data.address;
    curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
    sumSize = curSize;

    while (sumSize < size)
    {
        dataIsValid = m_Table.TraverseNext(&data, &ctx);

        if (!dataIsValid)
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if (data.address != (curAddr + curSize))
        {
            if (IsLinearMapPhysicalAddress(curAddr))
            {
                if (curSize >= 4)
                {
                    size_t readSize = RoundDown(curSize, 4);
                    if (!nn::kern::CopyMemoryToUserAlign4(buf, GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), readSize))
                    {
                        return nn::svc::ResultInvalidPointer();
                    }
                    buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + readSize);
                    curAddr += readSize;
                    curSize -= readSize;
                }
                if (curSize > 0)
                {
                    if (!nn::kern::CopyMemoryToUser(buf, GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize))
                    {
                        return nn::svc::ResultInvalidPointer();
                    }
                    buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + curSize);
                }
            }
            else
            {
                if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
                if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                if (!nn::kern::CopyMemoryToUser(buf, GetUntypedPointer(procAddr), curSize))
                {
                    return nn::svc::ResultInvalidPointer();
                }
                buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + curSize);
            }

            curAddr = data.address;
            curSize = data.size;
        }
        else
        {
            curSize += data.size;
        }
        sumSize += data.size;
    }

    if (sumSize > size)
    {
        curSize -= sumSize - size;
    }
    if (IsLinearMapPhysicalAddress(curAddr))
    {
        if (curSize >= 4)
        {
            size_t readSize = RoundDown(curSize, 4);
            if (!nn::kern::CopyMemoryToUserAlign4(buf, GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), readSize))
            {
                return nn::svc::ResultInvalidPointer();
            }
            buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + readSize);
            curAddr += readSize;
            curSize -= readSize;
        }
        if (curSize > 0)
        {
            if (!nn::kern::CopyMemoryToUser(buf, GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize))
            {
                return nn::svc::ResultInvalidPointer();
            }
        }
    }
    else
    {
        if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
        if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if (!nn::kern::CopyMemoryToUser(buf, GetUntypedPointer(procAddr), curSize))
        {
            return nn::svc::ResultInvalidPointer();
        }
        buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + curSize);
    }

    return ResultSuccess();
}

Result KPageTableBase::WriteDebugMemory(KProcessAddress addr, const void* buf, size_t size)
{
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    if (CheckMemoryStateContiguous(addr, size, 0, 0, KMemoryPermission_UserReadWrite, KMemoryPermission_UserReadWrite, 0, 0).IsFailure())
    {
        if (CheckMemoryStateContiguous(addr, size, KMemoryState_FlagsDebug, KMemoryState_FlagsDebug, 0, 0, 0, 0).IsFailure())
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
    }

    TraverseContext ctx;
    TraverseData data;
    KPhysicalAddress curAddr;
    size_t curSize;
    size_t sumSize;
    bool dataIsValid;

    dataIsValid = m_Table.TraverseBegin(&data, &ctx, addr);
    if (!dataIsValid)
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    curAddr = data.address;
    curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
    sumSize = curSize;

    while (sumSize < size)
    {
        dataIsValid = m_Table.TraverseNext(&data, &ctx);

        if (!dataIsValid)
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if (data.address != (curAddr + curSize))
        {
            if (!IsLinearMapPhysicalAddress(curAddr))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }

            if (curSize >= 4)
            {
                size_t writeSize = RoundDown(curSize, 4);
                if (!nn::kern::CopyMemoryFromUserAlign4(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), buf, writeSize))
                {
                    return nn::svc::ResultInvalidPointer();
                }
                KCPU::StoreDataCache(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), writeSize);
                buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + writeSize);
                curAddr += writeSize;
                curSize -= writeSize;
            }
            if (curSize > 0)
            {
                if (!nn::kern::CopyMemoryFromUser(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), buf, curSize))
                {
                    return nn::svc::ResultInvalidPointer();
                }
                KCPU::StoreDataCache(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize);
                buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + curSize);
            }
            curAddr = data.address;
            curSize = data.size;
        }
        else
        {
            curSize += data.size;
        }
        sumSize += data.size;
    }

    if (sumSize > size)
    {
        curSize -= sumSize - size;
    }
    if (!IsLinearMapPhysicalAddress(curAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if (curSize >= 4)
    {
        size_t writeSize = RoundDown(curSize, 4);
        if (!nn::kern::CopyMemoryFromUserAlign4(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), buf, writeSize))
        {
            return nn::svc::ResultInvalidPointer();
        }
        KCPU::StoreDataCache(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), writeSize);
        buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + writeSize);
        curAddr += writeSize;
        curSize -= writeSize;
    }
    if (curSize > 0)
    {
        if (!nn::kern::CopyMemoryFromUser(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), buf, curSize))
        {
            return nn::svc::ResultInvalidPointer();
        }
        KCPU::StoreDataCache(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize);
    }
    KCPU::InvalidateEntireInstructionCache();

    return ResultSuccess();
}

// Current Address Spaceの非特領域にコピーする
//   fromのPagetableがthis
Result KPageTableBase::CopyMemoryToUser(KProcessAddress toAddr, size_t size,
        KProcessAddress fromAddr,
        Bit32 fromStateMask, Bit32 fromState, KMemoryPermission fromTestPermission,
        Bit32 fromAttributeMask, Bit32 fromAttribute)
{
    if (!IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    {
        KScopedLightLock locker(&m_Mutex);

        Result result = CheckMemoryStateContiguous(fromAddr, size, fromStateMask, fromState, fromTestPermission, fromTestPermission, fromAttributeMask | KMemoryAttribute_Uncached, fromAttribute);
        if (result.IsFailure())
        {
            return result;
        }

        TraverseContext ctx;
        TraverseData data;
        KPhysicalAddress curAddr;
        size_t curSize;
        size_t sumSize;
        bool dataIsValid;

        dataIsValid = m_Table.TraverseBegin(&data, &ctx, fromAddr);
        NN_KERN_ABORT_UNLESS(dataIsValid);

        curAddr = data.address;
        curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
        sumSize = curSize;

        while (sumSize < size)
        {
            dataIsValid = m_Table.TraverseNext(&data, &ctx);
            NN_KERN_ASSERT(dataIsValid);

            if (data.address != (curAddr + curSize))
            {
                if (IsLinearMapPhysicalAddress(curAddr))
                {
                    if (curSize >= 4)
                    {
                        size_t copySize = (curSize & ~3ul);
                        if (!nn::kern::CopyMemoryToUserAlign4(GetUntypedPointer(toAddr), GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), copySize))
                        {
                            return nn::svc::ResultInvalidCurrentMemory();
                        }
                        toAddr += copySize;
                        curAddr += copySize;
                        curSize -= copySize;
                    }
                    if (curSize && !nn::kern::CopyMemoryToUser(GetUntypedPointer(toAddr), GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                }
                else if (curSize)
                {
                    if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
                    if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    if (!nn::kern::CopyMemoryToUser(GetUntypedPointer(toAddr), GetUntypedPointer(procAddr), curSize))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                }

                toAddr += curSize;
                curAddr = data.address;
                curSize = data.size;
            }
            else
            {
                curSize += data.size;
            }
            sumSize += data.size;
        }

        if (sumSize > size)
        {
            curSize -= sumSize - size;
        }

        if (IsLinearMapPhysicalAddress(curAddr))
        {
            if (curSize >= 4)
            {
                size_t copySize = (curSize & ~3ul);
                if (!nn::kern::CopyMemoryToUserAlign4(GetUntypedPointer(toAddr), GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), copySize))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                toAddr += copySize;
                curAddr += copySize;
                curSize -= copySize;
            }
            if (curSize && !nn::kern::CopyMemoryToUser(GetUntypedPointer(toAddr), GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
        }
        else if (curSize)
        {
            if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
            KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
            if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
            if (!nn::kern::CopyMemoryToUser(GetUntypedPointer(toAddr), GetUntypedPointer(procAddr), curSize))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
        }
    }

    return ResultSuccess();
}

// Current Address Spaceの特権領域にコピーする
//   fromのPagetableがthis
Result KPageTableBase::CopyMemoryToKernel(KProcessAddress toAddr, size_t size,
        KProcessAddress fromAddr,
        Bit32 fromStateMask, Bit32 fromState, KMemoryPermission fromTestPermission,
        Bit32 fromAttributeMask, Bit32 fromAttribute)
{
    if (!IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    {
        KScopedLightLock locker(&m_Mutex);

        Result result = CheckMemoryStateContiguous(fromAddr, size, fromStateMask, fromState, fromTestPermission, fromTestPermission, fromAttributeMask | KMemoryAttribute_Uncached, fromAttribute);
        if (result.IsFailure())
        {
            return result;
        }

        TraverseContext ctx;
        TraverseData data;
        KPhysicalAddress curAddr;
        size_t curSize;
        size_t sumSize;
        bool dataIsValid;

        dataIsValid = m_Table.TraverseBegin(&data, &ctx, fromAddr);
        NN_KERN_ABORT_UNLESS(dataIsValid);

        curAddr = data.address;
        curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
        sumSize = curSize;

        while (sumSize < size)
        {
            dataIsValid = m_Table.TraverseNext(&data, &ctx);
            NN_KERN_ASSERT(dataIsValid);

            if (data.address != (curAddr + curSize))
            {
                if (IsLinearMapPhysicalAddress(curAddr))
                {
                    std::memcpy(GetUntypedPointer(toAddr), GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize);
                }
                else if (curSize)
                {
                    if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
                    if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    std::memcpy(GetUntypedPointer(toAddr), GetUntypedPointer(procAddr), curSize);
                }

                toAddr += curSize;
                curAddr = data.address;
                curSize = data.size;
            }
            else
            {
                curSize += data.size;
            }
            sumSize += data.size;
        }

        if (sumSize > size)
        {
            curSize -= sumSize - size;
        }

        if (curSize)
        {
            if (IsLinearMapPhysicalAddress(curAddr))
            {
                std::memcpy(GetUntypedPointer(toAddr), GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), curSize);
            }
            else
            {
                if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
                if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                std::memcpy(GetUntypedPointer(toAddr), GetUntypedPointer(procAddr), curSize);
            }
        }
    }

    return ResultSuccess();
}


// Current Address Spaceの非特領域からコピーする
//   toのPagetableがthis
Result KPageTableBase::CopyMemoryFromUser(KProcessAddress toAddr, size_t size,
        Bit32 toStateMask, Bit32 toState, KMemoryPermission toTestPermission,
        Bit32 toAttributeMask, Bit32 toAttribute,
        KProcessAddress fromAddr)
{
    if (!IsInRange(toAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    {
        KScopedLightLock locker(&m_Mutex);

        Result result = CheckMemoryStateContiguous(toAddr, size, toStateMask, toState, toTestPermission, toTestPermission, toAttributeMask | KMemoryAttribute_Uncached, toAttribute);
        if (result.IsFailure())
        {
            return result;
        }

        TraverseContext ctx;
        TraverseData data;
        KPhysicalAddress curAddr;
        size_t curSize;
        size_t sumSize;
        bool dataIsValid;

        dataIsValid = m_Table.TraverseBegin(&data, &ctx, toAddr);
        NN_KERN_ABORT_UNLESS(dataIsValid);

        curAddr = data.address;
        curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
        sumSize = curSize;

        while (sumSize < size)
        {
            dataIsValid = m_Table.TraverseNext(&data, &ctx);
            NN_KERN_ASSERT(dataIsValid);

            if (data.address != (curAddr + curSize))
            {
                if (IsLinearMapPhysicalAddress(curAddr))
                {
                    if (curSize >= 4)
                    {
                        size_t copySize = (curSize & ~3ul);
                        if (!nn::kern::CopyMemoryFromUserAlign4(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), GetUntypedPointer(fromAddr), copySize))
                        {
                            return nn::svc::ResultInvalidCurrentMemory();
                        }
                        fromAddr += copySize;
                        curAddr += copySize;
                        curSize -= copySize;
                    }
                    if (curSize && !nn::kern::CopyMemoryFromUser(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), GetUntypedPointer(fromAddr), curSize))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                }
                else if (curSize)
                {
                    if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
                    if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    if (!nn::kern::CopyMemoryFromUser(GetUntypedPointer(procAddr), GetUntypedPointer(fromAddr), curSize))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                }

                fromAddr += curSize;
                curAddr = data.address;
                curSize = data.size;
            }
            else
            {
                curSize += data.size;
            }
            sumSize += data.size;
        }

        if (sumSize > size)
        {
            curSize -= sumSize - size;
        }

        if (IsLinearMapPhysicalAddress(curAddr))
        {
            if (curSize >= 4)
            {
                size_t copySize = (curSize & ~3ul);
                if (!nn::kern::CopyMemoryFromUserAlign4(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), GetUntypedPointer(fromAddr), copySize))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                fromAddr += copySize;
                curAddr += copySize;
                curSize -= copySize;
            }
            if (curSize && !nn::kern::CopyMemoryFromUser(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), GetUntypedPointer(fromAddr), curSize))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
        }
        else if (curSize)
        {
            if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
            KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
            if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
            if (!nn::kern::CopyMemoryFromUser(GetUntypedPointer(procAddr), GetUntypedPointer(fromAddr), curSize))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
        }
    }

    return ResultSuccess();
}

// Current Address Spaceの特権領域からコピーする
//   toのPagetableがthis
Result KPageTableBase::CopyMemoryFromKernel(KProcessAddress toAddr, size_t size,
        Bit32 toStateMask, Bit32 toState, KMemoryPermission toTestPermission,
        Bit32 toAttributeMask, Bit32 toAttribute,
        KProcessAddress fromAddr)
{
    if (!IsInRange(toAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    {
        KScopedLightLock locker(&m_Mutex);

        Result result = CheckMemoryStateContiguous(toAddr, size, toStateMask, toState, toTestPermission, toTestPermission, toAttributeMask | KMemoryAttribute_Uncached, toAttribute);
        if (result.IsFailure())
        {
            return result;
        }

        TraverseContext ctx;
        TraverseData data;
        KPhysicalAddress curAddr;
        size_t curSize;
        size_t sumSize;
        bool dataIsValid;

        dataIsValid = m_Table.TraverseBegin(&data, &ctx, toAddr);
        NN_KERN_ABORT_UNLESS(dataIsValid);

        curAddr = data.address;
        curSize = data.size - (GetAsInteger(data.address) & (data.size - 1));
        sumSize = curSize;

        while (sumSize < size)
        {
            dataIsValid = m_Table.TraverseNext(&data, &ctx);
            NN_KERN_ASSERT(dataIsValid);

            if (data.address != (curAddr + curSize))
            {
                if (IsLinearMapPhysicalAddress(curAddr))
                {
                    std::memcpy(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), GetUntypedPointer(fromAddr), curSize);
                }
                else if (curSize)
                {
                    if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
                    if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
                    {
                        return nn::svc::ResultInvalidCurrentMemory();
                    }
                    std::memcpy(GetUntypedPointer(procAddr), GetUntypedPointer(fromAddr), curSize);
                }

                fromAddr += curSize;
                curAddr = data.address;
                curSize = data.size;
            }
            else
            {
                curSize += data.size;
            }
            sumSize += data.size;
        }

        if (sumSize > size)
        {
            curSize -= sumSize - size;
        }

        if (curSize)
        {
            if (IsLinearMapPhysicalAddress(curAddr))
            {
                std::memcpy(GetUntypedPointer(GetLinearMapVirtualAddress(curAddr)), GetUntypedPointer(fromAddr), curSize);
            }
            else
            {
                if (!(KMemoryLayout::InSlabRegionPhysical(curAddr) && KMemoryLayout::InSlabRegionPhysical(curAddr + curSize - 1)))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                KProcessAddress procAddr = (KMemoryLayout::GetSlabRegionBegin() + (curAddr - KMemoryLayout::GetSlabRegionPhysicalBegin()));
                if (!(KPageBuffer::GetSlabAddr() <= GetAsInteger(procAddr) && GetAsInteger(procAddr) + curSize - 1 <= KPageBuffer::GetSlabAddr() + KPageBuffer::GetObjSize() * KPageBuffer::GetSlabSize() - 1))
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
                std::memcpy(GetUntypedPointer(procAddr), GetUntypedPointer(fromAddr), curSize);
            }
        }
    }

    return ResultSuccess();
}

Result KPageTableBase::MakePageGroupForIpcClient(KPageGroup* pOut, KProcessAddress fromAddr, size_t size, bool isSend, Bit32 allocateOption) const
{
    NN_KERN_ASSERT(m_Mutex.IsLockedByMe());
    Result result = ResultSuccess();

    KProcessAddress fromAddrBeginAligned = RoundDown(fromAddr, PageSize);
    KProcessAddress fromAddrMappingBegin = RoundUp(fromAddr, PageSize);
    KProcessAddress fromAddrMappingEnd   = RoundDown(fromAddr + size, PageSize);
    KProcessAddress fromAddrEndAligned   = RoundUp(fromAddr + size, PageSize);

    KVirtualAddress fromBeginFragment = Null<KVirtualAddress>();
    KVirtualAddress fromEndFragment = Null<KVirtualAddress>();
    const FillMemoryPattern fillPattern = m_IpcFillMemoryPattern;
#ifdef NN_KERN_CHECK_IPC_BOUNDARY
    if (size < PageSize)
    {
        fromBeginFragment = Kernel::GetKernelHeapManager().AllocateContinuous(1, 0, allocateOption);
        if (fromBeginFragment == Null<KVirtualAddress>())
        {
            result = nn::svc::ResultOutOfMemory();
            goto error_exit;
        }
        KPhysicalAddress pa;
        if (!GetPhysicalAddress(&pa, fromAddrBeginAligned))
        {
            result = nn::svc::ResultInvalidCurrentMemory();
            goto error_exit;
        }

        if (isSend)
        {
            size_t offset = fromAddr - fromAddrBeginAligned;
            std::memset(GetUntypedPointer(fromBeginFragment), fillPattern, PageSize - size);
            std::memcpy(GetUntypedPointer(fromBeginFragment + PageSize - size), GetUntypedPointer(GetHeapVirtualAddress(pa) + offset), size);
        }
        else
        {
            std::memset(GetUntypedPointer(fromBeginFragment), fillPattern, PageSize);
        }

        result = pOut->AddBlock(fromBeginFragment, 1);
        if (result.IsFailure())
        {
            result = nn::svc::ResultOutOfResource();
            goto error_exit;
        }

        return result;
    }
#endif

    if (fromAddrBeginAligned < fromAddrMappingBegin)
    {
        fromBeginFragment = Kernel::GetKernelHeapManager().AllocateContinuous(1, 0, allocateOption);
        if (fromBeginFragment == Null<KVirtualAddress>())
        {
            result = nn::svc::ResultOutOfMemory();
            goto error_exit;
        }
        KPhysicalAddress pa;
        if (!GetPhysicalAddress(&pa, fromAddrBeginAligned))
        {
            result = nn::svc::ResultInvalidCurrentMemory();
            goto error_exit;
        }

        if (isSend)
        {
            size_t offset = fromAddr - fromAddrBeginAligned;
            size_t copySize;
            size_t clearSize;
            if ((fromAddr + size) < fromAddrMappingBegin)
            {
                copySize = size;
                clearSize = fromAddrMappingBegin - (fromAddr + size);
            }
            else
            {
                copySize = fromAddrMappingBegin - fromAddr;
                clearSize = 0;
            }
            std::memset(GetUntypedPointer(fromBeginFragment), fillPattern, offset);
            std::memcpy(GetUntypedPointer(fromBeginFragment + offset), GetUntypedPointer(GetHeapVirtualAddress(pa) + offset), copySize);
            if (clearSize)
            {
                std::memset(GetUntypedPointer(fromBeginFragment + offset + copySize), fillPattern, clearSize);
            }
        }
        else
        {
            std::memset(GetUntypedPointer(fromBeginFragment), fillPattern, PageSize);
        }

        result = pOut->AddBlock(fromBeginFragment, 1);
        if (result.IsFailure())
        {
            result = nn::svc::ResultOutOfResource();
            goto error_exit;
        }
    }

    if (fromAddrMappingBegin < fromAddrMappingEnd)
    {
        result = MakePageGroupImpl(pOut, fromAddrMappingBegin, (fromAddrMappingEnd - fromAddrMappingBegin) / PageSize);
        if (result.IsFailure())
        {
            goto error_exit;
        }
    }

    if (fromAddrMappingEnd < fromAddrEndAligned && (fromAddrBeginAligned < fromAddrMappingEnd || fromAddrBeginAligned == fromAddrMappingBegin))
    {
        fromEndFragment = Kernel::GetKernelHeapManager().AllocateContinuous(1, 0, allocateOption);
        if (fromEndFragment == Null<KVirtualAddress>())
        {
            result = nn::svc::ResultOutOfMemory();
            goto error_exit;
        }
        KPhysicalAddress pa;
        if (!GetPhysicalAddress(&pa, fromAddrMappingEnd))
        {
            result = nn::svc::ResultInvalidCurrentMemory();
            goto error_exit;
        }

        if (isSend)
        {
            size_t copySize = (fromAddr + size) - fromAddrMappingEnd;
            size_t clearSize = PageSize - copySize;

            std::memcpy(GetUntypedPointer(fromEndFragment), GetUntypedPointer(GetHeapVirtualAddress(pa)), copySize);
            std::memset(GetUntypedPointer(fromEndFragment + copySize), fillPattern, clearSize);
        }
        else
        {
            std::memset(GetUntypedPointer(fromEndFragment), fillPattern, PageSize);
        }

        result = pOut->AddBlock(fromEndFragment, 1);
        if (result.IsFailure())
        {
            result = nn::svc::ResultOutOfResource();
            goto error_exit;
        }
    }

error_exit:
    if (result.IsFailure())
    {
        if (fromBeginFragment != Null<KVirtualAddress>())
        {
            Kernel::GetKernelHeapManager().Open(fromBeginFragment, 1);
            Kernel::GetKernelHeapManager().Close(fromBeginFragment, 1);
        }
        if (fromEndFragment != Null<KVirtualAddress>())
        {
            Kernel::GetKernelHeapManager().Open(fromEndFragment, 1);
            Kernel::GetKernelHeapManager().Close(fromEndFragment, 1);
        }
    }

    return result;
}

Result KPageTableBase::SetupForIpcClient(KPageGroup* pOut,
        KProcessAddress fromAddr, size_t size, KMemoryPermission testPermission, KMemoryState toState, bool isSend, Bit32 allocateOption)
{
    NN_KERN_ASSERT(testPermission == KMemoryPermission_UserReadWrite || testPermission == KMemoryPermission_UserRead);
    NN_KERN_ASSERT(pOut->GetTotalNumPages() == 0);
    Result result = ResultSuccess();

    if (!IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcessAddress fromAddrBeginAligned = RoundDown(fromAddr, PageSize);
    KProcessAddress fromAddrEndAligned   = RoundUp(fromAddr + size, PageSize);
    KProcessAddress fromAddrMappingBegin = RoundUp(fromAddr, PageSize);
    KProcessAddress fromAddrMappingEnd   = RoundDown(fromAddr + size, PageSize);

    {
        KMemoryPermission fromPermission = ((testPermission == KMemoryPermission_UserReadWrite)? KMemoryPermission_KernelReadWrite: KMemoryPermission_UserRead);

        Bit32 testState = 0;
        Bit32 testAttributeMask = 0;
        switch (toState)
        {
        case KMemoryState_Ipc:
            {
                testState = KMemoryState_FlagsIpc;
                testAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked | KMemoryAttribute_DeviceShared;
            }
            break;
        case KMemoryState_NonSecureIpc:
            {
                testState = KMemoryState_FlagsNonSecureIpc;
                testAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked;
            }
            break;
        case KMemoryState_NonDeviceIpc:
            {
                testState = KMemoryState_FlagsNonDeviceIpc;
                testAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked;
            }
            break;
        default:
            return nn::svc::ResultInvalidCombination();
        }

        KMemoryInfo mi;
        KProcessAddress lastAddr = fromAddrEndAligned - 1;

        KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }

        KScopedLightLock locker(&m_Mutex);

        KScopedPageTableUpdater updateData(this);
        size_t mappedTotalSize = 0;

        {
            KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(fromAddrBeginAligned);
            for (;;)
            {
                mi = it->GetMemoryInfo();
                result = CheckMemoryState(mi, testState, testState, testPermission, testPermission, testAttributeMask, 0);
                if (result.IsFailure())
                {
                    break;
                }

                if (fromAddrMappingBegin < fromAddrMappingEnd && GetAsInteger(fromAddrMappingBegin) < mi.baseAddress + mi.size && mi.baseAddress < GetAsInteger(fromAddrMappingEnd))
                {
                    KProcessAddress mapAddr;
                    size_t mapSize;
                    if (mi.baseAddress < GetAsInteger(fromAddrMappingBegin))
                    {
                        mapAddr = fromAddrMappingBegin;
                    }
                    else
                    {
                        mapAddr = mi.baseAddress;
                    }
                    if (GetAsInteger(fromAddrMappingEnd - 1) <= (mi.baseAddress + mi.size - 1))
                    {
                        mapSize = fromAddrMappingEnd - mapAddr;
                    }
                    else
                    {
                        mapSize = mi.baseAddress + mi.size - GetAsInteger(mapAddr);
                    }

                    if (mi.ipcLockCount == 0 && (mi.permission & KMemoryPermission_IpcLockChangeMask) != fromPermission)
                    {
                        const KPageProperty property = { fromPermission, false, false };
                        result = OperateImpl(updateData.GetDataPtr(), mapAddr, mapSize / PageSize, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangeProperties);
                        if (result.IsFailure())
                        {
                            break;
                        }
                    }
                    else
                    {
                        NN_KERN_ASSERT((mi.permission & KMemoryPermission_IpcLockChangeMask) == fromPermission);
                    }
                    mappedTotalSize += mapSize;
                }

                if (lastAddr <= (mi.baseAddress + mi.size - 1))
                {
                    break;
                }
                it++;
                NN_KERN_ABORT_UNLESS(it != m_MemoryBlocks.EndIterator());
            }
        }

        if (result.IsSuccess())
        {
            result = MakePageGroupForIpcClient(pOut, fromAddr, size, isSend, allocateOption);
        }

        if (result.IsFailure())
        {
            if (mappedTotalSize != 0)
            {
                KProcessAddress rollbackAddr;
                KProcessAddress rollbackLastAddr = fromAddrMappingBegin + mappedTotalSize - 1;
                KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(fromAddrMappingBegin);
                for (;;)
                {
                    size_t rollbackSize;
                    mi = it->GetMemoryInfo();
                    if (mi.baseAddress < GetAsInteger(fromAddrMappingBegin))
                    {
                        rollbackAddr = fromAddrMappingBegin;
                    }
                    else
                    {
                        rollbackAddr = mi.baseAddress;
                    }
                    if (rollbackLastAddr <= mi.baseAddress + mi.size - 1)
                    {
                        rollbackSize = rollbackLastAddr - rollbackAddr + 1;
                    }
                    else
                    {
                        rollbackSize = mi.baseAddress + mi.size - GetAsInteger(rollbackAddr);
                    }

                    if (mi.ipcLockCount == 0 && (mi.permission & KMemoryPermission_IpcLockChangeMask) != fromPermission)
                    {
                        const KPageProperty property = { mi.permission, false, false };
                        Result resultCheck = OperateImpl(updateData.GetDataPtr(), rollbackAddr, rollbackSize / PageSize, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangeProperties);
                        NN_KERN_ABORT_IF_FAILED(resultCheck);
                    }
                    else
                    {
                        NN_KERN_ASSERT((mi.permission & KMemoryPermission_IpcLockChangeMask) == fromPermission);
                    }

                    if (rollbackLastAddr <= (mi.baseAddress + mi.size - 1))
                    {
                        break;
                    }
                    it++;
                    NN_KERN_ABORT_UNLESS(it != m_MemoryBlocks.EndIterator());
                }
            }

            return result;
        }

        pOut->Open();
        if (mappedTotalSize)
        {
            NN_KERN_ASSERT(mappedTotalSize == static_cast<size_t>(fromAddrMappingEnd - fromAddrMappingBegin));
            m_MemoryBlocks.UpdateLock(&updateBuffer, fromAddrMappingBegin, mappedTotalSize / PageSize, &KMemoryBlock::IpcLock, fromPermission);
        }

        NN_KERN_ASSERT(!(fromAddrMappingBegin < fromAddrMappingEnd) || static_cast<size_t>(fromAddrMappingEnd - fromAddrMappingBegin) == mappedTotalSize);
#ifdef NN_KERN_CHECK_IPC_BOUNDARY
        if (size < PageSize)
        {
            NN_KERN_ASSERT(pOut->GetTotalNumPages() == 1);
        }
        else
#endif
        {
            NN_KERN_ASSERT(PageSize * pOut->GetTotalNumPages() == static_cast<size_t>(fromAddrEndAligned - fromAddrBeginAligned));
        }
    }

    return ResultSuccess();
}

Result KPageTableBase::SetupForIpcServer(KProcessAddress* pToAddr, size_t size,
        KProcessAddress fromAddr, KMemoryPermission testPermission, KMemoryState toState, const KPageGroup& pg)
{
    Result result;
    KProcessAddress regionBegin = GetIpcRegionBegin();
    size_t regionSize = GetIpcRegionSize();

    if (size >= regionSize)
    {
        return nn::svc::ResultOutOfAddressSpace();
    }

    KProcessAddress fromAddrBeginAligned = RoundDown(fromAddr, PageSize);
    KProcessAddress fromAddrEndAligned = RoundUp(fromAddr + size, PageSize);

    KScopedLightLock locker(&m_Mutex);
#ifdef NN_KERN_CHECK_IPC_BOUNDARY
    if (size < PageSize)
    {
        KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }

        KProcessAddress toAddrBeginAligned = FindFreeAreaInRegion(regionBegin, regionSize / PageSize, 1, PageSize, 0, GetGuardSize() / PageSize);
        if (toAddrBeginAligned == Null<KProcessAddress>())
        {
            return nn::svc::ResultOutOfAddressSpace();
        }
        NN_KERN_ASSERT(IsInRange(toAddrBeginAligned, PageSize, toState));

        KScopedPageTableUpdater updateData(this);

        const KPageProperty property = { testPermission, false, false };

        {
            KPageGroup::BlockInfoList::const_iterator it = pg.GetBlockBeginIter();
            NN_KERN_ASSERT(it->GetNumPages() == 1);
            result = OperateImpl(updateData.GetDataPtr(), toAddrBeginAligned, it->GetNumPages(), GetHeapPhysicalAddress(it->GetBlockAddr()), true, property, PageTableOperation_Map);
            NN_KERN_ASSERT(++it == pg.GetBlockEndIter());
        }

        if (result.IsSuccess())
        {
            *pToAddr = toAddrBeginAligned + (PageSize - size);
            m_MemoryBlocks.Update(&updateBuffer, toAddrBeginAligned, 1, toState, property.permission, KMemoryAttribute_None);
        }
    }
    else
#endif
    {
        KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }

        KProcessAddress toAddrBeginAligned = Null<KProcessAddress>();
        for (int szc = GetPageSizeCountMax(); szc >= 0; szc--)
        {
            size_t align = GetPageSize(szc);
            toAddrBeginAligned = FindFreeAreaInRegion(regionBegin, regionSize / PageSize, (fromAddrEndAligned - fromAddrBeginAligned) / PageSize, align, fromAddrBeginAligned & (align - 1), GetGuardSize() / PageSize);
            if (toAddrBeginAligned != Null<KProcessAddress>())
            {
                break;
            }
        }
        if (toAddrBeginAligned == Null<KProcessAddress>())
        {
            return nn::svc::ResultOutOfAddressSpace();
        }
        NN_KERN_ASSERT(IsInRange(toAddrBeginAligned, (fromAddrEndAligned - fromAddrBeginAligned), toState));

        KProcessAddress toAddrEndAligned = toAddrBeginAligned + (fromAddrEndAligned - fromAddrBeginAligned);

        KScopedPageTableUpdater updateData(this);

        const KPageProperty property = { testPermission, false, false };

        {
            KProcessAddress curAddr = toAddrBeginAligned;
            for (KPageGroup::BlockInfoList::const_iterator it = pg.GetBlockBeginIter(); it != pg.GetBlockEndIter(); it++)
            {
                result = OperateImpl(updateData.GetDataPtr(), curAddr, it->GetNumPages(), GetHeapPhysicalAddress(it->GetBlockAddr()), true, property, PageTableOperation_Map);
                if (result.IsFailure())
                {
                    break;
                }
                curAddr += it->GetNumPages() * PageSize;
            }

            if (result.IsFailure())
            {
                const KPageProperty freeProperty = { KMemoryPermission_None, false, false};

                // toAddrMappingBegin - curAddr までを巻き戻す
                if (curAddr - toAddrBeginAligned != 0)
                {
                    Result resultCheck = OperateImpl(
                            updateData.GetDataPtr(), toAddrBeginAligned, (curAddr - toAddrBeginAligned) / PageSize,
                            Null<KPhysicalAddress>(), false, freeProperty, PageTableOperation_Unmap);
                    NN_KERN_ABORT_IF_FAILED(resultCheck);
                }
            }
        }

        if (result.IsSuccess())
        {
            *pToAddr = toAddrBeginAligned + (fromAddr - fromAddrBeginAligned);
            m_MemoryBlocks.Update(&updateBuffer, toAddrBeginAligned, (toAddrEndAligned - toAddrBeginAligned) / PageSize, toState, property.permission, KMemoryAttribute_None);
        }
    }

    return result;
}

Result KPageTableBase::SetupForIpc(KProcessAddress* pToAddr, size_t size,
        KProcessAddress fromAddr, KPageTableBase* pFromTable,
        KMemoryPermission testPermission, KMemoryState toState,
        bool isSend)
{
    Result result = ResultSuccess();

    KScopedPageGroup pg(pFromTable->GetBlockInfoManager());

    result = pFromTable->SetupForIpcClient(&pg.GetPageGroup(), fromAddr, size, testPermission, toState, isSend, GetAllocateOption());
    if (result.IsFailure())
    {
        return result;
    }

    result = SetupForIpcServer(pToAddr, size, fromAddr, testPermission, toState, pg.GetPageGroup());
    if (result.IsFailure())
    {
        pFromTable->CleanupForIpcClient(fromAddr, size, toState);
    }

    pg.GetPageGroup().Close();
    return result;
}

Result KPageTableBase::CleanupForIpcServer(KProcessAddress addr, size_t size, KMemoryState toState)
{
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    KProcessAddress mappingBegin = RoundDown(addr, PageSize);
    KProcessAddress mappingEnd = RoundUp(addr + size, PageSize);

    KScopedLightLock locker(&m_Mutex);

    Result result;

    result = CheckMemoryState(addr, size,
            KMemoryState_All, toState,
            KMemoryPermission_UserRead, KMemoryPermission_UserRead,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    const KPageProperty property = { KMemoryPermission_None, false, false};
    result = OperateImpl(updateData.GetDataPtr(), mappingBegin, (mappingEnd - mappingBegin) / PageSize, Null<KPhysicalAddress>(), false, property, PageTableOperation_Unmap);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, mappingBegin, (mappingEnd - mappingBegin) / PageSize, KMemoryState_Free, property.permission, KMemoryAttribute_None);

    return ResultSuccess();
}

Result KPageTableBase::CleanupForIpcClient(KProcessAddress fromAddr, size_t size, KMemoryState toState)
{
    Result result = ResultSuccess();

    if (!IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcessAddress fromAddrMappingBegin = RoundUp(fromAddr, PageSize);
    KProcessAddress fromAddrMappingEnd = RoundDown(fromAddr + size, PageSize);

    if (fromAddrMappingBegin < fromAddrMappingEnd)
    {
        Bit32 testState = 0;
        Bit32 testAttributeMask = 0;
        switch (toState)
        {
        case KMemoryState_Ipc:
            {
                testState = KMemoryState_FlagsIpc;
                testAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked | KMemoryAttribute_DeviceShared;
            }
            break;
        case KMemoryState_NonSecureIpc:
            {
                testState = KMemoryState_FlagsNonSecureIpc;
                testAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked;
            }
            break;
        case KMemoryState_NonDeviceIpc:
            {
                testState = KMemoryState_FlagsNonDeviceIpc;
                testAttributeMask = KMemoryAttribute_Uncached | KMemoryAttribute_Locked;
            }
            break;
        default:
            return nn::svc::ResultInvalidCombination();
        }

        KMemoryInfo mi;
        KProcessAddress lastAddr = fromAddrMappingEnd - 1;

        KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }

        KScopedLightLock locker(&m_Mutex);

        KScopedPageTableUpdater updateData(this);
        KProcessAddress mapAddr;
        size_t mappedTotalSize = 0;

        {
            KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(fromAddrMappingBegin);
            for (;;)
            {
                size_t mapSize;
                mi = it->GetMemoryInfo();
                if (mi.baseAddress < GetAsInteger(fromAddrMappingBegin))
                {
                    mapAddr = fromAddrMappingBegin;
                }
                else
                {
                    mapAddr = mi.baseAddress;
                }
                if (lastAddr <= (mi.baseAddress + mi.size - 1))
                {
                    mapSize = lastAddr - mapAddr + 1;
                }
                else
                {
                    mapSize = mi.baseAddress + mi.size - GetAsInteger(mapAddr);
                }

                result = CheckMemoryState(mi, testState, testState, 0, 0, testAttributeMask | KMemoryAttribute_IpcLocked, KMemoryAttribute_IpcLocked);
                if (result.IsFailure())
                {
                    break;
                }

                if (mi.ipcLockCount == 1 && mi.permissionOrig != mi.permission)
                {
                    const KPageProperty property = { mi.permissionOrig, false, false };
                    result = OperateImpl(updateData.GetDataPtr(), mapAddr, mapSize / PageSize, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangeProperties);
                    if (result.IsFailure())
                    {
                        break;
                    }
                }
                mappedTotalSize += mapSize;

                if (lastAddr <= (mi.baseAddress + mi.size - 1))
                {
                    break;
                }
                it++;
                NN_KERN_ABORT_UNLESS(it != m_MemoryBlocks.EndIterator());
            }
        }

        if (result.IsFailure())
        {
            if (mappedTotalSize != 0)
            {
                KProcessAddress rollbackAddr;
                KProcessAddress rollbackLastAddr = fromAddrMappingBegin + mappedTotalSize - 1;
                KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(fromAddrMappingBegin);
                for (;;)
                {
                    size_t rollbackSize;
                    mi = it->GetMemoryInfo();
                    if (mi.baseAddress < GetAsInteger(fromAddrMappingBegin))
                    {
                        rollbackAddr = fromAddrMappingBegin;
                    }
                    else
                    {
                        rollbackAddr = mi.baseAddress;
                    }
                    if (rollbackLastAddr <= mi.baseAddress + mi.size - 1)
                    {
                        rollbackSize = rollbackLastAddr - rollbackAddr + 1;
                    }
                    else
                    {
                        rollbackSize = mi.baseAddress + mi.size - GetAsInteger(rollbackAddr);
                    }

                    if (mi.ipcLockCount == 1 && mi.permissionOrig != mi.permission)
                    {
                        const KPageProperty property = { mi.permission, false, false };
                        Result resultCheck = OperateImpl(updateData.GetDataPtr(), rollbackAddr, rollbackSize / PageSize, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangeProperties);
                        NN_KERN_ABORT_IF_FAILED(resultCheck);
                    }

                    if (rollbackLastAddr <= (mi.baseAddress + mi.size - 1))
                    {
                        break;
                    }
                    it++;
                    NN_KERN_ABORT_UNLESS(it != m_MemoryBlocks.EndIterator());
                }
            }

            return result;
        }

        m_MemoryBlocks.UpdateLock(&updateBuffer, fromAddrMappingBegin, (fromAddrMappingEnd - fromAddrMappingBegin) / PageSize, &KMemoryBlock::IpcUnlock, KMemoryPermission_None);
    }

    return ResultSuccess();
}


Result KPageTableBase::LockMemoryAndOpen(KPageGroup* pOut, KProcessAddress addr, size_t size,
        Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute,
        KMemoryPermission newPermission, Bit32 lockAttribute)
{
    size_t numPages = size / PageSize;
    NN_KERN_ASSERT((attribute & lockAttribute) == 0);
    NN_KERN_ASSERT((lockAttribute & (KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared)) == 0);
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);
    if (pOut)
    {
        NN_KERN_ASSERT(pOut->GetTotalNumPages() == 0);
    }
    Result result;

    KMemoryState oldState;
    KMemoryPermission oldPermission;
    KMemoryAttribute oldAttribute;
    result = CheckMemoryState(&oldState, &oldPermission, &oldAttribute,
            addr, size,
            stateMask | KMemoryState_FlagsMemory, state | KMemoryState_FlagsMemory,
            permissionMask, permission,
            attributeMask, attribute);
    if (result.IsFailure())
    {
        return result;
    }

    if (pOut)
    {
        result = MakePageGroupImpl(pOut, addr, size / PageSize);
        if (result.IsFailure())
        {
            return result;
        }
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    newPermission = ((newPermission == KMemoryPermission_None)? oldPermission: newPermission);
    KMemoryAttribute newAttribute = static_cast<KMemoryAttribute>(oldAttribute | lockAttribute);

    if (newPermission != oldPermission)
    {
        KScopedPageTableUpdater updateData(this);

        const KPageProperty property = { newPermission, false, (oldAttribute & KMemoryAttribute_Uncached) != 0};
        result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangeProperties);
        if (result.IsFailure())
        {
            return result;
        }
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, oldState, newPermission, newAttribute);

    if (pOut)
    {
        pOut->Open();
    }

    return result;
}

Result KPageTableBase::UnlockMemory(KProcessAddress addr, size_t size,
            Bit32 stateMask, Bit32 state, Bit32 permissionMask, Bit32 permission, Bit32 attributeMask, Bit32 attribute,
            KMemoryPermission newPermission, Bit32 lockAttribute, const KPageGroup* pPageGroup)
{
    NN_KERN_ASSERT((attributeMask & lockAttribute) == lockAttribute);
    NN_KERN_ASSERT((attribute & lockAttribute) == lockAttribute);
    size_t numPages = size / PageSize;
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);
    Result result;

    KMemoryState oldState;
    KMemoryPermission oldPermission;
    KMemoryAttribute oldAttribute;
    result = CheckMemoryState(&oldState, &oldPermission, &oldAttribute,
            addr, size,
            stateMask | KMemoryState_FlagsMemory, state | KMemoryState_FlagsMemory,
            permissionMask, permission,
            attributeMask, attribute);
    if (result.IsFailure())
    {
        return result;
    }

    if (pPageGroup)
    {
        KScopedPageGroup pg(GetBlockInfoManager());
        result = MakePageGroupImpl(&pg.GetPageGroup(), addr, numPages);
        if (result.IsFailure())
        {
            return result;
        }
        if (!pg.GetPageGroup().IsSamePages(*pPageGroup))
        {
            return nn::svc::ResultInvalidRegion();
        }
    }

    newPermission = ((newPermission == KMemoryPermission_None)? oldPermission: newPermission);
    KMemoryAttribute newAttribute = static_cast<KMemoryAttribute>(oldAttribute & ~lockAttribute);

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    if (newPermission != oldPermission)
    {
        KScopedPageTableUpdater updateData(this);

        const KPageProperty property = { newPermission, false, (oldAttribute & KMemoryAttribute_Uncached) != 0};
        result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_ChangeProperties);
        if (result.IsFailure())
        {
            return result;
        }
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, oldState, newPermission, newAttribute);

    return ResultSuccess();
}

Result KPageTableBase::LockForDeviceAddressSpace(KPageGroup* pOut, KProcessAddress addr, size_t size, KMemoryPermission permission, bool aligned)
{
    Result result;
    size_t numPages = size / PageSize;
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);
    if (pOut)
    {
        NN_KERN_ASSERT(pOut->GetTotalNumPages() == 0);
    }

    Bit32 testState = (aligned? KMemoryState_FlagsDeviceSharedAligned: KMemoryState_FlagsDeviceShared);
    result = CheckMemoryState(addr, size,
            testState, testState,
            permission, permission,
            (KMemoryAttribute_Locked | KMemoryAttribute_IpcLocked), 0, KMemoryAttribute_DeviceShared);
    if (result.IsFailure())
    {
        return result;
    }

    if (pOut)
    {
        result = MakePageGroupImpl(pOut, addr, size / PageSize);
        if (result.IsFailure())
        {
            return result;
        }
    }

    KMemoryBlockManagerUpdateBuffer lockUpdateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.UpdateLock(&lockUpdateBuffer, addr, numPages, &KMemoryBlock::DeviceShared, KMemoryPermission_None);

    if (pOut)
    {
        pOut->Open();
    }

    return ResultSuccess();
}

Result KPageTableBase::UnlockForDeviceAddressSpace(KProcessAddress addr, size_t size)
{
    size_t numPages = size / PageSize;
    if (!IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KScopedLightLock locker(&m_Mutex);

    Result result;
    result = CheckMemoryStateContiguous(addr, size,
            KMemoryState_FlagsDeviceShared, KMemoryState_FlagsDeviceShared,
            0, 0,
            KMemoryAttribute_Locked | KMemoryAttribute_DeviceShared, KMemoryAttribute_DeviceShared);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer lockUpdateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.UpdateLock(&lockUpdateBuffer, addr, numPages, &KMemoryBlock::DeviceUnshared, KMemoryPermission_None);

    return ResultSuccess();
}

Result KPageTableBase::LockForTransferMemory(KPageGroup* pOut, KProcessAddress addr, size_t size, KMemoryPermission permission)
{
    return LockMemoryAndOpen(pOut, addr, size,
            KMemoryState_FlagsTransfer, KMemoryState_FlagsTransfer,
            KMemoryPermission_All, KMemoryPermission_UserReadWrite,
            KMemoryAttribute_All, KMemoryAttribute_None,
            permission, KMemoryAttribute_Locked);
}

Result KPageTableBase::UnlockForTransferMemory(KProcessAddress addr, size_t size, const KPageGroup& pg)
{
    return UnlockMemory(addr, size,
            KMemoryState_FlagsTransfer, KMemoryState_FlagsTransfer,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_Locked,
            KMemoryPermission_UserReadWrite, KMemoryAttribute_Locked, &pg);
}

Result KPageTableBase::LockForIpcUserBuffer(KProcessAddress addr, size_t size)
{
    return LockMemoryAndOpen(nullptr, addr, size,
            KMemoryState_FlagsIpcBuffer, KMemoryState_FlagsIpcBuffer,
            KMemoryPermission_All, KMemoryPermission_UserReadWrite,
            KMemoryAttribute_All, KMemoryAttribute_None,
            KMemoryPermission_KernelReadWrite, KMemoryAttribute_Locked);
}

Result KPageTableBase::UnlockForIpcUserBuffer(KProcessAddress addr, size_t size)
{
    return UnlockMemory(addr, size,
            KMemoryState_FlagsIpcBuffer, KMemoryState_FlagsIpcBuffer,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_Locked,
            KMemoryPermission_UserReadWrite, KMemoryAttribute_Locked, nullptr);
}

Result KPageTableBase::LockForCodeMemory(KPageGroup* pOut, KProcessAddress addr, size_t size)
{
    return LockMemoryAndOpen(pOut, addr, size,
            KMemoryState_FlagsCodeMemory, KMemoryState_FlagsCodeMemory,
            KMemoryPermission_All, KMemoryPermission_UserReadWrite,
            KMemoryAttribute_All, KMemoryAttribute_None,
            KMemoryPermission_KernelReadWrite, KMemoryAttribute_Locked);
}

Result KPageTableBase::UnlockForCodeMemory(KProcessAddress addr, size_t size, const KPageGroup& pg)
{
    return UnlockMemory(addr, size,
            KMemoryState_FlagsCodeMemory, KMemoryState_FlagsCodeMemory,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_Locked,
            KMemoryPermission_UserReadWrite, KMemoryAttribute_Locked, &pg);
}

Result KPageTableBase::MapPageGroupImpl(void** ppUpdateData, KProcessAddress vaddr, const KPageGroup& pg, const KPageProperty property)
{
    NN_KERN_ASSERT(m_Mutex.IsLockedByMe());
    Result result = ResultSuccess();
    KProcessAddress vaddrTop = vaddr;
    KVirtualAddress heapBegin = KMemoryLayout::GetHeapRegionBegin();
    KVirtualAddress heapEnd = KMemoryLayout::GetHeapRegionEnd();

    for (KPageGroup::BlockInfoList::const_iterator it = pg.GetBlockBeginIter(); it != pg.GetBlockEndIter(); it++)
    {
        KVirtualAddress virtAddress = it->GetBlockAddr();
        size_t numPages = it->GetNumPages();
        KVirtualAddress virtAddressEnd = virtAddress + (numPages * PageSize);

        NN_KERN_ABORT_UNLESS(heapBegin <= virtAddress);
        NN_KERN_ABORT_UNLESS(virtAddress < virtAddressEnd - 1);
        NN_KERN_ABORT_UNLESS(virtAddressEnd - 1 <= heapEnd - 1);

        result = OperateImpl(ppUpdateData, vaddr, numPages, GetHeapPhysicalAddress(virtAddress), true, property, PageTableOperation_Map);
        if (result.IsFailure())
        {
            NN_WARNING(false, "failed to operate memory map vaddr: 0x%08X, numPages: %d, paddr: 0x%08X",
                    GetAsInteger(vaddr), it->GetNumPages(), GetAsInteger(it->GetBlockAddr()));
            break;
        }
        vaddr += it->GetNumPages() * PageSize;
    }

    if (result.IsFailure() && (vaddr - vaddrTop) != 0)
    {
        // vaddrTop - vaddr を 元に戻す
        KMemoryInfo mi;
        nn::svc::PageInfo pi;
        Result resultCheck = QueryInfoImpl(&mi, &pi, vaddr);
        NN_KERN_ABORT_UNLESS(resultCheck.IsSuccess());

        const KPageProperty prevProperty = { mi.permission, mi.state == KMemoryState_Io, (mi.attribute & KMemoryAttribute_Uncached) != 0};
        resultCheck = OperateImpl(ppUpdateData, vaddrTop, (vaddrTop - vaddr) / PageSize, Null<KPhysicalAddress>(), false, prevProperty, PageTableOperation_Unmap);
        NN_KERN_ABORT_UNLESS(resultCheck.IsSuccess());
    }

    return result;
}

KProcessAddress KPageTableBase::FindFreeAreaInRegion(KProcessAddress regionBegin, size_t regionNumPages, size_t numPages, size_t align, size_t alignOffset, size_t guardPages) const
{
    KProcessAddress addr = Null<KProcessAddress>();
    if (numPages <= regionNumPages)
    {
        if (IsAslr())
        {
            for (int i = 0; i < 8; i++)
            {
                size_t offsetPages = KSystemControl::GetRandomValue(0, (regionNumPages - numPages - guardPages) * PageSize / align) * align;
                KProcessAddress testAddr = ((regionBegin + offsetPages) & ~(align - 1)) + alignOffset;

                KMemoryInfo mi;
                nn::svc::PageInfo pi;

                Result result = QueryInfoImpl(&mi, &pi, GetAsInteger(testAddr));
                NN_KERN_ABORT_UNLESS(result.IsSuccess());
                if (mi.state == KMemoryState_Free)
                {
                    if (regionBegin <= GetAsInteger(testAddr) &&
                            mi.baseAddress + PageSize * guardPages <= GetAsInteger(testAddr) &&
                            testAddr + (numPages + guardPages) * PageSize - 1 <= mi.baseAddress + mi.size - 1 &&
                            testAddr + (numPages + guardPages) * PageSize - 1 <= regionBegin + regionNumPages * PageSize - 1)
                    {
                        addr = testAddr;
                        break;
                    }
                }
            }
            if (addr == Null<KProcessAddress>())
            {
                size_t offsetPages = KSystemControl::GetRandomValue(0, regionNumPages - numPages);
                addr = m_MemoryBlocks.FindFreeAreaInRegion(regionBegin + offsetPages * PageSize, regionNumPages - offsetPages, numPages, align, alignOffset, guardPages);
            }
        }
        if (addr == Null<KProcessAddress>())
        {
            addr = m_MemoryBlocks.FindFreeAreaInRegion(regionBegin, regionNumPages, numPages, align, alignOffset, guardPages);
        }
    }

    return addr;
}

size_t KPageTableBase::GetCodeSize() const
{
    KScopedLightLock locker(&m_Mutex);
    size_t totalSize = 0;
    KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(m_AddressBegin);
    while (it != m_MemoryBlocks.EndIterator())
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        if (mi.state == KMemoryState_Code)
        {
            totalSize += mi.size;
        }
        it++;
    }

    return totalSize;
}

size_t KPageTableBase::GetCodeDataSize() const
{
    KScopedLightLock locker(&m_Mutex);
    size_t totalSize = 0;
    KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(m_AddressBegin);
    while (it != m_MemoryBlocks.EndIterator())
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        if (mi.state == KMemoryState_CodeData)
        {
            totalSize += mi.size;
        }
        it++;
    }

    return totalSize;
}

Result KPageTableBase::StackFillPattern(KProcessAddress stackEnd) const
{
    if (!(IsInRange(stackEnd - 1, 1, KMemoryState_Stack)))
    {
        return nn::svc::ResultInvalidRegion();
    }
    return FillMemoryBlock(stackEnd - 1, KMemoryState_Stack, m_StackFillMemoryPattern);
}

// addr の属するメモリブロックの先頭から addr まで pattern で埋める
Result KPageTableBase::FillMemoryBlock(KProcessAddress addr, KMemoryState state, FillMemoryPattern pattern) const
{
    Result result;

    if (!IsInRange(addr, 1))
    {
        return nn::svc::ResultInvalidAddress();
    }

    KScopedLightLock locker(&m_Mutex);

    KMemoryBlockManager::MemoryBlockList::const_iterator it0 = m_MemoryBlocks.FindIterator(addr);

    KMemoryInfo mi = it0->GetMemoryInfo();
    result = CheckMemoryState(mi,
            KMemoryState_All, state,
            KMemoryPermission_All, KMemoryPermission_UserReadWrite,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageGroup pg(GetBlockInfoManager());

    KProcessAddress mbStart = mi.baseAddress;
    size_t numPages = mi.size / PageSize;

    result = MakePageGroupImpl(&pg.GetPageGroup(), mbStart, numPages);
    if (result.IsFailure())
    {
        return result;
    }

    size_t leftSize = (GetAsInteger(addr) - mi.baseAddress + 1);
    for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter() && leftSize != 0; it++)
    {
        size_t fillSize = it->GetNumPages() * PageSize;
        if (leftSize < fillSize)
        {
            fillSize = leftSize;
        }
        ::std::memset(GetUntypedPointer(it->GetBlockAddr()), pattern, fillSize);
        leftSize -= fillSize;
    }

    return ResultSuccess();
}

// addr 以前にあるメモリブロックの先頭から pattern で埋められている
Result KPageTableBase::NotCmpMemoryBlock(size_t* pFillSize, size_t* pSize, KProcessAddress addr, KMemoryState state, FillMemoryPattern pattern) const
{
    Result result;

    if (!IsInRange(addr - 1, 1))
    {
        return nn::svc::ResultInvalidAddress();
    }

    KScopedLightLock locker(&m_Mutex);

    // stateのメモリブロック探索
    KProcessAddress mbEnd = addr;
    for (KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(addr); it != m_MemoryBlocks.EndIterator(); it++)
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        if (CheckMemoryState(mi, KMemoryState_All, state, 0, 0, 0, 0).IsFailure())
        {
            break;
        }
        mbEnd = mi.baseAddress + mi.size;
    }
    KProcessAddress prevMbEnd = addr;
    for (;;)
    {
        KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(prevMbEnd - 1);
        KMemoryInfo mi = it->GetMemoryInfo();
        if (CheckMemoryState(mi, KMemoryState_All, state, 0, 0, 0, 0).IsFailure())
        {
            break;
        }
        prevMbEnd = mi.baseAddress;
    }
    if (mbEnd == addr && prevMbEnd == addr)
    {
        return nn::svc::ResultInvalidAddress();
    }

    KProcessAddress mbStart = prevMbEnd;

    size_t size = mbEnd - mbStart;
    size_t numPages = (size + PageSize - 1) / PageSize;

    KScopedPageGroup pg(GetBlockInfoManager());
    result = MakePageGroupImpl(&pg.GetPageGroup(), mbStart, numPages);
    if (result.IsFailure())
    {
        return result;
    }

    size_t leftSize = addr - mbStart;
    KProcessAddress addrTop = mbStart;
    for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter() && leftSize != 0; it++)
    {
        size_t mapSize = it->GetNumPages() * PageSize;
        if (leftSize < mapSize)
        {
            mapSize = leftSize;
        }
        const Bit8* pStack = GetTypedPointer<const Bit8>(it->GetBlockAddr());
        size_t i;
        for (i = 0; i < mapSize; i++)
        {
            if (pStack[i] != pattern)
            {
                break;
            }
        }
        addrTop += i;
        if (i != mapSize)
        {
            break;
        }
        leftSize -= mapSize;
    }
    *pFillSize = addrTop - mbStart;
    *pSize = size;

    return ResultSuccess();
}

Result KPageTableBase::GetStackUsage(size_t* pUsageSize, size_t* pStackSize, KProcessAddress stackAddr) const
{
    size_t fillSize;
    size_t stackSize;
    Result result = NotCmpMemoryBlock(&fillSize, &stackSize, stackAddr, KMemoryState_Stack, m_StackFillMemoryPattern);
    if (result.IsFailure())
    {
        return result;
    }
    *pUsageSize = stackSize - fillSize;
    *pStackSize = stackSize;
    return ResultSuccess();
}

Result KPageTableBase::MapPhysicalMemory(KProcessAddress addr, size_t size)
{
    KScopedLightLock mapPhyslocker(&m_MapPhysMutex);
    KProcessAddress lastAddr = addr + size - 1;
    KProcessAddress curAddr;
    size_t mappedSize;

retry:
    {
        KScopedLightLock locker(&m_Mutex);
        curAddr = addr;
        mappedSize = 0;

        KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(addr);
        NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());

        for (;;)
        {
            KMemoryInfo mi = it->GetMemoryInfo();

            if (lastAddr <= (mi.baseAddress + mi.size - 1))
            {
                if (mi.state != KMemoryState_Free)
                {
                    mappedSize += (lastAddr - curAddr + 1);
                }
                break;
            }

            if (mi.state != KMemoryState_Free)
            {
                mappedSize += (KProcessAddress(mi.baseAddress + mi.size) - curAddr);
            }
            curAddr = mi.baseAddress + mi.size;

            it++;
            NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
        }

        // 既にすべてマップ済み
        if (size == mappedSize)
        {
            return ResultSuccess();
        }
    }

    {
        KScopedResourceLimitTester tester(GetCurrentProcessPointer(), nn::svc::LimitableResource_PhysicalMemoryMax, size - mappedSize);
        if (tester.IsTestFailed())
        {
            return nn::svc::ResultLimit();
        }

        KScopedPageGroup pg(GetBlockInfoManager());
        Result result = Kernel::GetKernelHeapManager().Allocate(&pg.GetPageGroup(), (size - mappedSize) / PageSize, GetAllocateOption());
        if (result.IsFailure())
        {
            NN_WARNING(false, "failed to allocate memory, numPages = %d", (size - mappedSize) / PageSize);
            return result;
        }
        pg.GetPageGroup().Open();
        NN_UTIL_SCOPE_EXIT { pg.GetPageGroup().Close(); };

        for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter(); it++)
        {
            ::std::memset(GetUntypedPointer(it->GetBlockAddr()), m_AllocateFillMemoryPattern, it->GetNumPages() * PageSize);
        }

        {
            KScopedLightLock locker(&m_Mutex);
            size_t mappedSizeForCheck = 0;

            curAddr = addr;

            KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(addr);
            NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());

            for (;;)
            {
                KMemoryInfo mi = it->GetMemoryInfo();

                if (lastAddr <= (mi.baseAddress + mi.size - 1))
                {
                    if (mi.state != KMemoryState_Free)
                    {
                        mappedSizeForCheck += (lastAddr - curAddr + 1);
                    }
                    break;
                }

                if (mi.state != KMemoryState_Free)
                {
                    mappedSizeForCheck += (KProcessAddress(mi.baseAddress + mi.size) - curAddr);
                }
                curAddr = mi.baseAddress + mi.size;

                it++;
                NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
            }

            // 既にすべてマップ済み
            if (mappedSize != mappedSizeForCheck)
            {
                goto retry;
            }

            Result result;

            KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
            if (result.IsFailure())
            {
                return result;
            }

            KScopedPageTableUpdater updateData(this);

            curAddr = addr;

            KPageGroup::BlockInfoList::const_iterator bit = pg.GetPageGroup().GetBlockBeginIter();
            KVirtualAddress blockVaddr = bit->GetBlockAddr();
            size_t leftBlockPages = bit->GetNumPages();

            it = m_MemoryBlocks.FindIterator(addr);
            NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());

            for (;;)
            {
                KMemoryInfo mi = it->GetMemoryInfo();

                if (mi.state == KMemoryState_Free)
                {
                    size_t numPages = std::min((KProcessAddress(mi.baseAddress + mi.size) - curAddr), (lastAddr - curAddr + 1)) / PageSize;
                    const KPageProperty property = { KMemoryPermission_UserReadWrite, false, false};

                    while (numPages)
                    {
                        if (leftBlockPages == 0)
                        {
                            NN_KERN_ASSERT(bit != pg.GetPageGroup().GetBlockEndIter());
                            bit++;
                            blockVaddr = bit->GetBlockAddr();
                            leftBlockPages = bit->GetNumPages();
                        }

                        size_t opPages = std::min(numPages, leftBlockPages);
                        result = OperateImpl(updateData.GetDataPtr(), curAddr, opPages, GetHeapPhysicalAddress(blockVaddr), true, property, PageTableOperation_Map);
                        if (result.IsFailure())
                        {
                            goto error_exit;
                        }

                        curAddr += opPages * PageSize;
                        numPages -= opPages;
                        blockVaddr += opPages * PageSize;
                        leftBlockPages -= opPages;
                    }
                }

                if (lastAddr <= (mi.baseAddress + mi.size - 1))
                {
                    break;
                }
                curAddr = mi.baseAddress + mi.size;
                it++;
                NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
            }

            tester.Accepted();
            m_MapPhysicalMemorySize += size - mappedSize;

            m_MemoryBlocks.UpdateIfMatch(&updateBuffer, addr, size / PageSize,
                    KMemoryState_Free,   KMemoryPermission_None,          KMemoryAttribute_None,
                    KMemoryState_Normal, KMemoryPermission_UserReadWrite, KMemoryAttribute_None);

            return ResultSuccess();

error_exit:
            if (curAddr > addr)
            {
                lastAddr = curAddr - 1;
                curAddr = addr;

                it = m_MemoryBlocks.FindIterator(addr);
                NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());

                for (;;)
                {
                    KMemoryInfo mi = it->GetMemoryInfo();

                    if (mi.state == KMemoryState_Free)
                    {
                        size_t numPages = std::min((KProcessAddress(mi.baseAddress + mi.size) - curAddr), (lastAddr - curAddr + 1)) / PageSize;
                        const KPageProperty property = { KMemoryPermission_None, false, false};
                        Result resultCheck = OperateImpl(updateData.GetDataPtr(), curAddr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_Unmap);
                        NN_KERN_ABORT_IF_FAILED(resultCheck);
                    }

                    if (lastAddr <= (mi.baseAddress + mi.size - 1))
                    {
                        break;
                    }
                    curAddr = mi.baseAddress + mi.size;
                    it++;
                    NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
                }
            }

            return result;
        }
    }
}

Result KPageTableBase::UnmapPhysicalMemory(KProcessAddress addr, size_t size)
{
    KScopedLightLock mapPhyslocker(&m_MapPhysMutex);
    KScopedLightLock locker(&m_Mutex);

    KProcessAddress lastAddr = addr + size - 1;
    KProcessAddress curAddr = addr;
    size_t mappedSize = 0;
    Result result;

    KMemoryBlockManager::MemoryBlockList::const_iterator it = m_MemoryBlocks.FindIterator(addr);
    NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());

    for (;;)
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        if (!((mi.state == KMemoryState_Normal && mi.attribute == 0) || (mi.state == KMemoryState_Free)))
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }

        if (lastAddr <= (mi.baseAddress + mi.size - 1))
        {
            if (mi.state == KMemoryState_Normal)
            {
                mappedSize += (lastAddr - curAddr + 1);
            }
            break;
        }

        if (mi.state == KMemoryState_Normal)
        {
            mappedSize += (KProcessAddress(mi.baseAddress + mi.size) - curAddr);
        }
        curAddr = mi.baseAddress + mi.size;

        it++;
        NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
    }

    // 既に解放済み
    if (mappedSize == 0)
    {
        return ResultSuccess();
    }

    KScopedPageGroup pg(GetBlockInfoManager());

    {
        TraverseContext     context;
        TraverseData        current;
        bool                currentIsValid;
        TraverseData        next;
        bool                nextIsValid;
        size_t              sizeSum  = 0;

        current.address = 0;
        current.size    = 0;
        currentIsValid  = false;

        nextIsValid = m_Table.TraverseBegin(&next, &context, addr);
        next.size = next.size - (GetAsInteger(addr) & (next.size - 1));

        for (;;)
        {
            if ((nextIsValid && currentIsValid && next.address == current.address + current.size)
                    || (!currentIsValid && !nextIsValid ))
            {
                current.size += next.size;
            }
            else
            {
                if (currentIsValid)
                {
                    NN_KERN_ABORT_UNLESS(IsHeapPhysicalAddress(current.address));
                    result = pg.GetPageGroup().AddBlock(GetHeapVirtualAddress(current.address), current.size / PageSize);
                    if (result.IsFailure())
                    {
                        return result;
                    }
                }

                sizeSum += current.size;
                current = next;
                currentIsValid = nextIsValid;
            }

            if (sizeSum + current.size >= size)
            {
                break;
            }

            nextIsValid = m_Table.TraverseNext(&next, &context);
        }

        if (currentIsValid)
        {
            NN_KERN_ABORT_UNLESS(IsHeapPhysicalAddress(current.address));
            result = pg.GetPageGroup().AddBlock(GetHeapVirtualAddress(current.address), (size - sizeSum) / PageSize);
            if (result.IsFailure())
            {
                return result;
            }
        }
    }

    NN_KERN_ASSERT(pg.GetPageGroup().GetTotalNumPages() == mappedSize / PageSize);

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    curAddr = addr;

    it = m_MemoryBlocks.FindIterator(addr);
    NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());

    pg.GetPageGroup().Open();

    for (;;)
    {
        KMemoryInfo mi = it->GetMemoryInfo();

        if (mi.state == KMemoryState_Normal)
        {
            size_t numPages = std::min((KProcessAddress(mi.baseAddress + mi.size) - curAddr), (lastAddr - curAddr + 1)) / PageSize;
            const KPageProperty property = { KMemoryPermission_None, false, false};
            result = OperateImpl(updateData.GetDataPtr(), curAddr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_Unmap);
            if (result.IsFailure())
            {
                goto error_exit;
            }
        }

        if (lastAddr <= (mi.baseAddress + mi.size - 1))
        {
            break;
        }
        curAddr = mi.baseAddress + mi.size;
        it++;
        NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
    }

    pg.GetPageGroup().Close();

    m_MapPhysicalMemorySize -= mappedSize;
    GetCurrentProcessPointer()->ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, mappedSize);

    m_MemoryBlocks.Update(&updateBuffer, addr, size / PageSize, KMemoryState_Free, KMemoryPermission_None, KMemoryAttribute_None);

    return ResultSuccess();

error_exit:
    if (curAddr > addr)
    {
        lastAddr = curAddr - 1;
        curAddr = addr;

        KPageGroup::BlockInfoList::const_iterator bit = pg.GetPageGroup().GetBlockBeginIter();
        KVirtualAddress blockVaddr = bit->GetBlockAddr();
        size_t leftBlockPages = bit->GetNumPages();

        it = m_MemoryBlocks.FindIterator(addr);
        NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());

        for (;;)
        {
            KMemoryInfo mi = it->GetMemoryInfo();

            if (mi.state == KMemoryState_Normal)
            {
                size_t numPages = std::min((KProcessAddress(mi.baseAddress + mi.size) - curAddr), (lastAddr - curAddr + 1)) / PageSize;
                const KPageProperty property = { mi.permission, false, false};

                while (numPages)
                {
                    if (leftBlockPages == 0)
                    {
                        NN_KERN_ASSERT(bit != pg.GetPageGroup().GetBlockEndIter());
                        bit++;
                        blockVaddr = bit->GetBlockAddr();
                        leftBlockPages = bit->GetNumPages();
                    }

                    size_t opPages = std::min(numPages, leftBlockPages);
                    Result resultCheck = OperateImpl(updateData.GetDataPtr(), curAddr, opPages, GetHeapPhysicalAddress(blockVaddr), true, property, PageTableOperation_Map);
                    NN_KERN_ABORT_IF_FAILED(resultCheck);

                    curAddr += opPages * PageSize;
                    numPages -= opPages;
                    blockVaddr += opPages * PageSize;
                    leftBlockPages -= opPages;
                }
            }

            if (lastAddr <= (mi.baseAddress + mi.size - 1))
            {
                break;
            }
            curAddr = mi.baseAddress + mi.size;
            it++;
            NN_KERN_ASSERT(it != m_MemoryBlocks.EndIterator());
        }
    }

    pg.GetPageGroup().Close();
    return result;
}

Result KPageTableBase::MapPhysicalMemoryUnsafe(KProcessAddress addr, size_t size)
{
    if (!Kernel::GetUnsafeMemory().TestLimit(size))
    {
        return nn::svc::ResultLimit();
    }

    bool isSuccess = false;
    NN_UTIL_SCOPE_EXIT { if (!isSuccess) { Kernel::GetUnsafeMemory().ReleaseLimit(size); } };

    KScopedPageGroup pg(GetBlockInfoManager());
    Result result = Kernel::GetKernelHeapManager().Allocate(
            &pg.GetPageGroup(),
            size / PageSize,
            KMemoryManager::MakeAllocateOption(KMemoryManager::Region_Unsafe, KMemoryManager::From_Front));
    if (result.IsFailure())
    {
        NN_WARNING(false, "failed to allocate memory, numPages = %d", size / PageSize);
        return result;
    }

    pg.GetPageGroup().Open();
    NN_UTIL_SCOPE_EXIT { pg.GetPageGroup().Close(); };

    for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter(); it++)
    {
        ::std::memset(GetUntypedPointer(it->GetBlockAddr()), m_AllocateFillMemoryPattern, it->GetNumPages() * PageSize);
    }

    {
        KScopedLightLock locker(&m_Mutex);

        result = CheckMemoryState(addr, size, KMemoryState_All, KMemoryState_Free, 0, 0, 0, 0);
        if (result.IsFailure())
        {
            return result;
        }

        KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
        if (result.IsFailure())
        {
            return result;
        }

        KScopedPageTableUpdater updateData(this);

        size_t numPages = size / PageSize;
        const KPageProperty property = { KMemoryPermission_UserReadWrite, false, false};
        result = OperateImpl(updateData.GetDataPtr(), addr, numPages, pg.GetPageGroup(), property, PageTableOperation_MapRelocate);
        if (result.IsFailure())
        {
            return result;
        }

        isSuccess = true;

        m_MemoryBlocks.Update(&updateBuffer, addr, numPages, KMemoryState_Normal, property.permission, KMemoryAttribute_None);

        return ResultSuccess();
    }
}

Result KPageTableBase::UnmapPhysicalMemoryUnsafe(KProcessAddress addr, size_t size)
{
    KScopedLightLock locker(&m_Mutex);

    Result result = CheckMemoryState(addr, size,
            KMemoryState_All, KMemoryState_Normal,
            KMemoryPermission_All, KMemoryPermission_UserReadWrite,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    KMemoryBlockManagerUpdateBuffer updateBuffer(&result, GetMemoryBlockResourceManager());
    if (result.IsFailure())
    {
        return result;
    }

    KScopedPageTableUpdater updateData(this);

    size_t numPages = size / PageSize;
    const KPageProperty property = { KMemoryPermission_None, false, false};
    result = OperateImpl(updateData.GetDataPtr(), addr, numPages, Null<KPhysicalAddress>(), false, property, PageTableOperation_Unmap);
    if (result.IsFailure())
    {
        return result;
    }

    m_MemoryBlocks.Update(&updateBuffer, addr, numPages, KMemoryState_Free, property.permission, KMemoryAttribute_None);
    Kernel::GetUnsafeMemory().ReleaseLimit(size);

    return ResultSuccess();
}


}}
