﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include "kern_Platform.h"
#include "kern_Kernel.h"
#include "kern_Assert.h"
#include "kern_KPageHeap.h"
#include "kern_DebugString.h"
#include "kern_KPageGroup.h"
#include "kern_Utility.h"

namespace nn { namespace kern {

size_t KPageHeap::CalcManagementAreaSize(size_t heapSize, const size_t blockPageShift[], size_t numBlockPages)
{
    size_t size = 0;
    for (size_t index = 0; index < numBlockPages; index++)
    {
        int blockShift = blockPageShift[index];
        int nextBlockShift = (index == numBlockPages - 1)? 0: blockPageShift[index + 1];
        size += FreeBlocks::CalcSize(heapSize, blockShift, nextBlockShift);
    }

    return RoundUp(size, PageSize);
}

void KPageHeap::Initialize(KVirtualAddress heapStart, size_t heapSize, KVirtualAddress heapBackStart, KVirtualAddress managementStart, size_t managementSize, const size_t blockPageShift[], size_t numBlockPages)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT((PageSize != 0));
    NN_KERN_ASSERT(((PageSize - 1) & PageSize) == 0);   // pageSize は 2 の冪上でなければならない。
    NN_KERN_ASSERT(heapStart % PageSize == 0);
    NN_KERN_ASSERT(heapSize  % PageSize == 0);
    NN_KERN_ASSERT(numBlockPages <= MaxNumBlockPages);
    NN_KERN_ASSERT(numBlockPages > 0);
    NN_KERN_ASSERT(heapStart <= heapBackStart);
    KVirtualAddress managementEnd = managementStart + managementSize;

    m_HeapStart = heapStart;
    m_HeapSize  = heapSize;
    m_BackStart = heapBackStart;
    m_NumBlockPages  = numBlockPages;

    Bit64* pBitArray = GetTypedPointer<Bit64>(managementStart);
    for (size_t index = 0; index < numBlockPages; index++)
    {
        int blockShift = blockPageShift[index];
        int nextBlockShift = (index == numBlockPages - 1)? 0: blockPageShift[index + 1];
        pBitArray = m_FreeBlocks[index].Initialize(m_HeapStart, m_HeapSize, blockShift, nextBlockShift, m_BackStart, pBitArray);
    }
    NN_KERN_ABORT_UNLESS(KVirtualAddress(pBitArray) <= managementEnd);

    if (heapSize)
    {
        FreeImpl(heapStart, heapSize / PageSize);

#if defined NN_SDK_BUILD_DEVELOP || defined NN_SDK_BUILD_DEBUG
        size_t totalFree = GetNumFreePages() * PageSize;
        if (totalFree > 0x01000000)
        {
            NN_LOG("Heap Total %ld MB\n", totalFree / 0x00100000);
        }
        else
        {
            NN_LOG("Heap Total %ld KB\n", totalFree / 0x00000400);
        }
#endif
    }
}


Result KPageHeap::Allocate(KPageGroup* pOut, size_t numPages, bool fromBack)
{
    NN_KERN_THIS_ASSERT();

    if (numPages == 0)
    {
        NN_WARNING(false, "0 page is required.");
        return ResultSuccess();
    }

    if (numPages > GetNumFreePages())
    {
        return nn::svc::ResultOutOfMemory();
    }

    size_t leftPages = numPages;
    Result result = ResultSuccess();

    for (int index = AdaptiveBlock(leftPages); index >= 0 && leftPages > 0; index--)
    {
        size_t blockPages = m_FreeBlocks[index].GetBlockPages();

        while (leftPages >= blockPages)
        {
            KVirtualAddress addr = AllocateBlock(index, fromBack);
            if (addr == Null<KVirtualAddress>())
            {
                break;
            }
            result = pOut->AddBlock(addr, blockPages);
            if (result.IsFailure())
            {
                FreeImpl(addr, blockPages);
                goto error_exit;
            }
            leftPages -= blockPages;
        }
    }

    if (leftPages > 0)
    {
        result = nn::svc::ResultOutOfMemory();
    }
error_exit:
    if (leftPages > 0)
    {
        for (KPageGroup::BlockInfoList::const_iterator it = pOut->GetBlockBeginIter(); it != pOut->GetBlockEndIter(); it++)
        {
            FreeImpl(it->GetBlockAddr(), it->GetNumPages());
        }
    }

    return result;
}


KVirtualAddress KPageHeap::AllocateContinuous(size_t numPages, size_t pageAlign, bool fromBack)
{
    NN_KERN_THIS_ASSERT();
    NN_UNUSED(pageAlign);

    if (numPages == 0)
    {
        NN_WARNING(false, "0 page is required.");
        return Null<KVirtualAddress>();
    }

    int index = AlignedBlock(numPages);
    if (index < 0)
    {
        NN_WARNING(false, "AllocateContinuous numPages=%p pageAlign=%u", numPages, pageAlign);
        return Null<KVirtualAddress>();
    }

    size_t reqSize = PageSize * numPages;
    size_t allocSize = m_FreeBlocks[index].GetBlockSize();

    KVirtualAddress addr = AllocateBlock(index, fromBack);
    if (addr != Null<KVirtualAddress>() && allocSize > reqSize)
    {
        FreeImpl(addr + reqSize, (allocSize - reqSize) / PageSize);
    }
    return addr;
}

void KPageHeap::Free(KVirtualAddress addr, size_t numPages)
{
    NN_KERN_THIS_ASSERT();

    if (numPages == 0)
    {
        return;
    }

    FreeImpl(addr, numPages);
}

void KPageHeap::DumpFreeList()
{
    NN_KERN_THIS_ASSERT();

    NN_KERN_RELEASE_LOG("KPageHeap::DumpFreeList %p\n", this);
    for (int i = 0; i < m_NumBlockPages; i++)
    {
        size_t blockSize = m_FreeBlocks[i].GetBlockSize();
        const char *str =
            (blockSize >= 0x40000000)? "GiB":
            (blockSize >= 0x00100000)? "MiB":
            (blockSize >= 0x00000400)? "KiB": "B";
        size_t printSize =
            (blockSize >= 0x40000000)? blockSize / 0x40000000:
            (blockSize >= 0x00100000)? blockSize / 0x00100000:
            (blockSize >= 0x00000400)? blockSize / 0x00000400: blockSize;

        NN_UNUSED(blockSize);
        NN_UNUSED(str);
        NN_UNUSED(printSize);
        NN_KERN_RELEASE_LOG("    %4ld %s block x %ld (%ld, %ld, %ld)\n",
                printSize, str, m_FreeBlocks[i].GetNumFreeBlocks(),
                m_FreeBlocks[i].CountFreeBlocks(),
                m_FreeBlocks[i].CountFrontFreeBlocks(),
                m_FreeBlocks[i].CountBackFreeBlocks());
    }
}

size_t KPageHeap::GetNumFreePages() const
{
    NN_KERN_THIS_ASSERT();
    size_t numPages = 0;

    for (int i = 0; i < m_NumBlockPages; i++)
    {
        numPages += m_FreeBlocks[i].GetNumFreePages();
    }

    return numPages;
}

int KPageHeap::AdaptiveBlock(size_t numPages) const
{
    for (int index = m_NumBlockPages - 1; index >= 0; index--)
    {
        if (numPages >= m_FreeBlocks[index].GetBlockPages())
        {
            return index;
        }
    }
    return -1;
}

int KPageHeap::AlignedBlock(size_t numPages) const
{
    for (int index = 0; index < m_NumBlockPages; index++)
    {
        if (numPages <= m_FreeBlocks[index].GetBlockPages())
        {
            return index;
        }
    }
    return -1;
}

void KPageHeap::FreeBlock(KVirtualAddress addr, int index)
{
    do
    {
        addr = m_FreeBlocks[index++].Push(addr);
    } while (addr != Null<KVirtualAddress>());
}

KVirtualAddress KPageHeap::AllocateBlock(int index, bool fromBack)
{
    size_t reqSize = m_FreeBlocks[index].GetBlockSize();

    for (int i = index; i < m_NumBlockPages; i++)
    {
        KVirtualAddress addr = m_FreeBlocks[i].Pop(fromBack);
        if (addr != Null<KVirtualAddress>())
        {
            size_t allocSize = m_FreeBlocks[i].GetBlockSize();
            if (allocSize > reqSize)
            {
                FreeImpl(addr + reqSize, (allocSize - reqSize) / PageSize);
            }
            return addr;
        }
    }

    for (int i = index; i < m_NumBlockPages; i++)
    {
        KVirtualAddress addr = m_FreeBlocks[i].Pop(!fromBack);
        if (addr != Null<KVirtualAddress>())
        {
            size_t allocSize = m_FreeBlocks[i].GetBlockSize();
            if (allocSize > reqSize)
            {
                FreeImpl(addr + reqSize, (allocSize - reqSize) / PageSize);
            }
            return addr;
        }
    }

    return Null<KVirtualAddress>();
}


void KPageHeap::FreeImpl(KVirtualAddress addr, size_t numPages)
{
    KVirtualAddress freeStart = addr;
    KVirtualAddress freeEnd = addr + numPages * PageSize;
    int index;
    KVirtualAddress freeStart0 = freeStart;
    KVirtualAddress freeEnd0 = freeStart;
    KVirtualAddress freeStart1 = freeEnd;
    KVirtualAddress freeEnd1 = freeEnd;
    for (index = m_NumBlockPages - 1; index >= 0; index--)
    {
        size_t blockSize = m_FreeBlocks[index].GetBlockSize();
        KVirtualAddress largeBlockStart = RoundUp(freeStart, blockSize);
        KVirtualAddress largeBlockEnd = RoundDown(freeEnd, blockSize);
        if (largeBlockStart < largeBlockEnd)
        {
            for (KVirtualAddress a = largeBlockStart; a < largeBlockEnd; a += blockSize)
            {
                FreeBlock(a, index);
            }
            freeEnd0 = largeBlockStart;
            freeStart1 = largeBlockEnd;
            break;
        }
    }
    NN_KERN_ASSERT(index >= 0);

    for (int i = index - 1; i >= 0; i--)
    {
        size_t blockSize = m_FreeBlocks[i].GetBlockSize();
        while (freeStart0 <= freeEnd0 - blockSize)
        {
            freeEnd0 -= blockSize;
            FreeBlock(freeEnd0, i);
        }
    }

    for (int i = index - 1; i >= 0; i--)
    {
        size_t blockSize = m_FreeBlocks[i].GetBlockSize();
        while (freeStart1 + blockSize <= freeEnd1)
        {
            FreeBlock(freeStart1, i);
            freeStart1 += blockSize;
        }
    }
}

}}

