﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_Types.h>
#include <nn/svc/svc_BaseType.h>
#include "kern_Assert.h"
#include "kern_KTaggedAddress.h"
#include "kern_MemoryMap.h"
#include "kern_KLightMutex.h"
#include "kern_KPageHeap.h"
#include <algorithm>

namespace nn { namespace kern {
class KPageGroup;
struct KHeapArrange;
class KPageHeapHeader;
struct KPageHeapParameter;
class KMemoryManager
{
public:
    static const size_t PageSize = NN_KERN_FINEST_PAGE_SIZE;
    enum Region
    {
        Region_Application, Region_Unsafe = Region_Application,
        Region_Applet,
        Region_SecureSystem,
        Region_NonSecureSystem,
        Region_Num,

        Region_Shift = 4,
        Region_Mask  = 0xF0,
    };

    enum From
    {
        From_Front,
        From_Back,

        From_Shift = 0,
        From_Mask  = 0xF,
    };

    static Bit32 MakeAllocateOption(Region region, From from)
    {
        return (region << Region_Shift) | (from << From_Shift);
    }

private:

    class KMemoryManagerImpl
    {
    private:
        KPageHeap           m_Heap;
        KLightMutex         m_Lock;
        uint16_t*           m_pRefCount;
    public:
        void Initialize(KVirtualAddress heapStart, size_t heapSize, KVirtualAddress heapBackStart, KVirtualAddress managementStart, size_t managementSize);
        KMemoryManagerImpl(): m_pRefCount(nullptr) {}

    public:
        static size_t CalcManagementAreaSize(size_t heapSize);
        Result Allocate(KPageGroup* pOut, size_t numPages, From from)
        {
            NN_KERN_THIS_ASSERT();
            NN_KERN_ASSERT(NN_KERN_HAS_MMU);

            KScopedLightLock locker(&m_Lock);

            return m_Heap.Allocate(pOut, numPages, (from == From_Back));
        }

        KVirtualAddress AllocateContinuous(size_t numPages, uint32_t pageAlign, From from)
        {
            NN_KERN_THIS_ASSERT();
            KScopedLightLock locker(&m_Lock);

            return m_Heap.AllocateContinuous(numPages, pageAlign, (from == From_Back));
        }

        void Open(KVirtualAddress addr, size_t numPages)
        {
            NN_KERN_THIS_ASSERT();
            KScopedLightLock locker(&m_Lock);

            size_t index      = (addr - m_Heap.GetHeapStartAddress()) / PageSize;
            size_t indexEnd   = index + numPages;
            while (index < indexEnd)
            {
                uint16_t refCount = ++m_pRefCount[index];
                NN_KERN_ABORT_UNLESS(refCount > 0);
                index++;
            }
        }

        void Close(KVirtualAddress addr, size_t numPages)
        {
            NN_KERN_THIS_ASSERT();
            KScopedLightLock locker(&m_Lock);

            size_t index      = (addr - m_Heap.GetHeapStartAddress()) / PageSize;
            size_t indexEnd   = index + numPages;

            size_t freeBegin = 0;
            size_t numFree = 0;

            while (index < indexEnd)
            {
                NN_KERN_ABORT_UNLESS(m_pRefCount[index] > 0);

                uint16_t refCount = --m_pRefCount[index];

                if (refCount == 0)
                {
                    if (numFree > 0)
                    {
                        numFree++;
                    }
                    else
                    {
                        freeBegin = index;
                        numFree = 1;
                    }
                }
                else
                {
                    if (numFree > 0)
                    {
                        m_Heap.Free(m_Heap.GetHeapStartAddress() + freeBegin * PageSize, numFree);
                        numFree = 0;
                    }
                }

                index++;
            }

            if (numFree > 0)
            {
                m_Heap.Free(m_Heap.GetHeapStartAddress() + freeBegin * PageSize, numFree);
            }
        }

        void DumpFreeList()
        {
            NN_KERN_THIS_ASSERT();
            KScopedLightLock locker(&m_Lock);

            m_Heap.DumpFreeList();
        }

        size_t GetFreeSize()
        {
            NN_KERN_THIS_ASSERT();
            KScopedLightLock locker(&m_Lock);

            return m_Heap.GetNumFreePages() * KPageHeap::PageSize;
        }

        bool Includes(KVirtualAddress a) const { return m_Heap.Includes(a); }

        size_t GetSize() const
        {
            NN_KERN_THIS_ASSERT();
            return m_Heap.GetHeapSize();
        }

        KVirtualAddress GetEnd() const
        {
            NN_KERN_THIS_ASSERT();
            return m_Heap.GetHeapEndAddress();
        }
    };

    KMemoryManagerImpl m_MemoryManager[Region_Num];
public:
    void Initialize(const KHeapArrange& arrange);

    Result Allocate(KPageGroup* pOut, size_t numPages, Bit32 option)
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ASSERT(NN_KERN_HAS_MMU);

        Region region = static_cast<Region>((option & Region_Mask) >> Region_Shift);
        From from = static_cast<From>((option & From_Mask) >> From_Shift);
        NN_KERN_ASSERT(region < Region_Num);
        return m_MemoryManager[region].Allocate(pOut, numPages, from);
    }

    KVirtualAddress AllocateContinuous(size_t numPages, uint32_t pageAlign, Bit32 option)
    {
        NN_KERN_THIS_ASSERT();

        Region region = static_cast<Region>((option & Region_Mask) >> Region_Shift);
        From from = static_cast<From>((option & From_Mask) >> From_Shift);
        NN_KERN_ASSERT(region < Region_Num);
        return m_MemoryManager[region].AllocateContinuous(numPages, pageAlign, from);
    }

    void Open(KVirtualAddress addr, size_t numPages)
    {
        NN_KERN_THIS_ASSERT();

        while (numPages)
        {
            KMemoryManagerImpl& mm = GetMemoryManager(addr);
            KVirtualAddress end = mm.GetEnd();
            size_t n = std::min(numPages, (end - addr) / PageSize);
            mm.Open(addr, n);
            numPages -= n;
            addr += n * PageSize;
        }
    }

    void Close(KVirtualAddress addr, size_t numPages)
    {
        NN_KERN_THIS_ASSERT();
        while (numPages)
        {
            KMemoryManagerImpl& mm = GetMemoryManager(addr);
            KVirtualAddress end = mm.GetEnd();
            size_t n = std::min(numPages, (end - addr) / PageSize);
            mm.Close(addr, n);
            numPages -= n;
            addr += n * PageSize;
        }
    }

    void DumpFreeList()
    {
        NN_KERN_THIS_ASSERT();

        for (size_t i = 0; i < NN_ARRAY_SIZE(m_MemoryManager); i++)
        {
            m_MemoryManager[i].DumpFreeList();
        }
    }

    size_t GetFreeSize()
    {
        NN_KERN_THIS_ASSERT();

        size_t sum = 0;
        for (size_t i = 0; i < NN_ARRAY_SIZE(m_MemoryManager); i++)
        {
            sum += m_MemoryManager[i].GetFreeSize();
        }

        return sum;
    }

    size_t GetSize()
    {
        NN_KERN_THIS_ASSERT();

        size_t sum = 0;
        for (size_t i = 0; i < NN_ARRAY_SIZE(m_MemoryManager); i++)
        {
            sum += m_MemoryManager[i].GetSize();
        }

        return sum;
    }

    void DumpFreeList(KMemoryManager::Region region)
    {
        NN_KERN_THIS_ASSERT();
        m_MemoryManager[region].DumpFreeList();
    }

    size_t GetFreeSize(KMemoryManager::Region region)
    {
        NN_KERN_THIS_ASSERT();
        return m_MemoryManager[region].GetFreeSize();
    }

    size_t GetSize(KMemoryManager::Region region)
    {
        NN_KERN_THIS_ASSERT();
        return m_MemoryManager[region].GetSize();
    }

    static size_t SizeToPage(size_t size)
    {
        return (size + PageSize - 1) / PageSize;
    }

    static size_t CalcManagementAreaSize(size_t heapSize)
    {
        return KMemoryManagerImpl::CalcManagementAreaSize(heapSize);
    }

private:
    KMemoryManagerImpl& GetMemoryManager(KVirtualAddress v)
    {
        for (size_t i = 0; i < NN_ARRAY_SIZE(m_MemoryManager); i++)
        {
            if (m_MemoryManager[i].Includes(v))
            {
                return m_MemoryManager[i];
            }
        }
        NN_KERN_ABORT();
    }
};


}}

