﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_Platform.h"
#include "kern_KMemoryBlockManager.h"
#include "kern_KPageTableBase.h"

#define DEBUG_ASSERT(x) // NN_KERN_ASSERT(x)

namespace nn { namespace kern {

namespace
{
#ifdef NN_KERN_FOR_DEVELOPMENT
void DumpMemoryInfo(const KMemoryInfo& mi)
{
    NN_UNUSED(mi);
    struct MemoryStateName
    {
        const char* pName;
        KMemoryState state;
    };
    static const MemoryStateName memoryStateName[] =
    {
        {"-------- Free", KMemoryState_Free },
        {"Io           ", KMemoryState_Io },
        {"Static       ", KMemoryState_Static },
        {"Code         ", KMemoryState_Code },
        {"CodeData     ", KMemoryState_CodeData },
        {"Normal       ", KMemoryState_Normal },
        {"Shared       ", KMemoryState_Shared },
        {"AliasCode    ", KMemoryState_AliasCode },
        {"AliasCodeData", KMemoryState_AliasCodeData },
        {"Ipc          ", KMemoryState_Ipc },
        {"Stack        ", KMemoryState_Stack },
        {"ThreadLocal  ", KMemoryState_ThreadLocal },
        {"Transfered   ", KMemoryState_Transfered },
        {"ShTransfered ", KMemoryState_SharedTransfered },
        {"SharedCode   ", KMemoryState_SharedCode },
        {"NonSecureIpc ", KMemoryState_NonSecureIpc },
        {"NonDeviceIpc ", KMemoryState_NonDeviceIpc },
        {"CodeOut      ", KMemoryState_CodeOut },
        {"GeneratedCode", KMemoryState_GeneratedCode },
    };

    const char* stateString = "Unknown";
    for (size_t i = 0; i < sizeof(memoryStateName) / sizeof(*memoryStateName); i++)
    {
        if (mi.state == memoryStateName[i].state)
        {
            stateString = memoryStateName[i].pName;
            break;
        }
    }
    const char* rw = "---";
    const int32_t kb = mi.size / 1024;

    if( mi.state == KMemoryState_Free )
    {
        rw = "   ";
    }
    else
    {
        if( mi.permission == KMemoryPermission_UserReadExecute )
        {
            rw = "r-x";
        }
        else if( mi.permission == KMemoryPermission_UserReadWrite )
        {
            rw = "rw-";
        }
        else if( mi.permission == KMemoryPermission_UserRead )
        {
            rw = "r--";
        }
    }

    NN_KERN_RELEASE_LOG("%p - %p %9d KB %s %s %s%s%s%s (%d,%d)\n",
            mi.baseAddress, mi.baseAddress + mi.size - 1, kb, rw, stateString,
            (mi.attribute & KMemoryAttribute_Locked)?       "L": "-",
            (mi.attribute & KMemoryAttribute_IpcLocked)?    "I": "-",
            (mi.attribute & KMemoryAttribute_DeviceShared)? "D": "-",
            (mi.attribute & KMemoryAttribute_Uncached)?     "U": "-",
            mi.ipcLockCount, mi.deviceSharedCount);
}
#endif
}

Result KMemoryBlockManager::Initialize(KProcessAddress begin, KProcessAddress end, KMemoryBlockResourceManager* pMemoryBlockManager)
{
    // 全領域にあたる MemoryBlock を作る
    KMemoryBlock* pMemoryBlock = pMemoryBlockManager->Allocate();
    if (!pMemoryBlock)
    {
        return nn::svc::ResultOutOfResource();
    }

    pMemoryBlock->Initialize(begin, (end - begin) / NN_KERN_FINEST_PAGE_SIZE, KMemoryState_Free, KMemoryPermission_None, KMemoryAttribute_None);
    m_MemoryBlockList.insert(*pMemoryBlock);
    m_Begin = begin;
    m_End = end;

    return ResultSuccess();
}

void KMemoryBlockManager::Finalize(KMemoryBlockResourceManager* pMemoryBlockManager)
{
    // 全ての MemoryBlock を削除/解放する
    MemoryBlockList::iterator it = m_MemoryBlockList.begin();
    while (it != m_MemoryBlockList.end())
    {
        KMemoryBlock* p = &*it;
        it = m_MemoryBlockList.erase(it);
        pMemoryBlockManager->Free(static_cast<KMemoryBlock*>(p));
    }
    NN_KERN_ASSERT(m_MemoryBlockList.empty());
}

void KMemoryBlockManager::Update(KMemoryBlockManagerUpdateBuffer* pBuffer, KProcessAddress addr, size_t numPages, KMemoryState state, KMemoryPermission permission, KMemoryAttribute attribute)
{
    DEBUG_ASSERT(Check());
    NN_KERN_ASSERT((attribute & (KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared)) == 0);
    NN_KERN_ASSERT(addr % NN_KERN_FINEST_PAGE_SIZE == 0);

    KProcessAddress curAddr = addr;
    MemoryBlockList::iterator it = FindIterator(addr);

    size_t leftPages = numPages;

    while (leftPages)
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        if (it->Is(state, permission, attribute))
        {
            if (curAddr + leftPages * NN_KERN_FINEST_PAGE_SIZE < mi.baseAddress + mi.size)
            {
                leftPages = 0;
                curAddr += leftPages * NN_KERN_FINEST_PAGE_SIZE;
            }
            else
            {
                leftPages = ((curAddr + leftPages * NN_KERN_FINEST_PAGE_SIZE) - (mi.baseAddress + mi.size)) / NN_KERN_FINEST_PAGE_SIZE;
                curAddr = mi.baseAddress + mi.size;
            }
        }
        else
        {
            if (mi.baseAddress != GetAsInteger(curAddr))
            {
                KMemoryBlock* pMb = pBuffer->Allocate();

                it->Split(pMb, curAddr);
                it = m_MemoryBlockList.insert(*pMb);
                it++;
                mi = it->GetMemoryInfo();
                curAddr = mi.baseAddress;
            }

            if (mi.size > leftPages * NN_KERN_FINEST_PAGE_SIZE)
            {
                KMemoryBlock* pMb = pBuffer->Allocate();
                it->Split(pMb, curAddr + leftPages * NN_KERN_FINEST_PAGE_SIZE);
                it = m_MemoryBlockList.insert(*pMb);
                mi = it->GetMemoryInfo();
            }
            it->Update(state, permission, attribute);
            leftPages -= mi.size / NN_KERN_FINEST_PAGE_SIZE;
            curAddr += mi.size;
        }
        it++;
    }

    it = FindIterator(addr);
    if (addr != m_Begin)
    {
        it--;
    }

    for (;;)
    {
        MemoryBlockList::iterator prev = it++;
        if (it == m_MemoryBlockList.end())
        {
            break;
        }
        if (prev->IsSameProperty(*it))
        {
            size_t addPages = it->GetNumPages();
            KMemoryBlock* p = &*it;
            m_MemoryBlockList.erase(it);
            pBuffer->Free(p);
            prev->Add(addPages);
            it = prev;
        }
        KMemoryInfo mi = it->GetMemoryInfo();
        if (KProcessAddress(mi.baseAddress + mi.size) > addr + numPages * NN_KERN_FINEST_PAGE_SIZE)
        {
            break;
        }
    }

    DEBUG_ASSERT(Check());
}

void KMemoryBlockManager::UpdateIfMatch(KMemoryBlockManagerUpdateBuffer* pBuffer, KProcessAddress addr, size_t numPages,
            KMemoryState testState, KMemoryPermission testPermission, KMemoryAttribute testAttribute,
            KMemoryState state, KMemoryPermission permission, KMemoryAttribute attribute)
{
    DEBUG_ASSERT(Check());
    NN_KERN_ASSERT((attribute & (KMemoryAttribute_IpcLocked | KMemoryAttribute_DeviceShared)) == 0);
    NN_KERN_ASSERT(addr % NN_KERN_FINEST_PAGE_SIZE == 0);

    KProcessAddress curAddr = addr;
    MemoryBlockList::iterator it = FindIterator(addr);

    size_t leftPages = numPages;

    while (leftPages)
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        if (it->Is(testState, testPermission, testAttribute) && !it->Is(state, permission, attribute))
        {
            if (mi.baseAddress != GetAsInteger(curAddr))
            {
                KMemoryBlock* pMb = pBuffer->Allocate();

                it->Split(pMb, curAddr);
                it = m_MemoryBlockList.insert(*pMb);
                it++;
                mi = it->GetMemoryInfo();
                curAddr = mi.baseAddress;
            }

            if (mi.size > leftPages * NN_KERN_FINEST_PAGE_SIZE)
            {
                KMemoryBlock* pMb = pBuffer->Allocate();
                it->Split(pMb, curAddr + leftPages * NN_KERN_FINEST_PAGE_SIZE);
                it = m_MemoryBlockList.insert(*pMb);
                mi = it->GetMemoryInfo();
            }
            it->Update(state, permission, attribute);
            leftPages -= mi.size / NN_KERN_FINEST_PAGE_SIZE;
            curAddr += mi.size;
        }
        else
        {
            if (curAddr + leftPages * NN_KERN_FINEST_PAGE_SIZE < mi.baseAddress + mi.size)
            {
                leftPages = 0;
                curAddr += leftPages * NN_KERN_FINEST_PAGE_SIZE;
            }
            else
            {
                leftPages = ((curAddr + leftPages * NN_KERN_FINEST_PAGE_SIZE) - (mi.baseAddress + mi.size)) / NN_KERN_FINEST_PAGE_SIZE;
                curAddr = mi.baseAddress + mi.size;
            }
        }
        it++;
    }

    it = FindIterator(addr);
    if (addr != m_Begin)
    {
        it--;
    }

    for (;;)
    {
        MemoryBlockList::iterator prev = it++;
        if (it == m_MemoryBlockList.end())
        {
            break;
        }
        if (prev->IsSameProperty(*it))
        {
            size_t addPages = it->GetNumPages();
            KMemoryBlock* p = &*it;
            m_MemoryBlockList.erase(it);
            pBuffer->Free(p);
            prev->Add(addPages);
            it = prev;
        }
        KMemoryInfo mi = it->GetMemoryInfo();
        if (KProcessAddress(mi.baseAddress + mi.size) > addr + numPages * NN_KERN_FINEST_PAGE_SIZE)
        {
            break;
        }
    }

    DEBUG_ASSERT(Check());
}

const KMemoryBlock* KMemoryBlockManager::FindByAddress(KProcessAddress addr) const
{
    MemoryBlockList::const_iterator it = FindIterator(addr);

    if (it != m_MemoryBlockList.end())
    {
        return &*it;
    }

    return nullptr;
}

KMemoryBlockManager::MemoryBlockList::iterator KMemoryBlockManager::FindIterator(KProcessAddress addr) const
{
    KMemoryBlock dummy;
    dummy.Initialize(addr, 1, KMemoryState_Free, KMemoryPermission_None, KMemoryAttribute_None);

    return m_MemoryBlockList.find(dummy);
}

KProcessAddress KMemoryBlockManager::FindFreeAreaInRegion(KProcessAddress regionBegin, size_t regionNumPages, size_t numPages, size_t align, size_t alignOffset, size_t guardPages) const
{
    if (numPages)
    {
        KMemoryInfo info;
        KProcessAddress begin = GetAsInteger(regionBegin);
        KProcessAddress end   = GetAsInteger(regionBegin + regionNumPages * NN_KERN_FINEST_PAGE_SIZE) - 1;

        for (MemoryBlockList::const_iterator it = FindIterator(regionBegin); it != m_MemoryBlockList.end(); it++)
        {
            info = it->GetMemoryInfo();
            if (end < info.baseAddress)
            {
                break;
            }
            if (info.state == KMemoryState_Free)
            {
                KProcessAddress allocBase = (begin < info.baseAddress) ? info.baseAddress : begin;

                allocBase += guardPages * NN_KERN_FINEST_PAGE_SIZE;

                KProcessAddress tmp = (allocBase & ~(align - 1)) + alignOffset;

                allocBase = (allocBase <= tmp)? tmp : tmp + align;

                KProcessAddress allocEnd  = allocBase + numPages * NN_KERN_FINEST_PAGE_SIZE + (guardPages * NN_KERN_FINEST_PAGE_SIZE) - 1;

                if (info.baseAddress <= GetAsInteger(allocBase) && allocBase < allocEnd && allocEnd <= end && allocEnd <= info.baseAddress + info.size - 1)
                {
                    return allocBase;
                }
            }
        }
    }

    return Null<KProcessAddress>();
}

void KMemoryBlockManager::UpdateLock(KMemoryBlockManagerUpdateBuffer* pBuffer, KProcessAddress addr, size_t numPages,
        void (KMemoryBlock::*pLock)(KMemoryPermission newPermission), KMemoryPermission newPermission)
{
    DEBUG_ASSERT(Check());
    NN_KERN_ASSERT(addr % NN_KERN_FINEST_PAGE_SIZE == 0);

    KProcessAddress curAddr = addr;
    MemoryBlockList::iterator it = FindIterator(addr);
    MemoryBlockList::iterator prev = it;
    MemoryBlockList::iterator next = it;
    bool checkMergePrev = false;
    bool checkMergeNext = false;

    size_t leftPages = numPages;

    while (leftPages)
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        if (mi.baseAddress != GetAsInteger(curAddr))
        {
            KMemoryBlock* pMb = pBuffer->Allocate();

            it->Split(pMb, curAddr);
            it = m_MemoryBlockList.insert(*pMb);
            it++;
            mi = it->GetMemoryInfo();
            curAddr = mi.baseAddress;
        }
        else if (curAddr == addr && m_Begin != curAddr)
        {
            checkMergePrev = true;
            --prev;
        }

        if (mi.size > leftPages * NN_KERN_FINEST_PAGE_SIZE)
        {
            KMemoryBlock* pMb = pBuffer->Allocate();
            it->Split(pMb, curAddr + leftPages * NN_KERN_FINEST_PAGE_SIZE);
            it = m_MemoryBlockList.insert(*pMb);
            mi = it->GetMemoryInfo();
        }
        else if (mi.size == leftPages * NN_KERN_FINEST_PAGE_SIZE)
        {
            next = it;
            ++next;
            if (next != m_MemoryBlockList.end())
            {
                checkMergeNext = true;
            }
        }
        ((&*it)->*pLock)(newPermission);
        leftPages -= mi.size / NN_KERN_FINEST_PAGE_SIZE;
        curAddr += mi.size;
        it++;
    }

    if (checkMergePrev)
    {
        it = prev;
        it++;
        if (prev->IsSameProperty(*it))
        {
            size_t addPages = it->GetNumPages();
            KMemoryBlock* p = &*it;
            m_MemoryBlockList.erase(it);
            pBuffer->Free(p);
            prev->Add(addPages);
        }
    }
    if (checkMergeNext)
    {
        it = next;
        it--;
        if (it->IsSameProperty(*next))
        {
            size_t addPages = next->GetNumPages();
            KMemoryBlock* p = &*next;
            m_MemoryBlockList.erase(next);
            pBuffer->Free(p);
            it->Add(addPages);
        }
    }

    DEBUG_ASSERT(Check());
}

void KMemoryBlockManager::Dump() const
{
#ifdef NN_KERN_FOR_DEVELOPMENT
    for (MemoryBlockList::const_iterator it = m_MemoryBlockList.begin(); it != m_MemoryBlockList.end(); it++)
    {
        KMemoryInfo mi = it->GetMemoryInfo();
        DumpMemoryInfo(mi);
    }
#endif
}

bool KMemoryBlockManager::Check() const
{
    MemoryBlockList::const_iterator it = m_MemoryBlockList.begin();
    MemoryBlockList::const_iterator prev = it++;
    while (it != m_MemoryBlockList.end())
    {
        KMemoryInfo prevInfo = prev->GetMemoryInfo();
        KMemoryInfo info = it->GetMemoryInfo();
        if (prev->IsSameProperty(*it))
        {
            Dump();
            return false;
        }
        if (prevInfo.baseAddress + prevInfo.size != info.baseAddress)
        {
            Dump();
            return false;
        }
        if ((info.attribute & (KMemoryAttribute_IpcLocked)) != 0 && info.ipcLockCount == 0)
        {
            Dump();
            return false;
        }
        if ((info.attribute & (KMemoryAttribute_DeviceShared)) != 0 && info.deviceSharedCount == 0)
        {
            Dump();
            return false;
        }
        if ((info.attribute & (KMemoryAttribute_IpcLocked)) == 0 && info.ipcLockCount != 0)
        {
            Dump();
            return false;
        }
        if ((info.attribute & (KMemoryAttribute_DeviceShared)) == 0 && info.deviceSharedCount != 0)
        {
            Dump();
            return false;
        }
        prev = it++;
    }
    return true;
}

}}

