﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_KMemoryBlock.h"

namespace nn { namespace kern {

bool KMemoryBlock::Overlaps(KProcessAddress addr, size_t numPages) const
{
    NN_KERN_THIS_ASSERT();
    // m_Addr + m_NumPages * PageSize が 0x1_0000_0000 になると 0 になることへの対策
    KProcessAddress farRight  = (addr - 1) + numPages * NN_KERN_FINEST_PAGE_SIZE;
    KProcessAddress farLeft   = addr;

    KProcessAddress farRightSelf = (m_Addr - 1) + m_NumPages * NN_KERN_FINEST_PAGE_SIZE;
    KProcessAddress farLeftSelf  = m_Addr;

    return !(farRightSelf <= farLeft || farRight <= farLeftSelf);
}

bool KMemoryBlock::Includes(KProcessAddress addr, size_t numPages) const
{
    NN_KERN_THIS_ASSERT();
    // m_Addr + m_NumPages * PageSize が 0x1_0000_0000 になると 0 になることへの対策
    KProcessAddress farRight  = (addr - 1) + numPages * NN_KERN_FINEST_PAGE_SIZE;
    KProcessAddress farLeft   = addr;

    KProcessAddress farRightSelf = (m_Addr - 1) + m_NumPages * NN_KERN_FINEST_PAGE_SIZE;
    KProcessAddress farLeftSelf  = m_Addr;

    return (farLeftSelf <= farLeft && farRight <= farRightSelf);
}

bool KMemoryBlock::Includes(KProcessAddress addr) const
{
    NN_KERN_THIS_ASSERT();
    // m_Addr + m_NumPages * PageSize が 0x1_0000_0000 になると 0 になることへの対策
    const KProcessAddress lastAddr = (m_Addr - 1) + m_NumPages * NN_KERN_FINEST_PAGE_SIZE;
    return m_Addr <= addr && addr <= lastAddr;
}

bool KMemoryBlock::IsIncluded(KProcessAddress addr, size_t numPages) const
{
    NN_KERN_THIS_ASSERT();
    // m_Addr + m_NumPages * PageSize が 0x1_0000_0000 になると 0 になることへの対策
    KProcessAddress farRight  = (addr - 1) + numPages * NN_KERN_FINEST_PAGE_SIZE;
    KProcessAddress farLeft   = addr;

    KProcessAddress farRightSelf = (m_Addr - 1) + m_NumPages * NN_KERN_FINEST_PAGE_SIZE;
    KProcessAddress farLeftSelf  = m_Addr;

    return (farLeft <= farLeftSelf && farRightSelf <= farRight);
}


KMemoryInfo KMemoryBlock::GetMemoryInfo() const
{
    NN_KERN_THIS_ASSERT();
    KMemoryInfo info;
    info.baseAddress        = GetAsInteger(m_Addr);
    info.permission         = m_Permission;
    info.attribute          = m_Attribute;
    info.permissionOrig     = m_PermissionOrig;
    info.size               = m_NumPages * NN_KERN_FINEST_PAGE_SIZE;
    info.state              = m_MemoryState;
    info.ipcLockCount       = m_IpcLockCount;
    info.deviceSharedCount  = m_DeviceSharedCount;

    return info;
}

void KMemoryBlock::Split(KMemoryBlock* pBlock, KProcessAddress addr)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(m_Addr < addr);
    NN_KERN_ASSERT(addr <= m_Addr + m_NumPages * NN_KERN_FINEST_PAGE_SIZE - 1);
    pBlock->m_Addr = m_Addr;
    pBlock->m_NumPages = (addr - m_Addr) / NN_KERN_FINEST_PAGE_SIZE;
    pBlock->m_Permission = m_Permission;
    pBlock->m_PermissionOrig = m_PermissionOrig;
    pBlock->m_MemoryState = m_MemoryState;
    pBlock->m_Attribute = m_Attribute;
    pBlock->m_IpcLockCount = m_IpcLockCount;
    pBlock->m_DeviceSharedCount = m_DeviceSharedCount;

    m_Addr = GetAsInteger(addr);
    m_NumPages -= pBlock->m_NumPages;
}

void KMemoryBlock::Add(size_t numPages)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(numPages > 0);
    NN_KERN_ASSERT(m_Addr + m_NumPages * NN_KERN_FINEST_PAGE_SIZE - 1 < m_Addr + (m_NumPages + numPages) * NN_KERN_FINEST_PAGE_SIZE - 1);

    m_NumPages += numPages;
}

}}
