﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*!
    @file

    @brief LightMutex に関する API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_InterlockedSelect.h"
#include "kern_Assert.h"
#include "kern_Current.h"
#include "kern_CPUSelect.h"

namespace nn { namespace kern {

class KThread;

/*!
    @brief      LightMutex クラスの定義です。

    LightMutexとは、スレッドやCPU間で排他制御をするための同期プリミティブです。

    プログラムの特定の箇所について複数のスレッドから同時実行されることを抑止します。
    これまでのプラットフォームでは "割り込みの禁止" という形で保護してきたものです。

    スレッド優先度の継承処理を実現しています。

*/

class KLightMutex
{
private:
    InterlockedVariable<uintptr_t> m_pHeldThread;
    struct UpdateLock
    {
        uintptr_t val;
        uintptr_t old;
        bool operator()(uintptr_t* x)
        {
            this->old = *x;
            if (*x == 0)
            {
                *x = this->val;
            }
            else
            {
                *x |= 0x1;
            }
            return true;
        }
    };


public:
    KLightMutex() : m_pHeldThread(0) {}
    ~KLightMutex(){}

    void    Lock()
    {
#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
        LockAssert();
#endif
        uintptr_t pCurrentThread = reinterpret_cast<uintptr_t>(&GetCurrentThread());
        for (;;)
        {
            UpdateLock f;
            f.val = pCurrentThread;
            m_pHeldThread.AtomicUpdateConditional(&f);
            if (NN_LIKELY((f.old == 0) || ((f.old | 0x1) == (pCurrentThread | 0x1))))
            {
                break;
            }

            Lock2((f.old | 0x1), pCurrentThread);
        }
        KCPU::DataMemoryBarrier();
    }

    void  Unlock()
    {
        NN_KERN_THIS_ASSERT();

        // 現在のスレッドがこの Mutex を Lock していなければならない
        NN_KERN_ASSERT(IsLocked());
        NN_KERN_ASSERT(IsLockedByMe());

        KCPU::DataMemoryBarrier();
        uintptr_t pCurrentThread = reinterpret_cast<uintptr_t>(&GetCurrentThread());
        if (m_pHeldThread.CompareAndSwap(pCurrentThread, 0) != pCurrentThread)
        {
            Unlock2(pCurrentThread);
        }
    }

    // CHECK: 関数にconstの追加を推奨します
    KThread*    GetOwnerThread() const { return reinterpret_cast<KThread*>(m_pHeldThread & ~0x1ul); }
    bool        IsLocked() const { return !!m_pHeldThread; }
    bool        IsLockedByMe() const { return (m_pHeldThread | 0x1) == (reinterpret_cast<uintptr_t>(&GetCurrentThread()) | 0x1); }

private:
    //!< オブジェクトの診断
    // CHECK: 関数にconstの追加を推奨します
    void    LockAssert();

    //!< ロックしているスレッドが存在する場合の追加処理
    void    Lock2(uintptr_t ownerValue, uintptr_t currentValue);

    //!< ロックの解除待ちスレッドが存在する場合の追加処理
    void    Unlock2(uintptr_t currentValue);
};

/*!
    @brief      スコープつき LightMutex クラスの定義です。

    オブジェクトが存在する間だけ LightMutex を Lock 状態にします。

*/

class KScopedLightLock
{
private:
    KLightMutex* m_pMutex;

public:
    explicit KScopedLightLock(KLightMutex* pMutex) : m_pMutex(pMutex)
    {
        NN_KERN_POINTER_ASSERT(pMutex);
        m_pMutex->Lock();
    }

    ~KScopedLightLock()
    {
        m_pMutex->Unlock();
    }
};


}}

