﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*!
    @file

    @brief LightMutex に関する API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_InterlockedSelect.h"
#include "kern_Assert.h"
#include "kern_Current.h"
#include "kern_CPUSelect.h"
#include "kern_KLightMutex.h"
#include "kern_KThreadQueue.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KScopedSchedulingLockAndSleep.h"

namespace nn { namespace kern {

class KLightConditionvariable
{
public:
    void Wait(KLightMutex* pMutex, int64_t timeout)
    {
        WaitImpl(pMutex, timeout);
        pMutex->Lock();
    }

    void Broadcast()
    {
        KScopedSchedulingLock locker;
        while (m_ThreadQueue.WakupFrontThread() != nullptr) {}
    }

private:
    void WaitImpl(KLightMutex* pMutex, int64_t timeout)
    {
        // WaitConditionVariable が呼ばれるのは lock 獲得中.
        KThread* pOwnerThread = &GetCurrentThread();
        KHardwareTimer* pTimer;
        {
            KScopedSchedulingLockAndSleep sleep(&pTimer, pOwnerThread, timeout);
            pMutex->Unlock();

            if (!m_ThreadQueue.SleepThread(pOwnerThread))
            {
                sleep.Cancel();
                return;
            }
        }

        if (pTimer != nullptr)
        {
            pTimer->CancelTask(pOwnerThread);
        }
    }

private:
    KThreadQueue m_ThreadQueue;
};

}}

