﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*!
    @file

    @brief 割り込みイベントに関する API の宣言

*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Assert.h"
#include "kern_KReadableEvent.h"
#include "kern_KInterruptTask.h"
#include "kern_KObjectAdaptor.h"
#include "kern_KAutoObject.h"
#include "kern_KUseSlabAllocator.h"

namespace nn { namespace kern {

class KProcess;

/*!
  @brief      割り込みイベント クラスの定義です。

  割り込みイベント とは、VBlank割り込み、タイマー割り込みなど 個々の割り込みのことです。
  IRQ 番号に対して 1 つずつ割り当てされます。

*/
class KInterruptEventTask;
class KInterruptEvent :
    public KObjectAdaptor<KInterruptEvent, KReadableEvent>
{
public:
    explicit KInterruptEvent() : m_pTask(nullptr), m_IrqNo(-1), m_IsInitialized(false)
    {
        NN_KERN_THIS_ASSERT();
    }
    virtual ~KInterruptEvent() {}

    Result Initialize(int32_t name, nn::svc::InterruptType type);
    virtual void Finalize();

    virtual Result Reset();

    virtual bool IsInitialized() const { return m_IsInitialized; }
    static void PostFinalize(uintptr_t arg) { NN_UNUSED(arg); }

    int GetIrqNo() const { return m_IrqNo; }
private:
    KInterruptEventTask*    m_pTask;
    int32_t                 m_IrqNo;
    bool                    m_IsInitialized;

    //! KAutoObjectのプリセット関数セット定義です。クラスの末尾に記述する必要があります
    NN_AUTOOBJECT_DERIVED_FUNCSET(KInterruptEvent, KReadableEvent)
};



class KInterruptEvent;
class KInterruptEventTask :
    public KUseSlabAllocator<KInterruptEventTask>,
    public KInterruptTask
{
public:
    KInterruptEventTask(): m_pEvent(nullptr), m_InterruptNo(-1)
    {
        NN_KERN_THIS_ASSERT();
    }
    ~KInterruptEventTask()
    {
        NN_KERN_THIS_ASSERT();
    }
    virtual void  DoInterruptTask();
    virtual KInterruptTask* OnInterrupt(int32_t interruptNo);
    static Result Register(KInterruptEventTask** pOut, int32_t interruptNo, bool isLevel, KInterruptEvent* pEvent);
    void Unregister();
private:
    KInterruptEvent* m_pEvent;
    int32_t m_InterruptNo;
};


}}

