﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KInterruptEvent.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_KProcess.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KLightMutex.h"

namespace nn { namespace kern {
namespace {
KLightMutex s_Lock;
KInterruptEventTask* s_InterruptTaskTable[KInterruptController::NumInterrupts];
}
NN_AUTOOBJECT_DEFINE_TYPE_NAME(KInterruptEvent);

Result KInterruptEvent::Initialize(int32_t name, nn::svc::InterruptType type)
{
    NN_KERN_THIS_ASSERT();

    m_IrqNo = name;
    KReadableEvent::Initialize(nullptr);

    Result result = KInterruptEventTask::Register(&m_pTask, m_IrqNo, type == nn::svc::InterruptType_Level, this);
    if (result.IsSuccess())
    {
        m_IsInitialized = true;
    }

    return result;
}

void KInterruptEvent::Finalize()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(m_pTask);
    m_pTask->Unregister();
}

Result KInterruptEvent::Reset()
{
    NN_KERN_THIS_ASSERT();

    KScopedLightLock locker(&s_Lock);

    Result result = KReadableEvent::Reset();

    if (result.IsSuccess())
    {
        Kernel::GetInterruptManager().ClearInterrupt(m_IrqNo);
    }
    return result;
}

Result KInterruptEventTask::Register(KInterruptEventTask** pOut, int32_t interruptNo, bool isLevel, KInterruptEvent* pEvent)
{
    // Capability で制限されているのでこないはず。
    if (!(Kernel::GetInterruptManager().IsDefinedInterrupt(interruptNo) && Kernel::GetInterruptManager().IsGlobal(interruptNo)))
    {
        return nn::svc::ResultOutOfRange();
    }

    KScopedLightLock locker(&s_Lock);

    KInterruptEventTask* pTask = s_InterruptTaskTable[interruptNo];
    bool isAllocate = false;
    if (pTask)
    {
        if (pTask->m_pEvent)
        {
            return nn::svc::ResultBusy();
        }
    }
    else
    {
        pTask = KInterruptEventTask::Allocate();
        if (!pTask)
        {
            return nn::svc::ResultOutOfResource();
        }
        isAllocate = true;
    }

    const int32_t coreNo = GetCurrentCpuNo();

    Result result = Kernel::GetInterruptManager().BindHandler(pTask, interruptNo, coreNo, KInterruptController::PriorityLevel_DevInterrupt, true, isLevel);
    if (result.IsSuccess())
    {
        pTask->m_pEvent = pEvent;

        if (isAllocate)
        {
            pTask->m_InterruptNo = interruptNo;
            s_InterruptTaskTable[interruptNo] = pTask;
        }
        *pOut = pTask;
    }
    else
    {
        if (isAllocate)
        {
            KInterruptEventTask::Free(pTask);
        }
    }

    return result;
}

void KInterruptEventTask::Unregister()
{
    NN_KERN_THIS_ASSERT();
    KScopedLightLock locker(&s_Lock);

    NN_KERN_ABORT_UNLESS(this == s_InterruptTaskTable[m_InterruptNo]);
    NN_KERN_ABORT_UNLESS(m_pEvent);
    m_pEvent = nullptr;
    const int32_t coreNo = GetCurrentCpuNo();
    Kernel::GetInterruptManager().UnbindHandler(m_InterruptNo, coreNo);
}

KInterruptTask* KInterruptEventTask::OnInterrupt(int32_t)
{
    NN_KERN_THIS_ASSERT();
    return this;
}

void KInterruptEventTask::DoInterruptTask()
{
    NN_KERN_THIS_ASSERT();

    KScopedLightLock locker(&s_Lock);

    if (m_pEvent)
    {
        m_pEvent->Signal();
    }
}


}}
