﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Assert.h"
#include "kern_KAutoObject.h"
#include "kern_KSimpleLock.h"
#include "kern_KScopedDisableDispatch.h"

namespace nn { namespace kern {

inline Bit32 HandleToBit32(nn::svc::Handle handle)
{
    return reinterpret_cast<Bit32&>(handle);
}

/*! @file
    @brief      ハンドルテーブル クラスの定義です。

*/

class KHandleTable
{
private:
#define MAKE_BIT_MASK(bits, shift) ((~0u >> (32 - bits - shift)) << shift)
    static const int32_t HANDLE_INDEX_SHIFT     =  0;
    static const int32_t HANDLE_SERIAL_ID_SHIFT = 15;
    static const int32_t HANDLE_FLAGS_SHIFT     = 30;

    static const int32_t HANDLE_INDEX_BITS      = 15;
    static const int32_t HANDLE_SERIAL_ID_BITS  = 15;
    static const int32_t HANDLE_FLAGS_BITS      =  2;

    static const Bit32 HANDLE_INDEX_MASK      = MAKE_BIT_MASK(HANDLE_INDEX_BITS,     HANDLE_INDEX_SHIFT);
    static const Bit32 HANDLE_SERIAL_ID_MASK  = MAKE_BIT_MASK(HANDLE_SERIAL_ID_BITS, HANDLE_SERIAL_ID_SHIFT);
    static const Bit32 HANDLE_FLAGS_MASK      = MAKE_BIT_MASK(HANDLE_FLAGS_BITS,     HANDLE_FLAGS_SHIFT);

    static const int32_t SERIAL_ID_MAX          = static_cast<int32_t>(MAKE_BIT_MASK(HANDLE_SERIAL_ID_BITS, 0));

    // Index(15bit), serialId(15bit), flags(2bit) からハンドル値を生成します
    static Bit32 MakeHandleValue(int32_t index, int32_t serialId, int32_t flags)
    {
        return ( ((index    << HANDLE_INDEX_SHIFT)     & HANDLE_INDEX_MASK)
               | ((serialId << HANDLE_SERIAL_ID_SHIFT) & HANDLE_SERIAL_ID_MASK)
               | ((flags    << HANDLE_FLAGS_SHIFT)     & HANDLE_FLAGS_MASK) );
    }
    //! ハンドル値から Index を取得します
    static int32_t GetHandleIndex(Bit32 handleValue)
    {
        return (handleValue & HANDLE_INDEX_MASK) >> HANDLE_INDEX_SHIFT;
    }
    //! ハンドル値から SerialId を取得します
    static int32_t GetHandleSerialId(Bit32 handleValue)
    {
        return (handleValue & HANDLE_SERIAL_ID_MASK) >> HANDLE_SERIAL_ID_SHIFT;
    }
    //! ハンドル値から Flags を取得します
    static int32_t GetHandleFlags(Bit32 handleValue)
    {
        return (handleValue & HANDLE_FLAGS_MASK) >> HANDLE_FLAGS_SHIFT;
    }

    struct TableEntry
    {
    private:
        union
        {
            struct
            {
                uint16_t serialId;
                Bit16    type;
            }
            objectInfo;

            TableEntry* pNextFreeEntry;
        }
        m_Info;

        KAutoObject* m_pObject;

    public:
        /*!
            @brief  テーブルエントリーをフリー状態にします

            @param[in]  pNext   フリーエントリー

        */
        void SetFree(TableEntry* pNext)
        {
            NN_KERN_THIS_ASSERT();
            m_pObject = nullptr;
            m_Info.pNextFreeEntry = pNext;
            NN_KERN_ASSERT(GetFreeEntry() == pNext);
        }

        /*!
            @brief  テーブルエントリーを使用中の状態にします

            @param[in]  p           テーブルに関連付けするオブジェクト
            @param[in]  serialId    シリアルID
            @param[in]  type        オブジェクトの種別

        */
        void SetUsing(KAutoObject* p, uint16_t serialId, Bit16 type)
        {
            NN_KERN_THIS_ASSERT();
            m_pObject = p;
            m_Info.objectInfo.serialId = serialId;
            m_Info.objectInfo.type = type;
            NN_KERN_ASSERT(GetObject() == p);
        }

        KAutoObject* GetObject() const { return m_pObject; }
        TableEntry* GetFreeEntry() const { return m_Info.pNextFreeEntry; }
        uint16_t GetSerialId() const { return m_Info.objectInfo.serialId; }
        Bit16 GetType() const { return m_Info.objectInfo.type; }
    };

private:
    TableEntry*     m_pTable;               //!< テーブルのエントリー
    TableEntry*     m_pFreeHead;            //!< テーブルのフリーエントリー
    TableEntry      m_Table[NN_KERN_HANDLE_TABLE_SIZE];
    uint16_t        m_TableSize;            //!< テーブル数
    uint16_t        m_MaxCount;             //!< テーブルの使用上限数
    uint16_t        m_NextSerialId;         //!< 次に使用するシリアルID
    uint16_t        m_Count;                //!< テーブルの使用数
    mutable KSimpleLock m_Lock;

private:

    /*!
        @brief     ハンドル値からテーブルエントリーを取得します

        @param[in]  handle   ハンドル

        @return    ハンドル値が正しい場合テーブルエントリーを、誤っている場合nullptrを返します。

    */
    TableEntry*     FindEntry(nn::svc::Handle handle) const
    {
        NN_KERN_THIS_ASSERT();

        const Bit32 handleValue = HandleToBit32(handle);
        const int32_t index    = GetHandleIndex(handleValue);
        const int32_t serialId = GetHandleSerialId(handleValue);

        NN_KERN_ASSERT(GetHandleFlags(handleValue) == 0);

        // ハンドル値を確認します
        // ありえない数字になっているものはnullptrを返します
        if (handleValue == 0)
        {
            return nullptr;
        }
        if (serialId == 0)
        {
            return nullptr;
        }
        if (index >= m_TableSize)
        {
            return nullptr;
        }

        // エントリ取得
        TableEntry& entry = m_pTable[index];

        // ハンドルエントリーの使用状況を確認する
        if (entry.GetObject() == nullptr)
        {
            // このハンドルは使用されていない
            return nullptr;
        }

        // シリアルIDに相違がある
        if (entry.GetSerialId() != serialId)
        {
            // 以前使用していたハンドルでアクセスしようとしたため
            // オブジェクトを得ることができないようにしておく
            return nullptr;
        }

        return &entry;
    }

    /*!
        @brief     ハンドルに対応したオブジェクトを取得します

        @param[in]  handle   ハンドル

        @return    ハンドル値が正しい場合、対応するオブジェクトを返します。ハンドルが不正な場合はnullptrを返します。

    */
    KAutoObject* GetObjectImpl(nn::svc::Handle handle) const
    {
        NN_KERN_THIS_ASSERT();

        if (GetHandleFlags(HandleToBit32(handle)) != 0)
        {
            return nullptr;
        }

        TableEntry* pEntry = FindEntry(handle);

        if (pEntry != nullptr)
        {
            return pEntry->GetObject();
        }
        else
        {
            return nullptr;
        }
    }

    // デバッグ用関数
    KAutoObject* GetObjectByIndexImpl(nn::svc::Handle* pHandle, int index) const
    {
        NN_KERN_THIS_ASSERT();

        if (index >= m_TableSize || !m_pTable)
        {
            return nullptr;
        }

        TableEntry& entry = m_pTable[index];
        if (entry.GetObject() == nullptr)
        {
            return nullptr;
        }

        int serialId = entry.GetSerialId();
        const Bit32 handleValue = MakeHandleValue(index, serialId, 0);

        *pHandle = nn::svc::Handle(handleValue);

        return entry.GetObject();
    }

    /*!
        @brief     オブジェクトを登録します

        @param[out]  pOut   オブジェクトに対応するハンドル
        @param[in]  obj     登録するオブジェクト
        @param[in]  type    登録するオブジェクトの型情報

        @return    正常に登録できた場合は成功を、テーブルが溢れた場合は SUMMARY_OUT_OF_RESOURCE を返します。

    */
    Result          Add(nn::svc::Handle* pOut, KAutoObject* obj, Bit16 type);
    void            Register(nn::svc::Handle handle, KAutoObject* obj, Bit16 type);

    /*!
        @brief     テーブルエントリーを確保します

        呼び出す前に必ずフリーエントリーがあるか確認する必要があります。

        @return     確保できたテーブル
    */
    TableEntry&     AllocateEntry();

    /*!
        @brief     フリーリストに追加します

        エントリーをフリーし、フリーリストの先頭にエントリーを追加します。

        @param[in]  pEntry フリー対象/リスト追加対象のエントリー

    */
    void            FreeEntry(TableEntry* pEntry);
    void            FreeEntryImpl(TableEntry* pEntry);

public:
    explicit KHandleTable();
    KHandleTable& operator=(const KHandleTable&) = delete;
    KHandleTable(const KHandleTable&) = delete;

    /*!
        @brief     テーブルを初期化します

        @param[in]  tableSize   テーブル数

        @return    初期化が正常に行われた場合は成功を、テーブル数が大きすぎる場合は SUMMARY_OUT_OF_RESOURCE を返します。

    */
    Result  Initialize(int32_t tableSize);

    /*!
        @brief     終了処理

        @return    SUMMARY_NOT_SUPPORTEDを返します。

    */
    Result  Finalize();

    /*!
        @brief     ハンドルを削除します

        @param[in]  handle   削除対象のハンドル

        @return    正常に削除できたら true を返します。

    */
    bool    Remove(nn::svc::Handle handle);

    /*!
        @brief     総テーブル数を取得します

        @return    総テーブル数

        CHECK: 関数にconstの追加を推奨します
    */
    int32_t     GetTableSize() { return m_TableSize; }

    /*!
        @brief     使用中のテーブル数を取得します

        @return    使用中のテーブル数

        CHECK: 関数にconstの追加を推奨します
    */
    int32_t     GetCount()     { return m_Count; }

    /*!
        @brief     最大テーブル数(テーブル使用数のピーク)を取得します

        @return    最大テーブル数

        CHECK: 関数にconstの追加を推奨します
    */
    int32_t     GetMaxCount()  { return m_MaxCount; }

#if NN_KERN_ENABLE_OBJECT_INFO
    /*!
        @brief     指定したタイプ IDを持つオブジェクトの数を取得します

                   TORIAEZU: 方法が泥臭いので後で何とかする

        @param[in] typeId    オブジェクトのタイプ ID
        @param[in] processId プロセス ID
                             0 以外を指定するとそのプロセスが作成したオブジェクトのみが対象になります。

        @return    オブジェクトの数
    */
    int32_t     GetObjectCount(Bit16 typeId, Bit32 processId = 0) const;
#endif

    /*!
        @brief     ハンドルに対応したオブジェクトを取得します

        @param[in]  handle   ハンドル

        @return    ハンドル値が正しい場合、対応するオブジェクトを返します。ハンドルが不正な場合はnullptrを返します。

    */
    KAutoObject* GetObject(nn::svc::Handle handle) const
    {
        NN_KERN_THIS_ASSERT();
        KScopedDisableDispatch dd;
        KScopedSimpleLock lock(&m_Lock);
        KAutoObject* pObject = GetObjectImpl(handle);
        if (pObject)
        {
            pObject->Open();
        }
        return pObject;
    }

    KAutoObject* GetObjectForIpc(nn::svc::Handle handle) const;

    // デバッグ用関数
    KAutoObject* GetObjectByIndex(nn::svc::Handle* pHandle, int index) const
    {
        NN_KERN_THIS_ASSERT();
        KScopedDisableDispatch dd;
        KScopedSimpleLock lock(&m_Lock);
        KAutoObject* pObject = GetObjectByIndexImpl(pHandle, index);
        if (pObject)
        {
            pObject->Open();
        }
        return pObject;
    }

    /*!
        @brief     オブジェクトを登録します

        @param[out]  pOut   オブジェクトに対応するハンドル
        @param[in]  pObj    登録するオブジェクト

        @return    正常に登録できた場合は成功を、テーブルが溢れた場合は SUMMARY_OUT_OF_RESOURCE を返します。

    */
    template <typename T>
    Result Add(nn::svc::Handle* pOut, T* pObj)
    {
        return Add(pOut, pObj, pObj->GetTypeObj().GetTypeID());
    }

    Result Reserve(nn::svc::Handle* pOut);
    template <typename T>
    void Register(nn::svc::Handle handle, T* pObj)
    {
        Register(handle, pObj, pObj->GetTypeObj().GetTypeID());
    }
    void Unreserve(nn::svc::Handle handle);


    /*!
        @brief     ハンドルに対応したオブジェクトを取得します

        @param[in]  handle   ハンドル

        @return    ハンドル値が正しい場合、対応するオブジェクトを返します。ハンドルが不正な場合はnullptrを返します。

    */
    template <class T>
    NN_FORCEINLINE T* GetObject(nn::svc::Handle handle) const
    {
        KScopedDisableDispatch dd;
        KScopedSimpleLock lock(&m_Lock);
        KAutoObject* pObject = GetObjectImpl(handle);
        if (pObject == nullptr)
        {
            return nullptr;
        }

        T* pT = pObject->DynamicCast<T*>();
        if( pT )
        {
            pObject->Open();
        }

        return pT;
    }

    template <class T>
    NN_FORCEINLINE bool GetObject(T** pSyncObjects, const nn::svc::Handle* pHandle, size_t numHandle) const
    {
        size_t i;
        {
            KScopedDisableDispatch dd;
            KScopedSimpleLock lock(&m_Lock);
            for (i = 0; i < numHandle; i++)
            {
                nn::svc::Handle handle = pHandle[i];
                asm volatile ("":::"memory");
                KAutoObject* pObject = GetObjectImpl(handle);
                if (NN_UNLIKELY(!pObject))
                {
                    break;
                }

                T* pT = pObject->DynamicCast<T*>();
                if (NN_UNLIKELY(!pT))
                {
                    break;
                }

                pObject->Open();
                pSyncObjects[i] = pT;
            }
        }

        if (NN_LIKELY(i == numHandle))
        {
            return true;
        }

        for (size_t j = 0; j < i; j++)
        {
            pSyncObjects[j]->Close();
        }

        return false;
    }

};
}}

