﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/* @file
  @brief 排他制御された変数を扱うクラスの宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/TargetConfigs/build_Cpu.h>
#include <type_traits>

#if defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
    #include "ip/ARMv7A/kern_Interlocked.h"
    namespace nn {
        namespace kern {
            using ARMv7A::Interlocked;
        }
    }
#elif   defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    #include "ip/ARMv8A/kern_Interlocked.h"
    namespace nn {
        namespace kern {
            using ARMv8A::Interlocked;
        }
    }
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7M)
    #include "ip/ARMv7M/kern_Interlocked.h"
    namespace nn {
        namespace kern {
            using ARMv7M::Interlocked;
        }
    }
#elif defined(NN_BUILD_CONFIG_CPU_PPC32)
    #include "PPC32/kern_Interlocked.h"
    namespace nn {
        namespace kern {
            using PPC32::Interlocked;
        }
    }
#else
    #error processor not selected
#endif

namespace nn { namespace kern {

namespace detail
{
    template <typename U, typename = void> struct StorageSelector;

    template <typename U> struct StorageSelector<U, typename std::enable_if<sizeof(U) == sizeof(int64_t)>::type>
    {
        typedef int64_t Type;
    };

    template <typename U> struct StorageSelector<U, typename std::enable_if<sizeof(U) == sizeof(int32_t)>::type>
    {
        typedef int32_t Type;
    };

    template <typename U> struct StorageSelector<U, typename std::enable_if<sizeof(U) == sizeof(int16_t)>::type>
    {
        typedef int16_t Type;
    };

    template <typename U> struct StorageSelector<U, typename std::enable_if<sizeof(U) == sizeof(int8_t)>::type>
    {
        typedef int8_t Type;
    };
}


/*
    @brief 排他制御された変数を扱うクラステンプレートです。

           テンプレート引数 T は POD である必要があります。
*/

template <typename T>
class InterlockedVariable
{
private:
    template <typename U> struct StorageSelector : public detail::StorageSelector<U>
    {
    };

private:
    volatile T m_v; //!< バリア変数

    // AtomicUpdateConditional 用関数オブジェクトの定義

#define NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC(F, op)                       \
    struct F                                                                \
    {                                                                       \
        T m_operand;                                                        \
        template <typename U> F(const U& operand) : m_operand(operand) {}   \
        bool operator()(T* x) { (*x) op m_operand; return true; }              \
    };

    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC(AssignFunc, =)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC(PlusFunc, +=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC(MinusFunc, -=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC(OrFunc, |=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC(AndFunc, &=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC(XorFunc, ^=)

#undef NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_FUNC

#define NN_UTIL_INTERLOCKED_DEFINE_UNARY_FUNC(F, preop, postop) \
    struct F                                                    \
    {                                                           \
        T result;                                               \
        bool operator()(T* x) { result = preop (*x) postop; return true; }   \
    };

    NN_UTIL_INTERLOCKED_DEFINE_UNARY_FUNC(PreIncFunc,  ++,   )
    NN_UTIL_INTERLOCKED_DEFINE_UNARY_FUNC(PreDecFunc,  --,   )
    NN_UTIL_INTERLOCKED_DEFINE_UNARY_FUNC(PostIncFunc,   , ++)
    NN_UTIL_INTERLOCKED_DEFINE_UNARY_FUNC(PostDecFunc,   , --)

#undef NN_UTIL_INTERLOCKED_DEFINE_UNARY_FUNC

    struct CompareAndSwapFunc
    {
        T m_comparand;
        T m_value;
        T m_result;
        bool operator()(T* x)
        {
            m_result = *x;
            if (*x == m_comparand)
            {
                *x = m_value;
                return true;
            }
            else
            {
                return false;
            }
        }
        CompareAndSwapFunc(T comparand, T value) : m_comparand(comparand), m_value(value) {}
    };

public:
    /*
        @brief コンストラクタです。

        @param[in] v 初期値
    */
    constexpr InterlockedVariable() : m_v() {}
    constexpr explicit InterlockedVariable(T v) : m_v(v) {}

    NN_IMPLICIT operator T() const
    {
        const volatile T& x = m_v;
        return x;
    }
    T operator ->()
    {
        const volatile T& x = m_v;
        return x;
    }

    /*
        @brief バリア変数を取得します。

        @return バリア変数を返します。
    */
    T Read() const { return *this; }

    /*
        @brief バリア変数を排他制御なしで変更します。

        @param[in] v 変更後の値
    */
    void WriteNotAtomic (T v) { m_v = v; }

    /*
        @brief 関数オブジェクトを排他制御下で実行します。

        @param[in] updater 関数オブジェクト

        @return 関数オブジェクトの実行の結果を返します。
    */
    template <class Updater>
    bool AtomicUpdateConditional(Updater* pUpdater) { return Interlocked::AtomicUpdate(&m_v, pUpdater); }

    /*
        @brief バリア変数と comparand を比較し、等しければバリア変数を value に変更します。

        @param[in] comparand 比較する値
        @param[in] value 等しい場合に設定する値

        @return 比較、代入を行う直前でのバリア変数を返します。
    */
    T CompareAndSwap(T comparand, T value)
    {
        CompareAndSwapFunc f(comparand, value);
        AtomicUpdateConditional(&f);
        return f.m_result;
    }

    /*
        @brief バリア変数と comparand を比較し、等しければバリア変数を value に変更します。

        @param[in] comparand 比較する値
        @param[in] value 等しい場合に設定する値

        @return value への変更が成功すれば 0 を返し、失敗すれば 1 を返します。
    */
    int CompareAndSwapWeak(T comparand, T value)
    {
        return Interlocked::CompareAndSwapWeak(&m_v, comparand, value);
    }

    /*
        @brief バリア変数と cmpValue が等しくなるまで待ち、等しくなれば内部を newValue に変更します。

        @param[in] comparand 比較する値
        @param[in] newValue 等しい場合に設定する値
    */
    void CompareAndSpinWaitAndSwap(T cmpValue, T newValue)
    {
        for(;;)
        {
            if( m_v == cmpValue )
            {
                if( CompareAndSwapWeak(cmpValue, newValue) == 0 )
                {
                    break;
                }
            }
        }
    }

    // 演算子のオーバーロード

#define NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR(F, op)                                       \
    template <typename V>                                                                       \
    void NN_IMPLICIT operator op(V v) { F f(v); AtomicUpdateConditional(&f); }

    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR(AssignFunc, =)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR(PlusFunc, +=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR(MinusFunc, -=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR(OrFunc, |=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR(AndFunc, &=)
    NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR(XorFunc, ^=)

#undef NN_UTIL_INTERLOCKED_DEFINE_ASSIGN_OPERATOR

#define NN_UTIL_INTERLOCKED_DEFINE_UNARY_OPERATOR(F, op, premarker)                                     \
    InterlockedVariable NN_IMPLICIT operator op(premarker) { F f; AtomicUpdateConditional(&f); return InterlockedVariable(f.result); }    \

    NN_UTIL_INTERLOCKED_DEFINE_UNARY_OPERATOR(PreIncFunc,  ++,    )
    NN_UTIL_INTERLOCKED_DEFINE_UNARY_OPERATOR(PreDecFunc,  --,    )
    NN_UTIL_INTERLOCKED_DEFINE_UNARY_OPERATOR(PostIncFunc, ++, int)
    NN_UTIL_INTERLOCKED_DEFINE_UNARY_OPERATOR(PostDecFunc, --, int)

#undef NN_UTIL_INTERLOCKED_DEFINE_UNARY_OPERATOR
};

}}

