﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_Platform.h"
#include "kern_InterlockedSelect.h"
#include "kern_Kernel.h"
#include "kern_Result.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_InterruptNameSelect.h"
#include "kern_KInterruptHandler.h"
#include "kern_CPUSelect.h"
#include "kern_DebugString.h"
#include "kern_MemoryMap.h"
#include "kern_KSimpleLock.h"
#include "kern_SystemControl.h"

#include <cstring>
#include <cstdarg>

#define NN_KERN_ENABLE_PRINT

#ifdef NN_KERN_ENABLE_PRINT
namespace nn {
namespace
{
    // 実際に文字列を出力します。
    void PutStringImpl(const char *str);

    // printf用一時バッファ
    static const int BUFFER_SIZE = 256;
    char s_CommonBuffer[nn::kern::KCPU::NUM_CORE][BUFFER_SIZE];
    bool g_Initialized = false;
    nn::kern::KSimpleLock s_Lock;
    uintptr_t g_UartBase;

#if defined NN_BUILD_CONFIG_HARDWARE_KZMA9  // ifdef NN_DEBUGGER_KMC_PARTNER
#define UART1_THR (*reinterpret_cast<volatile Bit8  *>(g_UartBase + 0x00))
#define UART1_IER (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x04))
#define UART1_IIR (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x08))
#define UART1_FCR (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x0c))
#define UART1_LCR (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x10))
#define UART1_MCR (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x14))
#define UART1_LSR (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x18))
#define UART1_MSR (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x1c))
#define UART1_SCR (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x20))
#define UART1_DLL (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x24))
#define UART1_DLM (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x28))
#define UART1_HCR0 (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x2c))
#define UART1_HCR2 (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x30))
#define UART1_HCR3 (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x34))
    void InitUart()
    {
        UART1_IER = 0;
        UART1_FCR = 0x27;
        UART1_LCR = 0x80;
        UART1_DLL = 0x1f;
        UART1_DLM = 0x00;
        UART1_LCR = 0x03;
        nn::kern::KCPU::DataSynchronizationBarrier();
    }
    void FlushUart()
    {
        while ((UART1_LSR & 0x40) == 0) {}
    }
    void PutCharImpl(char c)
    {
        while ((UART1_HCR3 & 0x7F) >= 0x3F) {}
        UART1_THR = c;
        nn::kern::KCPU::DataSynchronizationBarrier();
    }
    void SaveUart() {}
    void RestoreUart() {}

#elif \
    defined NN_BUILD_CONFIG_HARDWARE_SMMA53   || \
    defined NN_BUILD_CONFIG_HARDWARE_JUNO
#define UART0_DR   (*reinterpret_cast<volatile Bit8  *>(g_UartBase + 0x00))
#define UART0_RSR  (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x04))
#define UART0_FR   (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x18))
#define UART0_IBRD (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x24))
#define UART0_FBRD (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x28))
#define UART0_LCRH (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x2c))
#define UART0_CR   (*reinterpret_cast<volatile Bit32 *>(g_UartBase + 0x30))
    void InitUart()
    {
        const uint32_t baudRate = 115200;
#if defined NN_BUILD_CONFIG_HARDWARE_SMMA53
        const uint32_t uartClock = 24000000;
#elif defined NN_BUILD_CONFIG_HARDWARE_JUNO
        const uint32_t uartClock =  7273800;
#endif

        UART0_CR    = 0;
        UART0_IBRD  = 0;
        UART0_FBRD  = 0;
        UART0_LCRH  = 0;
        nn::kern::KCPU::DataSynchronizationBarrier();

        UART0_RSR   = 0;
        UART0_IBRD  = uartClock / (16 * baudRate);
        UART0_FBRD  = (64 * (uartClock % (16 * baudRate)) + 8 * baudRate) / (16 * baudRate);
        UART0_LCRH  = (3<<5) | (1<<4);    // 8bit, FIFO enable
        nn::kern::KCPU::DataSynchronizationBarrier();
        UART0_CR    = (1<<8) | (1<<0);    // send enable, uart enable
        nn::kern::KCPU::DataSynchronizationBarrier();
    }
    void FlushUart()
    {
        while (UART0_FR & (1<<3)) {}
    }
    void PutCharImpl(char c)
    {
        while (UART0_FR & (1<<5)) {}
        UART0_DR = c;
        nn::kern::KCPU::DataSynchronizationBarrier();
    }
    void SaveUart() {}
    void RestoreUart() {}

#elif defined NN_BUILD_CONFIG_HARDWARE_JETSONTK1 || \
      defined NN_BUILD_CONFIG_HARDWARE_JETSONTK2 || \
      defined NN_BUILD_CONFIG_HARDWARE_JETSONTX2 || \
      defined NN_BUILD_CONFIG_HARDWARE_NX
#if defined NN_BUILD_CONFIG_HARDWARE_JETSONTK1
#define UART_REGISTER_OFFSET    0x300
#else
#if  defined ENABLE_SECONDARY_UART_LOG && defined NN_BUILD_CONFIG_HARDWARE_NX
#define UART_REGISTER_OFFSET    0x040
#else
#define UART_REGISTER_OFFSET    0x000
#endif
#endif

#if defined ENABLE_SECONDARY_UART_LOG && !defined NN_BUILD_CONFIG_HARDWARE_NX
#error ENABLE_SECONDARY_UART_LOG is not supported
#endif

#define UART_THR        (*reinterpret_cast<volatile Bit32 *>(g_UartBase + UART_REGISTER_OFFSET + 0x00))
#define UART_IER        (*reinterpret_cast<volatile Bit32 *>(g_UartBase + UART_REGISTER_OFFSET + 0x04))
#define UART_FCR        (*reinterpret_cast<volatile Bit32 *>(g_UartBase + UART_REGISTER_OFFSET + 0x08))
#define UART_LCR        (*reinterpret_cast<volatile Bit32 *>(g_UartBase + UART_REGISTER_OFFSET + 0x0c))
#define UART_LSR        (*reinterpret_cast<volatile Bit32 *>(g_UartBase + UART_REGISTER_OFFSET + 0x14))
#define UART_DLL        (*reinterpret_cast<volatile Bit32 *>(g_UartBase + UART_REGISTER_OFFSET + 0x00))
#define UART_DLH        (*reinterpret_cast<volatile Bit32 *>(g_UartBase + UART_REGISTER_OFFSET + 0x04))
    void InitUart()
    {
        const uint32_t baudRate = 115200;
        const uint32_t uartClock = 408000000;
        const uint32_t divisor = (uartClock + (16 * baudRate) / 2) / (16 * baudRate);

        // baud rate setting
        UART_LCR = 0x80;
        UART_LCR;
        UART_DLL = (divisor & 0xFF);
        UART_DLH = (divisor >> 8) & 0xFF;
        UART_DLH;
        UART_LCR = 0x03;
        UART_LCR;
        UART_IER = 0;
        UART_FCR = 0x03;
        UART_FCR;
    }
    void FlushUart()
    {
        while ((UART_LSR & 0x40) == 0) {}
    }
    void PutCharImpl(char c)
    {
        while (UART_LSR & 0x100)
        {
            asm volatile("yield":::"memory");
        }
        UART_THR = c;
        nn::kern::KCPU::DataSynchronizationBarrier();
    }

    Bit32 g_SaveLcr;
    Bit32 g_SaveIer;
    Bit32 g_SaveFcr;
    Bit32 g_SaveDll;
    Bit32 g_SaveDlh;

    void SaveUart()
    {
        g_SaveLcr = UART_LCR;
        g_SaveIer = UART_IER;
        g_SaveFcr = UART_FCR;

        UART_LCR = 0x80;
        UART_LCR;
        g_SaveDll = UART_DLL;
        g_SaveDlh = UART_DLH;
        UART_LCR = g_SaveLcr;
        UART_LCR;
    }

    void RestoreUart()
    {
        const Bit32 TX_CLR = 1 << 2;
        const Bit32 RX_CLR = 1 << 1;

        UART_LCR = 0x80;
        UART_LCR;
        UART_DLL = g_SaveDll;
        UART_DLH = g_SaveDlh;
        UART_DLH;
        UART_LCR = g_SaveLcr;
        UART_LCR;
        UART_IER = g_SaveIer;
        UART_FCR = g_SaveFcr | TX_CLR | RX_CLR;
        UART_FCR;
    }

#elif defined NN_BUILD_CONFIG_HARDWARE_BDSLIMX6
#define UART_URXD (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x00))
#define UART_UTXD (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x40))
#define UART_UCR1 (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x80))
#define UART_UCR2 (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x84))
#define UART_UCR3 (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x88))
#define UART_UCR4 (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x8C))
#define UART_UFCR (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x90))
#define UART_USR1 (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x94))
#define UART_USR2 (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x98))
#define UART_UESC (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0x9C))
#define UART_UTIM (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0xA0))
#define UART_UBIR (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0xA4))
#define UART_UBMR (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0xA8))
#define UART_UBRC (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0xAC))
#define UART_UTS  (*reinterpret_cast<volatile Bit32*>(g_UartBase + 0xB4))
    void InitUart()
    {
        const uint32_t baudRate = 115200;
        const uint32_t uartClock = 80000000;
        UART_UCR1 = 0;
        UART_UCR2 = 0;
        nn::kern::KCPU::DataSynchronizationBarrier();
        while (UART_UTS & (1<<0)) {}
        UART_UCR3 = 0x0704;
        UART_UCR4 = 0x8000;
        UART_UESC = 0x002b;
        UART_UTS  = 0;
        UART_UFCR = 4 << 7;
        UART_UBIR = 0xf;
        UART_UBMR = uartClock / (2 * baudRate);

        UART_UCR2 = (1<<14) | (1<<5) | (1<<2) | (1<<0);
        nn::kern::KCPU::DataSynchronizationBarrier();

        UART_UCR1 = (1<<0);
        nn::kern::KCPU::DataSynchronizationBarrier();
    }
    void FlushUart()
    {
        while ((UART_UTS & (1<<6)) == 0) {}   // (TxFIFO empty)
    }
    void PutCharImpl(char c)
    {
        while ((UART_UTS & (1<<4)) != 0) {}   // !(TxFIFO Full)
        UART_UTXD = c;
        nn::kern::KCPU::DataSynchronizationBarrier();
    }

    void SaveUart() {}
    void RestoreUart() {}
#else
    NN_FUNC_ATTR_PRIVATE_SECTION
    asm void PutStringImpl(const char *str NN_IS_UNUSED_VAR)
    {
            push    {r4,lr}
            mov     r1, r0
            mov     r0, #0x04
            swi     0x123456
            pop     {r4,pc}
    }
    void SaveRegisters()
    {
    }
    void RestoreRegisters()
    {
    }
#endif  // #ifdef NN_DEBUGGER_KMC_PARTNER else


#if defined NN_BUILD_CONFIG_HARDWARE_KZMA9      || \
    defined NN_BUILD_CONFIG_HARDWARE_SMMA53     || \
    defined NN_BUILD_CONFIG_HARDWARE_JUNO       || \
    defined NN_BUILD_CONFIG_HARDWARE_JETSONTK1  || \
    defined NN_BUILD_CONFIG_HARDWARE_JETSONTK2  || \
    defined NN_BUILD_CONFIG_HARDWARE_JETSONTX2  || \
    defined NN_BUILD_CONFIG_HARDWARE_NX         || \
    defined NN_BUILD_CONFIG_HARDWARE_BDSLIMX6

    void PutStringImpl(const char *str)
    {
        if (!g_Initialized)
        {
            InitUart();
            g_Initialized = true;
        }
        while (*str)
        {
            if (*str == '\n')
            {
                PutCharImpl('\r');
            }
            PutCharImpl(*str++);
        }
        FlushUart();
    }

#if defined NN_KERN_ENABLE_USER_PRINT
    Result PutUserStringImpl(nn::kern::svc::KUserPointer<const char*> textUser, size_t byteSize)
    {
        if (!g_Initialized)
        {
            InitUart();
            g_Initialized = true;
        }

        for (size_t i = 0; i < byteSize; i++)
        {
            char c;
            Result result = textUser.CopyArrayEntryTo(&c, i);
            if (result.IsFailure())
            {
                return nn::svc::ResultInvalidPointer();
            }
            if (c == '\n')
            {
                PutCharImpl('\r');
            }
            PutCharImpl(c);
        }

        FlushUart();

        return ResultSuccess();
    }
#endif

    void SaveRegisters()
    {
        if (g_Initialized)
        {
            SaveUart();
        }
    }

    void RestoreRegisters()
    {
        if (g_Initialized)
        {
            RestoreUart();
        }
    }

#endif
}
}
#endif


extern "C" {
    void nndbgPrintWarning_(const char* filename, int lineno, const char* fmt, ...)
    {
        ::std::va_list vlist;
        NN_UNUSED(filename);
        NN_UNUSED(lineno);

        va_start(vlist, fmt);
        NN_LOG("%s:%d [WARN] ", filename, lineno);
        NN_VLOG(fmt, vlist);
        NN_LOG("\n");
        va_end(vlist);
    }
}

namespace nn { namespace kern {
#ifdef NN_KERN_ENABLE_PRINT
namespace {
    typedef char CharT;

    class TVSNPrintfImpl
    {
    private:
        struct dst_string
        {
            size_t  len;
            CharT   *cur;
            CharT   *base;
        };

        static void string_put_char(dst_string * p, CharT c)
        {
            if (p->len > 0)
            {
                *p->cur = c, --p->len;
            }
            ++p->cur;
        }

        static void string_fill_char(dst_string * p, CharT c, int n)
        {
            if (n > 0)
            {
                size_t  i, k = p->len;
                if (k > (size_t) n)
                {
                    k = (size_t) n;
                }
                for (i = 0; i < k; ++i)
                {
                    p->cur[i] = c;
                }
                p->len -= k;
                p->cur += n;
            }
        }

        static void string_put_string(dst_string * p, const CharT *s, int n)
        {
            if (n > 0)
            {
                size_t  i, k = p->len;
                if (k > (size_t) n)
                {
                    k = (size_t) n;
                }
                for (i = 0; i < k; ++i)
                {
                    p->cur[i] = s[i];
                }
                p->len -= k;
                p->cur += n;
            }
        }

    public:
        static void TVSNPrintf(CharT *dst, size_t len, const CharT *fmt, ::std::va_list vlist)
        {
            CharT   buf[24];
            int     n_buf;
            CharT   prefix[2];
            int     n_prefix;

            const CharT *s = fmt;

            dst_string str;
            str.len = len, str.cur = str.base = dst;

            while (*s)
            {
                if (*s != '%')
                {
                    string_put_char(&str, *s++);
                }
                else
                {
                    /* output with format */
                    enum
                    {
                        flag_blank = 000001,   /* ' ' */
                        flag_plus = 000002,    /* '+' */
                        flag_sharp = 000004,   /* '#' */
                        flag_minus = 000010,   /* '-' */
                        flag_zero = 000020,    /* '0' */
                        flag_l1 = 000040,      /* "l" */
                        flag_h1 = 000100,      /* "h" */
                        flag_l2 = 000200,      /* "ll" */
                        flag_h2 = 000400,      /* "hh" */
                        flag_p  = 001000,      /* %p */
                        flag_unsigned = 010000, /* 'o', 'u', ... */
                        flag_end
                    };
                    int     flag = 0, width = 0, precision = -1, radix = 10;
                    CharT    hex_char = 'a' - 10;
                    const CharT *p_start = s;
                    /* flags */
                    for (;;)
                    {
                        switch (*++s)
                        {
                        case '+':
                            if (s[-1] != ' ')
                            {
                                break;
                            }
                            flag |= flag_plus;
                            continue;
                        case ' ':
                            flag |= flag_blank;
                            continue;
                        case '-':
                            flag |= flag_minus;
                            continue;
                        case '0':
                            flag |= flag_zero;
                            continue;
                        default:
                            break;
                        }
                        break;
                    }
                    /* width */
                    if (*s == '*')
                    {
                        ++s, width = va_arg(vlist, int);
                        if (width < 0)
                        {
                            width = -width, flag |= flag_minus;
                        }
                    }
                    else
                    {
                        while ((*s >= '0') && (*s <= '9'))
                        {
                            width = (width * 10) + *s++ - '0';
                        }
                    }
                    /* precision */
                    if (*s == '.')
                    {
                        ++s, precision = 0;
                        if (*s == '*')
                        {
                            ++s, precision = va_arg(vlist, int);
                            if (precision < 0)
                            {
                                precision = -1;
                            }
                        }
                        else
                        {
                            while ((*s >= '0') && (*s <= '9'))
                            {
                                precision = (precision * 10) + *s++ - '0';
                            }
                        }
                    }
                    /* option */
                    switch (*s)
                    {
                    case 'h':
                        if (*++s != 'h')
                        {
                            flag |= flag_h1;
                        }
                        else
                        {
                            ++s, flag |= flag_h2;
                        }
                        break;
                    case 'l':
                        if (*++s != 'l')
                        {
                            flag |= flag_l1;
                        }
                        else
                        {
                            ++s, flag |= flag_l2;
                        }
                        break;
                    default:
                        break;
                    }

                    /* type */
                    switch (*s)
                    {
                    case 'd':                 /* signed decimal */
                    case 'i':                 /* signed decimal */
                        goto put_integer;
                    case 'o':                 /* unsigned octal */
                        radix = 8;
                        flag |= flag_unsigned;
                        goto put_integer;
                    case 'u':                 /* unsigned decimal */
                        flag |= flag_unsigned;
                        goto put_integer;
                    case 'X':                 /* unsigned hexadecimal */
                        hex_char = 'A' - 10;
                        goto put_hexadecimal;
                    case 'x':                 /* unsigned hexadecimal */
                        goto put_hexadecimal;
                    case 'p':                 /* pointer */
                        flag |= flag_p;
                        if (sizeof(long) == sizeof(void *))
                        {
                            flag |= flag_l1;
                        }
                        else if (sizeof(long long) == sizeof(void *))
                        {
                            flag |= flag_l2;
                        }
                        precision = sizeof(void *) * 2;
                        goto put_hexadecimal;

                    case 'c':                 /* character */
                        if (precision >= 0)
                        {
                            goto put_invalid;
                        }
                        {
                            int     c = va_arg(vlist, int);
                            width -= 1;
                            if (flag & flag_minus)
                            {
                                string_put_char(&str, (CharT)c);
                                string_fill_char(&str, ' ', width);
                            }
                            else
                            {
                                CharT    pad = (CharT)((flag & flag_zero) ? '0' : ' ');
                                string_fill_char(&str, pad, width);
                                string_put_char(&str, (CharT)c);
                            }
                            ++s;
                        }
                        break;

                    case 's':                 /* string */
                        {
                            int     n_buf2 = 0;
                            const CharT *p_buf = va_arg(vlist, const CharT *);
                            if (precision < 0)
                            {
                                while (p_buf[n_buf2])
                                {
                                    ++n_buf2;
                                }
                            }
                            else
                            {
                                while ((n_buf2 < precision) && p_buf[n_buf2])
                                {
                                    ++n_buf2;
                                }
                            }
                            width -= n_buf2;
                            if (flag & flag_minus)
                            {
                                string_put_string(&str, p_buf, n_buf2);
                                string_fill_char(&str, ' ', width);
                            }
                            else
                            {
                                CharT    pad = (CharT)((flag & flag_zero) ? '0' : ' ');
                                string_fill_char(&str, pad, width);
                                string_put_string(&str, p_buf, n_buf2);
                            }
                            ++s;
                        }
                        break;

                    case '%':                 /* output '%' */
                        if (p_start + 1 != s)
                        {
                            goto put_invalid;
                        }
                        string_put_char(&str, *s++);
                        break;

                    default:                  /* invalid type */
                        goto put_invalid;

                      put_invalid:
                        string_put_string(&str, p_start, s - p_start);
                        break;

                      put_hexadecimal:
                        radix = 16;
                        flag |= flag_unsigned;
                      put_integer:
                        {
                            unsigned long long val = 0;
                            n_prefix = 0;

                            if (flag & flag_minus)
                            {
                                flag &= ~flag_zero;
                            }
                            if (precision < 0)
                            {
                                precision = 1;
                            }
                            else
                            {
                                flag &= ~flag_zero;
                            }

                            if (flag & flag_unsigned)
                            {
                                if (flag & flag_h2)
                                {
                                    val = static_cast<unsigned char>(va_arg(vlist, int));
                                }
                                else if (flag & flag_h1)
                                {
                                    val = static_cast<unsigned short>(va_arg(vlist, int));
                                }
                                else if (flag & flag_l2)
                                {
                                    val = va_arg(vlist, unsigned long long);
                                }
                                else if (flag & flag_l1)
                                {
                                    val = va_arg(vlist, unsigned long);
                                }
                                else
                                {
                                    val = va_arg(vlist, unsigned);
                                }
                                flag &= ~(flag_plus | flag_blank);
                                if (flag & (flag_sharp | flag_p))
                                {
                                    if (radix == 16)
                                    {
                                        if (val != 0 || (flag & flag_p))
                                        {
                                            prefix[0] = (CharT)(hex_char + (10 + 'x' - 'a'));
                                            prefix[1] = '0';
                                            n_prefix = 2;
                                        }
                                    }
                                    else if (radix == 8)
                                    {
                                        prefix[0] = '0';
                                        n_prefix = 1;
                                    }
                                }
                            }
                            else
                            {
                                if (flag & flag_h2)
                                {
                                    val = static_cast<char>(va_arg(vlist, int));
                                }
                                else if (flag & flag_h1)
                                {
                                    val = static_cast<short>(va_arg(vlist, int));
                                }
                                else if (flag & flag_l2)
                                {
                                    val = va_arg(vlist, unsigned long long);
                                }
                                else if (flag & flag_l1)
                                {
                                    val = va_arg(vlist, long);
                                }
                                else
                                {
                                    val = va_arg(vlist, int);
                                }
                                if ((val >> 32) & 0x80000000)
                                {
                                    val = ~val + 1;
                                    prefix[0] = '-';
                                    n_prefix = 1;
                                }
                                else
                                {
                                    if (val || precision)
                                    {
                                        if (flag & flag_plus)
                                        {
                                            prefix[0] = '+';
                                            n_prefix = 1;
                                        }
                                        else if (flag & flag_blank)
                                        {
                                            prefix[0] = ' ';
                                            n_prefix = 1;
                                        }
                                    }
                                }
                            }
                            n_buf = 0;
                            switch (radix)
                            {
                            case 8:
                                while (val != 0)
                                {
                                    int d = static_cast<int>(val & 0x07);
                                    val >>= 3;
                                    buf[n_buf++] = (CharT)(d + '0');
                                }
                                break;
                            case 10:
                                if ((val >> 32) == 0)
                                {
                                    uint32_t v = static_cast<uint32_t>(val);
                                    while (v != 0)
                                    {
                                        // uint32_t と定数の除算であれば、コンパイラが自動的に
                                        // マジックナンバーを使用した積算に変換する
                                        uint32_t r = v / 10;
                                        int d = static_cast<int>(v - (r * 10));
                                        v = r;
                                        buf[n_buf++] = (CharT)(d + '0');
                                    }
                                }
                                else
                                {
                                    while (val != 0)
                                    {
                                        uint64_t r = val / 10;
                                        int d = static_cast<int>(val - (r * 10));
                                        val = r;
                                        buf[n_buf++] = (CharT)(d + '0');
                                    }
                                }
                                break;
                            case 16:
                                while (val != 0)
                                {
                                    int d = static_cast<int>(val & 0x0f);
                                    val >>= 4;
                                    buf[n_buf++] = (CharT)((d < 10) ? (d + '0') : (d + hex_char));
                                }
                                break;
                            default:
                                break;
                            }
                            if ((n_prefix > 0) && (prefix[0] == '0'))
                            {
                                n_prefix = 0;
                                buf[n_buf++] = '0';
                            }
                        }
                        goto put_to_stream;

                      put_to_stream:
                        {
                            int n_pad = precision - n_buf;
                            if (flag & flag_zero)
                            {
                                if (n_pad < width - n_buf - n_prefix)
                                {
                                    n_pad = width - n_buf - n_prefix;
                                }
                            }
                            if (n_pad > 0)
                            {
                                width -= n_pad;
                            }
                            width -= n_prefix + n_buf;
                            if (!(flag & flag_minus))
                            {
                                string_fill_char(&str, ' ', width);
                            }
                            while (n_prefix > 0)
                            {
                                string_put_char(&str, prefix[--n_prefix]);
                            }
                            string_fill_char(&str, '0', n_pad);
                            while (n_buf > 0)
                            {
                                string_put_char(&str, buf[--n_buf]);
                            }
                            if (flag & flag_minus)
                            {
                                string_fill_char(&str, ' ', width);
                            }
                            ++s;
                        }
                        break;
                    }
                }
            }

            if (str.len > 0)
            {
                *str.cur = '\0';
            }
            else if (len > 0)
            {
                str.base[len - 1] = '\0';
            }
        }   // NOLINT(readability/fn_size)
    };

    void KOutputString(const char* text)
    {
#ifdef NN_KERN_ENABLE_PRINT
        PutStringImpl(text);
#endif
    }
}
#endif

    Result KOutputUserString(nn::kern::svc::KUserPointer<const char*> textUser, size_t byteSize)
    {
        Result result = ResultSuccess();
        NN_UNUSED(textUser);
        NN_UNUSED(byteSize);
#if defined NN_KERN_ENABLE_USER_PRINT
#ifdef NN_KERN_ENABLE_PRINT
        if (KTargetSystem::IsPrintEnabled())
        {
            KDisableInterrupt di;
            KScopedSimpleLock lock(&s_Lock);
            result = PutUserStringImpl(textUser, byteSize);
        }
#endif
#endif
        return result;
    }

    void KVPrintf(const char *fmt, ::std::va_list vlist)
    {
#ifdef NN_KERN_ENABLE_PRINT
        if (KTargetSystem::IsPrintEnabled())
        {
            KDisableInterrupt di;
            KScopedSimpleLock lock(&s_Lock);
            char* pBuffer = s_CommonBuffer[0];
            TVSNPrintfImpl::TVSNPrintf(pBuffer, sizeof(s_CommonBuffer[0]), fmt, vlist);
            KOutputString(pBuffer);
        }
#else
        NN_UNUSED(fmt);
        NN_UNUSED(vlist);
#endif
    }

    void KPrintf(const char* fmt, ...)
    {
#ifdef NN_KERN_ENABLE_PRINT
        if (KTargetSystem::IsPrintEnabled())
        {
            ::std::va_list vlist;
            va_start(vlist, fmt);
            KVPrintf(fmt, vlist);
            va_end(vlist);
        }
#else
        NN_UNUSED(fmt);
#endif
    }

    void SavePrintRegisters()
    {
#ifdef NN_KERN_ENABLE_PRINT
        if (KTargetSystem::IsPrintEnabled())
        {
            KDisableInterrupt di;
            KScopedSimpleLock lock(&s_Lock);
            SaveRegisters();
        }
#endif
    }

    void RestorePrintRegisters()
    {
#ifdef NN_KERN_ENABLE_PRINT
        if (KTargetSystem::IsPrintEnabled())
        {
            KDisableInterrupt di;
            KScopedSimpleLock lock(&s_Lock);
            RestoreRegisters();
        }
#endif
    }

    void InitializeDebugLog(uintptr_t base)
    {
        g_UartBase = base;
    }
}}


