﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include "../../kern_Platform.h"
#include "kern_KDevicePageTable.h"
#include "../../kern_CPUSelect.h"
#include "kern_MemoryMap.h"
#include "../../kern_Kernel.h"
#include "../../kern_KLightMutex.h"
#include "../../kern_KTaggedAddress.h"
#include "../../kern_KInterruptTask.h"
#include "../../kern_Kernel.h"
#include "../../kern_KPageTableManager.h"
#include "kern_SystemControl.h"
#include "../../kern_Utility.h"
#include <cstring>

// Jetson TK1, TK2以外の Debug/Develop で割り込みを有効化する
#if defined NN_SDK_BUILD_DEVELOP || defined NN_SDK_BUILD_DEBUG
#if !defined NN_BUILD_CONFIG_HARDWARE_JETSONTK2 && !defined NN_BUILD_CONFIG_HARDWARE_JETSONTK1
#define NN_KERN_MC_INTERRUPT_ENABLE
#endif
#endif

namespace nn { namespace kern { namespace tegra {
namespace
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
void WriteMcRegister(size_t offset, Bit32 value)
{
    Bit32 v;
    KSystemControl::ReadWriteRegisterFromKernel(&v, (NN_KERN_P_ADDR_MC_REGISTER + offset), 0xFFFFFFFF, value);
}

Bit32 ReadMcRegister(size_t offset)
{
    Bit32 v;
    KSystemControl::ReadWriteRegisterFromKernel(&v, (NN_KERN_P_ADDR_MC_REGISTER + offset), 0, 0);
    return v;
}

#else
KProcessAddress g_RegisterBase;
void WriteMcRegister(size_t offset, Bit32 value)
{
    *GetTypedPointer<volatile Bit32>(g_RegisterBase + offset) = value;
}

Bit32 ReadMcRegister(size_t offset)
{
    return *GetTypedPointer<volatile Bit32>(g_RegisterBase + offset);
}
#endif


#define NN_KERN_MC_INTSTATUS_OFFSET                    (0x00)
#define NN_KERN_MC_INTMASK_OFFSET                      (0x04)
#define NN_KERN_MC_ERRSTATUS_OFFSET                    (0x08)
#define NN_KERN_MC_ERRADR_OFFSET                       (0x0c)
#define NN_KERN_SMMU_CONFIG_OFFSET                     (0x10)
#define NN_KERN_SMMU_PTB_ASID_OFFSET                   (0x1c)
#define NN_KERN_SMMU_PTB_DATA_OFFSET                   (0x20)
#define NN_KERN_SMMU_TLB_FLUSH_OFFSET                  (0x30)
#define NN_KERN_SMMU_PTC_FLUSH_0_OFFSET                (0x34)
#define NN_KERN_SMMU_PTC_FLUSH_1_OFFSET                (0x9b8)

#define NN_KERN_SMMU_AFI_ASID_OFFSET            (0x238)
#define NN_KERN_SMMU_AVPC_ASID_OFFSET           (0x23c)
#define NN_KERN_SMMU_DC_ASID_OFFSET             (0x240)
#define NN_KERN_SMMU_DCB_ASID_OFFSET            (0x244)
#define NN_KERN_SMMU_HC_ASID_OFFSET             (0x250)
#define NN_KERN_SMMU_HDA_ASID_OFFSET            (0x254)
#define NN_KERN_SMMU_ISP2_ASID_OFFSET           (0x258)
#define NN_KERN_SMMU_MSENC_ASID_OFFSET          (0x264)
#define NN_KERN_SMMU_NVENC_ASID_OFFSET          (0x264)
#define NN_KERN_SMMU_NV_ASID_OFFSET             (0x268)
#define NN_KERN_SMMU_NV2_ASID_OFFSET            (0x26c)
#define NN_KERN_SMMU_PPCS_ASID_OFFSET           (0x270)
#define NN_KERN_SMMU_SATA_ASID_OFFSET           (0x274)
#define NN_KERN_SMMU_VDE_ASID_OFFSET            (0x27c)
#define NN_KERN_SMMU_VI_ASID_OFFSET             (0x280)
#define NN_KERN_SMMU_VIC_ASID_OFFSET            (0x284)
#define NN_KERN_SMMU_XUSB_HOST_ASID_OFFSET      (0x288)
#define NN_KERN_SMMU_XUSB_DEV_ASID_OFFSET       (0x28c)
#define NN_KERN_SMMU_TSEC_ASID_OFFSET           (0x294)
#define NN_KERN_SMMU_PPCS1_ASID_OFFSET          (0x298)
#define NN_KERN_SMMU_DC1_ASID_OFFSET            (0xa88)
#define NN_KERN_SMMU_SDMMC1A_ASID_OFFSET        (0xa94)
#define NN_KERN_SMMU_SDMMC2A_ASID_OFFSET        (0xa98)
#define NN_KERN_SMMU_SDMMC3A_ASID_OFFSET        (0xa9c)
#define NN_KERN_SMMU_SDMMC4A_ASID_OFFSET        (0xaa0)
#define NN_KERN_SMMU_ISP2B_ASID_OFFSET          (0xaa4)
#define NN_KERN_SMMU_GPU_ASID_OFFSET            (0xaa8)
#define NN_KERN_SMMU_GPUB_ASID_OFFSET           (0xaac)
#define NN_KERN_SMMU_PPCS2_ASID_OFFSET          (0xab0)
#define NN_KERN_SMMU_NVDEC_ASID_OFFSET          (0xab4)
#define NN_KERN_SMMU_APE_ASID_OFFSET            (0xab8)
#define NN_KERN_SMMU_SE_ASID_OFFSET             (0xabc)
#define NN_KERN_SMMU_NVJPG_ASID_OFFSET          (0xac0)
#define NN_KERN_SMMU_HC1_ASID_OFFSET            (0xac4)
#define NN_KERN_SMMU_SE1_ASID_OFFSET            (0xac8)
#define NN_KERN_SMMU_AXIAP_ASID_OFFSET          (0xacc)
#define NN_KERN_SMMU_ETR_ASID_OFFSET            (0xad0)
#define NN_KERN_SMMU_TSECB_ASID_OFFSET          (0xad4)
#define NN_KERN_SMMU_TSEC1_ASID_OFFSET          (0xad8)
#define NN_KERN_SMMU_TSECB1_ASID_OFFSET         (0xadc)
#define NN_KERN_SMMU_NVDEC1_ASID_OFFSET         (0xae0)


int GetAsidRegister(nn::svc::DeviceName deviceName)
{
    static const uint32_t AsidRegistersOffset[] =
    {
#if defined NN_BUILD_CONFIG_SOC_TEGRA_K1
        NN_KERN_SMMU_AFI_ASID_OFFSET,
        NN_KERN_SMMU_AVPC_ASID_OFFSET,
        NN_KERN_SMMU_DC_ASID_OFFSET,
        NN_KERN_SMMU_DCB_ASID_OFFSET,
        NN_KERN_SMMU_HC_ASID_OFFSET,
        NN_KERN_SMMU_HDA_ASID_OFFSET,
        NN_KERN_SMMU_ISP2_ASID_OFFSET,
        NN_KERN_SMMU_MSENC_ASID_OFFSET,
        NN_KERN_SMMU_NV_ASID_OFFSET,
        NN_KERN_SMMU_NV2_ASID_OFFSET,
        NN_KERN_SMMU_PPCS_ASID_OFFSET,
        NN_KERN_SMMU_SATA_ASID_OFFSET,
        NN_KERN_SMMU_VDE_ASID_OFFSET,
        NN_KERN_SMMU_VI_ASID_OFFSET,
        NN_KERN_SMMU_VIC_ASID_OFFSET,
        NN_KERN_SMMU_XUSB_HOST_ASID_OFFSET,
        NN_KERN_SMMU_XUSB_DEV_ASID_OFFSET,
        NN_KERN_SMMU_TSEC_ASID_OFFSET,
        NN_KERN_SMMU_PPCS1_ASID_OFFSET,
        NN_KERN_SMMU_DC1_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC1A_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC2A_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC3A_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC4A_ASID_OFFSET,
        NN_KERN_SMMU_ISP2B_ASID_OFFSET,
        NN_KERN_SMMU_GPU_ASID_OFFSET,
        NN_KERN_SMMU_GPUB_ASID_OFFSET,
        NN_KERN_SMMU_PPCS2_ASID_OFFSET

#elif defined NN_BUILD_CONFIG_SOC_TEGRA_X1
        NN_KERN_SMMU_AFI_ASID_OFFSET,
        NN_KERN_SMMU_AVPC_ASID_OFFSET,
        NN_KERN_SMMU_DC_ASID_OFFSET,
        NN_KERN_SMMU_DCB_ASID_OFFSET,
        NN_KERN_SMMU_HC_ASID_OFFSET,
        NN_KERN_SMMU_HDA_ASID_OFFSET,
        NN_KERN_SMMU_ISP2_ASID_OFFSET,
        NN_KERN_SMMU_NVENC_ASID_OFFSET,
        NN_KERN_SMMU_NV_ASID_OFFSET,
        NN_KERN_SMMU_NV2_ASID_OFFSET,
        NN_KERN_SMMU_PPCS_ASID_OFFSET,
        NN_KERN_SMMU_SATA_ASID_OFFSET,
        NN_KERN_SMMU_VI_ASID_OFFSET,
        NN_KERN_SMMU_VIC_ASID_OFFSET,
        NN_KERN_SMMU_XUSB_HOST_ASID_OFFSET,
        NN_KERN_SMMU_XUSB_DEV_ASID_OFFSET,
        NN_KERN_SMMU_TSEC_ASID_OFFSET,
        NN_KERN_SMMU_PPCS1_ASID_OFFSET,
        NN_KERN_SMMU_DC1_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC1A_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC2A_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC3A_ASID_OFFSET,
        NN_KERN_SMMU_SDMMC4A_ASID_OFFSET,
        NN_KERN_SMMU_ISP2B_ASID_OFFSET,
        NN_KERN_SMMU_GPU_ASID_OFFSET,
        NN_KERN_SMMU_GPUB_ASID_OFFSET,
        NN_KERN_SMMU_PPCS2_ASID_OFFSET,
        NN_KERN_SMMU_NVDEC_ASID_OFFSET,
        NN_KERN_SMMU_APE_ASID_OFFSET,
        NN_KERN_SMMU_SE_ASID_OFFSET,
        NN_KERN_SMMU_NVJPG_ASID_OFFSET,
        NN_KERN_SMMU_HC1_ASID_OFFSET,
        NN_KERN_SMMU_SE1_ASID_OFFSET,
        NN_KERN_SMMU_AXIAP_ASID_OFFSET,
        NN_KERN_SMMU_ETR_ASID_OFFSET,
        NN_KERN_SMMU_TSECB_ASID_OFFSET,
        NN_KERN_SMMU_TSEC1_ASID_OFFSET,
        NN_KERN_SMMU_TSECB1_ASID_OFFSET,
        NN_KERN_SMMU_NVDEC1_ASID_OFFSET,

#else
#error not defined NN_BUILD_CONFIG_SOC
#endif
    };
    static_assert(sizeof(AsidRegistersOffset) / sizeof(AsidRegistersOffset[0]) == nn::svc::DeviceName_Num, "");
    if (deviceName < nn::svc::DeviceName_Num)
    {
        return AsidRegistersOffset[deviceName];
    }
    else
    {
        return -1;
    }
}

const Bit64 HighAddressSupportDevice = 0
#if defined NN_BUILD_CONFIG_SOC_TEGRA_K1
#elif defined NN_BUILD_CONFIG_SOC_TEGRA_X1
    | (1ull << nn::svc::DeviceName_Afi)
    | (1ull << nn::svc::DeviceName_Dc)
    | (1ull << nn::svc::DeviceName_Dcb)
    | (1ull << nn::svc::DeviceName_Hda)
    | (1ull << nn::svc::DeviceName_Isp2)
    | (1ull << nn::svc::DeviceName_Sata)
    | (1ull << nn::svc::DeviceName_Vi)
    | (1ull << nn::svc::DeviceName_XusbHost)
    | (1ull << nn::svc::DeviceName_XusbDev)
    | (1ull << nn::svc::DeviceName_Tsec)
    | (1ull << nn::svc::DeviceName_Dc1)
    | (1ull << nn::svc::DeviceName_Sdmmc1a)
    | (1ull << nn::svc::DeviceName_Sdmmc2a)
    | (1ull << nn::svc::DeviceName_Sdmmc3a)
    | (1ull << nn::svc::DeviceName_Sdmmc4a)
    | (1ull << nn::svc::DeviceName_Isp2b)
    | (1ull << nn::svc::DeviceName_Gpu)
    | (1ull << nn::svc::DeviceName_Gpub)
    | (1ull << nn::svc::DeviceName_Axiap)
    | (1ull << nn::svc::DeviceName_Etr)
    | (1ull << nn::svc::DeviceName_Tsecb)
    | (1ull << nn::svc::DeviceName_Tsec1)
    | (1ull << nn::svc::DeviceName_Tsecb1)
#else
#error not defined NN_BUILD_CONFIG_SOC
#endif
    ;

const struct
{
    uint64_t start;
    uint64_t end;
} SmmuSupportRanges[nn::svc::DeviceName_Num] = {
#if defined NN_BUILD_CONFIG_SOC_TEGRA_K1
    [nn::svc::DeviceName_Afi] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Avpc] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Dc] =          { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Dcb] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Hc] =          { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Hda] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Isp2] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Msenc] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Nv] =          { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Nv2] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Ppcs] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Sata] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Vde] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Vi] =          { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Vic] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_XusbHost] =    { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_XusbDev] =     { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Tsec] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Ppcs1] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Dc1] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Sdmmc1a] =     { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Sdmmc2a] =     { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Sdmmc3a] =     { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Sdmmc4a] =     { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Isp2b] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Gpu] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Gpub] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Ppcs2] =       { 0x000000000ull, 0x0ffffffffull},

#elif defined NN_BUILD_CONFIG_SOC_TEGRA_X1
    [nn::svc::DeviceName_Afi] =         { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Avpc] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Dc] =          { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Dcb] =         { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Hc] =          { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Hda] =         { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Isp2] =        { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Nvenc] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Nv] =          { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Nv2] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Ppcs] =        { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Sata] =        { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Vi] =          { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Vic] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_XusbHost] =    { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_XusbDev] =     { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Tsec] =        { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Ppcs1] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Dc1] =         { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Sdmmc1a] =     { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Sdmmc2a] =     { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Sdmmc3a] =     { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Sdmmc4a] =     { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Isp2b] =       { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Gpu] =         { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Gpub] =        { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Ppcs2] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Nvdec] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Ape] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Se] =          { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Nvjpg] =       { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Hc1] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Se1] =         { 0x000000000ull, 0x0ffffffffull},
    [nn::svc::DeviceName_Axiap] =       { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Etr] =         { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Tsecb] =       { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Tsec1] =       { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Tsecb1] =      { 0x000000000ull, 0x3ffffffffull},
    [nn::svc::DeviceName_Nvdec1] =      { 0x000000000ull, 0x0ffffffffull},
#else
#error not defined NN_BUILD_CONFIG_SOC
#endif
};

#ifdef NN_KERN_MC_INTERRUPT_ENABLE
const char* GetSmmuClientName(size_t id)
{
    static const char* s_ClientName[138] = {
        /* 0 */   "csr_ptcr(ptc)",
        /* 1 */   "csr_display0a(dc)",
        /* 2 */   "csr_display0ab(dcb)",
        /* 3 */   "csr_display0b(dc)",
        /* 4 */   "csr_display0bb(dcb)",
        /* 5 */   "csr_display0c(dc)",
        /* 6 */   "csr_display0cb(dcb)",
        /* 7 */   "Unknown Client",
        /* 8 */   "Unknown Client",
        /* 9 */   "Unknown Client",
        /* 10 */  "Unknown Client",
        /* 11 */  "Unknown Client",
        /* 12 */  "Unknown Client",
        /* 13 */  "Unknown Client",
        /* 14 */  "csr_afir(afi)",
        /* 15 */  "csr_avpcarm7r(avpc)",
        /* 16 */  "csr_displayhc(dc)",
        /* 17 */  "csr_displayhcb(dcb)",
        /* 18 */  "Unknown Client",
        /* 19 */  "Unknown Client",
        /* 20 */  "Unknown Client",
        /* 21 */  "csr_hdar(hda)",
        /* 22 */  "csr_host1xdmar(hc)",
        /* 23 */  "csr_host1xr(hc)",
        /* 24 */  "Unknown Client",
        /* 25 */  "Unknown Client",
        /* 26 */  "Unknown Client",
        /* 27 */  "Unknown Client",
        /* 28 */  "csr_nvencsrd(nvenc)",
        /* 29 */  "csr_ppcsahbdmar(ppcs)",
        /* 30 */  "csr_ppcsahbslvr(ppcs)",
        /* 31 */  "csr_satar(sata)",
        /* 32 */  "Unknown Client",
        /* 33 */  "Unknown Client",
        /* 34 */  "Unknown Client",
        /* 35 */  "Unknown Client",
        /* 36 */  "Unknown Client",
        /* 37 */  "Unknown Client",
        /* 38 */  "Unknown Client",
        /* 39 */  "csr_mpcorer(cpu)",
        /* 40 */  "Unknown Client",
        /* 41 */  "Unknown Client",
        /* 42 */  "Unknown Client",
        /* 43 */  "csw_nvencswr(nvenc)",
        /* 44 */  "Unknown Client",
        /* 45 */  "Unknown Client",
        /* 46 */  "Unknown Client",
        /* 47 */  "Unknown Client",
        /* 48 */  "Unknown Client",
        /* 49 */  "csw_afiw(afi)",
        /* 50 */  "csw_avpcarm7w(avp)",
        /* 51 */  "Unknown Client",
        /* 52 */  "Unknown Client",
        /* 53 */  "csw_hdaw(hda)",
        /* 54 */  "csw_host1xw(hc)",
        /* 55 */  "Unknown Client",
        /* 56 */  "Unknown Client",
        /* 57 */  "csw_mpcorew(cpu)",
        /* 58 */  "Unknown Client",
        /* 59 */  "csw_ppcsahbdmaw(ppcs)",
        /* 60 */  "csw_ppcsahbslvw(ppcs)",
        /* 61 */  "csw_sataw(sata)",
        /* 62 */  "Unknown Client",
        /* 63 */  "Unknown Client",
        /* 64 */  "Unknown Client",
        /* 65 */  "Unknown Client",
        /* 66 */  "Unknown Client",
        /* 67 */  "Unknown Client",
        /* 68 */  "csr_ispra(isp2)",
        /* 69 */  "Unknown Client",
        /* 70 */  "csw_ispwa(isp2)",
        /* 71 */  "csw_ispwb(isp2)",
        /* 72 */  "Unknown Client",
        /* 73 */  "Unknown Client",
        /* 74 */  "csr_xusb_hostr(xusb_host)",
        /* 75 */  "csw_xusb_hostw(xusb_host)",
        /* 76 */  "csr_xusb_devr(xusb_dev)",
        /* 77 */  "csw_xusb_devw(xusb_dev)",
        /* 78 */  "csr_isprab(isp2b)",
        /* 79 */  "Unknown Client",
        /* 80 */  "csw_ispwab(isp2b)",
        /* 81 */  "csw_ispwbb(isp2b)",
        /* 82 */  "Unknown Client",
        /* 83 */  "Unknown Client",
        /* 84 */  "csr_tsecsrd(tsec)",
        /* 85 */  "csw_tsecswr(tsec)",
        /* 86 */  "csr_a9avpscr(a9avp)",
        /* 87 */  "csw_a9avpscw(a9avp)",
        /* 88 */  "csr_gpusrd(gpu)",
        /* 89 */  "csw_gpuswr(gpub)",
        /* 90 */  "csr_displayt(dc)",
        /* 91 */  "Unknown Client",
        /* 92 */  "Unknown Client",
        /* 93 */  "Unknown Client",
        /* 94 */  "Unknown Client",
        /* 95 */  "Unknown Client",
        /* 96 */  "csr_sdmmcra(sdmmc1a)",
        /* 97 */  "csr_sdmmcraa(sdmmc2a)",
        /* 98 */  "csr_sdmmcr(sdmmc3a)",
        /* 99 */  "csr_sdmmcrab(sdmmc4a)",
        /* 100 */ "csw_sdmmcwa(sdmmc1a)",
        /* 101 */ "csw_sdmmcwaa(sdmmc2a)",
        /* 102 */ "csw_sdmmcw(sdmmc3a)",
        /* 103 */ "csw_sdmmcwab(sdmmc4a)",
        /* 104 */ "Unknown Client",
        /* 105 */ "Unknown Client",
        /* 106 */ "Unknown Client",
        /* 107 */ "Unknown Client",
        /* 108 */ "csr_vicsrd(vic)",
        /* 109 */ "csw_vicswr(vic)",
        /* 110 */ "Unknown Client",
        /* 111 */ "Unknown Client",
        /* 112 */ "Unknown Client",
        /* 113 */ "Unknown Client",
        /* 114 */ "csw_viw(vi)",
        /* 115 */ "csr_displayd(dc)",
        /* 116 */ "Unknown Client",
        /* 117 */ "Unknown Client",
        /* 118 */ "Unknown Client",
        /* 119 */ "Unknown Client",
        /* 120 */ "csr_nvdecsrd(nvdec)",
        /* 121 */ "csw_nvdecswr(nvdec)",
        /* 122 */ "csr_aper(ape)",
        /* 123 */ "csw_apew(ape)",
        /* 124 */ "Unknown Client",
        /* 125 */ "Unknown Client",
        /* 126 */ "csr_nvjpgsrd(nvjpg)",
        /* 127 */ "csw_nvjpgswr(nvjpg)",
        /* 128 */ "csr_sesrd(se)",
        /* 129 */ "csw_seswr(se)",
        /* 130 */ "csr_axiapr(axiap)",
        /* 131 */ "csw_axiapw(axiap)",
        /* 132 */ "csr_etrr(etr)",
        /* 133 */ "csw_etrw(etr)",
        /* 134 */ "csr_tsecsrdb(tsecb)",
        /* 135 */ "csw_tsecswrb(testcb)",
        /* 136 */ "csr_gpusrd2(gpu)",
        /* 137 */ "csw_gpusw(gpu)",
    };
    if (id < sizeof(s_ClientName) / sizeof(s_ClientName[0]))
    {
        return s_ClientName[id];
    }
    return "Unknown Client";
}
#endif

bool IsAttachable(nn::svc::DeviceName deviceName, uint64_t spaceAddress, uint64_t spaceSize)
{
    if (0 <= deviceName && deviceName < nn::svc::DeviceName_Num)
    {
        return SmmuSupportRanges[deviceName].start <= spaceAddress &&
            (spaceAddress + spaceSize - 1) <= SmmuSupportRanges[deviceName].end;
    }
    return false;
}


const size_t   NumOfAsid                    = 128;
const uint64_t PhysicalAddressWidth         = 34;
const uint64_t PhysicalAddressMask          = (1ull << PhysicalAddressWidth) - 1;
#if defined NN_BUILD_CONFIG_SOC_TEGRA_K1
const uint64_t DeviceVirtualAddressWidth    = 32;
#elif defined NN_BUILD_CONFIG_SOC_TEGRA_X1
const uint64_t DeviceVirtualAddressWidth    = 34;
#else
#error not defined NN_BUILD_CONFIG_SOC
#endif
const uint64_t DeviceVirtualAddressMask     = (1ull << DeviceVirtualAddressWidth) - 1;
const size_t   DevicePageShift              = 12;
const uint64_t DevicePageSize               = 1ull << DevicePageShift;
const size_t   DeviceLargePageShift         = 22;
const uint64_t DeviceLargePageSize          = 1ull << DeviceLargePageShift;
const uint64_t PageDirectorySize            = (1ull << 12);
const uint64_t PageTableSize                = (1ull << 12);
const uint64_t DeviceRootRegionSize         = (1ull << 32);

const Bit32    ReadableBit                  = (1u << 31);
const Bit32    WriteableBit                 = (1u << 30);
const Bit32    NonSecureBit                 = (1u << 29);
const Bit32    TableBit                     = (1u << 28);

KLightMutex g_Lock;
Bit8 g_ReservedAsid;
KPhysicalAddress g_ReservedTablePhysicalAddress;
Bit32 g_SavedPageTable[NumOfAsid];
Bit32 g_SavedAsidRegister[nn::svc::DeviceName_Num];


class PageDirectoryEntry
{
public:
    bool IsValid()     const { return m_Entry & (ReadableBit | WriteableBit); }
    bool IsReadable()  const { return m_Entry & ReadableBit;  }
    bool IsWriteable() const { return m_Entry & WriteableBit; }
    bool IsNonSecure() const { return m_Entry & NonSecureBit; }
    bool IsTable()     const { return m_Entry & TableBit; }
    KPhysicalAddress GetPhysicalAddress() const { return ((static_cast<uint64_t>(m_Entry) << DevicePageShift) & PhysicalAddressMask); }
    Bit32 GetAttr() const { return (m_Entry & (ReadableBit | WriteableBit | NonSecureBit)); }
    void SetTable(bool isReadable, bool isWriteable, bool isNonSecure, KPhysicalAddress pa)
    {
        NN_KERN_ASSERT((static_cast<uint64_t>(GetAsInteger(pa)) & ~PhysicalAddressMask) == 0);
        NN_KERN_ASSERT((pa & (DevicePageSize - 1)) == 0);
        NN_KERN_ASSERT(m_Entry == 0);
        asm volatile ("":::"memory");
        m_Entry = ((isReadable? ReadableBit: 0) | (isWriteable? WriteableBit: 0) | (isNonSecure? NonSecureBit: 0) | TableBit | (pa >> DevicePageShift));
        asm volatile ("":::"memory");
    }
    void SetLargePage(bool isReadable, bool isWriteable, bool isNonSecure, KPhysicalAddress pa)
    {
        NN_KERN_ASSERT((static_cast<uint64_t>(GetAsInteger(pa)) & ~PhysicalAddressMask) == 0);
        NN_KERN_ASSERT((pa & (DeviceLargePageSize - 1)) == 0);
        NN_KERN_ASSERT(m_Entry == 0);
        asm volatile ("":::"memory");
        m_Entry = ((isReadable? ReadableBit: 0) | (isWriteable? WriteableBit: 0) | (isNonSecure? NonSecureBit: 0) | (pa >> DevicePageShift));
        asm volatile ("":::"memory");
    }
    void Invalidate()
    {
        asm volatile ("":::"memory");
        m_Entry = 0;
        asm volatile ("":::"memory");
    }
private:
    Bit32 m_Entry;
};

struct PageTableEntry
{
public:
    bool IsValid()     const { return m_Entry & (ReadableBit | WriteableBit); }
    bool IsReadable()  const { return m_Entry & ReadableBit;  }
    bool IsWriteable() const { return m_Entry & WriteableBit; }
    bool IsNonSecure() const { return m_Entry & NonSecureBit; }
    KPhysicalAddress GetPhysicalAddress() const { return ((static_cast<uint64_t>(m_Entry) << DevicePageShift) & PhysicalAddressMask); }
    Bit32 GetAttr() const { return (m_Entry & (ReadableBit | WriteableBit | NonSecureBit)); }
    void SetPage(bool isReadable, bool isWriteable, bool isNonSecure, KPhysicalAddress pa)
    {
        NN_KERN_ASSERT((static_cast<uint64_t>(GetAsInteger(pa)) & ~PhysicalAddressMask) == 0);
        NN_KERN_ASSERT((pa & (DevicePageSize - 1)) == 0);
        NN_KERN_ASSERT(m_Entry == 0);
        asm volatile ("":::"memory");
        m_Entry = ((isReadable? ReadableBit: 0) | (isWriteable? WriteableBit: 0) | (isNonSecure? NonSecureBit: 0) | (pa >> DevicePageShift));
        asm volatile ("":::"memory");
    }
    void Invalidate()
    {
        asm volatile ("":::"memory");
        m_Entry = 0;
        asm volatile ("":::"memory");
    }
private:
    Bit32 m_Entry;
};

void SmmuSynchronizationBarrier()
{
    ReadMcRegister(NN_KERN_SMMU_CONFIG_OFFSET);
}

void InvalidatePtc()
{
    WriteMcRegister(NN_KERN_SMMU_PTC_FLUSH_0_OFFSET, 0);
}

void InvalidatePtc(KPhysicalAddress pa)
{
    WriteMcRegister(NN_KERN_SMMU_PTC_FLUSH_1_OFFSET, (static_cast<uint64_t>(GetAsInteger(pa)) >> 32));
    WriteMcRegister(NN_KERN_SMMU_PTC_FLUSH_0_OFFSET, (((pa) & 0xfffffff0) | (1 << 0)));
}

void InvalidateTlb()
{
    WriteMcRegister(NN_KERN_SMMU_TLB_FLUSH_OFFSET, ((0u << 31) | (0 << 24) | (0u << 2) | (0u << 0)));
}

void InvalidateTlb(Bit8 asid)
{
    WriteMcRegister(NN_KERN_SMMU_TLB_FLUSH_OFFSET, ((1u << 31) | ((asid & 0x7F) << 24) | (0u << 2) | (0u << 0)));
}

void InvalidateTlbSection(Bit8 asid, KDeviceVirtualAddress addr)
{
    WriteMcRegister(NN_KERN_SMMU_TLB_FLUSH_OFFSET, ((1u << 31) | ((asid & 0x7F) << 24) | ((addr & 0xffc00000) >> DevicePageShift) | (2u << 0)));
}

void SetTable(Bit8 asid, KPhysicalAddress tablePhysicalAddress)
{
    {
        KScopedLightLock locker(&g_Lock);
        // ASIDを切り替え
        WriteMcRegister(NN_KERN_SMMU_PTB_ASID_OFFSET, asid);

        // 切り替えたASIDのページテーブルを設定
        WriteMcRegister(NN_KERN_SMMU_PTB_DATA_OFFSET, (ReadableBit | WriteableBit | NonSecureBit | (tablePhysicalAddress >> DevicePageShift)));
        SmmuSynchronizationBarrier();
    }

    InvalidatePtc();

    // ASID の TLB を flush all
    InvalidateTlb(asid);
    SmmuSynchronizationBarrier();
}

#if defined NN_BUILD_CONFIG_HARDWARE_NX
const Bit8 g_ReservedByBoot[] = {0, 1, 2, 3};
#endif

class AsidManager
{
public:
    static const size_t NumWords = NumOfAsid / 32;

private:
    Bit32       m_AsidMap[NumWords];
    KLightMutex m_Lock;

    void ReleaseImpl(Bit8 asid)
    {
        Bit32 x = (1u << (asid % 32));
        m_AsidMap[asid / 32] &= ~x;
    }

#if defined NN_BUILD_CONFIG_HARDWARE_NX
    void Reserve(Bit8 asid)
    {
        Bit32 x = (1u << (asid % 32));
        m_AsidMap[asid / 32] |= x;
    }
#endif

public:
    AsidManager()
    {
        std::memset(m_AsidMap, 0, sizeof(m_AsidMap));

#if defined NN_BUILD_CONFIG_HARDWARE_NX
        for (size_t i = 0; i < NN_ARRAY_SIZE(g_ReservedByBoot); i++)
        {
            Reserve(g_ReservedByBoot[i]);
        }
#endif
    }

    Result Acquire(Bit8* pArray, size_t n)
    {
        KScopedLightLock locker(&m_Lock);

        size_t j = 0;
        for (size_t i = 0; i < NumWords; ++i)
        {
            while (m_AsidMap[i] != 0xFFFFFFFF)
            {
                Bit32 zeroBit = (m_AsidMap[i] + 1) ^ (m_AsidMap[i]);
                size_t x = 31 - __builtin_clz(zeroBit);
                pArray[j++] = 32 * i + x;
                m_AsidMap[i] |= zeroBit;
                if (j == n)
                {
                    return ResultSuccess();
                }
            }
        }
        for (size_t i = 0; i < j; ++i)
        {
            ReleaseImpl(pArray[i]);
        }

        return nn::svc::ResultOutOfResource();
    }
    void Release(Bit8 asid)
    {
        KScopedLightLock locker(&m_Lock);
        ReleaseImpl(asid);
    }
};

AsidManager s_AsidManager;

class KSmmuHandler: public KInterruptTask
{
private:

public:
    KSmmuHandler() {}
    virtual KInterruptTask* OnInterrupt(int32_t interruptRequestNo)
    {
        NN_UNUSED(interruptRequestNo);
        return this;
    }
    virtual void DoInterruptTask()
    {
#ifdef NN_KERN_MC_INTERRUPT_ENABLE
        Bit32 intstatus;
        Bit32 errstatus;
        Bit32 erradr;
        {
            intstatus = ReadMcRegister(NN_KERN_MC_INTSTATUS_OFFSET);
            errstatus = ReadMcRegister(NN_KERN_MC_ERRSTATUS_OFFSET);
            erradr = ReadMcRegister(NN_KERN_MC_ERRADR_OFFSET);
            WriteMcRegister(NN_KERN_MC_INTSTATUS_OFFSET, intstatus);
        }
        static const char* errType[] =
        {
            "RSVD",
            "Unknown",
            "DECERR_EMEM",
            "SECURITY_TRUSTZONE",
            "SECURITY_CARVEOUT",
            "Unknown",
            "INVALID_SMMU_PAGE",
            "Unknown"
        };

        NN_UNUSED(intstatus);
        NN_UNUSED(errstatus);
        NN_UNUSED(erradr);
        NN_UNUSED(errType);

        NN_KERN_RELEASE_LOG("sMMU error interrupt\n");
        NN_KERN_RELEASE_LOG("    MC_INTSTATUS=%08x\n", intstatus);
        NN_KERN_RELEASE_LOG("        DECERR_GENERALIZED_CARVEOUT=%d\n", ((intstatus >> 17) & 1));
        NN_KERN_RELEASE_LOG("        DECERR_MTS=%d\n", ((intstatus >> 16) & 1));
        NN_KERN_RELEASE_LOG("        SECERR_SEC=%d\n", ((intstatus >> 13) & 1));
        NN_KERN_RELEASE_LOG("        DECERR_VPR=%d\n", ((intstatus >> 12) & 1));
        NN_KERN_RELEASE_LOG("        INVALID_APB_ASID_UPDATE=%d\n", ((intstatus >> 11) & 1));
        NN_KERN_RELEASE_LOG("        INVALID_SMMU_PAGE=%d\n", ((intstatus >> 10) & 1));
        NN_KERN_RELEASE_LOG("        ARBITRATION_EMEM=%d\n", ((intstatus >> 9) & 1));
        NN_KERN_RELEASE_LOG("        SECURITY_VIOLATION=%d\n", ((intstatus >> 8) & 1));
        NN_KERN_RELEASE_LOG("        DECERR_EMEM=%d\n", ((intstatus >> 6) & 1));
        NN_KERN_RELEASE_LOG("    MC_ERRSTATUS=%08x\n", errstatus);
        NN_KERN_RELEASE_LOG("        ERR_TYPE=%d (%s)\n", ((errstatus >> 28) & 7), errType[((errstatus >> 28) & 7)]);
        NN_KERN_RELEASE_LOG("        ERR_INVALID_SMMU_PAGE_READABLE=%d\n", ((errstatus >> 27) & 1));
        NN_KERN_RELEASE_LOG("        ERR_INVALID_SMMU_PAGE_WRITABLE=%d\n", ((errstatus >> 26) & 1));
        NN_KERN_RELEASE_LOG("        ERR_INVALID_SMMU_NONSECURE=%d\n", ((errstatus >> 25) & 1));
        NN_KERN_RELEASE_LOG("        ERR_ADR_HI=%x\n", ((errstatus >> 20) & 3));
        NN_KERN_RELEASE_LOG("        ERR_SWAP=%d\n", ((errstatus >> 18) & 1));
        NN_KERN_RELEASE_LOG("        ERR_SECURITY=%d %s\n", ((errstatus >> 17) & 1), (((errstatus >> 17) & 1)? "SECURE":"NON SECURE"));
        NN_KERN_RELEASE_LOG("        ERR_RW=%d %s\n", ((errstatus >> 16) & 1), (((errstatus >> 16) & 1)? "WRITE": "READ"));
        NN_KERN_RELEASE_LOG("        ERR_ADR1=%x\n", ((errstatus >> 12) & 7));
        NN_KERN_RELEASE_LOG("        ERR_ID=%d %s\n", ((errstatus >> 0) & 0xFF), GetSmmuClientName(((errstatus >> 0) & 0xFF)));
        NN_KERN_RELEASE_LOG("    MC_ERRADR=%08x\n", erradr);
        NN_KERN_RELEASE_LOG("        ERR_ADR=%llx\n", (static_cast<uint64_t>((errstatus >> 20) & 3) << 32) | erradr);
        NN_KERN_RELEASE_LOG("\n");
        Kernel::GetInterruptManager().ClearInterrupt(KInterruptName::INTR_CORE_ID_MC);
#endif
    }
};
KSmmuHandler s_SmmuHandler;

#if 0
bool CheckDirectoryFree(PageDirectoryEntry* pTable)
{
    for (int i = 0; i < PageDirectorySize / sizeof(PageDirectoryEntry); i++)
    {
        if (pTable[i].IsValid())
        {
            return false;
        }
    }
    return true;
}

bool CheckTableFree(PageTableEntry* pTable)
{
    for (int i = 0; i < PageTableSize / sizeof(PageTableEntry); i++)
    {
        if (pTable[i].IsValid())
        {
            return false;
        }
    }
    return true;
}
#endif

void ClearPageTable(KVirtualAddress addr)
{
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    for (size_t offset = 0; offset < PageTableSize; offset += KCPU::DATA_CACHE_LINE_SIZE)
    {
        uint64_t vaddr = (GetAsInteger(addr) + offset);
        asm volatile ("DC ZVA, %0"::"r"(vaddr):"memory");
    }
#else
    std::memset(GetUntypedPointer(addr), 0, PageTableSize);
#endif
}
void ClearPageDirectory(KVirtualAddress addr)
{
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
    for (size_t offset = 0; offset < PageDirectorySize; offset += KCPU::DATA_CACHE_LINE_SIZE)
    {
        uint64_t vaddr = (GetAsInteger(addr) + offset);
        asm volatile ("DC ZVA, %0"::"r"(vaddr):"memory");
    }
#else
    std::memset(GetUntypedPointer(addr), 0, PageDirectorySize);
#endif
}
}

void KDevicePageTable::PrepareInitialize(const KProcessAddress* pRegisterBase)
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_UNUSED(pRegisterBase);
    NN_KERN_ABORT();
#else
    g_RegisterBase = *pRegisterBase;
#endif
}
KPhysicalAddress KDevicePageTable::GetRegisterAddress(int index)
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_UNUSED(index);
    NN_KERN_ABORT();
    return Null<KPhysicalAddress>();
#else
    switch (index)
    {
    case 0: return NN_KERN_P_ADDR_MC_REGISTER;
    default:
            NN_KERN_ABORT();
            break;
    }
    return Null<KPhysicalAddress>();
#endif
}
int KDevicePageTable::GetRegisterNum()
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    return 0;
#else
    return 1;
#endif
}
size_t KDevicePageTable::GetRegisterSize(int index)
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_UNUSED(index);
    NN_KERN_ABORT();
    return 0;
#else
    switch (index)
    {
    case 0: return NN_KERN_P_ADDR_MC_REGISTER_SIZE;
    default:
            NN_KERN_ABORT();
            break;
    }
    return 0;
#endif
}

void KDevicePageTable::Initialize()
{
    KPageTableManager& pm = Kernel::GetPageTableManager();
    NN_KERN_ASSERT(pm.PageTableSize == PageTableSize);
    NN_KERN_ASSERT(pm.PageTableSize == PageDirectorySize);
    const KVirtualAddress tableVAddress = pm.Allocate();
    if (tableVAddress == Null<KVirtualAddress>())
    {
        NN_KERNEL_PANIC("reserved table allocate faild");
    }
    const KPhysicalAddress tablePAddress = GetPageTablePhysicalAddress(tableVAddress);

    NN_KERN_ASSERT((static_cast<uint64_t>(GetAsInteger(tablePAddress) - 1) & ~PhysicalAddressMask) == 0);
    pm.Open(tableVAddress, 1);
    ClearPageDirectory(tableVAddress);
    KCPU::StoreDataCache(GetUntypedPointer(tableVAddress), PageDirectorySize);
    g_ReservedTablePhysicalAddress = tablePAddress;

    Result result = s_AsidManager.Acquire(&g_ReservedAsid, 1);
    if (result.IsFailure())
    {
        NN_KERNEL_PANIC("reserved asid allocate faild");
    }

    // すべてのASIDは無効なテーブルを指す
    for (size_t i = 0; i < NumOfAsid; i++)
    {
        SetTable(i, g_ReservedTablePhysicalAddress);
    }

    // すべてデタッチ状態にする
    for (size_t i = 0; i < nn::svc::DeviceName_Num; i++)
    {
        Bit32 value = 0;

        value |= (1u << 31);
        if (HighAddressSupportDevice & (1ull << i))
        {
            for (size_t j = 0; j < NumOfTable; j++)
            {
                value |= (g_ReservedAsid << (8 * j));
            }
        }
        else
        {
            value |= g_ReservedAsid;
        }

        WriteMcRegister(GetAsidRegister(static_cast<nn::svc::DeviceName>(i)), value);
        SmmuSynchronizationBarrier();
    }

    InvalidatePtc();
    InvalidateTlb();
    SmmuSynchronizationBarrier();

    WriteMcRegister(NN_KERN_MC_INTSTATUS_OFFSET, ReadMcRegister(NN_KERN_MC_INTSTATUS_OFFSET));

#ifdef NN_KERN_MC_INTERRUPT_ENABLE
    WriteMcRegister(NN_KERN_MC_INTMASK_OFFSET,(0
        | (1u << 17)
        | (1u << 16)
        | (1u << 13)
        | (1u << 12)
        | (1u << 11)
        | (1u << 10)
        | (0u << 9)
        | (1u << 8)
        | (1u << 6)));
#endif

    WriteMcRegister(NN_KERN_SMMU_CONFIG_OFFSET, 1);
    SmmuSynchronizationBarrier();

#ifdef NN_KERN_MC_INTERRUPT_ENABLE
    Kernel::GetInterruptManager().BindHandler(&s_SmmuHandler, KInterruptName::INTR_CORE_ID_MC, GetCurrentCpuNo(), KInterruptController::PriorityLevel_DevInterrupt, true, true);
#endif
}

void KDevicePageTable::Lock()
{
    g_Lock.Lock();
}

void KDevicePageTable::Unlock()
{
    g_Lock.Unlock();
}

void KDevicePageTable::Sleep()
{
    for (size_t i = 0; i < NumOfAsid; i++)
    {
        WriteMcRegister(NN_KERN_SMMU_PTB_ASID_OFFSET, i);
        SmmuSynchronizationBarrier();
        g_SavedPageTable[i] = ReadMcRegister(NN_KERN_SMMU_PTB_DATA_OFFSET);
    }
    for (size_t i = 0; i < nn::svc::DeviceName_Num; i++)
    {
        g_SavedAsidRegister[i] = ReadMcRegister(GetAsidRegister(static_cast<nn::svc::DeviceName>(i)));
    }
}

void KDevicePageTable::Wakeup()
{
    InvalidatePtc();
    InvalidateTlb();
    SmmuSynchronizationBarrier();

    WriteMcRegister(NN_KERN_SMMU_CONFIG_OFFSET, 0);
    for (size_t i = 0; i < NumOfAsid; i++)
    {
        WriteMcRegister(NN_KERN_SMMU_PTB_ASID_OFFSET, i);
        SmmuSynchronizationBarrier();
        WriteMcRegister(NN_KERN_SMMU_PTB_DATA_OFFSET, g_SavedPageTable[i]);
    }
    SmmuSynchronizationBarrier();
    for (size_t i = 0; i < nn::svc::DeviceName_Num; i++)
    {
        WriteMcRegister(GetAsidRegister(static_cast<nn::svc::DeviceName>(i)), g_SavedAsidRegister[i]);
        SmmuSynchronizationBarrier();
    }
    InvalidatePtc();
    InvalidateTlb();
    SmmuSynchronizationBarrier();
    WriteMcRegister(NN_KERN_SMMU_CONFIG_OFFSET, 1);
    SmmuSynchronizationBarrier();
}

void KDevicePageTable::Dump()
{
    for (size_t i = 0; i < NumOfAsid; i++)
    {
        WriteMcRegister(NN_KERN_SMMU_PTB_ASID_OFFSET, i);
        SmmuSynchronizationBarrier();
        NN_LOG("KDevicePageTable:Dump ASID=%d PTB=%08x\n", i, ReadMcRegister(NN_KERN_SMMU_PTB_DATA_OFFSET));
    }
    for (size_t i = 0; i < nn::svc::DeviceName_Num; i++)
    {
        g_SavedAsidRegister[i] = ReadMcRegister(GetAsidRegister(static_cast<nn::svc::DeviceName>(i)));
        NN_LOG("KDevicePageTable:Dump dev=%d Reg=%08x\n", i, g_SavedAsidRegister[i]);
    }
    SmmuSynchronizationBarrier();
}

Result KDevicePageTable::Initialize(uint64_t spaceAddress, uint64_t spaceSize)
{
    if ((spaceAddress + spaceSize - 1) & ~DeviceVirtualAddressMask)
    {
        return nn::svc::ResultInvalidRegion();
    }

    Result result = ResultSuccess();

    size_t startIndex = spaceAddress / DeviceRootRegionSize;
    size_t endIndex = (spaceAddress + spaceSize - 1) / DeviceRootRegionSize;

    for (size_t i = 0; i < sizeof(m_Table) / sizeof(*m_Table); i++)
    {
        m_Table[i] = Null<KVirtualAddress>();
    }

    KPageTableManager& pm = Kernel::GetPageTableManager();
    for (size_t i = startIndex; i <= endIndex ; i++)
    {
        const KVirtualAddress tableVAddress = pm.Allocate();
        if (tableVAddress == Null<KVirtualAddress>())
        {
            for (size_t j = 0; j < i; j++)
            {
                if (pm.Close(m_Table[j], 1))
                {
                    pm.Free(m_Table[j]);
                }
            }
            return nn::svc::ResultOutOfMemory();
        }
        NN_KERN_ASSERT((static_cast<uint64_t>(GetAsInteger(GetPageTablePhysicalAddress(tableVAddress) - 1)) & ~PhysicalAddressMask) == 0);
        pm.Open(tableVAddress, 1);
        ClearPageDirectory(tableVAddress);
        KCPU::StoreDataCache(GetUntypedPointer(tableVAddress), PageDirectorySize);
        m_Table[i] = tableVAddress;
    }

    for (size_t i = 0; i < NumOfTable; i++)
    {
        m_Asid[i] = g_ReservedAsid;
    }
    result = s_AsidManager.Acquire(&m_Asid[startIndex], endIndex - startIndex + 1);

    if (result.IsFailure())
    {
        for (size_t i = startIndex; i <= endIndex; i++)
        {
            if (pm.Close(m_Table[i], 1))
            {
                pm.Free(m_Table[i]);
            }
        }
        return result;
    }

    // ここで ASID と テーブルを結びつける
    for (size_t i = startIndex; i <= endIndex; i++)
    {
        SetTable(m_Asid[i], GetPageTablePhysicalAddress(m_Table[i]));
    }

    m_AttachedDevice = 0;

    m_AttachedValue = (1u << 31);
    m_DetachedValue = 0;
    m_HighSupportAttachedValue = (1u << 31);
    m_HighSupportDetachedValue = 0;

    m_DetachedValue |= (1u << 31);
    m_HighSupportDetachedValue |= (1u << 31);

    for (size_t i = 0; i < NumOfTable; i++)
    {
        m_HighSupportAttachedValue |= ((m_Asid[i]) << (8 * i));
        m_HighSupportDetachedValue |= (g_ReservedAsid << (8 * i));
    }

    m_AttachedValue |= m_Asid[0];
    m_DetachedValue |= g_ReservedAsid;

    return ResultSuccess();
}

void KDevicePageTable::Finalize()
{
    KPageTableManager& pm = Kernel::GetPageTableManager();

    // デタッチ
    {
        KScopedLightLock locker(&g_Lock);
        for (size_t i = 0; i < nn::svc::DeviceName_Num; i++)
        {
            if (m_AttachedDevice & (1ull << i))
            {
                WriteMcRegister(GetAsidRegister(static_cast<nn::svc::DeviceName>(i)), ((HighAddressSupportDevice & (1ull << i))? m_HighSupportDetachedValue: m_DetachedValue));
                SmmuSynchronizationBarrier();
            }
        }
    }

    // page table の解除
    Unmap(0, 1ull << DeviceVirtualAddressWidth, true);

    for (size_t i = 0; i < NumOfTable ; i++)
    {
        if (m_Asid[i] != g_ReservedAsid)
        {
            SetTable(m_Asid[i], g_ReservedTablePhysicalAddress);

            //NN_KERN_ASSERT(CheckDirectoryFree(GetTypedPointer<PageDirectoryEntry>(m_Table[i])));
            KVirtualAddress tableAddress = m_Table[i];
            NN_KERN_ASSERT(pm.GetRefCount(tableAddress) == 1);
            NN_KERN_ABORT_UNLESS(pm.Close(tableAddress, 1));
            pm.Free(tableAddress);
            s_AsidManager.Release(m_Asid[i]);
        }
    }
}

Result KDevicePageTable::Attach(nn::svc::DeviceName deviceName, uint64_t spaceAddress, uint64_t spaceSize)
{
    if (deviceName < 0 || nn::svc::DeviceName_Num <= deviceName)
    {
        return nn::svc::ResultNotFound();
    }

    // アタッチ済みチェック
    if (m_AttachedDevice & (1ull << deviceName))
    {
        return nn::svc::ResultBusy();
    }

    // 34Bitテーブルに 34Bitをサポートしていないデバイス設定しないかチェック
    size_t endIndex = (spaceAddress + spaceSize - 1) / DeviceRootRegionSize;
    if (endIndex >= 1 && (HighAddressSupportDevice & (1ull << deviceName)) == 0)
    {
        return nn::svc::ResultInvalidCombination();
    }
    if (!IsAttachable(deviceName, spaceAddress, spaceSize))
    {
        return nn::svc::ResultInvalidCombination();
    }

    int offset = GetAsidRegister(deviceName);
    if (offset < 0)
    {
        return nn::svc::ResultNotFound();
    }

    Bit32 expectValue = ((HighAddressSupportDevice & (1ull << deviceName))? m_HighSupportDetachedValue: m_DetachedValue);
    Bit32 newValue = ((HighAddressSupportDevice & (1ull << deviceName))? m_HighSupportAttachedValue: m_AttachedValue);

    {
        KScopedLightLock locker(&g_Lock);
        // 割り当て済み?
        if ((ReadMcRegister(offset) | (1u << 31)) != (expectValue | (1u << 31)))
        {
            return nn::svc::ResultBusy();
        }

        WriteMcRegister(offset, newValue);
        SmmuSynchronizationBarrier();

        if (ReadMcRegister(offset) != newValue)
        {
            // 書き込みに失敗(Secure or hard code disable)
            WriteMcRegister(offset, expectValue);
            SmmuSynchronizationBarrier();
            return nn::svc::ResultNotFound();
        }
    }

    m_AttachedDevice |= (1ull << deviceName);

    return ResultSuccess();
}

Result KDevicePageTable::Detach(nn::svc::DeviceName deviceName)
{
    if (deviceName < 0 || nn::svc::DeviceName_Num <= deviceName)
    {
        return nn::svc::ResultNotFound();
    }
    if ((m_AttachedDevice & (1ull << deviceName)) == 0)
    {
        return nn::svc::ResultInvalidState();
    }

    int offset = GetAsidRegister(deviceName);
    NN_KERN_ASSERT(offset >= 0);

    Bit32 expectValue = ((HighAddressSupportDevice & (1ull << deviceName))? m_HighSupportAttachedValue: m_AttachedValue);
    NN_UNUSED(expectValue);
    Bit32 newValue = ((HighAddressSupportDevice & (1ull << deviceName))? m_HighSupportDetachedValue: m_DetachedValue);

    {
        KScopedLightLock locker(&g_Lock);
        // 割り当て済み?
        NN_KERN_ASSERT(ReadMcRegister(offset) == expectValue);

        WriteMcRegister(offset, newValue);
        SmmuSynchronizationBarrier();
        NN_KERN_ASSERT((ReadMcRegister(offset) | (1u << 31)) == (newValue | (1u << 31)));
    }

    m_AttachedDevice &= ~(1ull << deviceName);

    return ResultSuccess();
}

bool KDevicePageTable::IsFree(KDeviceVirtualAddress addr, uint64_t size) const
{
    NN_KERN_ASSERT((addr & ~DeviceVirtualAddressMask) == 0);
    NN_KERN_ASSERT(((addr + size - 1) & ~DeviceVirtualAddressMask) == 0);
    uint64_t leftSize = size;

    while (leftSize)
    {
        size_t l0Index = (addr / DeviceRootRegionSize);
        size_t l1Index = ((addr % DeviceRootRegionSize) / DeviceLargePageSize);
        size_t l2Index = ((addr % DeviceRootRegionSize) % DeviceLargePageSize) / DevicePageSize;
        PageDirectoryEntry* pL1 = GetTypedPointer<PageDirectoryEntry>(m_Table[l0Index]);

        if (!pL1 || !pL1[l1Index].IsValid())
        {
            size_t leftEntry = (PageTableSize / sizeof(PageTableEntry)) - l2Index;
            size_t numMap = ((leftEntry > (leftSize / DevicePageSize))? (leftSize / DevicePageSize): leftEntry);

            addr     += DevicePageSize * numMap;
            leftSize -= DevicePageSize * numMap;
        }
        else if (pL1[l1Index].IsTable())
        {
            PageTableEntry* pL2 = GetTypedPointer<PageTableEntry>(GetPageTableVirtualAddress(pL1[l1Index].GetPhysicalAddress()));
            size_t leftEntry = (PageTableSize / sizeof(PageTableEntry)) - l2Index;
            size_t numMap = ((leftEntry > (leftSize / DevicePageSize))? (leftSize / DevicePageSize): leftEntry);

            for (size_t i = 0; i < numMap; i++)
            {
                if (pL2[l2Index + i].IsValid())
                {
                    return false;
                }
            }

            addr     += DevicePageSize * numMap;
            leftSize -= DevicePageSize * numMap;
        }
        else
        {
            return false;
        }
    }
    return true;
}

Result KDevicePageTable::MapDevicePage(size_t* pMapSize, int* pNumPageTable, int pageTableLimit, KPhysicalAddress paddr, uint64_t size, KDeviceVirtualAddress addr, nn::svc::MemoryPermission devicePermission)
{
    *pMapSize = 0;
    if ((static_cast<uint64_t>(GetAsInteger(paddr)) + size - 1) & ~PhysicalAddressMask)
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    NN_KERN_ASSERT((addr & ~DeviceVirtualAddressMask) == 0);
    NN_KERN_ASSERT(((addr + size - 1) & ~DeviceVirtualAddressMask) == 0);

    uint64_t leftSize = size;

    KPageTableManager& pm = Kernel::GetPageTableManager();
    KMemoryManager& mm = Kernel::GetKernelHeapManager();

    while (leftSize)
    {
        size_t l0Index = (addr / DeviceRootRegionSize);
        size_t l1Index = ((addr % DeviceRootRegionSize) / DeviceLargePageSize);
        size_t l2Index = ((addr % DeviceRootRegionSize) % DeviceLargePageSize) / DevicePageSize;
        PageDirectoryEntry* pL1 = GetTypedPointer<PageDirectoryEntry>(m_Table[l0Index]);
        NN_KERN_ASSERT(pL1);

        if (!pL1[l1Index].IsTable())
        {
            NN_KERN_ASSERT(!pL1[l1Index].IsValid());

            if (l2Index == 0 && ((paddr & (DeviceLargePageSize - 1)) == 0) && leftSize >= DeviceLargePageSize)
            {
                pL1[l1Index].SetLargePage(devicePermission & nn::svc::MemoryPermission_Read, devicePermission & nn::svc::MemoryPermission_Write, true, paddr);
                KCPU::StoreDataCache(&pL1[l1Index], sizeof(PageDirectoryEntry));
                InvalidatePtc(GetPageTablePhysicalAddress(KVirtualAddress(&pL1[l1Index])));
                InvalidateTlbSection(m_Asid[l0Index], addr);
                SmmuSynchronizationBarrier();

                mm.Open(GetPageTableVirtualAddress(paddr), DeviceLargePageSize / pm.PageTableSize);
                paddr    += DeviceLargePageSize;
                addr     += DeviceLargePageSize;
                leftSize -= DeviceLargePageSize;
                *pMapSize += DeviceLargePageSize;
                continue;
            }
            else if (*pNumPageTable == pageTableLimit)
            {
                break;
            }
            else
            {
                const KVirtualAddress tableVAddress = pm.Allocate();
                if (tableVAddress == Null<KVirtualAddress>())
                {
                    return nn::svc::ResultOutOfMemory();
                }
                NN_KERN_ASSERT((static_cast<uint64_t>(GetAsInteger(GetPageTablePhysicalAddress(tableVAddress) - 1)) & ~PhysicalAddressMask) == 0);
                ClearPageTable(tableVAddress);
                KCPU::StoreDataCache(GetUntypedPointer(tableVAddress), PageTableSize);

                pL1[l1Index].SetTable(true, true, true, GetPageTablePhysicalAddress(tableVAddress));
                KCPU::StoreDataCache(&pL1[l1Index], sizeof(PageDirectoryEntry));
                InvalidatePtc(GetPageTablePhysicalAddress(KVirtualAddress(&pL1[l1Index])));
                InvalidateTlbSection(m_Asid[l0Index], addr);
                SmmuSynchronizationBarrier();
                (*pNumPageTable)++;
            }
        }

        NN_KERN_ASSERT(pL1[l1Index].IsTable());
        {
            PageTableEntry* pL2 = GetTypedPointer<PageTableEntry>(GetPageTableVirtualAddress(pL1[l1Index].GetPhysicalAddress()));
            size_t leftEntry = (PageTableSize / sizeof(PageTableEntry)) - l2Index;
            size_t numMap = ((leftEntry > (leftSize / DevicePageSize))? (leftSize / DevicePageSize): leftEntry);

            for (size_t i = 0; i < numMap; i++)
            {
                NN_KERN_ASSERT(!pL2[l2Index + i].IsValid());
                pL2[l2Index + i].SetPage(devicePermission & nn::svc::MemoryPermission_Read, devicePermission & nn::svc::MemoryPermission_Write, true, paddr + DevicePageSize * i);
                pm.Open(KVirtualAddress(reinterpret_cast<uintptr_t>(pL2)), 1);
            }
            KCPU::StoreDataCache(&pL2[l2Index], sizeof(PageTableEntry) * numMap);

            for (size_t i = (l2Index & ~0x3); i <= ((l2Index + numMap - 1) & ~0x3); i += 4)
            {
                InvalidatePtc(GetPageTablePhysicalAddress(KVirtualAddress(&pL2[i])));
            }

            InvalidateTlbSection(m_Asid[l0Index], addr);
            SmmuSynchronizationBarrier();

            mm.Open(GetHeapVirtualAddress(paddr), DevicePageSize * numMap / NN_KERN_FINEST_PAGE_SIZE);
            paddr    += DevicePageSize * numMap;
            addr     += DevicePageSize * numMap;
            leftSize -= DevicePageSize * numMap;
            *pMapSize += DevicePageSize * numMap;
        }
    }
    return ResultSuccess();
}

void KDevicePageTable::Unmap(KDeviceVirtualAddress addr, uint64_t size, bool force)
{
    NN_UNUSED(force);
    uint64_t leftSize = size;
    NN_KERN_ASSERT((addr & ~DeviceVirtualAddressMask) == 0);
    NN_KERN_ASSERT(((addr + size - 1) & ~DeviceVirtualAddressMask) == 0);

    KScopedPageGroup pg(&Kernel::GetBlockInfoManager());

    KMemoryManager& mm = Kernel::GetKernelHeapManager();
    KPageTableManager& pm = Kernel::GetPageTableManager();

    while (leftSize)
    {
        size_t l0Index = (addr / DeviceRootRegionSize);
        size_t l1Index = ((addr % DeviceRootRegionSize) / DeviceLargePageSize);
        size_t l2Index = ((addr % DeviceRootRegionSize) % DeviceLargePageSize) / DevicePageSize;
        PageDirectoryEntry* pL1 = GetTypedPointer<PageDirectoryEntry>(m_Table[l0Index]);

        if (!pL1 || !pL1[l1Index].IsValid())
        {
            NN_KERN_ASSERT(force);
            size_t leftEntry = (PageTableSize / sizeof(PageTableEntry)) - l2Index;
            size_t numMap = ((leftEntry > (leftSize / DevicePageSize))? (leftSize / DevicePageSize): leftEntry);

            addr     += DevicePageSize * numMap;
            leftSize -= DevicePageSize * numMap;
        }
        else if (pL1[l1Index].IsTable())
        {
            PageTableEntry* pL2 = GetTypedPointer<PageTableEntry>(GetPageTableVirtualAddress(pL1[l1Index].GetPhysicalAddress()));
            size_t leftEntry = (PageTableSize / sizeof(PageTableEntry)) - l2Index;
            size_t numMap = ((leftEntry > (leftSize / DevicePageSize))? (leftSize / DevicePageSize): leftEntry);
            size_t numClose = 0;
            bool tlbInvalidated = false;

            for (size_t i = 0; i < numMap; i++)
            {
                if (pL2[l2Index + i].IsValid())
                {
                    KPhysicalAddress paddr = pL2[l2Index + i].GetPhysicalAddress();
                    NN_KERN_ASSERT(IsHeapPhysicalAddress(paddr));
                    pL2[l2Index + i].Invalidate();
                    numClose++;
                    Result result = pg.GetPageGroup().AddBlock(GetHeapVirtualAddress(paddr), DevicePageSize / NN_KERN_FINEST_PAGE_SIZE);
                    if (result.IsFailure())
                    {
                        KCPU::StoreDataCache(&pL2[l2Index + i], sizeof(PageTableEntry));
                        InvalidatePtc(GetPageTablePhysicalAddress(KVirtualAddress(&pL2[l2Index + i])));
                        SmmuSynchronizationBarrier();
                        mm.Close(GetHeapVirtualAddress(paddr), DeviceLargePageSize / NN_KERN_FINEST_PAGE_SIZE);
                    }
                }
                else
                {
                    NN_KERN_ASSERT(force);
                }
            }
            KCPU::StoreDataCache(&pL2[l2Index], sizeof(PageTableEntry) * numMap);

            for (size_t i = (l2Index & ~0x3); i <= ((l2Index + numMap - 1) & ~0x3); i += 4)
            {
                InvalidatePtc(GetPageTablePhysicalAddress(KVirtualAddress(&pL2[i])));
            }
            SmmuSynchronizationBarrier();

            if (pm.Close(KVirtualAddress(reinterpret_cast<uintptr_t>(pL2)), numClose))
            {
                pL1[l1Index].Invalidate();
                KCPU::StoreDataCache(&pL1[l1Index], sizeof(PageDirectoryEntry));

                InvalidatePtc(GetPageTablePhysicalAddress(KVirtualAddress(&pL1[l1Index])));
                InvalidateTlbSection(m_Asid[l0Index], addr);
                SmmuSynchronizationBarrier();
                tlbInvalidated = true;
                pm.Free(KVirtualAddress(reinterpret_cast<uintptr_t>(pL2)));
                // NN_KERN_ASSERT(CheckTableFree(pL2));
            }

            if (!tlbInvalidated)
            {
                InvalidateTlbSection(m_Asid[l0Index], addr);
                SmmuSynchronizationBarrier();
            }

            addr     += DevicePageSize * numMap;
            leftSize -= DevicePageSize * numMap;
        }
        else
        {
            NN_KERN_ASSERT(l2Index == 0);
            KPhysicalAddress paddr = pL1[l1Index].GetPhysicalAddress();
            NN_KERN_ASSERT(IsHeapPhysicalAddress(paddr));
            pL1[l1Index].Invalidate();
            KCPU::StoreDataCache(&pL1[l1Index], sizeof(PageDirectoryEntry));
            InvalidatePtc(GetPageTablePhysicalAddress(KVirtualAddress(&pL1[l1Index])));
            InvalidateTlbSection(m_Asid[l0Index], addr);
            SmmuSynchronizationBarrier();
            mm.Close(GetHeapVirtualAddress(paddr), DeviceLargePageSize / NN_KERN_FINEST_PAGE_SIZE);

            addr     += DeviceLargePageSize;
            leftSize -= DeviceLargePageSize;
        }
    }
    pg.GetPageGroup().Close();
}

Result KDevicePageTable::MapImpl(size_t* pMapSize, int* pNumPageTable, int pageTableLimit, const KPageGroup& pageGroup, KDeviceVirtualAddress deviceAddress, nn::svc::MemoryPermission devicePermission)
{
    *pMapSize = 0;
    uint64_t totalSize = pageGroup.GetTotalNumPages() * NN_KERN_FINEST_PAGE_SIZE;
    NN_KERN_ASSERT((deviceAddress & ~DeviceVirtualAddressMask) == 0);
    NN_KERN_ASSERT(((deviceAddress + totalSize - 1) & ~DeviceVirtualAddressMask) == 0);

    if (!IsFree(deviceAddress, totalSize))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KDeviceVirtualAddress addr = deviceAddress;
    Result result = ResultSuccess();
    for (KPageGroup::BlockInfoList::const_iterator it = pageGroup.GetBlockBeginIter(); it != pageGroup.GetBlockEndIter(); it++)
    {
        if (!IsHeapVirtualAddress(it->GetBlockAddr()))
        {
            result = nn::svc::ResultInvalidCurrentMemory();
            break;
        }

        KPhysicalAddress paddr = GetHeapPhysicalAddress(it->GetBlockAddr());
        uint64_t size = it->GetNumPages() * NN_KERN_FINEST_PAGE_SIZE;
        size_t mapSize = 0;
        result = MapDevicePage(&mapSize, pNumPageTable, pageTableLimit, paddr, size, addr, devicePermission);
        if (result.IsFailure())
        {
            break;
        }
        addr += size;
        *pMapSize += mapSize;
        if (mapSize < size)
        {
            break;
        }
    }
    if (result.IsFailure())
    {
        Unmap(deviceAddress, totalSize, true);
    }
    return result;
}

Result KDevicePageTable::Map(size_t* pMapSize, const KPageGroup& pageGroup, KDeviceVirtualAddress deviceAddress, nn::svc::MemoryPermission devicePermission, bool breakAllocate)
{
    *pMapSize = 0;
    int numPageTable = 0;
    return MapImpl(pMapSize, &numPageTable, (breakAllocate? 1: INT_MAX), pageGroup, deviceAddress, devicePermission);
}

Result KDevicePageTable::MakePageGroup(KPageGroup* pOut, KDeviceVirtualAddress addr, uint64_t size) const
{
    uint64_t leftSize = size;
    NN_KERN_ASSERT((addr & ~DeviceVirtualAddressMask) == 0);
    NN_KERN_ASSERT(((addr + size - 1) & ~DeviceVirtualAddressMask) == 0);

    bool first = true;
    Bit32 attr = 0;

    while (leftSize)
    {
        size_t l0Index = (addr / DeviceRootRegionSize);
        size_t l1Index = ((addr % DeviceRootRegionSize) / DeviceLargePageSize);
        size_t l2Index = ((addr % DeviceRootRegionSize) % DeviceLargePageSize) / DevicePageSize;
        PageDirectoryEntry* pL1 = GetTypedPointer<PageDirectoryEntry>(m_Table[l0Index]);

        if (!pL1 || !pL1[l1Index].IsValid())
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        else if (pL1[l1Index].IsTable())
        {
            const PageTableEntry* pL2 = GetTypedPointer<PageTableEntry>(GetPageTableVirtualAddress(pL1[l1Index].GetPhysicalAddress()));
            size_t leftEntry = (PageTableSize / sizeof(PageTableEntry)) - l2Index;
            size_t numMap = ((leftEntry > (leftSize / DevicePageSize))? (leftSize / DevicePageSize): leftEntry);

            for (size_t i = 0; i < numMap; i++)
            {
                if (pL2[l2Index + i].IsValid())
                {
                    KPhysicalAddress paddr = pL2[l2Index + i].GetPhysicalAddress();
                    NN_KERN_ASSERT(IsHeapPhysicalAddress(paddr));
                    Result result = pOut->AddBlock(GetHeapVirtualAddress(paddr), DevicePageSize / NN_KERN_FINEST_PAGE_SIZE);
                    if (result.IsFailure())
                    {
                        return result;
                    }
                    if (first)
                    {
                        attr = pL2[l2Index + i].GetAttr();
                        first = false;
                    }
                    else
                    {
                        if (pL2[l2Index + i].GetAttr() != attr)
                        {
                            return nn::svc::ResultInvalidCurrentMemory();
                        }
                    }
                }
                else
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
            }
            addr     += DevicePageSize * numMap;
            leftSize -= DevicePageSize * numMap;
        }
        else
        {
            if (l2Index != 0)
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
            if (leftSize < DeviceLargePageSize)
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
            KPhysicalAddress paddr = pL1[l1Index].GetPhysicalAddress();
            NN_KERN_ASSERT(IsHeapPhysicalAddress(paddr));
            Result result = pOut->AddBlock(GetHeapVirtualAddress(paddr), DeviceLargePageSize / NN_KERN_FINEST_PAGE_SIZE);
            if (result.IsFailure())
            {
                return result;
            }
            if (first)
            {
                attr = pL1[l1Index].GetAttr();
                first = false;
            }
            else
            {
                if (pL1[l1Index].GetAttr() != attr)
                {
                    return nn::svc::ResultInvalidCurrentMemory();
                }
            }

            addr     += DeviceLargePageSize;
            leftSize -= DeviceLargePageSize;
        }
    }
    return ResultSuccess();
}

bool KDevicePageTable::Compare(const KPageGroup& pageGroup, KDeviceVirtualAddress addr) const
{
    bool isSame = false;

    KScopedPageGroup pg(&Kernel::GetBlockInfoManager());

    Result result = MakePageGroup(&pg.GetPageGroup(), addr, pageGroup.GetTotalNumPages() * NN_KERN_FINEST_PAGE_SIZE);
    if (result.IsSuccess())
    {
        if (pg.GetPageGroup().IsSamePages(pageGroup))
        {
            isSame = true;
        }
    }

    return isSame;
}

Result KDevicePageTable::Unmap(const KPageGroup& pageGroup, KDeviceVirtualAddress deviceAddress)
{
    uint64_t totalSize = pageGroup.GetTotalNumPages() * NN_KERN_FINEST_PAGE_SIZE;
    NN_KERN_ASSERT((deviceAddress & ~DeviceVirtualAddressMask) == 0);
    NN_KERN_ASSERT(((deviceAddress + totalSize - 1) & ~DeviceVirtualAddressMask) == 0);

    if (!Compare(pageGroup, deviceAddress))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    Unmap(deviceAddress, totalSize, false);

    return ResultSuccess();
}

}}}
