﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "../../kern_InterlockedSelect.h"
#include <nn/nn_BitTypes.h>

#include "../../kern_Result.h"
#include "../../kern_Assert.h"
#include "../../kern_KLinkedList.h"
#include "../../kern_KTaggedAddress.h"
#include "../../kern_CPUSelect.h"

#include "../../kern_KPageTableBase.h"
#include "../../kern_KPageGroup.h"
#include "../../kern_KMemoryBlock.h"
#include "../../kern_KMemoryManager.h"
#include "../../kern_KInterruptTask.h"
#include "../../kern_InterruptManagerSelect.h"

#include "kern_KPageTableBody.h"
#include "kern_KMemoryBlockManager.h"

namespace nn { namespace kern {

namespace ARMv8A
{

namespace {
    class KPageEntryAttr;
}

class KPageTable
    : public KPageTableBase
{
public:
    typedef KPageTableBody::TraverseContext TraverseContext;
    typedef KPageTableBody::TraverseData TraverseData;

private:
    enum Szc
    {
        Szc_PageSize,
        Szc_ContiguousPageSize,
        Szc_L2BlockSize,
#if defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
        Szc_Smmu4MBlockSize,
#endif
        Szc_ContiguousL2BlockSize,
        Szc_L1BlockSize,
        Szc_Max = Szc_L1BlockSize,
    };
public:
    static const size_t ContiguousPageSize          = PageSize * 16;
    static const size_t L2BlockSize                 = PageSize * 512;
#if defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
    static const size_t Smmu4MBlockSize             = L2BlockSize * 2;
#endif
    static const size_t ContiguousL2BlockSize       = L2BlockSize * 16;
    static const size_t L1BlockSize                 = L2BlockSize * 512;

    static size_t GetPageSizeFromSzc(int szc)
    {
        NN_KERN_ABORT_UNLESS(0 <= szc && szc <= Szc_Max);
        static const size_t pageSizeArray[] = {
            PageSize,
            ContiguousPageSize,
            L2BlockSize,
#if defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
            Smmu4MBlockSize,
#endif
            ContiguousL2BlockSize,
            L1BlockSize
        };
        return pageSizeArray[szc];
    }

    static int GetSzcFromPageSize(size_t pageSize)
    {
        switch (pageSize)
        {
        case PageSize:
            return Szc_PageSize;
        case ContiguousPageSize:
            return Szc_ContiguousPageSize;
        case L2BlockSize:
            return Szc_L2BlockSize;
#if defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
        case Smmu4MBlockSize:
            return Szc_Smmu4MBlockSize;
#endif
        case ContiguousL2BlockSize:
            return Szc_ContiguousL2BlockSize;
        case L1BlockSize:
            return Szc_L1BlockSize;
        default:
            NN_KERN_ABORT();
            return 0;
        }
    }

    size_t GetPageSize(int szc) const
    {
        return GetPageSizeFromSzc(szc);
    }
    int GetPageSizeCountMax() const { return Szc_Max; }

private:
    static const Bit64 PteTypeMask                  = 0x3;
    static const Bit64 PteTypeBlock                 = 0x1;
    static const Bit64 PteTypeTable                 = 0x3;
    static const Bit64 PteTypePage                  = 0x3;

    // readonly
    KPageTableManager*          m_pPageTableManager;
    Bit64                       m_TtbrValue;
    Bit64                       m_TcrValue;
    uint8_t                     m_Asid;

public:
    KPageTable() {}
    ~KPageTable(){}
    KPageTable& operator=(const KPageTable&) = delete;
    KPageTable(const KPageTable&) = delete;

    Result  Initialize(Bit32 spaceId, nn::svc::CreateProcessParameterFlag addressSapceType, bool enableAslr, bool fromBack, KMemoryManager::Region region, KProcessAddress codeRegionAddr, size_t codeRegionSize, KMemoryBlockResourceManager* pMemoryBlockManager, KBlockInfoManager* pBlockInfoManager, KPageTableManager* pPageTableManager);
    Result  InitializeForKernel(void* pTable, KProcessAddress begin, KProcessAddress end);
    Result  Finalize();
    bool GetStaticMap(KProcessAddress* pOut, KPhysicalAddress paddr, size_t numPages);

    // static
    static void Initialize(int32_t coreNo);
    static void Activate(KPageTable* pTable, Bit64 id)
    {
        NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
        KCPU::DataSynchronizationBarrier();
        KCPU::SwitchProcess(pTable->m_TcrValue, pTable->m_TtbrValue, id);
    }
    // for HardwareBreak
    uint8_t GetAsid() { return m_Asid; }
    void Info();
protected:
    Result  OperateImpl(void** ppUpdateData,
            KProcessAddress vaddr, size_t numPages, KPhysicalAddress paddr, bool paIsValid,
            const KPageProperty property, PageTableOperation operation);
    Result  OperateImpl(void** ppUpdateData,
            KProcessAddress vaddr, size_t numPages, const KPageGroup& pg,
            const KPageProperty property, PageTableOperation operation);
    void PageTableUpdaterFinalize(void** ppData);

private:
    Bit64 GetEntryTemplate(const KPageProperty property) const
    {
        Bit64 entryTemplate = HW_MMU_AF | HW_MMU_SH_ISHARED | HW_MMU_PXN;

        if (!IsKernel())
        {
            entryTemplate |= HW_MMU_NGLOBAL;
        }

        if (property.isIo)
        {
            NN_KERN_ABORT_UNLESS(!property.isUncached);
            NN_KERN_ABORT_UNLESS(!(property.permission & (KMemoryPermission_KernelExecute | KMemoryPermission_UserExecute)));

            entryTemplate |= HW_MMU_XN | HW_MMU_ATTRINDX(HW_MMU_ATTRINDEX_DEVICE_NGNRE);
        }
        else if (property.isUncached)
        {
            NN_KERN_ABORT_UNLESS(!(property.permission & (KMemoryPermission_KernelExecute | KMemoryPermission_UserExecute)));

            entryTemplate |= HW_MMU_ATTRINDX(HW_MMU_ATTRINDEX_NORMAL_NC);
        }
        else
        {
            entryTemplate |= HW_MMU_ATTRINDX(HW_MMU_ATTRINDEX_NORMAL);
        }

        if (property.permission != KMemoryPermission_UserReadExecute)
        {
            NN_KERN_ABORT_UNLESS(!(property.permission & (KMemoryPermission_KernelExecute | KMemoryPermission_UserExecute)));
            entryTemplate |= HW_MMU_XN;
        }

        switch ((property.permission & KMemoryPermission_UserReadWrite))
        {
        case KMemoryPermission_UserReadWrite:
            {
                entryTemplate |= HW_MMU_AP_S_RW_U_RW;
            }
            break;
        case KMemoryPermission_KernelReadWrite:
            {
                entryTemplate |= HW_MMU_AP_S_RW_U_NA;
            }
            break;
        case KMemoryPermission_UserRead:
            {
                entryTemplate |= HW_MMU_AP_S_RO_U_RO;
            }
            break;
        case KMemoryPermission_KernelRead:
            {
                entryTemplate |= HW_MMU_AP_S_RO_U_NA;
            }
            break;
        default:
            {
                NN_KERNEL_PANIC("invalid property.permission %x", property.permission);
            }
            break;
        }
        return entryTemplate;
    }
    KPageTableManager& GetPageTableManager() const { return *m_pPageTableManager; }
    Result  MapWithTemplate(
            KProcessAddress vAddr, KPhysicalAddress pAddr,
            size_t numPages, Bit64 entryTemplate, size_t pageSize, void** ppData)
    {
        switch (pageSize)
        {
        case PageSize:
            return MapPagesWithTemplate(vAddr, pAddr, numPages, entryTemplate, ppData);
        case ContiguousPageSize:
            return MapPagesWithTemplate(vAddr, pAddr, numPages, entryTemplate | HW_MMU_CONTIGUOUS, ppData);
#if defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
        case Smmu4MBlockSize:
#endif
        case L2BlockSize:
            return MapL2BlocksWithTemplate(vAddr, pAddr, numPages, entryTemplate, ppData);
        case ContiguousL2BlockSize:
            return MapL2BlocksWithTemplate(vAddr, pAddr, numPages, entryTemplate | HW_MMU_CONTIGUOUS, ppData);
        case L1BlockSize:
            return MapL1BlocksWithTemplate(vAddr, pAddr, numPages, entryTemplate, ppData);
        default:
            {
                NN_KERN_ABORT();
            }
            return nn::svc::ResultOutOfRange();
        }
    }
    void    PteSynchronizationBarrier()
    {
        asm volatile ("dsb ish":::"memory");
    }

    Result  MapAdaptiveWithTemplate(
            KProcessAddress vAddr, const KPageGroup& pg,
            size_t numPages, Bit64 entryTemplate, void** ppData);
    Result  MapAdaptiveContinuousWithTemplate(
            KProcessAddress vAddr, KPhysicalAddress pAddr,
            size_t numPages, Bit64 entryTemplate, void** ppData);

    Result  MapL1BlocksWithTemplate(
            KProcessAddress vAddr, KPhysicalAddress pAddr,
            size_t numPages, Bit64 entryTemplate, void** ppData);
    Result  MapL2BlocksWithTemplate(
            KProcessAddress vAddr, KPhysicalAddress pAddr,
            size_t numPages, Bit64 entryTemplate, void** ppData);
    Result  MapPagesWithTemplate(
            KProcessAddress vAddr, KPhysicalAddress pAddr,
            size_t numPages, Bit64 entryTemplate, void** ppData);

    Result  SeparatePage(KProcessAddress vAddr, size_t pageSize, void** ppData);
    bool    MergePage(KProcessAddress vAddr, void** ppData);
    Result  Unmap(KProcessAddress vAddr, size_t numPages, KPageGroup* pPgToBeClose, void** ppData, bool force);

    Result  UpdateProperties(KProcessAddress vAddr, size_t numPages, Bit64 entryTemplate, bool breakBeforeMake, void** ppData, bool recursive);
    Result  FreePages(KProcessAddress vAddr, size_t numPages, void** pData);
    void    MarkUpdated();
    void    MarkUpdated(KProcessAddress va);
    void    MarkUpdatedForSvStack(KProcessAddress va);

    void    OnTableUpdated()
    {
        KCPU::InvalidateTlbByAsid(m_Asid);
    }
    void    OnTableUpdated(KProcessAddress va)
    {
        KCPU::InvalidateTlbByVaAsid(m_Asid, va);
    }
    void    OnKernelTableUpdated()
    {
        KCPU::InvalidateEntireTlbNoCodeModification();
    }
    void    OnKernelTableUpdatedForSvStack(KProcessAddress va)
    {
        KCPU::InvalidateTlbByMvaNoCodeModification(va);
    }

    KVirtualAddress AllocatePageTable(void** ppData);
    void FreePageTable(void** ppData, KVirtualAddress tableAddress);
};

}

}}

