﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>

#include "../../kern_KSynchronizationObject.h"
#include "../../kern_KSpinLockMutex.h"
#include "../../kern_KDebugBase.h"
#include "../../kern_KObjectAdaptor.h"
#include "../../kern_KWorkerTask.h"
#include "../../kern_PageTableSelect.h"
#include "../../kern_KLightMutex.h"

namespace nn { namespace kern {
    class KProcess;
    class KThread;
    namespace ARMv8A {

class KDebug :
    public KObjectAdaptor<KDebug, KDebugBase>
{
private:
    enum PseudoBreakInstruction
    {
        PseudoBreakInstruction_A64 = 0xE7FFFFFF,
        PseudoBreakInstruction_A32 = 0xE7FFDEFE,
        PseudoBreakInstruction_T32 = 0xB68E
    };

    Result              GetFPUContext(nn::svc::ThreadContext* pContext, KThread* pThread, const Bit32 controlFlags);
    Result              SetFPUContext(const nn::svc::ThreadContext& context, KThread* pThread, const Bit32 controlFlags);

public:
    explicit            KDebug() {}
    virtual             ~KDebug(){}
public:
    static void         PostFinalize(uintptr_t arg) { NN_UNUSED(arg); }
    static uintptr_t    RetrivePc(const KThread& thread);
    Result              GetThreadContext_core(nn::svc::ThreadContext* pContext, KThread* pThread, const Bit32 controlFlags);
    Result              SetThreadContext_core(const nn::svc::ThreadContext& context, KThread* pThread, const Bit32 controlFlags);
    static void         SetPreviousPc();
    static void         PrintRegister(KThread* pThread = nullptr);
    static void         PrintBacktrace(KThread* pThread = nullptr);
    static void         GetCurrentBacktrace(uintptr_t* pOut, size_t num);

    static bool         IsBreakInstruction(Bit32 code, Bit32 cpsr)
    {
        if ((cpsr & HW_PSR_AARCH_MASK) == HW_PSR_AARCH64)
        {
            return (code == PseudoBreakInstruction_A64);
        }
        else
        {
            if (cpsr & HW_PSR_THUMB_STATE)
            {
                return (code == PseudoBreakInstruction_T32);
            }
            else
            {
                return (code == PseudoBreakInstruction_A32);
            }
        }
    }
    static Result SetHardwareBreakPoint(nn::svc::HardwareBreakPointRegisterName regNo, Bit64 control, Bit64 value);
    static Result BreakWhenAttached(const nn::svc::BreakReason reason, uintptr_t pData, size_t byteSize);

#ifdef NN_KERN_ENABLE_NX_PROFILER
    Result        GetRunningThreadInfo(nn::svc::LastThreadContext* pContext, nn::Bit64* pThreadId);
#endif

    NN_AUTOOBJECT_DERIVED_FUNCSET(KDebug, KDebugBase)
};
    }
}}

