﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/svc/svc_BaseId.autogen.h>

#include "../../kern_Platform.h"
#include "../ARM64/kern_RegisterAccess.h"
#include "../../kern_DebugSelect.h"
#include "../ARM64/kern_KContext.h"
#include "../../kern_Kernel.h"
#include "kern_MemoryMap.h"
#include "../../kern_KProcess.h"
#include "../../kern_KSynchronization.h"
#include "../../kern_KTaggedAddress.h"
#include "../../kern_KScheduler.h"
#include "kern_Config.h"
#include "kern_ExceptionHandler.h"
#include "../../kern_DebugString.h"
#include "../../kern_KScopedSchedulingLock.h"
#include "../../kern_PageTableSelect.h"
#include "kern_KPageTableBody.h"
#include "../ARM64/kern_MemoryCopy.h"
#include "kern_MemoryMap.h"
#include "../../kern_DpcManager.h"
#include "../../kern_MemoryCopySelect.h"
#include "../../kern_Utility.h"

#define KDEBUG_INFO_ONE     //KEventInfoの作成を1回に減少

using namespace nn::svc;
using namespace nn::kern::ARM64;

namespace nn { namespace kern { namespace ARMv8A {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KDebug);

uintptr_t KDebug::RetrivePc(const KThread& thread)
{
    return GetExceptionContext(&thread)->pc;
}

Result KDebug::GetThreadContext_core(ThreadContext* pContext, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    const ExceptionContext* exceptionContext = GetExceptionContext(pThread);

    if (controlFlags & nn::svc::ContextFlag_General)
    {
        if (!pThread->IsCallingSvc() || pThread->GetSvcNo() == NN_SVC_ID_RETURN_FROM_EXCEPTION)
        {
            if (Is64Bit())
            {
                // r0...r28
                for (int i = 0; i < 29; i++)
                {
                    pContext->r[i] = exceptionContext->x[i];
                }
            }
            else
            {
                // r0...r12
                for (int i = 0; i < 13; i++)
                {
                    pContext->r[i] = exceptionContext->x[i] & 0xFFFFFFFF;
                }
            }
        }
    }

    if (controlFlags & nn::svc::ContextFlag_Control)
    {
        if (Is64Bit())
        {
            pContext->fp = exceptionContext->x[29];
            pContext->lr = exceptionContext->x[30];
            pContext->sp = exceptionContext->sp;
            pContext->pc = exceptionContext->pc;
            pContext->pstate = (exceptionContext->psr & HW_PSR_EL0_MASK);
            if (exceptionContext->write == 0 && pThread->IsCallingSvc())
            {
                pContext->pc -= 4;
            }
            pContext->tpidr = exceptionContext->tpidr;
        }
        else
        {
            pContext->r[11] = exceptionContext->x[11] & 0xFFFFFFFF;
            pContext->r[13] = exceptionContext->x[13] & 0xFFFFFFFF;
            pContext->r[14] = exceptionContext->x[14] & 0xFFFFFFFF;
            pContext->lr = 0;
            pContext->sp = 0;
            pContext->pc = exceptionContext->pc;
            pContext->pstate = (exceptionContext->psr & HW_PSR_EL0_MASK);
            if (exceptionContext->write == 0 && pThread->IsCallingSvc())
            {
                pContext->pc -= (pContext->pstate & HW_PSR_THUMB_STATE)? 2: 4;
            }
            pContext->tpidr = exceptionContext->tpidr & 0xFFFFFFFF;
        }
    }

    return GetFPUContext(pContext, pThread, controlFlags);
}

Result KDebug::SetThreadContext_core(const ThreadContext& context, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    ExceptionContext* exceptionContext = GetExceptionContext(pThread);

    if (controlFlags & nn::svc::ContextFlag_General)
    {
        if (Is64Bit())
        {
            for (int i = 0; i < 29; i++)
            {
                exceptionContext->x[i] = context.r[i];
            }
        }
        else
        {
            for (size_t i = 0; i < 13; i++)
            {
                exceptionContext->x[i] = context.r[i] & 0xFFFFFFFF;
            }
        }
    }

    if (controlFlags & nn::svc::ContextFlag_Control)
    {
        exceptionContext->write = 1;
        if (Is64Bit())
        {
            exceptionContext->x[29] = context.fp;
            exceptionContext->x[30] = context.lr;
            exceptionContext->sp  = context.sp;
            exceptionContext->pc  = context.pc;
            exceptionContext->psr = ((context.pstate & HW_PSR_EL0_MASK) | (exceptionContext->psr & ~ HW_PSR_EL0_MASK));
            exceptionContext->tpidr = context.tpidr;
        }
        else
        {
            exceptionContext->x[13] = context.r[13] & 0xFFFFFFFF;
            exceptionContext->x[14] = context.r[14] & 0xFFFFFFFF;
            exceptionContext->x[30] = 0;
            exceptionContext->sp  = 0;
            exceptionContext->pc  = context.pc & 0xFFFFFFFF;
            exceptionContext->psr = ((context.pstate & HW_PSR_EL0_MASK) | (exceptionContext->psr & ~ HW_PSR_EL0_MASK));
            exceptionContext->tpidr = context.tpidr;
        }
    }

    return SetFPUContext(context, pThread, controlFlags);
}

Result KDebug::GetFPUContext(ThreadContext* pContext, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(pThread != &GetCurrentThread());
    if (!(controlFlags & (nn::svc::ContextFlag_FpuControl | nn::svc::ContextFlag_Fpu)))
    {
        return ResultSuccess();
    }

    KContext*  pThreadContext = pThread->GetContext();
    if (controlFlags & nn::svc::ContextFlag_FpuControl)
    {
        pContext->fpsr = pThreadContext->GetFpsr();
        pContext->fpcr = pThreadContext->GetFpcr();
    }

    if (controlFlags & nn::svc::ContextFlag_Fpu)
    {
        const Bit128 *fpr = pThreadContext->GetFpuRegisters();
        if (Is64Bit())
        {
            for (size_t i = 0; i < pThreadContext->GetNumOfVfpRegisters(); i++)
            {
                pContext->v[i] = fpr[i];
            }
        }
        else
        {
            for (size_t i = 0; i < pThreadContext->GetNumOfVfpRegisters() / 2; i++)
            {
                pContext->v[i] = fpr[i];
            }
            for (size_t i = pThreadContext->GetNumOfVfpRegisters() / 2; i < pThreadContext->GetNumOfVfpRegisters(); i++)
            {
                pContext->v[i] = 0;
            }
        }
    }

    return ResultSuccess();
}

Result KDebug::SetFPUContext(const ThreadContext& context, KThread* pThread, const Bit32 controlFlags)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(pThread != &GetCurrentThread());

    if (!(controlFlags & (nn::svc::ContextFlag_FpuControl | nn::svc::ContextFlag_Fpu)))
    {
        return ResultSuccess();
    }

    KContext* pThreadContext = pThread->GetContext();
    if (controlFlags & nn::svc::ContextFlag_FpuControl)
    {
        pThreadContext->SetFpsr(context.fpsr);
        pThreadContext->SetFpcr(context.fpcr);
    }

    if (controlFlags & nn::svc::ContextFlag_Fpu)
    {
        pThreadContext->SetFpuRegisters(context.v, Is64Bit());
    }

    return ResultSuccess();
}

void KDebug::SetPreviousPc()
{
    KThread* pThread = &GetCurrentThread();
    NN_KERN_ASSERT(pThread->IsCallingSvc());

    ExceptionContext* pContext = GetExceptionContext(pThread);

    if (pContext->write == 0)
    {
        if (pThread->GetParent().Is64Bit())
        {
            pContext->pc -= 4;
        }
        else
        {
            pContext->pc -= (pContext->psr & HW_PSR_THUMB_STATE)? 2: 4;
        }
        pContext->write = 1;
    }
}

void KDebug::PrintRegister(KThread* pThread)
{
#ifdef NN_KERN_FOR_DEVELOPMENT
    if (!pThread)
    {
        pThread = &GetCurrentThread();
    }
    ExceptionContext* pContext = GetExceptionContext(pThread);
    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess)
    {
        KScopedLightLock procLocker(&pProcess->GetStateMutex());
        KScopedLightLock procListLocker(&pProcess->GetListMutex());

        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->SuspendRequest(KThread::SuspendType_BackTrace);
                }
            }
        }

        if (pContext->psr & 0x10)
        {
            // aarch32
            NN_KERN_RELEASE_LOG("R0:    0x%08x\n", static_cast<Bit32>(pContext->x[0]));
            NN_KERN_RELEASE_LOG("R1:    0x%08x\n", static_cast<Bit32>(pContext->x[1]));
            NN_KERN_RELEASE_LOG("R2:    0x%08x\n", static_cast<Bit32>(pContext->x[2]));
            NN_KERN_RELEASE_LOG("R3:    0x%08x\n", static_cast<Bit32>(pContext->x[3]));
            NN_KERN_RELEASE_LOG("R4:    0x%08x\n", static_cast<Bit32>(pContext->x[4]));
            NN_KERN_RELEASE_LOG("R5:    0x%08x\n", static_cast<Bit32>(pContext->x[5]));
            NN_KERN_RELEASE_LOG("R6:    0x%08x\n", static_cast<Bit32>(pContext->x[6]));
            NN_KERN_RELEASE_LOG("R7:    0x%08x\n", static_cast<Bit32>(pContext->x[7]));
            NN_KERN_RELEASE_LOG("R8:    0x%08x\n", static_cast<Bit32>(pContext->x[8]));
            NN_KERN_RELEASE_LOG("R9:    0x%08x\n", static_cast<Bit32>(pContext->x[9]));
            NN_KERN_RELEASE_LOG("R10:   0x%08x\n", static_cast<Bit32>(pContext->x[10]));
            NN_KERN_RELEASE_LOG("R11:   0x%08x\n", static_cast<Bit32>(pContext->x[11]));
            NN_KERN_RELEASE_LOG("R12:   0x%08x\n", static_cast<Bit32>(pContext->x[12]));
            NN_KERN_RELEASE_LOG("SP:    0x%08x\n", static_cast<Bit32>(pContext->x[13]));
            NN_KERN_RELEASE_LOG("LR:    0x%08x\n", static_cast<Bit32>(pContext->x[14]));
            NN_KERN_RELEASE_LOG("PC:    0x%08x\n", static_cast<Bit32>(pContext->pc) - ((pContext->psr & HW_PSR_THUMB_STATE)? 2: 4));
            NN_KERN_RELEASE_LOG("PSR:   0x%08x\n", static_cast<Bit32>(pContext->psr));
            NN_KERN_RELEASE_LOG("TPIDR: 0x%08x\n", static_cast<Bit32>(pContext->tpidr));
        }
        else
        {
            NN_KERN_RELEASE_LOG("X0:        0x%016lx\n", pContext->x[0]);
            NN_KERN_RELEASE_LOG("X1:        0x%016lx\n", pContext->x[1]);
            NN_KERN_RELEASE_LOG("X2:        0x%016lx\n", pContext->x[2]);
            NN_KERN_RELEASE_LOG("X3:        0x%016lx\n", pContext->x[3]);
            NN_KERN_RELEASE_LOG("X4:        0x%016lx\n", pContext->x[4]);
            NN_KERN_RELEASE_LOG("X5:        0x%016lx\n", pContext->x[5]);
            NN_KERN_RELEASE_LOG("X6:        0x%016lx\n", pContext->x[6]);
            NN_KERN_RELEASE_LOG("X7:        0x%016lx\n", pContext->x[7]);
            NN_KERN_RELEASE_LOG("X8:        0x%016lx\n", pContext->x[8]);
            NN_KERN_RELEASE_LOG("X9:        0x%016lx\n", pContext->x[9]);
            NN_KERN_RELEASE_LOG("X10:       0x%016lx\n", pContext->x[10]);
            NN_KERN_RELEASE_LOG("X11:       0x%016lx\n", pContext->x[11]);
            NN_KERN_RELEASE_LOG("X12:       0x%016lx\n", pContext->x[12]);
            NN_KERN_RELEASE_LOG("X13:       0x%016lx\n", pContext->x[13]);
            NN_KERN_RELEASE_LOG("X14:       0x%016lx\n", pContext->x[14]);
            NN_KERN_RELEASE_LOG("X15:       0x%016lx\n", pContext->x[15]);
            NN_KERN_RELEASE_LOG("X16:       0x%016lx\n", pContext->x[16]);
            NN_KERN_RELEASE_LOG("X17:       0x%016lx\n", pContext->x[17]);
            NN_KERN_RELEASE_LOG("X18:       0x%016lx\n", pContext->x[18]);
            NN_KERN_RELEASE_LOG("X19:       0x%016lx\n", pContext->x[19]);
            NN_KERN_RELEASE_LOG("X20:       0x%016lx\n", pContext->x[20]);
            NN_KERN_RELEASE_LOG("X21:       0x%016lx\n", pContext->x[21]);
            NN_KERN_RELEASE_LOG("X22:       0x%016lx\n", pContext->x[22]);
            NN_KERN_RELEASE_LOG("X23:       0x%016lx\n", pContext->x[23]);
            NN_KERN_RELEASE_LOG("X24:       0x%016lx\n", pContext->x[24]);
            NN_KERN_RELEASE_LOG("X25:       0x%016lx\n", pContext->x[25]);
            NN_KERN_RELEASE_LOG("X26:       0x%016lx\n", pContext->x[26]);
            NN_KERN_RELEASE_LOG("X27:       0x%016lx\n", pContext->x[27]);
            NN_KERN_RELEASE_LOG("X28:       0x%016lx\n", pContext->x[28]);
            NN_KERN_RELEASE_LOG("X29:       0x%016lx\n", pContext->x[29]);
            NN_KERN_RELEASE_LOG("X30:       0x%016lx\n", pContext->x[30]);
            NN_KERN_RELEASE_LOG("SP:        0x%016lx\n", pContext->sp);
            NN_KERN_RELEASE_LOG("PC:        0x%016lx\n", pContext->pc - 4);
            NN_KERN_RELEASE_LOG("PSR:       0x%08x\n", static_cast<Bit32>(pContext->psr));
            NN_KERN_RELEASE_LOG("TPIDR_EL0: 0x%016lx\n", pContext->tpidr);
        }

        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->Resume(KThread::SuspendType_BackTrace);
                }
            }
        }
    }
#else
    NN_UNUSED(pThread);
#endif
}

#ifdef NN_KERN_FOR_DEVELOPMENT
namespace {
template <typename T>
bool ReadValue(T* pOut, KProcess* pProcess, uintptr_t addr)
{
    KPhysicalAddress pa;
    KVirtualAddress va;
    KMemoryInfo mi;
    PageInfo pi;

    if (addr & (sizeof(T) - 1))
    {
        return false;
    }
    if (pProcess->GetPageTable().QueryInfo(&mi, &pi, addr).IsFailure())
    {
        return false;
    }
    if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
    {
        return false;
    }
    if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, addr))
    {
        return false;
    }
    if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
    {
        return false;
    }
    va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
    *pOut = *GetTypedPointer<T>(va);
    return true;
}

bool GetModuleName(char* pBuffer, KProcess* pProcess, uintptr_t baseAddr, size_t n)
{
    KMemoryInfo mi;
    PageInfo pi;
    Bit32 dw;

    for (;;)
    {
        if (pProcess->GetPageTable().QueryInfo(&mi, &pi, baseAddr).IsFailure())
        {
            return false;
        }
        if (mi.state != KMemoryState_Code)
        {
            return false;
        }
        if (mi.permission == KMemoryPermission_UserRead)
        {
            break;
        }
        baseAddr = mi.baseAddress + mi.size;
    }

    if (!ReadValue(&dw, pProcess, baseAddr))
    {
        return false;
    }
    if (dw != 0)
    {
        return false;
    }

    size_t nameLen;
    if (!ReadValue(&dw, pProcess, baseAddr + 4))
    {
        return false;
    }
    if (dw == 0 || dw >= n)
    {
        return false;
    }
    nameLen = dw;

    for (size_t i = 0; i < nameLen; i++)
    {
        if (!ReadValue(pBuffer + i, pProcess, baseAddr + 8 + i))
        {
            return false;
        }
        if (!(0 < pBuffer[i] && pBuffer[i] <= 0x7f))
        {
            return false;
        }
    }
    pBuffer[nameLen] = 0;

    return true;
}

void PrintCodeAddress(KProcess* pProcess, uintptr_t addr, bool isLinkRegister = true)
{
    // nso の先頭を調べる
    uintptr_t baseAddr = addr;
    uintptr_t dynAddr = 0;
    uintptr_t symtab = 0;
    uintptr_t strtab = 0;
    size_t numsym = 0;
    KMemoryInfo mi;
    PageInfo pi;
    Bit32 dw;
    Bit64 qw;
    char moduleName[32] = {0};
    bool hasModuleName = false;

    for (;;)
    {
        if (pProcess->GetPageTable().QueryInfo(&mi, &pi, baseAddr).IsFailure())
        {
            goto print_addr;
        }
        if (mi.state != KMemoryState_Code)
        {
            goto print_addr;
        }
        if (mi.permission != KMemoryPermission_UserReadExecute)
        {
            goto print_addr;
        }
        baseAddr = mi.baseAddress;

        if (pProcess->GetPageTable().QueryInfo(&mi, &pi, baseAddr - 1).IsFailure())
        {
            goto print_addr;
        }
        if (mi.state != KMemoryState_Code)
        {
            break;
        }
        if (mi.permission != KMemoryPermission_UserReadExecute)
        {
            break;
        }
        baseAddr = mi.baseAddress;
    }

    if (!ReadValue(&dw, pProcess, baseAddr))
    {
        goto print_addr;
    }

    hasModuleName = GetModuleName(moduleName, pProcess, baseAddr, sizeof(moduleName));

    if (!pProcess->Is64Bit())
    {
        goto print_base_addr;
    }
    if (dw == 0)
    {
        // rocrt を仮定している
        Bit32 rocrtOffset;
        if (!ReadValue(&rocrtOffset, pProcess, baseAddr + 4))
        {
            goto print_base_addr;
        }
        if (!ReadValue(&dw, pProcess, baseAddr + rocrtOffset))
        {
            goto print_base_addr;
        }
        if (dw != 0x30444f4d) // MOD0
        {
            goto print_base_addr;
        }
        if (!ReadValue(&dw, pProcess, baseAddr + rocrtOffset + 4))
        {
            goto print_base_addr;
        }
        dynAddr = baseAddr + rocrtOffset + dw;
    }
    else if (dw == 0x14000002)
    {
        // rtld を仮定している
        if (!ReadValue(&dw, pProcess, baseAddr + 0x5c))
        {
            goto print_base_addr;
        }
        if (dw != 0x94000002)
        {
            goto print_base_addr;
        }
        if (!ReadValue(&dw, pProcess, baseAddr + 0x60))
        {
            goto print_base_addr;
        }
        dynAddr = baseAddr + 0x60 + dw;
    }
    else
    {
        goto print_base_addr;
    }

    for (size_t offset = 0; ; offset += 16)
    {
        if (!ReadValue(&qw, pProcess, dynAddr + offset))
        {
            goto print_base_addr;
        }
        if (qw == 0)
        {
            break;
        }
        else if (qw == 4)
        {
            if (!ReadValue(&qw, pProcess, dynAddr + offset + 8))
            {
                goto print_base_addr;
            }
            if (!ReadValue(&dw, pProcess, baseAddr + qw + 4))
            {
                goto print_base_addr;
            }
            numsym = dw;
        }
        else if (qw == 5)
        {
            if (!ReadValue(&qw, pProcess, dynAddr + offset + 8))
            {
                goto print_base_addr;
            }
            strtab = baseAddr + qw;
        }
        else if (qw == 6)
        {
            if (!ReadValue(&qw, pProcess, dynAddr + offset + 8))
            {
                goto print_base_addr;
            }
            symtab = baseAddr + qw;
        }
    }
    if (!(symtab && strtab && numsym))
    {
        goto print_base_addr;
    }

    for (size_t i = 0; i < numsym; i++)
    {
        struct
        {
            uint32_t name;
            uint8_t info;
            uint8_t other;
            uint16_t shndx;
            uint64_t value;
            uint64_t size;
        } sym;
        Bit64 x[3];

        if (!ReadValue(&x[0], pProcess, symtab + 24 * i + 8 * 0))
        {
            goto print_base_addr;
        }
        if (!ReadValue(&x[1], pProcess, symtab + 24 * i + 8 * 1))
        {
            goto print_base_addr;
        }
        if (!ReadValue(&x[2], pProcess, symtab + 24 * i + 8 * 2))
        {
            goto print_base_addr;
        }
        std::memcpy(&sym, x, sizeof(sym));
        if (sym.shndx == 0)
        {
            continue;
        }
        if ((sym.shndx & 0xff00) == 0xff00)
        {
            continue;
        }
        if ((sym.info & 0xF) != 2)
        {
            continue;
        }
        uintptr_t testAddr = addr;
        if (isLinkRegister)
        {
            testAddr -= 4;
        }
        if (baseAddr + sym.value <= testAddr && testAddr < baseAddr + sym.value + sym.size)
        {
            uintptr_t symAddr = strtab + sym.name;
            Bit8 uc[0x80] = {0};
            for (size_t j = 0; j < sizeof(uc) - 1; j++)
            {
                if (!ReadValue(&uc[j], pProcess, symAddr + j))
                {
                    goto print_base_addr;
                }
                if (uc[j] == 0)
                {
                    break;
                }
            }
            if (hasModuleName)
            {
                NN_KERN_RELEASE_LOG("   %p [%10s + %8lx] (%s + %lx)\n", addr, moduleName, addr - baseAddr, uc, addr - (baseAddr + sym.value));
            }
            else
            {
                NN_KERN_RELEASE_LOG("   %p [%10lx + %8lx] (%s + %lx)\n", addr, baseAddr, addr - baseAddr, uc, addr - (baseAddr + sym.value));
            }
            return;
        }
    }

print_base_addr:
    if (hasModuleName)
    {
        NN_KERN_RELEASE_LOG("   %p [%10s + %8lx]\n", addr, moduleName, addr - baseAddr);
    }
    else
    {
        NN_KERN_RELEASE_LOG("   %p [%10lx + %8lx]\n", addr, baseAddr, addr - baseAddr);
    }
    return;

print_addr:
    NN_KERN_RELEASE_LOG("   %p\n", addr);
    return;
}

}
#endif

void KDebug::PrintBacktrace(KThread* pThread)
{
#ifdef NN_KERN_FOR_DEVELOPMENT
    if (!pThread)
    {
        pThread = &GetCurrentThread();
    }
    ExceptionContext* pContext = GetExceptionContext(pThread);
    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess)
    {
        KScopedLightLock procLocker(&pProcess->GetStateMutex());
        KScopedLightLock procListLocker(&pProcess->GetListMutex());

        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->SuspendRequest(KThread::SuspendType_BackTrace);
                }
            }
        }

        if (pContext->psr & 0x10)
        {
            NN_KERN_RELEASE_LOG("User Backtrace\n");
            PrintCodeAddress(pProcess, pContext->pc, false);
            PrintCodeAddress(pProcess, pContext->x[14]);
            uintptr_t frameAddress = static_cast<uintptr_t>(pContext->x[11]);
            for (int i = 0; i < 32; i++)
            {
                if (!frameAddress)
                {
                    break;
                }
                if (frameAddress & 0x3)
                {
                    break;
                }
                struct
                {
                    uint32_t frame;
                    uint32_t lr;
                } frame;
                {
                    KMemoryInfo mi;
                    PageInfo pi;

                    KPhysicalAddress pa;
                    KVirtualAddress va;

                    if (pProcess->GetPageTable().QueryInfo(&mi, &pi, frameAddress).IsFailure())
                    {
                        break;
                    }
                    if (!(mi.state & KMemoryState_FlagsMemory))
                    {
                        break;
                    }
                    if (mi.attribute & KMemoryAttribute_Uncached)
                    {
                        break;
                    }
                    if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                    {
                        break;
                    }
                    if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, frameAddress))
                    {
                        break;
                    }
                    if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
                    {
                        break;
                    }
                    va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
                    uint32_t v = *GetTypedPointer<uint32_t>(va);
                    if (pContext->x[13] <= v && v < pContext->x[13] + NN_KERN_FINEST_PAGE_SIZE )
                    {
                        // clang
                        if (pProcess->GetPageTable().QueryInfo(&mi, &pi, frameAddress + 4).IsFailure())
                        {
                            break;
                        }
                        if (!(mi.state & KMemoryState_FlagsMemory))
                        {
                            break;
                        }
                        if (mi.attribute & KMemoryAttribute_Uncached)
                        {
                            break;
                        }
                        if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                        {
                            break;
                        }
                        frame.frame = *GetTypedPointer<uint32_t>(va);
                        if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, frameAddress + 4))
                        {
                            break;
                        }
                        if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
                        {
                            break;
                        }
                        va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
                        frame.lr = *GetTypedPointer<uint32_t>(va);
                    }
                    else
                    {
                        // gcc
                        if (pProcess->GetPageTable().QueryInfo(&mi, &pi, frameAddress - 4).IsFailure())
                        {
                            break;
                        }
                        if (!(mi.state & KMemoryState_FlagsMemory))
                        {
                            break;
                        }
                        if (mi.attribute & KMemoryAttribute_Uncached)
                        {
                            break;
                        }
                        if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                        {
                            break;
                        }
                        frame.lr = *GetTypedPointer<uint32_t>(va);
                        if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, frameAddress - 4))
                        {
                            break;
                        }
                        if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
                        {
                            break;
                        }
                        va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
                        frame.frame = *GetTypedPointer<uint32_t>(va);
                    }
                }
                PrintCodeAddress(pProcess, frame.lr);
                frameAddress = frame.frame;
            }
        }
        else
        {
            NN_KERN_RELEASE_LOG("User Backtrace\n");
            PrintCodeAddress(pProcess, pContext->pc, false);
            PrintCodeAddress(pProcess, pContext->x[30]);
            uintptr_t frameAddress = static_cast<uintptr_t>(pContext->x[29]);
            for (int i = 0; i < 32; i++)
            {
                if (!frameAddress)
                {
                    break;
                }
                if (frameAddress & 0xF)
                {
                    break;
                }
                struct
                {
                    uint64_t frame;
                    uint64_t lr;
                } frame;
                {
                    KMemoryInfo mi;
                    PageInfo pi;

                    KPhysicalAddress pa;
                    KVirtualAddress va;

                    if (pProcess->GetPageTable().QueryInfo(&mi, &pi, frameAddress).IsFailure())
                    {
                        break;
                    }
                    if (!(mi.state & KMemoryState_FlagsMemory))
                    {
                        break;
                    }
                    if (mi.attribute & KMemoryAttribute_Uncached)
                    {
                        break;
                    }
                    if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                    {
                        break;
                    }
                    if (!pProcess->GetPageTable().GetPhysicalAddress(&pa, frameAddress))
                    {
                        break;
                    }
                    if (!pProcess->GetPageTable().GetPageTable().IsHeapPhysicalAddress(pa))
                    {
                        break;
                    }
                    va = pProcess->GetPageTable().GetPageTable().GetHeapVirtualAddress(pa);
                    frame.frame = *GetTypedPointer<uint64_t>(va);
                    frame.lr = *GetTypedPointer<uint64_t>(va + sizeof(frame.frame));
                }
                PrintCodeAddress(pProcess, frame.lr);
                frameAddress = frame.frame;
            }
        }

        {
            KScopedSchedulingLock locker;
            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                if (&*it != &GetCurrentThread())
                {
                    it->Resume(KThread::SuspendType_BackTrace);
                }
            }
        }
    }
#else
    NN_UNUSED(pThread);
#endif
}

void KDebug::GetCurrentBacktrace(uintptr_t* pOut, size_t num)
{
    KThread* pThread = &GetCurrentThread();
    KProcess* pCurrentProcess = pThread->GetParentPointer();
    size_t i = 0;
    if (pCurrentProcess)
    {
        ExceptionContext* pContext = GetExceptionContext(pThread);
        if (pContext->psr & 0x10)
        {
            uintptr_t frameAddress = static_cast<uintptr_t>(pContext->x[11]);
            uintptr_t returnAddress = pContext->x[14];
            bool isClang = false;
            while (i < num)
            {
                pOut[i++] = returnAddress;

                if (!frameAddress)
                {
                    break;
                }
                if (frameAddress & 0x3)
                {
                    break;
                }

                if (i == 0)
                {
                    uint32_t v;
                    if (!CopyMemoryFromUser(&v, reinterpret_cast<const void*>(frameAddress), sizeof(v)))
                    {
                        break;
                    }

                    isClang = (pContext->x[13] <= v && v < pContext->x[13] + NN_KERN_FINEST_PAGE_SIZE);
                }

                struct
                {
                    uint32_t frame;
                    uint32_t lr;
                } frame;

                if (!CopyMemoryFromUser(&frame, reinterpret_cast<const void*>(frameAddress + (isClang ? 0: - 4)), sizeof(frame)))
                {
                    break;
                }

                frameAddress = frame.frame;
                returnAddress = frame.lr;
            }
        }
        else
        {
            uintptr_t frameAddress = static_cast<uintptr_t>(pContext->x[29]);
            uintptr_t returnAddress = pContext->x[30];

            while (i < num)
            {
                pOut[i++] = returnAddress;

                if (!frameAddress)
                {
                    break;
                }
                if (frameAddress & 0xF)
                {
                    break;
                }
                struct
                {
                    uint64_t frame;
                    uint64_t lr;
                } frame;

                if (!CopyMemoryFromUser(&frame, reinterpret_cast<const void*>(frameAddress), sizeof(frame)))
                {
                    break;
                }
                frameAddress = frame.frame;
                returnAddress = frame.lr;
            }
        }
    }
    for (size_t j = i; j < num; j++)
    {
        pOut[j] = 0;
    }
}

#define HARDWARE_BREAK_POINT(i, c, v)   \
    do                                  \
    { \
        asm volatile ("msr DBGBCR" #i "_el1, %0"::"r"(0ul):"memory"); \
        asm volatile ("dsb sy; isb":::"memory"); \
        asm volatile ("msr DBGBVR" #i "_el1, %0"::"r"(v):"memory"); \
        asm volatile ("dsb sy; isb":::"memory"); \
        asm volatile ("msr DBGBCR" #i "_el1, %0"::"r"(c):"memory"); \
        asm volatile ("dsb sy; isb":::"memory"); \
    } while (0)

#define HARDWARE_WATCH_POINT(i, c, v)   \
    do                                  \
    { \
        asm volatile ("msr DBGWCR" #i "_el1, %0"::"r"(0ul):"memory"); \
        asm volatile ("dsb sy; isb":::"memory"); \
        asm volatile ("msr DBGWVR" #i "_el1, %0"::"r"(v):"memory"); \
        asm volatile ("dsb sy; isb":::"memory"); \
        asm volatile ("msr DBGWCR" #i "_el1, %0"::"r"(c):"memory"); \
        asm volatile ("dsb sy; isb":::"memory"); \
    } while (0)


Result KDebug::SetHardwareBreakPoint(nn::svc::HardwareBreakPointRegisterName regNo, Bit64 control, Bit64 value)
{
    KProcess *pProcess = nullptr;
    Bit64 dfr0;

    HW_GET_ID_AA64DFR0_EL1(dfr0);
    uint32_t wrps = ((dfr0 >> 20) & 0xF);
    uint32_t brps = ((dfr0 >> 12) & 0xF);
    uint32_t ctxcmp = ((dfr0 >> 28) & 0xF);

    if (nn::svc::HardwareBreakPointRegisterName_I0 <= regNo && regNo <= nn::svc::HardwareBreakPointRegisterName_I15)
    {
        const Bit64 sbz = (0xFFFFFFFFul << 32) | (0x1FF << 23) | (0x7F << 9) | (0xF << 1);
        if ((regNo - nn::svc::HardwareBreakPointRegisterName_I0) > brps)
        {
            return nn::svc::ResultNotSupported();
        }
        if (control & 1)
        {
            if (sbz & control)
            {
                return nn::svc::ResultInvalidCombination();
            }
            if (((control >> 20) & 0x3) == 0)
            {
                return nn::svc::ResultInvalidCombination();
            }
            if ((control >> 20) & 0x2)
            {
                if ((regNo - nn::svc::HardwareBreakPointRegisterName_I0) < (brps - ctxcmp))
                {
                    return nn::svc::ResultNotSupported();
                }
                if ((control >> 20) & 0x4)
                {
                    return nn::svc::ResultInvalidCombination();
                }
                KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
                KDebug* pDebug = handleTable.GetObject<KDebug>(nn::svc::Handle(value));
                if (!pDebug)
                {
                    return nn::svc::ResultInvalidHandle();
                }
                KScopedAutoObject<KDebug> autoCloser(pDebug);

                pProcess = pDebug->GetProcess();
                if (!pProcess)
                {
                    return nn::svc::ResultProcessTerminated();
                }

                value = pProcess->GetId() & 0xffffffff;
            }
            control |= ((1 << 14) | (2 << 1));
        }
        else
        {
            control = 0;
            value = 0;
        }
        switch (regNo)
        {
        case nn::svc::HardwareBreakPointRegisterName_I0: { HARDWARE_BREAK_POINT(0, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I1: { HARDWARE_BREAK_POINT(1, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I2: { HARDWARE_BREAK_POINT(2, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I3: { HARDWARE_BREAK_POINT(3, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I4: { HARDWARE_BREAK_POINT(4, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I5: { HARDWARE_BREAK_POINT(5, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I6: { HARDWARE_BREAK_POINT(6, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I7: { HARDWARE_BREAK_POINT(7, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I8: { HARDWARE_BREAK_POINT(8, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I9: { HARDWARE_BREAK_POINT(9, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I10: { HARDWARE_BREAK_POINT(10, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I11: { HARDWARE_BREAK_POINT(11, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I12: { HARDWARE_BREAK_POINT(12, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I13: { HARDWARE_BREAK_POINT(13, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I14: { HARDWARE_BREAK_POINT(14, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_I15: { HARDWARE_BREAK_POINT(15, control, value); } break;
        default: break;
        }

        if (pProcess)
        {
            pProcess->Close();
        }
    }
    else if (nn::svc::HardwareBreakPointRegisterName_D0 <= regNo && regNo <= nn::svc::HardwareBreakPointRegisterName_D15)
    {
        const Bit64 sbz = (0xFFFFFFFFul << 32) | (0xF << 20) | (0x7 << 13) | (0x3 << 1);
        if ((regNo - nn::svc::HardwareBreakPointRegisterName_D0) > wrps)
        {
            return nn::svc::ResultNotSupported();
        }
        if (control & 1)
        {
            if (sbz & control)
            {
                return nn::svc::ResultInvalidCombination();
            }

            control |= ((1 << 20) | (1 << 14) | (2 << 1));
        }
        else
        {
            control = 0;
            value = 0;
        }
        switch (regNo)
        {
        case nn::svc::HardwareBreakPointRegisterName_D0: { HARDWARE_WATCH_POINT(0, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D1: { HARDWARE_WATCH_POINT(1, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D2: { HARDWARE_WATCH_POINT(2, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D3: { HARDWARE_WATCH_POINT(3, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D4: { HARDWARE_WATCH_POINT(4, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D5: { HARDWARE_WATCH_POINT(5, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D6: { HARDWARE_WATCH_POINT(6, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D7: { HARDWARE_WATCH_POINT(7, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D8: { HARDWARE_WATCH_POINT(8, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D9: { HARDWARE_WATCH_POINT(9, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D10: { HARDWARE_WATCH_POINT(10, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D11: { HARDWARE_WATCH_POINT(11, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D12: { HARDWARE_WATCH_POINT(12, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D13: { HARDWARE_WATCH_POINT(13, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D14: { HARDWARE_WATCH_POINT(14, control, value); } break;
        case nn::svc::HardwareBreakPointRegisterName_D15: { HARDWARE_WATCH_POINT(15, control, value); } break;
        default: break;
        }
    }
    else
    {
        return nn::svc::ResultInvalidEnum();
    }

    return ResultSuccess();
}

Result KDebug::BreakWhenAttached(const nn::svc::BreakReason reason, uintptr_t pData, size_t byteSize)
{
    return OnDebugEvent(nn::svc::DebugEvent_Exception, nn::svc::DebugException_UserBreak, RetrivePc(GetCurrentThread()), reason, pData, byteSize);
}

#ifdef NN_KERN_ENABLE_NX_PROFILER
Result KDebug::GetRunningThreadInfo(nn::svc::LastThreadContext* pContext, nn::Bit64* pThreadId)
{
    Result result;

    KProcess* pProcess = GetProcess();
    if (!pProcess)
    {
        return nn::svc::ResultProcessTerminated();
    }
    KScopedAutoObject<KProcess> autoCloser(pProcess);

    {
        KScopedSchedulingLock schlocker;

        int core = GetCurrentCpuNo();
        KThread* pThread = pProcess->GetRunningThread(core);
        if (pProcess->GetRunningThreadIdleCount(core) == Kernel::GetScheduler(core).GetIdleCount())
        {
            if (pThread != nullptr && static_cast<size_t>(pThread->GetRunningProcessor()) == core)
            {
                const nn::kern::ARM64::ExceptionContext* pExcContext = GetExceptionContext(pThread);
                if ((pExcContext->psr & HW_PSR_AARCH_MASK) == HW_PSR_AARCH32)
                {
                    pContext->fp = pExcContext->x[11];
                    pContext->sp = pExcContext->x[13];
                    pContext->lr = pExcContext->x[14];
                    pContext->pc = pExcContext->pc;
                }
                else
                {
                    pContext->fp = pExcContext->x[29];
                    pContext->sp = pExcContext->sp;
                    pContext->lr = pExcContext->x[30];
                    pContext->pc = pExcContext->pc;
                }
                *pThreadId = pThread->GetId();
                result = ResultSuccess();
            }
            else
            {
                result = nn::svc::ResultUnknownThread();
            }
        }
        else
        {
            result = nn::svc::ResultNoThread();
        }
    }
    return result;
}
#endif

}}}
