﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_MemoryMap.h"
#include "../../kern_KTaggedAddress.h"
#include "../ARM/kern_RegisterAccess.h"
#include "kern_KPageTableDefinition.h"
#include "../../kern_KMemoryLayout.h"
#include "kern_MemoryMap.h"

namespace nn {
    namespace kern {
        namespace ARMv7A {

class KPageTableBody
{
private:
    static const Bit32 HW_MMU7_T1_CORS_SB_MASK = 0x00000003;

private:
    Bit32*  m_pTable;
    size_t m_MinIndex;
    size_t m_MaxIndex;

public:
    struct TraverseData
    {
        KPhysicalAddress    address;
        size_t              size;
    };
    struct TraverseContext
    {
        const Bit32*    pL1;    // 次の L1 エントリへのポインタ
        const Bit32*    pL2;    // 次の L2 エントリへのポインタ
    };

private:
    static KVirtualAddress GetPageTableVirtualAddress(KPhysicalAddress pa)
    {
        return KMemoryLayout::ToLinearVirtualAddress(pa);
    }


private:
    static bool GetAddrFromL1Entry(size_t* pNumPages, KPhysicalAddress* pOut, const Bit32** ppL2, Bit32 entry, KProcessAddress srcAddr);
    static bool GetAddrFromL2Entry(size_t* pNumPages, KPhysicalAddress* pOut, Bit32 entry, KProcessAddress srcAddr);

public:
    void    InitializeForKernel(void* pTable, KProcessAddress begin, KProcessAddress end);
    void    InitializeForProcess(void* pTable, KProcessAddress begin, KProcessAddress end);
    Bit32*  Finalize();

    void    Dump(uintptr_t begin, size_t size) const;
    size_t CountPageTables() const { return 0; }

    bool TraverseBegin(TraverseData* pOut, TraverseContext* pContext, KProcessAddress addr) const;
    bool TraverseNext (TraverseData* pOut, TraverseContext* pContext) const;
    bool GetPhysicalAddress(KPhysicalAddress* pOut, KProcessAddress vaddr) const;

    // TORIAEZU: KPageTable でのキャッシュ操作のため
    void* GetAddress() const { return m_pTable; }


    // inline

    Bit32 GetMasterEntry(KProcessAddress addr) const
    {
        const Bit32 offset = (GetAsInteger(addr) >> HW_MMU7_T1_SEC_BASE_SFT);
        return m_pTable[offset];
    }
    void SetMasterEntry(KProcessAddress addr, Bit32 entry)
    {
        const Bit32 offset = (GetAsInteger(addr) >> HW_MMU7_T1_SEC_BASE_SFT);
        m_pTable[offset] = entry;
    }
    Bit32 GetCoarseEntry(KProcessAddress addr) const
    {
        const Bit32* pCoraseTable = GetTypedPointer<Bit32>(GetCoarseTableAddress(addr));
        return GetCoarseEntryFromCoarseTable(addr, pCoraseTable);
    }
    KPhysicalAddress SetCoarseEntry(KProcessAddress addr, Bit32 entry)
    {
        Bit32* pCoraseTable = GetTypedPointer<Bit32>(GetCoarseTableAddress(addr));
        return SetCoarseEntryToCoarseTable(addr, pCoraseTable, entry);
    }
    bool IsCoarse(KProcessAddress addr) const
    {
        const Bit32 entry = GetMasterEntry(addr);
        return (entry & HW_MMU7_T1_CORS_SB_MASK) == HW_MMU7_T1_CORS_SB1;
    }


    // static inline

    static Bit32 GetCoarseEntryFromCoarseTable(KProcessAddress addr, const Bit32* pTable)
    {
        return pTable[(GetAsInteger(addr) & ~HW_MMU7_T1_SEC_BASE_MASK) >> HW_MMU7_T2_SP_BASE_SFT];
    }
    static KPhysicalAddress SetCoarseEntryToCoarseTable(KProcessAddress addr, Bit32* pTable, Bit32 entryValue)
    {
        Bit32& entry = pTable[(GetAsInteger(addr) & ~HW_MMU7_T1_SEC_BASE_MASK) >> HW_MMU7_T2_SP_BASE_SFT];
        KPhysicalAddress old = (entry & HW_MMU7_T2_SP_BASE_MASK);
        entry = entryValue;
        return old;
    }
    static KPhysicalAddress GetCoarseTableAddressFromMasterEntry(Bit32 entry)
    {
        return KPhysicalAddress(entry & HW_MMU7_T1_CORS_MASK);
    }
    static Bit32 MakeMasterEntry(KPhysicalAddress paddr, Bit32 property)
    {
        NN_UNUSED(property);

        // TODO: property の仕様が未定なので、とりあえず RW 実行可能、キャッシュ可能で作ってしまう。
        const Bit32 entry = HW_MMU7_T1_SEC_PACK(
                                GetAsInteger(paddr),       // 物理アドレス
                                HW_MMU7_T1_APX_ALL,        // アクセス制限無し
                                HW_MMU7_T1_RGT_L1L2C_WB_WA, // キャッシュ可能書き込み割り当て
                                HW_MMU7_T1_NGLOBAL,        // ASID 付き
                                true,                      // 共有
                                false,                     // 実行不可でない
                                0 );                       // ドメイン 0
        return entry;
    }


private:
    KVirtualAddress GetCoarseTableAddress(KProcessAddress addr) const
    {
        const Bit32 masterEntry = GetMasterEntry(addr);
        const KPhysicalAddress coarseTableAddressP = GetCoarseTableAddressFromMasterEntry(masterEntry);
        const KVirtualAddress  coarseTableAddressV = GetPageTableVirtualAddress(coarseTableAddressP);

        return coarseTableAddressV;
    }
};

        }
    }
}


