﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Platform.h"
#include "kern_KPageTable.h"
#include "kern_MemoryMap.h"
#include "../../kern_InterruptManagerSelect.h"
#include "../../kern_InterruptNameSelect.h"
#include "../../kern_Kernel.h"
#include "../../kern_KProcess.h"
#include "../../kern_KPageTableManager.h"
#include "../../kern_KPageBuffer.h"
#include "../../kern_Utility.h"
#include <cstring>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace kern {

namespace ARMv7A {

namespace
{
    const Bit32 UNMAPPED_ENTRY = 0;  // ALL 0

    Bit32 MakeTtbr0Value(KPhysicalAddress tableAddresss)
    {
        const uintptr_t tableAddr = GetAsInteger(tableAddresss);
        const Bit32 ttb        = (tableAddr & HW_C2_0_T1_BASE_MASK_MAX);
        Bit32 ttbr;
        Bit32 ttbrHigh;
        NN_UNUSED(ttbrHigh);

        HW_GET_CP15_TTB0(ttbr, ttbrHigh);

        return (ttb | (ttbr & ~HW_C2_0_T1_BASE_MASK_MAX));
    }

    inline Bit32 Rightmost0(Bit32 x) { return ~x & (x + 1); }

    class AsidManager
    {
    public:
        static const uint8_t NULL_ASID = 0;

    private:
        static const int NUM_IDS    = 256;
        static const int NUM_WORDS  = NUM_IDS / 32;

    private:
        Bit32       m_AsidMap[NUM_WORDS];
        KLightMutex m_Lock;
        uint8_t          m_Index;

    public:
        AsidManager()
        {
            std::memset(m_AsidMap, 0, sizeof(m_AsidMap));
            m_Index = 0;

            // NULL_ASID を予約しておく
            SetBit(m_AsidMap, NULL_ASID);
        }

        uint8_t Acquire()
        {
            Bit32 id;

            KScopedLightLock lock(&m_Lock);

            if( GetBit(m_AsidMap, m_Index) )
            {
                id = FindClear(m_AsidMap);
                m_Index = id;
            }
            else
            {
                id = m_Index;
            }

            SetBit(m_AsidMap, id);
            m_Index++;              // 8 bit で回り込む

            return static_cast<uint8_t>(id);
        }

        void Release(uint8_t id)
        {
            NN_KERN_ASSERT( GetBit(m_AsidMap, id) );
            KScopedLightLock lock(&m_Lock);
            ClearBit(m_AsidMap, id);
        }

    private:
        static void SetBit(Bit32* pBits, uint32_t pos)
        {
            const int wordIndex = pos / 32;
            const int bitIndex  = pos % 32;
            pBits[wordIndex] |= (1u << bitIndex);
        }
        static void ClearBit(Bit32* pBits, uint32_t pos)
        {
            const int wordIndex = pos / 32;
            const int bitIndex  = pos % 32;
            pBits[wordIndex] &= ~(1u << bitIndex);
        }
        static bool GetBit(const Bit32* pBits, uint32_t pos)
        {
            const int wordIndex = pos / 32;
            const int bitIndex  = pos % 32;
            return (pBits[wordIndex] >> bitIndex) & 0x1;
        }
        static uint8_t FindClear(const Bit32* pBits)
        {
            for( int i = 0; i < NUM_WORDS; ++i )
            {
                uint8_t x = __builtin_clz(Rightmost0(pBits[i]));
                if( x < 32 )
                {
                    return 32 * i + (31 - x);
                }
            }

            NN_KERNEL_PANIC("too many asid");
            return 0;
        }
    };

    AsidManager s_AsidManager;

    class CoarsePageTable: public nn::util::IntrusiveListBaseNode<CoarsePageTable>
    {
    };
    class CoarsePageTableManager
    {
    private:
        // L2 PageTableに割り当てるサイズ。
        //  初期化時に決定し、拡張しない
        static const size_t CoarsePageTableSize  = HW_MMU7_T1_CORS_SIZE;

        nn::util::IntrusiveList<CoarsePageTable, nn::util::IntrusiveListBaseNodeTraits<CoarsePageTable>> m_FreeList;
        uint16_t*       m_pRefCount;
        KLightMutex     m_Lock;
        KVirtualAddress m_PageTableBegin;
        size_t          m_Used;
        size_t          m_Peak;

    public:
        size_t GetUsed() const { return m_Used; }
        size_t GetPeak() const { return m_Peak; }
        Result Initialize()
        {
            KVirtualAddress va;

            m_PageTableBegin = Kernel::GetPageTableManager().GetBase();
            size_t pageTableNum = Kernel::GetPageTableManager().GetSize() / CoarsePageTableSize;
            const size_t refCountNumPages = RoundUp(pageTableNum * sizeof(uint16_t), NN_KERN_FINEST_PAGE_SIZE) / NN_KERN_FINEST_PAGE_SIZE;

            NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, refCountNumPages * NN_KERN_FINEST_PAGE_SIZE));
            va = Kernel::GetKernelHeapManager().AllocateContinuous(refCountNumPages, 0, KMemoryManager::MakeAllocateOption(KMemoryManager::Region_SecureSystem, KMemoryManager::From_Front));
            NN_KERN_ABORT_UNLESS(va != Null<KVirtualAddress>());
            Kernel::GetKernelHeapManager().Open(va, refCountNumPages);
            m_pRefCount = GetTypedPointer<uint16_t>(va);

            for (size_t i = 0; i < pageTableNum; i++)
            {
                m_pRefCount[i] = 0;
            }

            return ResultSuccess();
        }

        KVirtualAddress Allocate()
        {
            KVirtualAddress  addr;

            KScopedLightLock lock(&m_Lock);

            if (m_FreeList.empty())
            {
                addr = Kernel::GetPageTableManager().Allocate();
                if (addr == Null<KVirtualAddress>())
                {
                    return addr;
                }
                for (size_t i = 1; i < Kernel::GetPageTableManager().PageTableSize / CoarsePageTableSize; i++)
                {
                    CoarsePageTable* pTable = new(GetUntypedPointer(addr + i * CoarsePageTableSize)) CoarsePageTable;
                    m_FreeList.push_front(*pTable);
                    NN_KERN_ASSERT(m_pRefCount[(addr + i * CoarsePageTableSize - m_PageTableBegin) / CoarsePageTableSize] == 0);
                }
                Kernel::GetPageTableManager().Open(addr, 1);
            }
            else
            {
                CoarsePageTable* pTable = &m_FreeList.front();
                m_FreeList.pop_front();
                addr = KVirtualAddress(pTable);
            }

            int32_t index = (addr - m_PageTableBegin) / CoarsePageTableSize;
            NN_KERN_ASSERT(m_pRefCount[index] == 0 );
            m_pRefCount[index]++;

            return addr;
        }

        bool IsPageTablePhysicalAddress(KPhysicalAddress pa)
        {
            return Kernel::GetPageTableManager().InPageTableHeap(KPageTable::GetPageTableVirtualAddress(pa));
        }
        void Open(KVirtualAddress addr)
        {
            KScopedLightLock lock(&m_Lock);

            NN_KERN_ASSERT((m_PageTableBegin <= addr) && (addr <= (m_PageTableBegin + Kernel::GetPageTableManager().GetSize() - 1)));

            int32_t index = (addr - m_PageTableBegin) / CoarsePageTableSize;
            NN_KERN_ASSERT(m_pRefCount[index] >= 0);
            m_pRefCount[index]++;
        }

        int32_t GetRefCount(KVirtualAddress addr)
        {
            NN_KERN_ASSERT((m_PageTableBegin <= addr) && (addr <= (m_PageTableBegin + Kernel::GetPageTableManager().GetSize() - 1)));
            size_t index = (addr - m_PageTableBegin) / CoarsePageTableSize;
            return m_pRefCount[index];
        }

        bool Close(KVirtualAddress addr)
        {
            KScopedLightLock lock(&m_Lock);

            NN_KERN_ASSERT((m_PageTableBegin <= addr) && (addr <= (m_PageTableBegin + Kernel::GetPageTableManager().GetSize() - 1)));

            size_t index = (addr - m_PageTableBegin) / CoarsePageTableSize;

            NN_KERN_ASSERT(m_pRefCount[index] > 0);

            m_pRefCount[index]--;

            if (m_pRefCount[index] == 0)
            {
                Free(addr);

                return true;
            }
            else
            {
                return false;
            }
        }
    private:
        void Free(KVirtualAddress addr)
        {
            NN_KERN_ASSERT((m_PageTableBegin <= addr) && (addr <= (m_PageTableBegin + Kernel::GetPageTableManager().GetSize() - 1)));

            size_t index = (addr - m_PageTableBegin) / CoarsePageTableSize;
            NN_UNUSED(index);

            NN_KERN_ASSERT( m_pRefCount[index] == 0 );
            CoarsePageTable* pCoarsePageTable = new(GetUntypedPointer(addr)) CoarsePageTable;
            m_FreeList.push_front(*pCoarsePageTable);

            size_t perPage = Kernel::GetPageTableManager().PageTableSize / CoarsePageTableSize;
            index = index / perPage * perPage;
            size_t i;
            for (i = 0; i < perPage; i++)
            {
                if (m_pRefCount[index + i])
                {
                    break;
                }
            }
            if (i == perPage)
            {
                addr = m_PageTableBegin + index * CoarsePageTableSize;
                for (i = 0; i < perPage; i++)
                {
                    CoarsePageTable* pTable = GetTypedPointer<CoarsePageTable>(addr + i * CoarsePageTableSize);
                    m_FreeList.erase(m_FreeList.iterator_to(*pTable));
                }
                Kernel::GetPageTableManager().Close(addr, 1);
                Kernel::GetPageTableManager().Free(addr);
            }
        }

    };

    CoarsePageTableManager s_CoarsePageTableManager;
}


/* =======================================================================
        初期化・終了処理
   ======================================================================== */

void
KPageTable::Initialize(int32_t coreNo)
{
    if (coreNo == 0)
    {
        Result result = s_CoarsePageTableManager.Initialize();
        if( result.IsFailure() )
        {
            NN_KERNEL_PANIC("no space for CoarsePageTableManager.");
        }
    }
}

Result
KPageTable::Initialize(Bit32 spaceId, nn::svc::CreateProcessParameterFlag addressSapceType, bool enableAslr, bool fromBack, KMemoryManager::Region region, KProcessAddress codeRegionAddr, size_t codeRegionSize, KMemoryBlockResourceManager* pMemoryBlockManager, KBlockInfoManager* pBlockInfoManager, KPageTableManager* pPageTableManager)
{
    m_Asid              = s_AsidManager.Acquire();

    // ページテーブル用メモリの確保、ポインタの設定
    KMemoryManager& mm = Kernel::GetKernelHeapManager();
    size_t numPages = (NN_KERN_MMU_L1_PAGE_TABLE_0_SIZE + PageSize - 1) / PageSize;
    NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, numPages * NN_KERN_FINEST_PAGE_SIZE));
    const KVirtualAddress tableVAddress = mm.AllocateContinuous(
        numPages,
        numPages,
        KMemoryManager::MakeAllocateOption(KMemoryManager::Region_SecureSystem, KMemoryManager::From_Front));
    Bit32* pTable = GetTypedPointer<Bit32>(tableVAddress);
    if(pTable == 0)
    {
        NN_WARNING(false, "Failed to allocate a process page table.");
        s_AsidManager.Release(m_Asid);
        return nn::svc::ResultOutOfMemory();
    }

    mm.Open(tableVAddress, (NN_KERN_MMU_L1_PAGE_TABLE_0_SIZE + PageSize - 1) / PageSize);
    m_TtbrValue = MakeTtbr0Value(GetPageTablePhysicalAddress(tableVAddress));

    Bit32 ttbcr;

    HW_GET_CP15_TTB_CONTROL(ttbcr);
    ttbcr &= ~HW_C2_2_PD0;
    m_TtbcrValue = ttbcr;

    Result result = KPageTableBase::InitializeForProcess(addressSapceType, enableAslr, fromBack, pTable, NN_KERN_V_ADDR_PROCESS, NN_KERN_V_ADDR_PROCESS_END, region, codeRegionAddr, codeRegionSize, pMemoryBlockManager, pBlockInfoManager);
    if (result.IsFailure())
    {
        mm.Close(tableVAddress, (NN_KERN_MMU_L1_PAGE_TABLE_0_SIZE + PageSize - 1) / PageSize);
        s_AsidManager.Release(m_Asid);
        return result;
    }

    MarkUpdated();

    return ResultSuccess();
}

Result KPageTable::InitializeForKernel(void* pTable, KProcessAddress begin, KProcessAddress end)
{
    m_Asid              = 0;
    m_TtbrValue         = 0;
    Bit32 ttbcr;

    HW_GET_CP15_TTB_CONTROL(ttbcr);
    ttbcr |= HW_C2_2_PD0;
    m_TtbcrValue = ttbcr;

    Result result = KPageTableBase::InitializeForKernel(false, pTable, begin, end);
    if (result.IsFailure())
    {
        NN_KERNEL_PANIC("failed to initialize page table");
    }

    return ResultSuccess();
}

Result KPageTable::Finalize()
{
    NN_KERN_ASSERT( ! IsKernel() );
    KPageTableBody& table = GetPageTable();

    // 全コアで ASID を無効化
    MarkUpdated();

    {
        KMemoryManager& mm = Kernel::GetKernelHeapManager();

        // 全てのページを解放
        {
            TraverseContext     context;
            TraverseData        current;
            bool                currentIsValid;
            TraverseData        next;
            bool                nextIsValid;
            const size_t        spaceSize = GetAddressEnd() - GetAddressBegin();
            size_t              sizeSum  = 0;

            current.address = 0;
            current.size    = 0;
            currentIsValid  = false;

            nextIsValid = table.TraverseBegin(&next, &context, GetAddressBegin());

            // ページを解放
            for(;;)
            {
                if( (nextIsValid && currentIsValid && next.address == current.address + current.size)
                 || ( !currentIsValid && !nextIsValid ) )
                {
                    current.size += next.size;
                }
                else
                {
                    if (currentIsValid)
                    {
                        if (IsHeapPhysicalAddress(current.address))
                        {
                            const KVirtualAddress  va = GetHeapVirtualAddress(current.address);
                            mm.Close( va, current.size / PageSize);
                        }
                        else if (KMemoryLayout::InSlabRegionPhysical(current.address))
                        {
                            // KPageBuffer が プロセスにマップされるとき、エイリアス不可な領域(現在、ThreadLocalのみ)
                            // MemoryBlock枯渇を救済する
                            KVirtualAddress va = GetLinearMapVirtualAddress(current.address);
                            KVirtualAddress slabBase = KPageBuffer::GetSlabAddr();
                            KVirtualAddress slabEnd = slabBase + KPageBuffer::GetSlabSize() * sizeof(KPageBuffer);
                            if ((slabBase <= va) && ((va + current.size - 1) <= (slabEnd - 1)))
                            {
                                for (size_t i = 0; i < current.size / sizeof(KPageBuffer); i++)
                                {
                                    KPageBuffer::Free(GetTypedPointer<KPageBuffer>(va + i * sizeof(KPageBuffer)));
                                }
                            }
                        }
                    }

                    sizeSum += current.size;
                    current = next;
                    currentIsValid = nextIsValid;
                }

                if( sizeSum + current.size >= spaceSize )
                {
                    break;
                }

                nextIsValid = table.TraverseNext(&next, &context);
            }

            if (currentIsValid)
            {
                if (IsHeapPhysicalAddress(current.address))
                {
                    const KVirtualAddress  va = GetHeapVirtualAddress(current.address);
                    mm.Close( va, current.size / PageSize);
                }
                else if (KMemoryLayout::InSlabRegionPhysical(current.address))
                {
                    // KPageBuffer が プロセスにマップされるとき、エイリアス不可な領域(現在、ThreadLocalのみ)
                    // MemoryBlock枯渇を救済する
                    KVirtualAddress va = GetLinearMapVirtualAddress(current.address);
                    KVirtualAddress slabBase = KPageBuffer::GetSlabAddr();
                    KVirtualAddress slabEnd = slabBase + KPageBuffer::GetSlabSize() * sizeof(KPageBuffer);
                    if ((slabBase <= va) && ((va + current.size - 1) <= (slabEnd - 1)))
                    {
                        for (size_t i = 0; i < current.size / sizeof(KPageBuffer); i++)
                        {
                            KPageBuffer::Free(GetTypedPointer<KPageBuffer>(va + i * sizeof(KPageBuffer)));
                        }
                    }
                }
            }
        }

        // 全ての L2 ページテーブルを解放
        {
            static const size_t SECTION_SIZE = 1024 * 1024;
            const size_t numSections = (GetAddressEnd() - GetAddressBegin()) / SECTION_SIZE;
            uintptr_t vAddr = GetAsInteger(GetAddressBegin());

            for( size_t i = 0; i < numSections; ++i )
            {
                if( table.IsCoarse(vAddr) )
                {
                    const Bit32 masterEntry = table.GetMasterEntry(vAddr);
                    const KPhysicalAddress coarsePAddress = table.GetCoarseTableAddressFromMasterEntry(masterEntry);
                    const KVirtualAddress  coarseVAddress = GetPageTableVirtualAddress(coarsePAddress);

                    if (Kernel::GetPageTableManager().InPageTableHeap(coarseVAddress))
                    {
                        while( ! s_CoarsePageTableManager.Close(coarseVAddress) ) {}
                    }
                }

                vAddr += SECTION_SIZE;
            }
        }

        // L1 ページテーブルに割り当てたメモリを解放
        {
            Bit32* pTable = table.Finalize();
            mm.Close(KVirtualAddress(pTable), (NN_KERN_MMU_L1_PAGE_TABLE_0_SIZE + PageSize - 1) / PageSize);
            Kernel::GetSystemResourceLimit().ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, (NN_KERN_MMU_L1_PAGE_TABLE_0_SIZE + PageSize - 1) / PageSize * PageSize);
        }

        KPageTableBase::Finalize();
    }

    // ASID を返却
    s_AsidManager.Release(m_Asid);

    return ResultSuccess();
}   // NOLINT(readability/fn_size)

bool KPageTable::GetStaticMap(KProcessAddress* pOut, KPhysicalAddress paddr, size_t size)
{
    return false;
}

void KPageTable::FreePageTable(void** ppData, KVirtualAddress pTable)
{
    NN_KERN_ASSERT(Kernel::GetPageTableManager().InPageTableHeap(pTable));
    NN_KERN_ASSERT(s_CoarsePageTableManager.GetRefCount(pTable) == 1);
    void* pNext = *ppData;
    *GetTypedPointer<void*>(pTable) = pNext;
    *ppData = GetUntypedPointer(pTable);
}

void KPageTable::PageTableUpdaterFinalize(void** ppData)
{
    while (*ppData)
    {
        KVirtualAddress pTable = KVirtualAddress(*ppData);
        *ppData = *static_cast<void**>(*ppData);
        NN_KERN_ASSERT(Kernel::GetPageTableManager().InPageTableHeap(pTable));
        NN_KERN_ASSERT(s_CoarsePageTableManager.GetRefCount(pTable) == 1);
        s_CoarsePageTableManager.Close(pTable);
    }
}

Result
KPageTable::OperateImpl(void** ppData,
        KProcessAddress vaddr, size_t numPages, const KPageGroup& pg,
        const KPageProperty property, PageTableOperation operation)
{
    Result result;
    NN_KERN_ASSERT(IsLockedByMe());
    NN_KERNEL_PAGE_ALIGN_ASSERT(GetAsInteger(vaddr));
    NN_KERN_ASSERT(numPages != 0);
    NN_KERN_ASSERT(numPages == pg.GetTotalNumPages());

    Bit32 entryTemplate = GetEntryTemplate(property);
    switch (operation)
    {
    case PageTableOperation_MapRelocate:
        {
            result = MapWithTemplate(vaddr, pg, numPages, entryTemplate, ppData);
            if (result.IsFailure())
            {
                NN_WARNING(false, "failed to map pages.");
                return result;
            }
        }
        break;

    default:
        {
            NN_KERN_ABORT();
        }
        break;
    }

    return ResultSuccess();
}

Result
KPageTable::OperateImpl(void** ppData,
        KProcessAddress vaddr, size_t numPages, KPhysicalAddress paddr, bool paIsValid,
        const KPageProperty property, PageTableOperation operation)
{
    Result result;
    NN_KERN_ASSERT(IsLockedByMe());
    NN_KERNEL_PAGE_ALIGN_ASSERT(GetAsInteger(vaddr));
    NN_KERN_ASSERT(numPages != 0);
    NN_KERN_ASSERT(IsPagesInRange(vaddr, numPages));

    if (operation == PageTableOperation_Map)
    {
        NN_KERN_ASSERT(paIsValid);
        NN_KERNEL_PAGE_ALIGN_ASSERT(GetAsInteger(paddr));
    }
    else
    {
        NN_KERN_ASSERT(!paIsValid);
    }

    if (operation != PageTableOperation_Unmap)
    {
        Bit32 entryTemplate = GetEntryTemplate(property);

        switch (operation)
        {
        case PageTableOperation_AllocateAndMap:
            {
                KScopedPageGroup pg(GetBlockInfoManager());
                KMemoryManager& mm = Kernel::GetKernelHeapManager();
                result = mm.Allocate(&pg.GetPageGroup(), numPages, GetAllocateOption());
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to allocate memory, numPages = %d", numPages);
                    return result;
                }

                for (KPageGroup::BlockInfoList::const_iterator it = pg.GetPageGroup().GetBlockBeginIter(); it != pg.GetPageGroup().GetBlockEndIter(); it++)
                {
                    ::std::memset(GetUntypedPointer(it->GetBlockAddr()), m_AllocateFillMemoryPattern, it->GetNumPages() * PageSize);
                }

                result = MapWithTemplate(vaddr, pg.GetPageGroup(), numPages, entryTemplate, ppData);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to map pages.");
                    return result;
                }
            }
            break;
        case PageTableOperation_Map:
            {
                result = MapContinuousWithTemplate(vaddr, paddr, numPages, entryTemplate, ppData);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to map pages.");
                    return result;
                }
            }
            break;
        case PageTableOperation_ChangeProperties:
            {
                result = UpdateProperties(vaddr, numPages, entryTemplate, false);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to change permission");
                    return result;
                }
            }
            break;
        case PageTableOperation_ChangePropertiesWithBreakBeforeMake:
            {
                result = UpdateProperties(vaddr, numPages, entryTemplate, true);
                if (result.IsFailure())
                {
                    NN_WARNING(false, "failed to change page attribute");
                    return result;
                }
            }
            break;
        default:
            {
                NN_KERN_ABORT();
            }
            break;
        }
    }
    else
    {
        // Unmap
        result = FreePages(vaddr, numPages, ppData);
        if (result.IsFailure())
        {
            NN_WARNING(false, "failed to free pages");
            return result;
        }
    }

    return ResultSuccess();
}

/* =======================================================================
        インスタンスメンバ / private
   ======================================================================== */

Result KPageTable::MapWithTemplate(
        KProcessAddress vAddr, const KPageGroup& pg,
        size_t numPages, Bit32 entryTemplate, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    const KProcessAddress cVirtualAddress = vAddr;
    Result result;
    size_t totalPages = 0;

    pg.Open();

    for (KPageGroup::BlockInfoList::const_iterator it = pg.GetBlockBeginIter(); it != pg.GetBlockEndIter(); it++)
    {
        size_t mapNumPages = it->GetNumPages();
        const KPhysicalAddress pMapAddr = GetHeapPhysicalAddress(it->GetBlockAddr());
        result = MapSmallPagesWithTemplate(vAddr, pMapAddr, mapNumPages, entryTemplate);
        if (result.IsFailure())
        {
            NN_KERN_ABORT_IF_FAILED(Unmap(cVirtualAddress, totalPages + mapNumPages, nullptr, ppData));
            pg.Close();
            return result;
        }

        vAddr += mapNumPages * PageSize;
        totalPages += mapNumPages;
    }

    NN_KERN_ASSERT(totalPages == numPages);

    return ResultSuccess();
}

Result KPageTable::MapContinuousWithTemplate(
        KProcessAddress vAddr, KPhysicalAddress pAddr,
        size_t numPages, Bit32 entryTemplate, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    const KProcessAddress cVirtualAddress = vAddr;
    const KPhysicalAddress cPhysicalAddr  = pAddr;
    Result result;

    result = MapSmallPagesWithTemplate(vAddr, pAddr, numPages, entryTemplate);
    if (result.IsFailure())
    {
        NN_KERN_ABORT_IF_FAILED(Unmap(cVirtualAddress, numPages, nullptr, ppData));
        return result;
    }

    if (IsHeapPhysicalAddress(cPhysicalAddr))
    {
        Kernel::GetKernelHeapManager().Open(GetHeapVirtualAddress(cPhysicalAddr), numPages);
    }

    return ResultSuccess();
}


Result KPageTable::MapSmallPagesWithTemplate(KProcessAddress vAddr, KPhysicalAddress pAddr, size_t numPages, Bit32 entryTemplate)
{
    NN_KERN_ASSERT(IsLockedByMe());
    KPageTableBody& table = GetPageTable();

    for (size_t i = 0; i < numPages; i++)
    {
        bool allocated = false;
        if (!table.IsCoarse(vAddr))
        {
            // コーステーブルが存在しなかったらアロケート
            KVirtualAddress pCoarseTable = s_CoarsePageTableManager.Allocate();
            if (pCoarseTable == Null<KVirtualAddress>())
            {
                return nn::svc::ResultOutOfResource();
            }

            std::memset(GetUntypedPointer(pCoarseTable), 0, HW_MMU7_T1_CORS_SIZE);
            KCPU::DataSynchronizationBarrier();

            // コースエントリを作って、マスタテーブルに書き込み。ドメインはとりあえず 0。
            table.SetMasterEntry(vAddr, HW_MMU7_T1_COURSE_PACK(GetAsInteger(GetPageTablePhysicalAddress(pCoarseTable)), 0));
            KCPU::DataSynchronizationBarrier();

            // TODO: 後でアロケートしたコーステーブルを削除できるような仕組みが必要。
            // TODO: 上書きしたエントリがスーパーセクションだった場合は、他の関連エントリ
            //       （15個あるはず）も削除しなければならない。
            allocated = true;
        }

        const Bit32 masterEntry = table.GetMasterEntry(vAddr);
        const KPhysicalAddress coarsePAddress = table.GetCoarseTableAddressFromMasterEntry(masterEntry);
        const KVirtualAddress  coarseVAddress = GetPageTableVirtualAddress(coarsePAddress);
        Bit32* pCoarseTable = GetTypedPointer<Bit32>(coarseVAddress);

        // TODO: 同上の理由で、ラージテーブルはとりあえず扱わない。
        const Bit32 smallPageEntry = (pAddr & HW_MMU7_T2_SP_BASE_MASK) | entryTemplate;
        table.SetCoarseEntryToCoarseTable(vAddr, pCoarseTable, smallPageEntry);
        if (!allocated && s_CoarsePageTableManager.IsPageTablePhysicalAddress(coarsePAddress))
        {
            s_CoarsePageTableManager.Open(coarseVAddress);
        }

        vAddr += PageSize;
        pAddr += PageSize;
    }

    return ResultSuccess();
}

Result KPageTable::UpdateProperties(KProcessAddress vAddr, size_t numPages, Bit32 entryTemplate, bool breakBeforeMake)
{
    NN_KERN_ASSERT(IsLockedByMe());
    KPageTableBody& table = GetPageTable();
    for (size_t i = 0; i < numPages; i++)
    {
        NN_KERN_ABORT_UNLESS(table.IsCoarse(vAddr));

        const Bit32 masterEntry = table.GetMasterEntry(vAddr);
        const KPhysicalAddress coarsePAddress = table.GetCoarseTableAddressFromMasterEntry(masterEntry);
        const KVirtualAddress  coarseVAddress = GetPageTableVirtualAddress(coarsePAddress);
        Bit32* pCoarseTable = GetTypedPointer<Bit32>(coarseVAddress);

        const Bit32 oldEntry = table.GetCoarseEntryFromCoarseTable(vAddr, pCoarseTable);
        NN_KERN_ABORT_UNLESS((oldEntry & HW_MMU7_T2_SP_SB1) == HW_MMU7_T2_SP_SB1);
        const Bit32 newEntry = (oldEntry & HW_MMU7_T2_SP_BASE_MASK) | entryTemplate;
        if (breakBeforeMake)
        {
            table.SetCoarseEntryToCoarseTable(vAddr, pCoarseTable, UNMAPPED_ENTRY);
            MarkUpdated(vAddr);
            if (IsHeapPhysicalAddress(oldEntry & HW_MMU7_T2_SP_BASE_MASK))
            {
                KCPU::FlushDataCache(GetUntypedPointer(GetHeapVirtualAddress((oldEntry & HW_MMU7_T2_SP_BASE_MASK))), PageSize);
            }
        }
        table.SetCoarseEntryToCoarseTable(vAddr, pCoarseTable, newEntry);

        vAddr += PageSize;
    }

    MarkUpdated();

    return ResultSuccess();
}

Result KPageTable::Unmap(KProcessAddress addr, size_t numPages, KPageGroup* pPgToBeClose, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());
    KPageTableBody& table = GetPageTable();
    uintptr_t vAddr = GetAsInteger(addr);
    const KProcessAddress cVirtualAddress = vAddr;

    for (size_t i = 0; i < numPages; i++)
    {
        if (!table.IsCoarse(vAddr))
        {
            uintptr_t nextAddr = ((vAddr + (0x100000 - 1)) & ~(0x100000 - 1));
            i += (nextAddr - vAddr) / PageSize;
            vAddr = nextAddr;
            continue;
        }

        const Bit32 masterEntry = table.GetMasterEntry(vAddr);
        const KPhysicalAddress coarsePAddress = table.GetCoarseTableAddressFromMasterEntry(masterEntry);
        const KVirtualAddress  coarseVAddress = GetPageTableVirtualAddress(coarsePAddress);
        Bit32* pCoarseTable = GetTypedPointer<Bit32>(coarseVAddress);

        const KPhysicalAddress pa = table.SetCoarseEntryToCoarseTable(vAddr, pCoarseTable, UNMAPPED_ENTRY);

        if (Kernel::GetPageTableManager().InPageTableHeap(coarseVAddress))
        {
            if (s_CoarsePageTableManager.GetRefCount(coarseVAddress) == 1)
            {
                table.SetMasterEntry(vAddr, UNMAPPED_ENTRY);
                MarkUpdated();
                FreePageTable(ppData, coarseVAddress);
            }
            else
            {
                s_CoarsePageTableManager.Close(coarseVAddress);
            }
        }
        if (pPgToBeClose && IsHeapPhysicalAddress(pa))
        {
            if (pPgToBeClose->AddBlock(GetHeapVirtualAddress(pa), 1).IsFailure())
            {
                MarkUpdated();
                Kernel::GetKernelHeapManager().Close(GetHeapVirtualAddress(pa), 1);
            }
        }

        vAddr += PageSize;
    }

    if (IsKernel() && (numPages == 1))
    {
        // スーパバイザスタック解放用最適化
        MarkUpdatedForSvStack(cVirtualAddress);
    }
    else
    {
        MarkUpdated();
    }
    return ResultSuccess();
}

Result KPageTable::FreePages(KProcessAddress vAddr, size_t numPages, void** ppData)
{
    NN_KERN_ASSERT(IsLockedByMe());

    KScopedPageGroup pg(GetBlockInfoManager());

    Result result = Unmap(vAddr, numPages, &pg.GetPageGroup(), ppData);
    if (result.IsFailure())
    {
        return result;
    }

    pg.GetPageGroup().Close();

    return ResultSuccess();
}

void KPageTable::MarkUpdatedForSvStack(KProcessAddress va)
{
    KCPU::DataSynchronizationBarrier();

    OnKernelTableUpdatedForSvStack(va);
}

void KPageTable::MarkUpdated()
{
    KCPU::DataSynchronizationBarrier();

    if (IsKernel())
    {
        OnKernelTableUpdated();
    }
    else
    {
        OnTableUpdated();
    }
}

void KPageTable::MarkUpdated(KProcessAddress va)
{
    KCPU::DataSynchronizationBarrier();

    if (IsKernel())
    {
        OnKernelTableUpdatedForSvStack(va);
    }
    else
    {
        OnTableUpdated(va);
    }
}

/* =======================================================================
        クラスメンバ / public
   ======================================================================== */

void KPageTable::Activate(KPageTable* pTable)
{
    // CP15 操作に挟まれた箇所では実行順序の最適化をしてくれないので
    // ここでロードしてよいことを明示する
    const Bit32 asid      = pTable->m_Asid;
    const Bit32 ttbrValue = pTable->m_TtbrValue;
    const Bit32 ttbcrValue = pTable->m_TtbcrValue;

    {
        // 割り込みを禁止することで ASID と TTBR の同期の問題を回避する
        KDisableInterrupt di;

        HW_SET_CP15_TTB_CONTROL(ttbcrValue | HW_C2_2_PD0);
        KCPU::DataSynchronizationBarrier();
        KCPU::SwitchProcess(asid, ttbrValue);
        if ((ttbcrValue | HW_C2_2_PD0) != ttbcrValue)
        {
            HW_SET_CP15_TTB_CONTROL(ttbcrValue);
            KCPU::DataSynchronizationBarrier();
        }
    }
}

void KPageTable::Info()
{
    NN_KERN_RELEASE_LOG("Page Table usage\n");
    NN_KERN_RELEASE_LOG("    used / peak / num\n");
    NN_KERN_RELEASE_LOG("    %ld / %ld / %ld\n", Kernel::GetPageTableManager().GetUsed(), Kernel::GetPageTableManager().GetPeak(), Kernel::GetPageTableManager().GetNum());
}

}
}}
