﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Result.h"
#include "../../kern_CPUSelect.h"
#include "../../kern_KInterruptManagerBase.h"
#include "../../kern_KSimpleLock.h"
#include "../../kern_InterruptControllerSelect.h"

namespace nn { namespace kern {
    namespace ARMv7A {

extern const Bit32 EXCEPTION_VECTOR[8];
extern "C" void ExceptionVector();

class KInterruptManager : public KInterruptManagerBase
{
private:
    struct CoreLocalInterruptTableEntry
    {
        KInterruptHandler*  pHandler;
        bool                isManualClear;
        bool                isWaitingClear;
        uint8_t             priority;
    };
    struct GlobalInterruptTableEntry
    {
        KInterruptHandler*  pHandler;
        bool                isManualClear;
        bool                isWaitingClear;
    };

private:
    CoreLocalInterruptTableEntry        m_CoreLocalInterruptTable[KInterruptController::NumLocalInterrupts];
    KInterruptController                m_InterruptController;
    KInterruptController::LocalState    m_LocalState;
    bool                                m_LocalSaved;

    static KSimpleLock                          s_Lock;
    static GlobalInterruptTableEntry            s_GlobalInterruptTable[KInterruptController::NumGlobalInterrupts];
    static KInterruptController::GlobalState    s_GlobalState;
    static bool                                 s_GlobalSaved;
public:
    KInterruptManager(): m_LocalSaved(false) {}
    void    Initialize(int coreNo);
    void    Finalize(int coreNo);

    Result  ClearInterrupt(int32_t irqNo, int32_t coreNo);
    Result  ClearInterrupt(int32_t irqNo);

    Result  BindHandler   (KInterruptHandler* pHandler, int32_t irqNo, int32_t coreNo, int32_t priority, bool isManualClear, bool isLevel);
    Result  UnbindHandler (int32_t irqNo, int32_t coreNo);

    static void InterruptRequestHandler(bool fromUserMode);

    void Save(int coreNo);
    void Restore(int coreNo);
    bool IsDefinedInterrupt(int irqNo) const
    {
        return m_InterruptController.IsDefinedInterrupt(irqNo);
    }
    void SendIpi(Bit64 cpuSet, int id) const
    {
        m_InterruptController.SendIpi(cpuSet, id);
    }
    void SendIpi(int id) const
    {
        m_InterruptController.SendIpi(id);
    }
    bool IsGlobal(int irqNo) const
    {
        return m_InterruptController.IsGlobal(irqNo);
    }
    bool IsLocal(int irqNo) const
    {
        return m_InterruptController.IsLocal(irqNo);
    }

    static Bit32  EnableInterrupt()
    {
        Bit32 psr;
        asm volatile("mrs     %0, cpsr":"=r"(psr)::);
        asm volatile("cpsie i");
        return psr;
    }
    static Bit32 DisableInterrupt()
    {
        Bit32 psr;
        asm volatile("mrs     %0, cpsr":"=r"(psr)::);
        asm volatile("cpsid i");
        return psr;
    }
    static void RestoreInterrupt(Bit32 psr)
    {
        asm volatile("msr     cpsr_c, %0"::"r"(psr):);
    }
    static bool IsInterruptEnabled()
    {
        Bit32 psr;
        asm volatile("mrs     %0, cpsr":"=r"(psr)::);
        return (psr & HW_PSR_IRQ_DISABLE) == 0;
    }

private:
    bool    OnInterruptRequest();
    Result  BindGlobal  (KInterruptHandler* pHandler, int32_t irqNo, int32_t coreNo, int32_t priority, bool isManualClear, bool isLevel);
    Result  BindLocal   (KInterruptHandler* pHandler, int32_t irqNo, int32_t priority, bool isManualClear);
    Result  UnbindGlobal(int32_t irqNo);
    Result  UnbindLocal (int32_t irqNo);
    Result  ClearGlobal (int32_t irqNo);
    Result  ClearLocal  (int32_t irqNo);

    KInterruptHandler*  ForceUnbindGlobal(int32_t irqNo);
};

    }
}}

