﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>

#include "../../kern_Platform.h"
#include "kern_ExceptionHandlerPrint.h"
#include "../../kern_DebugString.h"
#include "../../kern_KProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_InterruptManagerSelect.h"

namespace nn { namespace kern { namespace ARMv7A {

#ifdef NN_KERN_FOR_DEVELOPMENT
    namespace
    {
        int s_ExceptionCount = 0;

        void PrintException(ExceptionType type)
        {
            switch(type)
            {
            case EXCEPTION_TYPE_DABT:
                NN_KERN_RELEASE_LOG("========== Data Abort Exception ==========\n");
                break;

            case EXCEPTION_TYPE_PABT:
                NN_KERN_RELEASE_LOG("========== Prefetch Abort Exception ==========\n");
                break;

            case EXCEPTION_TYPE_UNDEF:
                NN_KERN_RELEASE_LOG("========== Undefined Instruction Exception ==========\n");
                break;

            case EXCEPTION_TYPE_INVALID_SVC:
                NN_KERN_RELEASE_LOG("========== Invalid SVC Call Exception ==========\n");
                break;

            default:
                NN_KERNEL_PANIC("unknown exception type(=%d).", type);
            }
        }

        void PrintAbortStatus( uintptr_t fa, Bit32 fs )
        {
            NN_KERN_RELEASE_LOG("-- abort status ----\n");
            NN_KERN_RELEASE_LOG("Fault Address         = 0x%08X\n", fa);
            NN_KERN_RELEASE_LOG("Fault Status Register = 0x%08X\n", fs);

            const char* fmt = "  %-30s";

            NN_KERN_RELEASE_LOG(fmt, "domain:");
            NN_KERN_RELEASE_LOG("%d\n", ((fs >> 4) & 0xF));

            NN_KERN_RELEASE_LOG(fmt, "external abort:");
            if( (fs & (1 << 12)) != 0 )
            {
                NN_KERN_RELEASE_LOG("slverr\n");
            }
            else
            {
                NN_KERN_RELEASE_LOG("decerr\n");
            }

            const Bit32 status = ( (((fs >> 10) & 0x1) << 4) | (fs & 0xF) );

            static const char* STATUS_MSGS[] =
            {
                "00000",
                "alignment fault",
                "debug event",
                "access section fault",
                "IC management error",
                "TLB section",
                "access page fault",
                "TLB page",
                "correct external abort",
                "domain section fault",
                "01010",
                "domain page fault",
                "external abort on TLB L1",
                "access section forbidden",
                "external abort on TLB L2",
                "access page forbidden",
                "10000",
                "10001",
                "10010",
                "10011",
                "10100",
                "10101",
                "incorrect external abort",
                "10111",
                "11000",
                "11001",
                "11010",
                "11011",
                "11100",
                "11101",
                "11110",
                "11111",
            };

            NN_KERN_RELEASE_LOG(fmt, "status:");
            NN_KERN_RELEASE_LOG("%s\n", STATUS_MSGS[status]);
            NN_KERN_RELEASE_LOG("\n");
        }

        void PrintGeneralRegisterInfo( const ARM::ExceptionContext& context )
        {
            int i = 0;
            for( ; i < 5; ++i )
            {
                NN_KERN_RELEASE_LOG("r%-2d=0x%08x ", i, context.r[i]);
            }
            NN_KERN_RELEASE_LOG("\n");
            for( ; i < 10; ++i )
            {
                NN_KERN_RELEASE_LOG("r%-2d=0x%08x ", i, context.r[i]);
            }
            NN_KERN_RELEASE_LOG("\n");
            for( ; i < 12; ++i )
            {
                NN_KERN_RELEASE_LOG("r%-2d=0x%08x ", i, context.r[i]);
            }
            NN_KERN_RELEASE_LOG("r%-2d=0x%08x ", 12, context.ip);
        }

        void PrintAbortInfo( const ARM::ExceptionContext& context, ExceptionType type, Bit32 fs, uintptr_t fa )
        {
            NN_KERN_RELEASE_LOG("-- registers ----\n");

            PrintGeneralRegisterInfo(context);

            NN_KERN_RELEASE_LOG("pc=0x%08x\n", context.pc);
            NN_KERN_RELEASE_LOG("sp_usr=0x%08x lr_usr=0x%08x cpsr=0x%08x\n", context.sp_usr, context.lr_usr, context.cpsr);
            NN_KERN_RELEASE_LOG("tpidr=0x%08x\n", context.tpidr);
            NN_KERN_RELEASE_LOG("\n");

            if (type == EXCEPTION_TYPE_DABT || type == EXCEPTION_TYPE_PABT)
            {
                PrintAbortStatus( fa, fs );
            }

            NN_KERN_RELEASE_LOG("-- addresses ----\n");
            const bool inUserMode = ((context.cpsr & HW_PSR_CPU_MODE_MASK) == HW_PSR_USR_MODE);
            NN_KERN_RELEASE_LOG("mode                          = %s\n", inUserMode ? "user": "privilege");
            NN_KERN_RELEASE_LOG("Abort Instruction Address     = 0x%08x\n", context.pc);
            NN_KERN_RELEASE_LOG("USR mode Latest Function Call = 0x%08x\n", context.lr_usr - 4);
            NN_KERN_RELEASE_LOG("\n");
        }

        void PrintCurrent()
        {
            KDisableInterrupt di;

            NN_KERN_RELEASE_LOG("-- current ----\n");
            KThread& currentThread       = GetCurrentThread();
            KProcess* pCurrentProcess    = GetCurrentProcessPointer();

            NN_KERN_RELEASE_LOG("cur thread               = %p (id = %lld)\n", &currentThread, currentThread.GetId());
            if( pCurrentProcess != NULL )
            {
                NN_KERN_RELEASE_LOG("cur process              = %p (id = %lld)\n", pCurrentProcess, pCurrentProcess->GetId());
            }
            else
            {
                NN_KERN_RELEASE_LOG("cur process              = NULL (id = 0)\n");
            }
            if( currentThread.GetParentPointer() != NULL )
            {
                NN_KERN_RELEASE_LOG("process of cur thread    = %p (id = %lld)\n", currentThread.GetParentPointer(), currentThread.GetParentPointer()->GetId());
            }
            else
            {
                NN_KERN_RELEASE_LOG("process of cur thread    = kernel\n");
            }
            if( pCurrentProcess != NULL )
            {
                NN_KERN_RELEASE_LOG("pagetable of cur process = %p\n", &pCurrentProcess->GetPageTable());
                pCurrentProcess->GetPageTable().GetPageTable().DumpMemoryBlocksLocked();
            }

            NN_KERN_RELEASE_LOG("\n");
        }
    }

    void CommonExceptionHandlerDebug(const ARM::ExceptionContext* pContext, ExceptionType type, Bit32 fs, Bit32 fa)
    {
        PrintException(type);

        const bool inUserMode = ((pContext->cpsr & HW_PSR_CPU_MODE_MASK) == HW_PSR_USR_MODE);
        if (!inUserMode)
        {
            s_ExceptionCount++;
            if( s_ExceptionCount >= 2 )
            {
                NN_KERNEL_PANIC("exception handler not implemented yet.");
            }
        }

        PrintAbortInfo(*pContext, type, fs, fa);
        PrintCurrent();

        //NN_KERNEL_PANIC("stop.");
    }

#endif

}}}


